/* dhcdbd.c
 *
 * Dynamic Host Configuration Client manager D-BUS Daemon  
 *
 * Copyright © 2006 Red Hat, Inc.  All rights reserved.
 *
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions of
 * the GNU General Public License v.2.  This program is distributed in the
 * hope that it will be useful, but WITHOUT ANY WARRANTY expressed or
 * implied, including the implied warranties of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.  You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301,
 * USA. Any Red Hat trademarks that are incorporated in the source code or
 * documentation are not subject to the GNU General Public License and may
 * only be used or replicated with the express permission of Red Hat, Inc.
 *
 * Red Hat Author(s):  Jason Vas Dias
 *                     David Cantrell
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <fcntl.h>
#include <ifaddrs.h>
#include <search.h>
#include <signal.h>
#include <syslog.h>
#include <time.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/wait.h>

#include "dbus_service.h"
#include "dhcdbd.h"
#include "dhcp_options.h"

typedef
        enum dhc_ifs_e {
        DHC_IF_INACTIVE,
        DHC_IF_UP_REQUEST,
        DHC_IF_UP_START,
        DHC_IF_UP,
        DHC_IF_DOWN_REQUEST,
        DHC_IF_DOWN_START,
        DHC_IF_DOWN,
        DHC_IF_RESTART,
        DHC_IF_RESTARTING,
        DHC_IF_RELEASED,
        DHC_IF_RELEASER
} DHC_IF_State;

typedef struct dhcdbd_olm_s {
        dbus_svc_MessageHandle msg;
        uint8_t ok;
} DHCDBD_Option_List_Message;

typedef
        struct dhcdbd_if_s {
        DBUS_SVC dbus;
        DHCO dhco;
        DHCP_Option_Type_Mapping mapping;
        char *name;
        uint32_t state;
        pid_t dhc_pid;
        uint32_t dhc_state;
        uint32_t dhc_exit_status;
        uint32_t dhc_si_code;
        uint32_t dhc_mode;
        uint32_t dhc_flags;
        struct timeval tv;                      /* time of last state change */
        uint32_t timeout;
        void *options;
        void *old_options;
        void *subscriptions;
        void *dead_subscribers;
        void *live_subscribers;
        struct dhcdbd_if_s *releasing;
        DHCDBD_Option_List_Message *list_msg;
} DHC_IF;

typedef struct dhcdbd_o_s {
        char *qname;
        DHC_IF *d_if;
        DHC_Option *option;
        char *output;
        char *value;
        uint32_t length;
} DHCDBD_Option;

typedef enum dhcdbd_sub_e {
        DHCDBD_SUB_BINARY,
        DHCDBD_SUB_DBUS,
        DHCDBD_SUB_TEXT
} DHCDBD_Subscription_Type;

typedef struct dhcdbd_subs_s {
        DHC_Option *opt;
        void *subscribers;
        DHCDBD_Option *dho;
} DHCDBD_Subscription;

typedef struct dhcdbd_subr_s {
        DHCDBD_Subscription *subs;
        uint32_t type;
        char *subscriber;
} DHCDBD_Subscriber;

static void *if_tree = 0L;
static void *dhc_tree = 0L;
static void *dead_dhc_tree = 0L;
static void *restart_dhc_tree = 0L;
static void *restarted_dhc_tree = 0L;
static void *timeout_dhc_tree = 0L;
static void *prospective_subscriptions = 0L;
static void *empty_prospective_subscriptions = 0L;
static void **subscription_copy = 0L;
static DHCDBD_Subscription *subscribers_copy = 0L;
static DHCDBD_Subscriber *subscriber_copy = 0L;

static char dhclient_bin[] = DHCDBD_DHCLIENT_BIN;

static uint8_t dhc_type_mapping[DHC_N_ATOMIC_TYPES] = {
        /* DHC_M_NONE */ '\0',
        /* DHC_M_IP_ADDRESS */ 'u',
        /* DHC_M_HEX_STRING */ 'S',
        /* DHC_M_TEXT */ 's',
        /* DHC_M_DOMAIN_NAME */ 's',
        /* DHC_M_INT32 */ 'u',
        /* DHC_M_UINT32 */ 'u',
        /* DHC_M_INT16 */ 'n',
        /* DHC_M_UINT16 */ 'q',
        /* DHC_M_CHAR */ 'y',
        /* DHC_M_UCHAR */ 'y',
        /* DHC_M_BOOL */ 'y',
        /* DHC_M_IMPLICIT_BOOL */ 'y',
        /* DHC_M_ENUMERATION */ 'u'
};

static dbus_svc_MessageHandle if_list_msg;

int
dhcdbd_log (const char *fmt, ...)
{
        va_list va;

        va_start (va, fmt);
        vsyslog (LOG_INFO, fmt, va);
        va_end (va);
        return 0;
}

int
dhcdbd_debug (const char *fmt, ...)
{
/*
    va_list va;
    va_start(va, fmt);
    vsyslog(LOG_INFO, fmt, va);
    va_end(va);
*/
        return 0;
}

static int
if_name_comparator (const void *p1, const void *p2)
{
        return strcmp ((const char *) (((const DHC_IF *) p1)->name),
                                   (const char *) (((const DHC_IF *) p2)->name)
                );
}

static int
dhc_pid_comparator (const void *p1, const void *p2)
{
        return ((((const DHC_IF *) p1)->dhc_pid
                         == ((const DHC_IF *) p2)->dhc_pid) ? 0
                        : ((((const DHC_IF *) p1)->dhc_pid
                                > ((const DHC_IF *) p2)->dhc_pid) ? 1 : -1)
                );
}

static int
dhc_subs_comparator (const void *p1, const void *p2)
{
        uint16_t oc1 = ((((const DHCDBD_Subscription *) p1)->opt->universe << 8)
                                        | ((const DHCDBD_Subscription *) p1)->opt->code),
                oc2 = ((((const DHCDBD_Subscription *) p2)->opt->universe << 8)
                           | ((const DHCDBD_Subscription *) p2)->opt->code);

        return ((oc1 == oc2)
                        ? 0 : ((oc1 > oc2)
                                   ? 1 : -1)
                );
}

static int
dhc_subr_comparator (const void *p1, const void *p2)
{
        return strcmp (((const DHCDBD_Subscriber *) p1)->subscriber,
                                   ((const DHCDBD_Subscriber *) p2)->subscriber);
}

static
        void
dhcdbd_if_state_change (DHC_IF * d_if, uint8_t state, char *state_str)
{
        char state_change_interface[1024];

        dhcdbd_debug ("StateChange: %s  %u -> %u %s", d_if->name, d_if->dhc_state,
                                  state, state_str);

        d_if->dhc_state = state;

        gettimeofday (&(d_if->tv), 0L);

        snprintf (state_change_interface, 1024, "%s.state",
                          dhcdbd_interface_prefix);

        if (!dbus_svc_send
                (d_if->dbus, SIGNAL, -1,
                 0L,
                 0L,
                 dhcdbd_object_path,
                 state_change_interface, d_if->name, TYPE_BYTE, &state, TYPE_INVALID)
                )
                dhcdbd_log ("dhcdbd_if_state_change: dbus_svc_send failed");
}

static
        void
dhc_reaper (int sig, siginfo_t * si, void *arg)
{
        DHC_IF *d, *const *dpp, ds;

        if (sig != SIGCHLD)
                return;
        ds.dhc_pid = si->si_pid;
        dpp = tfind (&ds, &dhc_tree, dhc_pid_comparator);
        if ((dpp == 0L) || ((d = *dpp) == 0L)) {        /* can't happen - but if it * 
                                                                                                 * does... */
                waitpid (si->si_pid, 0L, WNOHANG);
                return;
        }
        d->dhc_si_code = si->si_code;
        if ((si->si_code == CLD_EXITED)
                || (si->si_code == CLD_KILLED)
                || (si->si_code == CLD_DUMPED)
                )
                d->dhc_exit_status = si->si_status;
}

static void
state_trans (DHC_IF * d_if, uint32_t next_state)
{
        /* Uncomment to debug dhclient process state changes */
/*
    char cs[16], ns[16];
    if( d_if->state == next_state)
        return;
    sprintf(cs,"%u",d_if->state);
    sprintf(ns,"%u", next_state);
    dhcdbd_log("dhc %p %d %s -> %s",
               d_if, d_if->dhc_pid,
               (d_if->state == DHC_IF_INACTIVE)  
               ? "INACTIVE"
               : (d_if->state == DHC_IF_UP_REQUEST)
                 ? "UP_REQUEST"
                 : (d_if->state == DHC_IF_UP_START) 
                   ? "UP_START"
                   : (d_if->state == DHC_IF_UP) 
                     ? "UP"                  
                       : (d_if->state == DHC_IF_DOWN_REQUEST )
                         ? "DOWN_REQUEST"
                         : (d_if->state ==  DHC_IF_DOWN_START)
                           ? "DOWN_START"
                           : (d_if->state ==  DHC_IF_DOWN)
                             ?"DOWN"
                             : (d_if->state ==  DHC_IF_RESTART)
                             ? "RESTART"
                             : (d_if->state ==  DHC_IF_RESTARTING)
                               ? "RESTARTING"
                               : ( d_if->state == DHC_IF_RELEASED)
                                 ? "RELEASED"
                                 : ( d_if->state == DHC_IF_RELEASER)
                                   ? "RELEASER"
                                   : cs,
               
               (next_state == DHC_IF_INACTIVE)  
               ? "INACTIVE"
               : (next_state == DHC_IF_UP_REQUEST)
                 ? "UP_REQUEST"
                 : (next_state == DHC_IF_UP_START) 
                   ? "UP_START"
                   : (next_state == DHC_IF_UP) 
                     ? "UP"
                     : (next_state ==  DHC_IF_DOWN_REQUEST)
                     ? "DOWN_REQUEST"
                       : (next_state ==  DHC_IF_DOWN_START)
                         ? "DOWN_START"
                         : (next_state ==  DHC_IF_DOWN)
                           ? "DOWN"
                           : (next_state ==  DHC_IF_RESTART)
                             ? "RESTART"
                             : (next_state ==  DHC_IF_RESTARTING)
                               ? "RESTARTING"
                               : ( next_state == DHC_IF_RELEASED)
                                 ? "RELEASED"
                                 : ( next_state == DHC_IF_RELEASER)
                                   ? "RELEASER"        
                                   : ns
        );         
*/
}

void
dhcdbd_monitor (const void *p, const VISIT which, const int level)
{
        DHC_IF *d_if, *const *dpp = p;
        uint8_t dead = 0;
        uint32_t sig = 0, pid = 0;
        char pid_file[1024];
        FILE *f = 0L;
        int ws = 0, e = 0;
        time_t t;

        if ((dpp == 0L) || ((d_if = *dpp) == 0L)
                || ((which != postorder) && (which != leaf))
                )
                return;

        if (d_if->dhc_pid == 0L) {
                dhcdbd_log
                        ("huh ? dhclient %p (%s) has zero pid and should not be on dhc_tree!",
                         d_if, d_if->name);
                if (tfind (d_if, (&dead_dhc_tree), dhc_pid_comparator) == 0L)
                        tsearch (d_if, (&dead_dhc_tree), dhc_pid_comparator);
                return;
        }

        if (d_if->dhc_si_code != 0L) {  /* RIP dhclient */
                if (d_if->dhc_si_code == CLD_EXITED) {
                        dhcdbd_debug ("dhclient %d exited: %d", d_if->dhc_pid,
                                                  WEXITSTATUS (d_if->dhc_exit_status));
                        dhcdbd_if_state_change (d_if, DHC_END, "END");
                        dead = 1;
                } else if (d_if->dhc_si_code == CLD_KILLED) {
                        sig = WTERMSIG (d_if->dhc_exit_status);
                        dhcdbd_debug ("dhclient %d killed with signal %d", d_if->dhc_pid,
                                                  sig);
                        if (sig != SIGTERM)
                                dhcdbd_if_state_change (d_if, DHC_END, "ABEND");
                        else
                                dhcdbd_if_state_change (d_if, DHC_END, "END");
                        dead = 1;
                } else if (d_if->dhc_si_code == CLD_DUMPED) {
                        dhcdbd_log ("dhclient %d dumped core", d_if->dhc_pid);
                        dhcdbd_if_state_change (d_if, DHC_END, "ABEND");
                        dead = 1;
                } else if (d_if->dhc_si_code) {
                        dhcdbd_log ("dhclient %d stopped / traced: %d", d_if->dhc_pid,
                                                d_if->dhc_si_code);
                }
                if (dead) {
                        if ((ws = waitpid (d_if->dhc_pid, 0L, WNOHANG)) == d_if->dhc_pid) {
                                dhcdbd_debug ("Wait OK");
                                d_if->dhc_si_code = 0;
                                switch (d_if->state) {
                                        case DHC_IF_DOWN_START:
                                        case DHC_IF_RELEASED:
                                        case DHC_IF_RESTARTING:

                                                dhcdbd_debug ("dhc %d state %d", d_if->dhc_pid,
                                                                          d_if->state);

                                                snprintf (pid_file, 128, "%s%s%s",
                                                                  DHCDBD_DHCLIENT_PID_PFX, d_if->name,
                                                                  DHCDBD_DHCLIENT_PID_SFX);

                                                if (access (pid_file, F_OK) == 0) {
                                                        if ((f = fopen (pid_file, "r")) != 0L) {
                                                                if (fscanf (f, "%d", &pid) == 1) {
                                                                        fclose (f);
                                                                        if (pid == d_if->dhc_pid) {
                                                                                dhcdbd_debug
                                                                                        ("monitor: Removed pid file OK");
                                                                                unlink (pid_file);
                                                                        }
                                                                } else
                                                                        fclose (f);
                                                        }
                                                }
                                                break;

                                        case DHC_IF_RELEASER:
                                                if (d_if->releasing != 0L) {
                                                        if (d_if->releasing->state != DHC_IF_RESTARTING) {
                                                                dhcdbd_debug
                                                                        (" released pid %d state %d completed - remove from dhc_tree ",
                                                                         d_if->releasing->dhc_pid,
                                                                         d_if->releasing->state);
                                                                if (tfind
                                                                        (d_if->releasing, (&dead_dhc_tree),
                                                                         dhc_pid_comparator) == 0L)
                                                                        tsearch (d_if->releasing,
                                                                                         (&dead_dhc_tree),
                                                                                         dhc_pid_comparator);
                                                        }
                                                        d_if->releasing->releasing = 0L;
                                                }
                                                d_if->releasing = 0L;

                                                dhcdbd_debug ("RELEASER %d exited OK.",
                                                                          d_if->dhc_pid);
                                                break;

                                        default:
                                                dhcdbd_log ("Unrequested down ?:%d", d_if->state);
                                                /* XXX fixme: perhaps we should restart automatically 
                                                 * ? */
                                                break;
                                }

                                if (d_if->releasing == 0L) {
                                        dhcdbd_debug (" pid %d completed - remove from dhc_tree ",
                                                                  d_if->dhc_pid);
                                        if (tfind (d_if, (&dead_dhc_tree), dhc_pid_comparator) ==
                                                0L)
                                                tsearch (d_if, (&dead_dhc_tree), dhc_pid_comparator);
                                } else if (d_if->state != DHC_IF_RESTARTING) {
                                        state_trans (d_if, DHC_IF_DOWN_START);
                                        d_if->dhc_si_code = 0;
                                        d_if->dhc_exit_status = 0;
                                }
                                if (d_if->state == DHC_IF_RESTARTING) {
                                        dhcdbd_debug
                                                (" old dhclient %d finished - restarting - %d",
                                                 d_if->dhc_pid, d_if->dhc_si_code);
                                        if (tfind (d_if, (&restart_dhc_tree), dhc_pid_comparator)
                                                == 0L)
                                                tsearch (d_if, (&restart_dhc_tree),
                                                                 dhc_pid_comparator);
                                }
                        } else {
                                e = errno;
                                dhcdbd_debug (" waitpid: %d errno: %d", ws, e);
                                if (e == ECHILD) {
                                        d_if->dhc_si_code = 0;
                                        d_if->dhc_exit_status = 0;
                                        snprintf (pid_file, 1024, "/proc/%u", d_if->dhc_pid);
                                        if (access (pid_file, F_OK) == 0) {
                                                dhcdbd_log ("ECHILD for dead pid %d - removing",
                                                                        d_if->dhc_pid);
                                                if (tfind (d_if, (&dead_dhc_tree), dhc_pid_comparator)
                                                        == 0L)
                                                        tsearch (d_if, (&dead_dhc_tree),
                                                                         dhc_pid_comparator);
                                        }
                                }
                        }
                }
        } else if (d_if->releasing == 0L) {
                snprintf (pid_file, 1024, "/proc/%u", d_if->dhc_pid);
                if (access (pid_file, F_OK) == 0) {
                        if ((d_if->state == DHC_IF_UP_START) ||
                                (d_if->state == DHC_IF_RESTARTING)) {
                                dhcdbd_if_state_change (d_if, DHC_START, "START");
                        }
                        state_trans (d_if, DHC_IF_UP);
                        d_if->state = DHC_IF_UP;
                        if (((d_if->dhc_state == DHC_PREINIT) ||
                                 (d_if->dhc_state == DHC_START)) && (d_if->timeout > 0)) {
                                time (&t);
                                if ((t - d_if->tv.tv_sec) > d_if->timeout) {
                                        dhcdbd_if_state_change (d_if, DHC_TIMEOUT, "TIMEOUT");
                                        if (tfind (d_if, (&timeout_dhc_tree), dhc_pid_comparator)
                                                == 0L)
                                                tsearch (d_if, (&timeout_dhc_tree),
                                                                 dhc_pid_comparator);
                                }
                        }
                } else {
                        dhcdbd_log
                                (" dhclient %d down (%d) but si_code == 0 and releasing==0 !",
                                 d_if->dhc_pid, d_if->state);
                }
        }
}

void
dhcdbd_remove_pid (const void *p, const VISIT which, const int level)
{
        DHC_IF *d_if, *const *dpp = p;

        if ((dpp == 0L) || ((d_if = *dpp) == 0L)
                || ((which != postorder) && (which != leaf))
                )
                return;
        dhcdbd_debug ("REMOVE pid: %d", d_if->dhc_pid);
        d_if->dhc_si_code = 0;
        d_if->dhc_exit_status = 0;
        d_if->releasing = 0L;
        tdelete (d_if, &(dhc_tree), dhc_pid_comparator);
        if (d_if->state != DHC_IF_RESTARTING) {
                dhcdbd_debug ("Remove pid - NOT restarting %d %d", d_if->dhc_pid,
                                          d_if->state);
                d_if->dhc_pid = 0;
                state_trans (d_if, DHC_IF_DOWN);
                d_if->state = DHC_IF_DOWN;
        }
}

static int dhcdbd_run_dhclient (DHC_IF *);

void
dhcdbd_restart_dhc (const void *p, const VISIT which, const int level)
{
        DHC_IF *d_if, *const *dpp = p;

        if ((dpp == 0L) || ((d_if = *dpp) == 0L)
                || ((which != postorder) && (which != leaf))
                )
                return;
        if (d_if->releasing == 0L) {
                if (d_if->dhc_pid != 0L) {
                        if (tfind (d_if, &(dhc_tree), dhc_pid_comparator) != 0L) {
                                if (tfind (d_if, &(dead_dhc_tree), dhc_pid_comparator) == 0L)
                                        tsearch (d_if, &(dead_dhc_tree), dhc_pid_comparator);
                                dhcdbd_debug ("restart_dhc - remove old dhc %d",
                                                          d_if->dhc_pid);
                                return;
                        }
                }
                if (tfind (d_if, &(restarted_dhc_tree), dhc_pid_comparator) == 0L) {
                        tsearch (d_if, &(restarted_dhc_tree), dhc_pid_comparator);
                        dhcdbd_debug ("restart_dhc - add dhc %d to restarted tree",
                                                  d_if->dhc_pid);
                }
        }
}

void
dhcdbd_remove_restarted_pid (const void *p, const VISIT which,
                                                         const int level)
{
        DHC_IF *d_if, *const *dpp = p;

        if ((dpp == 0L) || ((d_if = *dpp) == 0L)
                || ((which != postorder) && (which != leaf))
                )
                return;
        dhcdbd_debug ("Remove Restart pid: %d", d_if->dhc_pid);
        state_trans (d_if, DHC_IF_DOWN);
        d_if->state = DHC_IF_DOWN;
        tdelete (d_if, &(restart_dhc_tree), dhc_pid_comparator);
        d_if->dhc_pid = 0L;
        state_trans (d_if, DHC_IF_UP_REQUEST);
        d_if->state = DHC_IF_UP_REQUEST;
        dhcdbd_debug ("RESTARTING %s", d_if->name);
        dhcdbd_run_dhclient (d_if);
}


void
dhcdbd_timeout_dhc (const void *p, const VISIT which, const int level)
{
        DHC_IF *d_if, *const *dpp = p;

        if ((dpp == 0L) || ((d_if = *dpp) == 0L)
                || ((which != postorder) && (which != leaf))
                )
                return;

        state_trans (d_if, DHC_IF_DOWN_REQUEST);
        d_if->state = DHC_IF_DOWN_REQUEST;
        dhcdbd_run_dhclient (d_if);
}

void
dhcdbd_no_free (void *p)
{
}

void
dhcdbd_work (DBUS_SVC dbus)
{
        sigset_t ss;

        sigemptyset (&ss);
        sigaddset (&ss, SIGCHLD);

        dead_dhc_tree = 0L;
        restarted_dhc_tree = 0L;
        if (restart_dhc_tree != 0L) {
                twalk (restart_dhc_tree, dhcdbd_restart_dhc);
                if (dead_dhc_tree != 0L) {
                        sigprocmask (SIG_BLOCK, &ss, 0L);
                        twalk (dead_dhc_tree, dhcdbd_remove_pid);
                        tdestroy (dead_dhc_tree, dhcdbd_no_free);
                        dead_dhc_tree = 0L;
                        sigprocmask (SIG_UNBLOCK, &ss, 0L);
                }
                if (restarted_dhc_tree != 0L) {
                        twalk (restarted_dhc_tree, dhcdbd_remove_restarted_pid);
                        tdestroy (restarted_dhc_tree, dhcdbd_no_free);
                        restarted_dhc_tree = 0L;
                }
        }

        sigprocmask (SIG_BLOCK, &ss, 0L);
        dead_dhc_tree = 0L;
        timeout_dhc_tree = 0L;
        twalk (dhc_tree, dhcdbd_monitor);

        if (dead_dhc_tree != 0L) {
                dhcdbd_debug ("dhcdbd_work: PIDS TO REMOVE!");
                twalk (dead_dhc_tree, dhcdbd_remove_pid);
                tdestroy (dead_dhc_tree, dhcdbd_no_free);
        }

        dead_dhc_tree = 0L;
        sigprocmask (SIG_UNBLOCK, &ss, 0L);

        if (timeout_dhc_tree != 0L) {
                twalk (timeout_dhc_tree, dhcdbd_timeout_dhc);
                tdestroy (timeout_dhc_tree, dhcdbd_no_free);
                timeout_dhc_tree = 0L;
        }
}

static pid_t
dhcdbd_find_existing_dhclient (char *if_name)
{
        char proc_file[128];
        char pid_file[128];
        pid_t pid = 0;
        FILE *f = 0L;

        snprintf (pid_file, 128, "%s%s%s", DHCDBD_DHCLIENT_PID_PFX, if_name,
                          DHCDBD_DHCLIENT_PID_SFX);

        if (access (pid_file, R_OK) != 0)
                return (0);

        if ((f = fopen (pid_file, "r")) == 0L)
                return (0);

        if (fscanf (f, "%d", &pid) != 1)
                pid = 0;

        fclose (f);

        if (pid == 0)
                return pid;

        snprintf (proc_file, 128, "/proc/%d", pid);

        if (access (proc_file, F_OK) != 0) {
                unlink (pid_file);
                return (0);
        }
        return pid;
}

static
        int
dhcdbd_kill_dhclient (DHC_IF * d_if, pid_t pid)
{
        char pid_file[128], proc_file[128];
        time_t tt, tn;
        struct timespec ts;
        int i;
        FILE *f;

        snprintf (proc_file, 128, "/proc/%d", pid);
        if (access (proc_file, F_OK) != 0) {
                if (d_if->dhc_pid == pid)
                        d_if->dhc_pid = 0;
                return 0;
        }

        dhcdbd_debug ("killing dhclient pid: %d", pid);
        tt = time (0L);
        i = 0;
        do {
                if (i <= 4)
                        kill (pid, SIGTERM);
                else if (i <= 6)
                        kill (pid, SIGABRT);
                else
                        kill (pid, SIGKILL);
                ts.tv_sec = 0;
                ts.tv_nsec = 200000000;
                nanosleep (&ts, 0L);

                dhcdbd_work (d_if->dbus);

                tn = time (0L);
                i++;
        } while (((tn - tt) < 2) && (access (proc_file, F_OK) == 0));

        if (access (proc_file, F_OK) == 0) {
                dhcdbd_log ("Unable to kill existing dhclient process %d", pid);
                return (-1);
        } else {
                dhcdbd_debug ("Killed existing dhclient process %d OK.", pid);
                snprintf (pid_file, 128, "%s%s%s", DHCDBD_DHCLIENT_PID_PFX,
                                  d_if->name, DHCDBD_DHCLIENT_PID_SFX);

                if (access (pid_file, F_OK) == 0) {
                        if ((f = fopen (pid_file, "r")) != 0L) {
                                if (fscanf (f, "%d", &pid) == 1) {
                                        fclose (f);
                                        if (pid == d_if->dhc_pid) {
                                                dhcdbd_debug ("kill: Removed pid file OK");
                                                unlink (pid_file);
                                        }
                                } else
                                        fclose (f);
                        }
                }
        }
        return 0;
}

static
        int
dhcdbd_run_dhclient (DHC_IF * d_if)
{
        char **dhc_args, **dhc_envp, *xtra_arg = 0L;
        uint8_t n_args = 12, arg;
        DHC_IF *rel_if = 0L;
        int i;
        sigset_t ss;
        pid_t kill_pid = 0;
        char dhc_if_conf_file[1024];

        sigemptyset (&ss);
        sigaddset (&ss, SIGCHLD);

        switch ((DHC_IF_State) d_if->state) {
                case DHC_IF_UP_REQUEST:
                        dhcdbd_debug ("dhcdbd_run_dhclient: IF_UP");
                        if ((d_if->dhc_flags & DHCLIENT_PERSISTENT) == 0) {
                                ++n_args;
                                xtra_arg = "-1";
                        }
                        kill_pid = dhcdbd_find_existing_dhclient (d_if->name);
                        state_trans (d_if, DHC_IF_UP_START);
                        d_if->state = DHC_IF_UP_START;
                        if (d_if->dhc_pid != 0) {
                                if (d_if->dhc_pid != kill_pid) {
                                        dhcdbd_debug ("dhcdbd_run_dhclient: IF_RESTARTING");
                                        d_if->state = DHC_IF_RESTARTING;
                                        if (dhcdbd_kill_dhclient (d_if, d_if->dhc_pid) == -1)
                                                return (-1);
                                }
                        } else if (kill_pid != 0)
                                d_if->dhc_pid = kill_pid;
                        break;

                case DHC_IF_RESTART:
                        dhcdbd_debug ("dhcbdb_run_dhclient: IF_RESTART");
                        state_trans (d_if, DHC_IF_RESTARTING);
                        d_if->state = DHC_IF_RESTARTING;
                case DHC_IF_DOWN_REQUEST:
                        dhcdbd_debug ("dhcbdb_run_dhclient: IF_DOWN");
                        if (d_if->releasing != 0L) {
                                dhcdbd_log ("dhcdbd_run_dhclient - already releasing");
                                return -1;
                        }
                        if (d_if->dhc_flags & DHCLIENT_RELEASE) {
                                ++n_args;
                                xtra_arg = "-r";
                                if (d_if->state == DHC_IF_DOWN_REQUEST) {
                                        state_trans (d_if, DHC_IF_RELEASED);
                                        d_if->state = DHC_IF_RELEASED;
                                }
                        } else {
                                /* no release; but we have to run dhclient_script in stop
                                 * mode */
                                if (d_if->state != DHC_IF_RESTARTING) {
                                        state_trans (d_if, DHC_IF_DOWN_START);
                                        d_if->state = DHC_IF_DOWN_START;
                                }
                                kill_pid = d_if->dhc_pid;
                        }
                        if ((rel_if = (DHC_IF *) malloc (sizeof (DHC_IF))) == 0L) {
                                dhcdbd_log ("dhcdbd_run_dhclient: out of memory");
                                return -1;
                        }
                        memcpy (rel_if, d_if, sizeof (DHC_IF));
                        d_if->releasing = rel_if;
                        rel_if->state = DHC_IF_RELEASER;
                        rel_if->releasing = d_if;
                        rel_if->dhc_pid = 0;
                        gettimeofday (&(rel_if->tv), 0L);
                        d_if = rel_if;
                        break;
                default:
                        dhcdbd_log ("dhcdbd_run_dhclient: unhandled init state %d",
                                                d_if->state);
                        return (-1);
        }

        if ((dhc_envp = (char **) calloc (3, sizeof (char *))) == 0L) {
                dhcdbd_log ("dhcdbd_run_dhclient: out of memory");
                return -1;
        }
        if ((dhc_envp[0] =
                 (char *) malloc (strlen ("PATH=/bin:/usr/bin:/sbin:/usr/sbin") +
                                                  1)) == 0L) {
                dhcdbd_log ("dhcdbd_run_dhclient: out of memory");
                return -1;
        }
        strcpy (dhc_envp[0], "PATH=/bin:/usr/bin:/sbin:/usr/sbin");
        if ((dhc_envp[1] =
                 (char *) malloc (strlen ("PWD=") +
                                                  strlen (DHCDBD_DHCLIENT_LEASE_DIR) + 1)) == 0L) {
                dhcdbd_log ("dhcdbd_run_dhclient: out of memory");
                return -1;
        }
        sprintf (dhc_envp[1], "PWD=%s", DHCDBD_DHCLIENT_LEASE_DIR);
        if ((d_if->state != DHC_IF_RELEASER)
                || ((d_if->releasing != 0L)
                        && (d_if->releasing->state == DHC_IF_RELEASED)
                )
                ) {
                dhcdbd_debug ("Running dhclient for IF_UP %s", d_if->name);

                snprintf (dhc_if_conf_file, 1024, "%s%s%s%s",
                                  DHCDBD_DHCLIENT_CONF_PFX, DHCDBD_DHCLIENT_CONF_SEP,
                                  d_if->name, DHCDBD_DHCLIENT_CONF_SFX);

                if (access (dhc_if_conf_file, R_OK) == 0)
                        n_args += 2;
                else
                        dhc_if_conf_file[0] = '\0';

#if DHCLIENT_EXTENDED_OPTION_ENVIRONMENT <= 0
                --n_args;
#endif

                if ((dhc_args = (char **) calloc (n_args, sizeof (char *))) == 0L)
                        return (-1);

                if ((dhc_args[0] = (char *) malloc (strlen (dhclient_bin) + 1)) == 0L)
                        return (-1);

                strcpy (dhc_args[0], dhclient_bin);
                arg = 0;

                if (xtra_arg != 0L) {
                        if ((dhc_args[++arg] = (char *) malloc (3)) == 0L)
                                return (-1);
                        strcpy (dhc_args[arg], xtra_arg);
                }

                if ((dhc_args[++arg] = (char *) malloc (4)) == 0L)
                        return -1;
                strcpy (dhc_args[arg], "-lf");
                if ((dhc_args[++arg] = (char *) malloc (strlen (d_if->name)
                                                                                                +
                                                                                                strlen
                                                                                                (DHCDBD_DHCLIENT_LEASE_DIR)
                                                                                                +
                                                                                                strlen
                                                                                                (DHCDBD_DHCLIENT_LEASE_PFX)
                                                                                                +
                                                                                                strlen
                                                                                                (DHCDBD_DHCLIENT_LEASE_SFX)
                                                                                                + 1)
                        ) == 0L)
                        return -1;
                sprintf (dhc_args[arg], "%s%s%s%s", DHCDBD_DHCLIENT_LEASE_DIR,
                                 DHCDBD_DHCLIENT_LEASE_PFX, d_if->name,
                                 DHCDBD_DHCLIENT_LEASE_SFX);

                if (((d_if->dhc_flags & DHCLIENT_NO_LEASES) == DHCLIENT_NO_LEASES)
                        && (d_if->state == DHC_IF_UP_START)
                        )
                        unlink (dhc_args[arg]);

                if (dhc_if_conf_file[0] != '\0') {
                        if ((dhc_args[++arg] = (char *) malloc (4)) == 0L)
                                return -1;
                        strcpy (dhc_args[arg], "-cf");
                        if ((dhc_args[++arg] =
                                 (char *) malloc (strlen (dhc_if_conf_file) + 1)) == 0L)
                                return -1;
                        strcpy (dhc_args[arg], dhc_if_conf_file);
                }

                if ((dhc_args[++arg] = (char *) malloc (4)) == 0L)
                        return -1;
                strcpy (dhc_args[arg], "-pf");
                if ((dhc_args[++arg] = (char *) malloc (strlen (d_if->name)
                                                                                                +
                                                                                                strlen
                                                                                                (DHCDBD_DHCLIENT_PID_PFX)
                                                                                                +
                                                                                                strlen
                                                                                                (DHCDBD_DHCLIENT_PID_SFX)
                                                                                                + 1)
                        ) == 0L)
                        return -1;
                sprintf (dhc_args[arg], "%s%s%s", DHCDBD_DHCLIENT_PID_PFX, d_if->name,
                                 DHCDBD_DHCLIENT_PID_SFX);

                if ((dhc_args[++arg] = (char *) malloc (3)) == 0L)
                        return -1;
                strcpy (dhc_args[arg], "-q");

                if ((dhc_args[++arg] = (char *) malloc (3)) == 0L)
                        return -1;
                strcpy (dhc_args[arg], "-e");
                if ((dhc_args[++arg] =
                         (char *) malloc (strlen ("dhc_dbus=xx") + 1)) == 0L)
                        return -1;
                sprintf (dhc_args[arg], "dhc_dbus=%-2u", d_if->dhc_mode);

#if DHCLIENT_EXTENDED_OPTION_ENVIRONMENT > 0
                if ((dhc_args[++arg] = (char *) malloc (3)) == 0L)
                        return -1;
                strcpy (dhc_args[arg], "-x");
#endif

                if ((dhc_args[++arg] = (char *) malloc (3)) == 0L)
                        return -1;
                strcpy (dhc_args[arg], "-d");

                if ((dhc_args[++arg] =
                         (char *) malloc (strlen (d_if->name) + 1)) == 0L)
                        return -1;
                strcpy (dhc_args[arg], d_if->name);

        } else {
                dhcdbd_debug ("Running dhclient-script in STOP mode for %s %d",
                                          d_if->name, d_if->dhc_pid);
                if ((dhc_args = (char **) calloc (4, sizeof (char *))) == 0L)
                        return (-1);
                arg = 0;
                if ((dhc_args[arg] = (char *) malloc (10)) == 0L)
                        return (-1);
                strcpy (dhc_args[arg], "/bin/bash");
                if ((dhc_args[++arg] = (char *) malloc (3)) == 0L)
                        return (-1);
                strcpy (dhc_args[arg], "-c");
                if ((dhc_args[++arg] = (char *)
                         malloc (strlen
                                         ("reason=STOP interface=  dhc_dbus=xx /sbin/dhclient-script")
                                         + strlen (d_if->name)
                                         + 1)
                        ) == 0L)
                        return -1;
                sprintf (dhc_args[arg],
                                 "reason=STOP interface=%s dhc_dbus=%-2u /sbin/dhclient-script",
                                 d_if->name, d_if->dhc_mode);

        }

        if (kill_pid != 0) {
                dhcdbd_debug ("Killing dhclient pid %d", kill_pid);
                if (dhcdbd_kill_dhclient (d_if, kill_pid) == -1) {
                        dhcdbd_log ("dhcdbd_kill_dhclient failed");
                        return (-1);
                }
        }

        sigprocmask (SIG_BLOCK, &ss, 0L);

        gettimeofday (&(d_if->tv), 0L);
        dhcdbd_debug ("whoami: uid: %lu gid: %lu euid: %lu egid: %lu",
                                  getuid (), getgid (), geteuid (), getegid ()
                );

        if ((d_if->dhc_pid = fork ()) < 0) {
                dhcdbd_log ("fork failed");
                return (-1);
        } else if (d_if->dhc_pid == 0) {
/* FIXME: NR_OPEN is defined in linux/limits.h, figure out why cpp sucks */
                for (i = 0; i < /* NR_OPEN */ 1024; i++)
                        close (i);
                if ((i = open ("/dev/null", O_RDONLY)) != -1) {
                        if (i != 0)
                                if (dup2 (i, 0) == 0) {
                                        close (i);
                                        i = 0;
                                }
                        if (i == 0)
                                fcntl (i, F_SETFD, 0L);
                }
                if ((i = open ("/dev/null", O_WRONLY)) != -1) {
                        if (i != 1)
                                if (dup2 (i, 1) == 1) {
                                        close (i);
                                        i = 1;
                                }
                        if (i == 1)
                                fcntl (i, F_SETFD, 0L);
                }
                if ((i = open ("/dev/null", O_WRONLY)) != -1) {
                        if (i != 2)
                                if (dup2 (i, 2) == 2) {
                                        close (i);
                                        i = 2;
                                }
                        if (i == 2)
                                fcntl (i, F_SETFD, 0L);
                }
                i = chdir (DHCDBD_DHCLIENT_LEASE_DIR);
                execve (dhc_args[0], dhc_args, dhc_envp);
                exit (1);
        }
        dhcdbd_debug ("new pid: %d", d_if->dhc_pid);
        if (tsearch (d_if, &(dhc_tree), dhc_pid_comparator) == 0L) {
                dhcdbd_log ("dhcdbd_run_dhclient: failed to record pid: %d",
                                        d_if->dhc_pid);
                return (-1);
        }
        sigprocmask (SIG_UNBLOCK, &ss, 0L);
        dhcdbd_work (d_if->dbus);
        return 0;
}

dbus_svc_HandlerResult
dhcdbd_if_up (DBusMsgHandlerArgs)
{
        DHC_IF *d_if = (DHC_IF *) object;
        char error_name[1024] = "", error_message[1024] = "";
        uint32_t success_response = 0, new_serial, flags = 0, mode = 0;
        pid_t kill_pid = 0;
        dhcdbd_debug ("dhcdbd_if_up: %p %s", d_if, d_if->name);

        if (strcmp (signature, "u") == 0)
                dbus_svc_get_args (dbus, msg, TYPE_UINT32, &mode, TYPE_INVALID);
        else if (strcmp (signature, "uu") == 0)
                dbus_svc_get_args (dbus, msg,
                                                   TYPE_UINT32, &mode,
                                                   TYPE_UINT32, &flags, TYPE_INVALID);
        else if (strcmp (signature, "uuu") == 0)
                dbus_svc_get_args (dbus, msg,
                                                   TYPE_UINT32, &mode,
                                                   TYPE_UINT32, &flags,
                                                   TYPE_UINT32, &(d_if->timeout), TYPE_INVALID);

        d_if->dhc_mode = mode;

        switch ((DHC_IF_State) d_if->state) {
                case DHC_IF_INACTIVE:
                case DHC_IF_DOWN:
                        state_trans (d_if, DHC_IF_UP_REQUEST);
                        d_if->state = DHC_IF_UP_REQUEST;
                        d_if->dhc_flags = flags;
                        break;
                case DHC_IF_UP:
                        state_trans (d_if, DHC_IF_RESTART);
                        d_if->state = DHC_IF_RESTART;
                        d_if->dhc_flags = flags;
                        break;

                case DHC_IF_UP_REQUEST:
                case DHC_IF_UP_START:
                        sprintf (error_message,
                                         "interface %s is being started up. Please try again later.",
                                         d_if->name);
                        break;

                case DHC_IF_DOWN_REQUEST:
                case DHC_IF_DOWN_START:
                        sprintf (error_message,
                                         "interface %s is being shut down. Please try again later.",
                                         d_if->name);
                        break;

                case DHC_IF_RESTARTING:
                case DHC_IF_RESTART:
                        sprintf (error_message,
                                         "interface %s is being restarted %s. Please try again later.",
                                         d_if->name,
                                         (d_if->releasing == 0L) ? "" : " - releasing");
                        /* LACIE: restart-release, same situation as next case */
                        if(d_if->releasing == 0L)
                            break;

                case DHC_IF_RELEASED:
                        /* "interface %s is being released %s. Please try again later"
                         * LACIE: An interface gets up but it was stuck at last release
                         * Kill any remaining dhclient and reset the state machine
                         * Problem is known but tagged as "won't fix"
                         * https://bugs.launchpad.net/ubuntu/+source/dhcdbd/+bug/88327
                         */
                        kill_pid = dhcdbd_find_existing_dhclient(d_if->name);
                        dhcdbd_debug ("LaCie: force release of interface %s", d_if->name);
                        if (kill_pid != 0) {
                            dhcdbd_debug ("Killing dhclient pid %d", kill_pid);
                            if (dhcdbd_kill_dhclient (d_if, kill_pid) == -1) {
                                    dhcdbd_log ("dhcdbd_kill_dhclient failed");
                            }
                        }
                        d_if->state = DHC_IF_UP_REQUEST;
                default:
                        break;
        }

        if (geteuid () != 0) {
                sprintf (error_message,
                                 "You must install this program setuid to root in order to run dhclient.\n");
                sprintf (error_name, "%s.NotSupported", dhcdbd_interface_prefix);
        } else if (getuid () != 0) {
                setuid (0);
                setgid (0);
        }

        if (error_message[0] == '\0')
                dhcdbd_run_dhclient (d_if);
        else if (error_name[0] == '\0')
                sprintf (error_name, "%s.OperationInProgress",
                                 dhcdbd_interface_prefix);

        if (reply_expected) {
                dhcdbd_debug ("Sending reply  %d to %s", serial, sender);
                if (error_name[0] != '\0')
                        dbus_svc_send (dbus, ERROR, serial, &new_serial, sender, path,
                                                   interface, member, TYPE_STRING, error_name,
                                                   TYPE_STRING, error_message, TYPE_INVALID);
                else
                        dbus_svc_send (dbus, RETURN, serial, &new_serial, sender, path,
                                                   interface, member, TYPE_UINT32, &success_response,
                                                   TYPE_INVALID);
        }
        return HANDLED;
}

dbus_svc_HandlerResult
dhcdbd_if_down (DBusMsgHandlerArgs)
{
        DHC_IF *d_if = (DHC_IF *) object;
        char error_name[1024] = "", error_message[1024] = "";
        uint32_t success_response = 0, new_serial;

        dhcdbd_debug ("dhcdbd_if_down: %p %s", d_if, d_if->name);


        switch ((DHC_IF_State) d_if->state) {
                case DHC_IF_UP:
                        state_trans (d_if, DHC_IF_DOWN_REQUEST);
                        d_if->state = DHC_IF_DOWN_REQUEST;
                        break;

                case DHC_IF_DOWN:
                case DHC_IF_DOWN_REQUEST:
                case DHC_IF_INACTIVE:
                        sprintf (error_message,
                                         ": InvalidOperation: interface %s is not UP and cannot be brought DOWN.",
                                         d_if->name);
                        break;

                case DHC_IF_UP_REQUEST:
                case DHC_IF_UP_START:
                        sprintf (error_message,
                                         "interface %s is being started up. Please try again later.",
                                         d_if->name);
                        break;

                case DHC_IF_DOWN_START:
                        sprintf (error_message,
                                         "interface %s is being shut down. Please try again later.",
                                         d_if->name);
                        break;

                case DHC_IF_RESTARTING:
                case DHC_IF_RESTART:
                        sprintf (error_message,
                                         "interface %s is being restarted %s. Please try again later.",
                                         d_if->name,
                                         (d_if->releasing == 0L) ? "" : " - releasing");
                        break;

                case DHC_IF_RELEASED:
                        sprintf (error_message,
                                         "interface %s is being released. Please try again later.",
                                         d_if->name);
                        break;
                default:
                        break;
        }

        if (error_message[0] == '\0')
                dhcdbd_run_dhclient (d_if);
        else if (error_message[0] == ':')
                sprintf (error_name, "%s.InvalidOperation", dhcdbd_interface_prefix);
        else
                sprintf (error_name, "%s.OperationInProgress",
                                 dhcdbd_interface_prefix);

        if (reply_expected) {
                dhcdbd_debug ("Sending reply  %d to %s", serial, sender);
                if (error_name[0] != '\0')
                        dbus_svc_send (dbus, ERROR, serial, &new_serial, sender, path,
                                                   interface, member, TYPE_STRING, error_name,
                                                   TYPE_STRING, error_message, TYPE_INVALID);
                else
                        dbus_svc_send (dbus, RETURN, serial, &new_serial, sender, path,
                                                   interface, member, TYPE_UINT32, &success_response,
                                                   TYPE_INVALID);
        }
        return HANDLED;
}

static
        int
dhcdbd_option_comparator (const void *p1, const void *p2)
{
        return strcmp ((const char *) (((const DHCDBD_Option *) p1)->qname),
                                   (const char *) (((const DHCDBD_Option *) p2)->qname)
                );
}

dbus_svc_HandlerResult
dhcdbd_text_get_option (DBusMsgHandlerArgs)
{
        DHCDBD_Option *dho = (DHCDBD_Option *) object;
        uint32_t new_serial;

        dhcdbd_debug ("dhcdbd_text_get_option: %s %s", dho->d_if->name,
                                  dho->option->name);

        if (reply_expected) {
                dhcdbd_debug ("Sending reply  %d to %s", serial, sender);
                dbus_svc_send (dho->d_if->dbus, RETURN, serial, &new_serial, sender,
                                           path, interface, member, TYPE_STRING, &(dho->output),
                                           TYPE_INVALID);
        }

        return HANDLED;
}

dbus_svc_HandlerResult
dhcdbd_text_get_format (DBusMsgHandlerArgs)
{
        DHCDBD_Option *dho = (DHCDBD_Option *) object;
        uint32_t new_serial;
        char code[32], ucode[32], *t[1] = { "1" }, *f[1] = {
        "0"}, *uc[1] = {
        ucode}, *cp[1] = {
        code};
        dhcdbd_debug ("dhcdbd_text_get_format: %s %s", dho->d_if->name,
                                  dho->option->name);

        if (reply_expected) {
                dhcdbd_debug ("Sending reply  %d to %s", serial, sender);
                snprintf (code, 32, "%u", (uint32_t) dho->option->code);
                snprintf (ucode, 32, "%u", (uint32_t) dho->option->universe);
                dbus_svc_send (dho->d_if->dbus, RETURN, serial, &new_serial, sender,
                                           path, interface, member, TYPE_STRING, cp, TYPE_STRING,
                                           &(dho->option->type), TYPE_STRING, uc, TYPE_STRING,
                                           &(dhcp_universes[dho->option->universe].name),
                                           TYPE_STRING, &(dho->option->name), TYPE_STRING,
                                           ((dho->option->flags & DHC_F_NEW) ? t : f),
                                           TYPE_INVALID);
        }

        return HANDLED;
}

dbus_svc_HandlerResult
dhcdbd_binary_get_option (DBusMsgHandlerArgs)
{
        DHCDBD_Option *dho = (DHCDBD_Option *) object;
        uint32_t new_serial;
        char *value[1] = { dho->value };

        dhcdbd_debug ("dhcdbd_binary_get_option: %s %s", dho->d_if->name,
                                  dho->option->name);

        if (reply_expected) {
                dhcdbd_debug ("Sending reply %d to %s - length: %u", serial, sender,
                                          dho->length);

                dbus_svc_send
                        (dbus, RETURN, serial, &new_serial, sender, path, interface,
                         member, TYPE_ARRAY, TYPE_BYTE, &value, dho->length,
                         TYPE_INVALID);
        }

        return HANDLED;
}

typedef struct dhcdb_dom_s {
        DBUS_SVC dbus;
        dbus_svc_MessageHandle msg;
        uint8_t ok;
} DHCDBD_DBus_Option_Message;


static void dhcdbd_dbus_option_type
        (uint8_t map_code, uint8_t dhc_code, void *value, unsigned length,
         void *arg) {
        DHCDBD_DBus_Option_Message *d = (DHCDBD_DBus_Option_Message *) arg;
        uint8_t *p, *endp, is_hex = 0, *vptr[1] = { value };

        if (d->ok == 0)
                return;

        /* handle HEX_STRING types containing non-ascii bytes */
        if (dhc_code == DHC_T_HEX_STRING)
                for (p = value, endp = value + length; p < endp; p++) {
                        if ((*p < 32) || (*p > 127)) {
                                is_hex = 1;
                                break;
                        }
                }
        if (is_hex) {
                d->ok =
                        dbus_svc_message_append_args
                        (d->dbus, d->msg,
                         TYPE_ARRAY, TYPE_BYTE, &vptr, length, TYPE_INVALID);
        } else {
                d->ok =
                        dbus_svc_message_append_args
                        (d->dbus, d->msg,
                         (int) map_code, (map_code == TYPE_STRING) ? vptr : value,
                         TYPE_INVALID);
        }
}

dbus_svc_HandlerResult
dhcdbd_dbus_get_option (DBusMsgHandlerArgs)
{
        DHCDBD_Option *dho = (DHCDBD_Option *) object;
        uint32_t new_serial;
        char error_name[1024] = "", error_message[1024] = "";
        DHCDBD_DBus_Option_Message dbo;

        dhcdbd_debug ("dhcdbd_dbus_get_option: dho:%p dho->d_if: %p", dho,
                                  dho->d_if);
        dhcdbd_debug ("dhcdbd_dbus_get_option: %s %s", dho->d_if->name,
                                  dho->option->name);
        if (reply_expected) {
                dhcdbd_debug ("Sending reply  %d to %s", serial, sender);

                if ((dbo.msg = dbus_svc_new_message
                         (dho->d_if->dbus, RETURN, serial,
                          sender, path, interface, member)
                        ) == 0L) {
                        dhcdbd_log ("dbus_svc_new_message failed");
                        sprintf (error_name, "%s.InternalError", dhcdbd_interface_prefix);
                        sprintf (error_message, "dbus_svc_new_message failed");
                        dbus_svc_send (dho->d_if->dbus, ERROR, serial, &new_serial,
                                                   sender, path, interface, member, TYPE_STRING,
                                                   error_name, TYPE_STRING, error_message,
                                                   TYPE_INVALID);
                        return HANDLED;
                }

                dbo.dbus = dho->d_if->dbus;
                dbo.ok = 1;

                dhco_map_option (dho->d_if->dhco, dho->option,
                                                 dho->value, dho->length,
                                                 dho->d_if->mapping, dhcdbd_dbus_option_type, &dbo);

                if (dbo.ok != 1) {
                        dhcdbd_log ("dbus_svc_message_append_args failed");
                        sprintf (error_name, "%s.InternalError", dhcdbd_interface_prefix);
                        sprintf (error_message, "dbus_svc_message_append_args failed");
                        dbus_svc_send (dho->d_if->dbus, ERROR, serial, &new_serial,
                                                   sender, path, interface, member, TYPE_STRING,
                                                   error_name, TYPE_STRING, error_message,
                                                   TYPE_INVALID);
                        return HANDLED;
                }
                if (!dbus_svc_send_message (dho->d_if->dbus, dbo.msg, &new_serial)) {
                        dhcdbd_log ("dbus_svc_send_message failed");
                        sprintf (error_name, "%s.InternalError", dhcdbd_interface_prefix);
                        sprintf (error_message, "dbus_svc_send_message failed");
                        dbus_svc_send (dho->d_if->dbus, ERROR, serial, &new_serial,
                                                   sender, path, interface, member, TYPE_STRING,
                                                   error_name, TYPE_STRING, error_message,
                                                   TYPE_INVALID);
                        return HANDLED;
                }
        }
        return HANDLED;
}

dbus_svc_HandlerResult
dhcdbd_dbus_get_format (DBusMsgHandlerArgs)
{
        DHCDBD_Option *dho = (DHCDBD_Option *) object;
        uint32_t new_serial;
        uint8_t new = ((dho->option->flags) & DHC_F_NEW);

        dhcdbd_debug ("dhcdbd_dbus_get_format: %s %s", dho->d_if->name,
                                  dho->option->name);

        if (reply_expected) {
                dhcdbd_debug ("Sending reply  %d to %s", serial, sender);
                dbus_svc_send (dho->d_if->dbus, RETURN, serial, &new_serial, sender,
                                           path, interface, member, TYPE_BYTE,
                                           &(dho->option->code), TYPE_STRING,
                                           &(dho->option->type), TYPE_BYTE,
                                           &(dho->option->universe), TYPE_STRING,
                                           &(dhcp_universes[dho->option->universe].name),
                                           TYPE_STRING, &(dho->option->name), TYPE_BYTE, &new,
                                           TYPE_INVALID);
        }

        return HANDLED;
}

static void dhcdbd_copy_subscription_to_ifs (DHCDBD_Subscription * subs,
                                                                                         DHCDBD_Subscriber * subr);

dbus_svc_HandlerResult
dhcdbd_if_subscribe (DBusMsgHandlerArgs)
{
        DHCO dhco;
        DHC_IF *d_if;
        DHC_Option *opt = 0L;
        DHCDBD_Subscription *subs = 0L, *const *subspp, sbs;
        DHCDBD_Subscriber *subr = 0L, *const *subrpp, sbr;
        void **subscriptions;
        uint32_t new_serial = 0, success_response = 0;
        char error_name[1024] = "", error_message[1024] = "", *opt_name = 0;
        uint32_t stype = ((strcmp (member, "binary") == 0)
                                          ? DHCDBD_SUB_BINARY : ((strcmp (member, "text") == 0)
                                                                                         ? DHCDBD_SUB_TEXT
                                                                                         : DHCDBD_SUB_DBUS)
                );

        if ((strcmp (suffix, "subscribe") == 0L) && (prefixObject == object)) {
                d_if = 0L;
                dhco = object;
                subscriptions = &(prospective_subscriptions);
        } else {
                d_if = object;
                dhco = d_if->dhco;
                subscriptions = &(d_if->subscriptions);
        }

        if (strcmp (signature, "s") == 0) {
                dbus_svc_get_args (dbus, msg, TYPE_STRING, &opt_name, TYPE_INVALID);
        } else {
                snprintf (error_name, 1024, "%s.ArgumentExpected",
                                  dhcdbd_interface_prefix);
                snprintf (error_message, 1024,
                                  "dhcdbd_if_subscribe requires an option name argument.");
                dbus_svc_send (dbus, ERROR, serial, &new_serial, sender, path,
                                           interface, member, TYPE_STRING, error_name,
                                           TYPE_STRING, error_message, TYPE_INVALID);
                return HANDLED;
        }

        if ((opt_name == 0L)
                || ((opt = dhco_find_by_name (dhco, opt_name)) == 0L)
                ) {
                snprintf (error_name, 1024, "%s.InvalidArgument",
                                  dhcdbd_interface_prefix);
                snprintf (error_message, 1024,
                                  "dhcdbd_if_subscribe: invalid option name argument: %s",
                                  opt_name);
                dbus_svc_send (dbus, ERROR, serial, &new_serial, sender, path,
                                           interface, member, TYPE_STRING, error_name,
                                           TYPE_STRING, error_message, TYPE_INVALID);
                return HANDLED;
        }

        sbs.opt = opt;
        if ((*subscriptions != 0L)
                && ((subspp = tfind (&sbs, subscriptions, dhc_subs_comparator)) != 0L)
                && ((subs = *subspp) != 0L)
                ) {
                sbr.subscriber = sender;
                if ((subs->subscribers == 0L)
                        ||
                        ((subrpp =
                          tfind (&sbr, &(subs->subscribers), dhc_subr_comparator)) == 0L)
                        || (*subrpp == 0L)
                        ) {
                        if (((subr =
                                  (DHCDBD_Subscriber *) malloc (sizeof (DHCDBD_Subscriber)))
                                 == 0L)
                                || ((subr->subscriber = (char *) malloc (strlen (sender) + 1))
                                        == 0L)
                                || (strcpy (subr->subscriber, sender) == 0L)
                                || ((subr->subs = subs) == 0L)
                                || (((subr->type = stype) > DHCDBD_SUB_TEXT))
                                || (tsearch (subr, &(subs->subscribers), dhc_subr_comparator)
                                        == 0L)
                                ) {
                                snprintf (error_name, 1024, "%s.InternalError",
                                                  dhcdbd_interface_prefix);
                                snprintf (error_message, 1024,
                                                  "dhcdbd_if_subscribe failed - out of memory?");
                                dbus_svc_send (dbus, ERROR, serial, &new_serial, sender, path,
                                                           interface, member, TYPE_STRING, error_name,
                                                           TYPE_STRING, error_message, TYPE_INVALID);
                                return HANDLED;
                        }
                }
        } else {
                if (((subs =
                          (DHCDBD_Subscription *) calloc (1,
                                                                                          sizeof (DHCDBD_Subscription)))
                         == 0L)
                        || ((subs->opt = opt) == 0L)
                        || (tsearch (subs, subscriptions, dhc_subs_comparator) == 0L)
                        ||
                        ((subr =
                          (DHCDBD_Subscriber *) malloc (sizeof (DHCDBD_Subscriber))) ==
                         0L)
                        || ((subr->subscriber = (char *) malloc (strlen (sender) + 1)) ==
                                0L)
                        || ((subr->subs = subs) == 0L)
                        || (((subr->type = stype) > DHCDBD_SUB_TEXT))
                        || (strcpy (subr->subscriber, sender) == 0L)
                        || (tsearch (subr, &(subs->subscribers), dhc_subr_comparator) ==
                                0L)
                        ) {
                        snprintf (error_name, 1024, "%s.InternalError",
                                          dhcdbd_interface_prefix);
                        snprintf (error_message, 1024,
                                          "dhcdbd_if_subscribe failed - out of memory.");
                        dbus_svc_send (dbus, ERROR, serial, &new_serial, sender, path,
                                                   interface, member, TYPE_STRING, error_name,
                                                   TYPE_STRING, error_message, TYPE_INVALID);
                        return HANDLED;
                }
        }

        if ((subscriptions == &(prospective_subscriptions))
                && (subs != 0L)
                && (subr != 0L)
                )
                dhcdbd_copy_subscription_to_ifs (subs, subr);

        dhcdbd_debug
                ("handled successful subscription: subs:%p subr:%p %s type: %d if:%s opt:%s %s ",
                 subs, subr, subr->subscriber, subr->type,
                 (d_if != 0L) ? d_if->name : "prospective", opt_name, if_suffix);

        dbus_svc_send (dbus, RETURN, serial, &new_serial, sender, path, interface,
                                   member, TYPE_UINT32, &success_response, TYPE_INVALID);
        return HANDLED;
}

static void
dhcdbd_copy_subscriber (const void *p, const VISIT which, const int level)
{
        DHCDBD_Subscriber *subr, *subrc, *const *sbrpp = p;

        if (((which != leaf) && (which != postorder))
                || (sbrpp == 0L)
                || ((subr = *sbrpp) == 0L)
                )
                return;

        if (((sbrpp =
                  tfind (subr, &(subscribers_copy->subscribers),
                                 dhc_subs_comparator)) != 0L)
                && ((subrc = *sbrpp) != 0L)
                && (subrc->type == subr->type)
                )
                return;

        subrc = (DHCDBD_Subscriber *) malloc (sizeof (DHCDBD_Subscriber));
        if (subrc == 0L) {
                dhcdbd_log ("Out of memory");
                return;
        }
        *subrc = *subr;
        subrc->subs = subscribers_copy;
        subrc->subscriber = (char *) malloc (strlen (subr->subscriber) + 1);
        if (subrc->subscriber == 0L) {
                dhcdbd_log ("Out of memory");
                return;
        }
        strcpy (subrc->subscriber, subr->subscriber);
        dhcdbd_debug ("copy new subscriber: %p %s of %p %s", subrc,
                                  subrc->subscriber, subrc->subs, subrc->subs->opt->name);
        tsearch (subrc, &(subscribers_copy->subscribers), dhc_subs_comparator);
}

static void
dhcdbd_copy_subscription (const void *p, const VISIT which, const int level)
{
        DHCDBD_Subscription *subs = 0L, *subc = 0L, *subcc = 0L, *const *spp = p;

        if (((which != leaf) && (which != postorder))
                || (spp == 0L)
                || ((subs = *spp) == 0L)
                )
                return;

        if (((spp = tfind (subs, subscription_copy, dhc_subs_comparator)) != 0L)
                && ((subcc = *spp) != 0L)
                ) {
                subscribers_copy = subcc;
                dhcdbd_debug ("Copy subs %p to subs %p of %s", subs, subcc,
                                          subs->opt->name);
                twalk (subs->subscribers, dhcdbd_copy_subscriber);
                subscribers_copy = 0L;
                return;
        }

        subc = (DHCDBD_Subscription *) malloc (sizeof (DHCDBD_Subscription));
        if (subc == 0L) {
                dhcdbd_log ("Out of memory");
                return;
        }

        *subc = *subs;
        subc->subscribers = 0L;

        spp = tsearch (subc, subscription_copy, dhc_subs_comparator);
        if ((spp == 0L) || ((subcc = *spp) == 0L)) {
                free (subc);
                dhcdbd_log ("Out of memory");
                return;
        }

        subscribers_copy = subc;
        dhcdbd_debug ("Copy subs %p to new subs %p of %s", subs, subc,
                                  subs->opt->name);
        twalk (subs->subscribers, dhcdbd_copy_subscriber);
        subscribers_copy = 0L;
}

void
dhcdbd_copy_subscriptions (DHC_IF * d_if)
{
        subscription_copy = &(d_if->subscriptions);
        twalk (prospective_subscriptions, dhcdbd_copy_subscription);
        subscription_copy = 0L;
}

static void
dhcdbd_copy_if_subscription (const void *p, const VISIT which,
                                                         const int level)
{
        DHC_IF *d_if, *const *ifpp = p;
        DHCDBD_Subscription *subs = subscribers_copy, *subsc, *const *spp;
        DHCDBD_Subscriber *subr = subscriber_copy, *subrc, *const *sbrpp;

        if (((which != leaf) && (which != postorder))
                || (ifpp == 0L)
                || ((d_if = *ifpp) == 0L)
                )
                return;

        if (((spp =
                  tfind (subs, &(d_if->subscriptions), dhc_subs_comparator)) == 0L)
                || ((subsc = *spp) == 0L)
                ) {
                subsc = (DHCDBD_Subscription *) malloc (sizeof (DHCDBD_Subscription));
                if (subsc == 0L) {
                        dhcdbd_log ("Out of memory");
                        return;
                }
                *subsc = *subs;
                subsc->subscribers = 0L;
                tsearch (subsc, &(d_if->subscriptions), dhc_subs_comparator);
        }

        if (((sbrpp =
                  tfind (subr, &(subsc->subscribers), dhc_subr_comparator)) == 0L)
                || ((subrc = *sbrpp) == 0L)
                ) {
                subrc = (DHCDBD_Subscriber *) malloc (sizeof (DHCDBD_Subscriber));
                if (subrc == 0L) {
                        dhcdbd_log ("Out of memory");
                        return;
                }
                *subrc = *subr;
                subrc->subscriber = (char *) malloc (strlen (subr->subscriber) + 1);
                if (subrc->subscriber == 0L) {
                        dhcdbd_log ("Out of memory");
                        return;
                }
                strcpy (subrc->subscriber, subr->subscriber);
                subrc->subs = subsc;
                tsearch (subrc, &(subsc->subscribers), dhc_subr_comparator);
                dhcdbd_debug ("copy if new subscriber: %p %s of %s", subrc,
                                          subrc->subscriber, subrc->subs->opt->name);
        }
}

static void
dhcdbd_copy_subscription_to_ifs (DHCDBD_Subscription * subs,
                                                                 DHCDBD_Subscriber * subr)
{
        subscribers_copy = subs;
        subscriber_copy = subr;
        twalk (if_tree, dhcdbd_copy_if_subscription);
        subscribers_copy = 0L;
        subscriber_copy = 0L;
}

void
dhcdbd_signal_subscribers (const void *p, const VISIT which, const int level)
{
        DHCDBD_Subscriber *subr, *const *sbrpp = (DHCDBD_Subscriber * const *) p;
        DHCDBD_Subscription *subs;
        char subs_interface[1024], path[1024];
        DHCDBD_DBus_Option_Message dbo;
        uint32_t new_serial;
        void *vptr[1];

        if ((sbrpp == 0L) || ((subr = *sbrpp) == 0L)
                || ((subs = subr->subs) == 0L)
                || (subs->dho == 0L)
                || ((which != leaf) && (which != postorder))
                )
                return;

        dhcdbd_debug ("Sending %s to Subscriber %s", subs->opt->name,
                                  subr->subscriber);

        snprintf (path, 1024, "%s/%s", dhcdbd_object_path, subs->dho->d_if->name);

        switch (subr->type) {
                case DHCDBD_SUB_DBUS:

                        snprintf (subs_interface, 1024, "%s.subscribe.dbus",
                                          dhcdbd_interface_prefix);

                        if ((dbo.msg =
                                 dbus_svc_new_message
                                 (subs->dho->d_if->dbus, SIGNAL, -1,
                                  subr->subscriber, path, subs_interface, subs->dho->qname)
                                ) == 0L) {
                                dhcdbd_log
                                        ("dhcdbd_signal_subscriber: dbus_svc_new_message failed");
                                return;
                        }

                        if (!dbus_svc_message_append_args
                                (subs->dho->d_if->dbus, dbo.msg,
                                 TYPE_STRING, &(subs->opt->name), TYPE_INVALID)
                                ) {
                                dhcdbd_log
                                        ("dhcdbd_signal_subscriber: dbus_svc_message_append_args failed");
                                return;
                        }

                        dbo.dbus = subs->dho->d_if->dbus;
                        dbo.ok = 1;

                        dhco_map_option (subs->dho->d_if->dhco, subs->opt,
                                                         subs->dho->value, subs->dho->length,
                                                         subs->dho->d_if->mapping,
                                                         dhcdbd_dbus_option_type, &dbo);

                        if (dbo.ok != 1) {
                                dhcdbd_log
                                        ("dhcdbd_signal_subscriber: dbus_svc_message_append_args failed");
                                return;
                        }

                        if (!dbus_svc_send_message
                                (subs->dho->d_if->dbus, dbo.msg, &new_serial))
                                dhcdbd_log
                                        ("dhcdbd_signal_subscriber: dbus_svc_send_message failed");
                        return;

                case DHCDBD_SUB_BINARY:
                        snprintf (subs_interface, 1024, "%s.subscribe.binary",
                                          dhcdbd_interface_prefix);
                        vptr[0] = subs->dho->value;
                        if (!dbus_svc_send
                                (subs->dho->d_if->dbus, SIGNAL, -1, &new_serial,
                                 subr->subscriber,
                                 path,
                                 subs_interface,
                                 subs->dho->qname,
                                 TYPE_STRING, &(subs->opt->name),
                                 TYPE_ARRAY, TYPE_BYTE, &vptr, subs->dho->length,
                                 TYPE_INVALID)
                                ) {
                                dhcdbd_log
                                        ("dhcdbd_signal_subscriber: dbus_svc_send_message failed");
                        }
                        return;

                case DHCDBD_SUB_TEXT:

                        snprintf (subs_interface, 1024, "%s.subscribe.text",
                                          dhcdbd_interface_prefix);

                        if (!dbus_svc_send
                                (subs->dho->d_if->dbus, SIGNAL, -1, &new_serial,
                                 subr->subscriber,
                                 path,
                                 subs_interface,
                                 subs->dho->qname,
                                 TYPE_STRING, &(subs->opt->name),
                                 TYPE_STRING, &(subs->dho->output)
                                )
                                ) {
                                dhcdbd_log
                                        ("dhcdbd_signal_subscriber: dbus_svc_send_message failed");
                        }
        }
}

void
dhcdbd_find_dead_subscribers (const void *p, const VISIT which,
                                                          const int level)
{
        DHCDBD_Subscriber *subr, *dead_subr, *const *sbrpp =
                (DHCDBD_Subscriber * const *) p;
        DHCDBD_Subscription *subs;
        uint32_t subscriberValid = 0;
        dbus_svc_MessageHandle msg;
        char *sp;

        if ((sbrpp == 0L) || ((subr = *sbrpp) == 0L)
                || ((subs = subr->subs) == 0L)
                || (subs->dho == 0L)
                || ((which != leaf) && (which != postorder))
                )
                return;

        subs = subr->subs;

        if (tfind
                (subr, &(subs->dho->d_if->live_subscribers),
                 dhc_subr_comparator) != 0L)
                return;

        if (((sbrpp =
                  tfind (subr, &(subs->dho->d_if->dead_subscribers),
                                 dhc_subr_comparator)) != 0L)
                && ((dead_subr = *sbrpp) != 0L)
                ) {
                dead_subr->subs = subs;
                return;
        }

        dhcdbd_debug ("Checking subscriber: %s %p %p", subr->subscriber,
                                  subs->dho->d_if->dead_subscribers,
                                  subs->dho->d_if->live_subscribers);

        if ((msg = dbus_svc_call
                 (subs->dho->d_if->dbus,
                  "org.freedesktop.DBus",
                  "/org/freedesktop/DBus",
                  "NameHasOwner",
                  "org.freedesktop.DBus",
                  TYPE_STRING, &(subr->subscriber), TYPE_INVALID)
                ) == 0L)
                dhcdbd_debug ("org.freedesktop.DBus.NameHasOwner call failed");
        else {
                if (!dbus_svc_get_args
                        (subs->dho->d_if->dbus,
                         msg, TYPE_BOOLEAN, &subscriberValid, TYPE_INVALID)
                        ) {
                        dhcdbd_log ("dbus_svc_get_args failed");
                }
        }

        if (subscriberValid == 0) {
                if (((dead_subr =
                          (DHCDBD_Subscriber *) malloc (sizeof (DHCDBD_Subscriber))) ==
                         0L)
                        || ((sp = (char *) malloc (strlen (subr->subscriber) + 1)) == 0L)
                        ) {
                        dhcdbd_log ("Out of memory");
                        return;
                }
                *dead_subr = *subr;
                dead_subr->subscriber = sp;
                strcpy (dead_subr->subscriber, subr->subscriber);
                tsearch (dead_subr, &(subs->dho->d_if->dead_subscribers),
                                 dhc_subr_comparator);
                dhcdbd_debug ("Subscriber %p %s DEAD", dead_subr,
                                          dead_subr->subscriber);
        } else {
                tsearch (subr, &(subs->dho->d_if->live_subscribers),
                                 dhc_subr_comparator);
                dhcdbd_debug ("Subscriber %s OK", subr->subscriber);
        }
}

void
dhcdbd_free_subscriber (void *p)
{
        DHCDBD_Subscriber *subr = p;

        dhcdbd_debug ("free subscriber %p", subr);
        free (subr->subscriber);
        free (subr);
}

void
dhcdbd_remove_dead_subscriber (const void *p, const VISIT which,
                                                           const int level)
{
        DHCDBD_Subscriber *dsubr, *subr = 0L, *const *sbrpp =
                (DHCDBD_Subscriber * const *) p;
        if ((sbrpp == 0L) || ((dsubr = *sbrpp) == 0L)
                || ((which != leaf) && (which != postorder))
                )
                return;
        if (dsubr->subs != 0L) {
                if (((sbrpp =
                          tfind (dsubr, &(dsubr->subs->subscribers),
                                         dhc_subr_comparator)) == 0L)
                        || ((subr = *sbrpp) == 0L)
                        )
                        return;
                dhcdbd_debug ("Removing dead subscriber %p %s from option %p %s",
                                          subr, subr->subscriber, subr->subs,
                                          subr->subs->dho->option->name);
                tdelete (subr, &(subr->subs->subscribers), dhc_subr_comparator);
                dsubr->subs = 0L;
        }
        if (subr != 0L)
                dhcdbd_free_subscriber (subr);
}

void
dhcdbd_inform_option_subscribers (DHCDBD_Option * opt)
{
        DHCDBD_Subscription *subs, *const *sbpp, sbs;

        sbs.opt = opt->option;
        if (((sbpp =
                  tfind (&sbs, &(opt->d_if->subscriptions),
                                 dhc_subs_comparator)) != 0L)
                && ((subs = *sbpp) != 0L)
                ) {
                if (subs->subscribers == 0L) {
                        dhcdbd_debug ("Removing empty subscription %p for %s", subs,
                                                  opt->option->name);
                        tdelete (subs, &(opt->d_if->subscriptions), dhc_subs_comparator);
                        free (subs);
                        return;
                }
                subs->dho = opt;
                dhcdbd_debug ("inform option %s subscribers", opt->option->name);
                twalk (subs->subscribers, dhcdbd_find_dead_subscribers);
                if (opt->d_if->dead_subscribers != 0L) {
                        twalk (opt->d_if->dead_subscribers,
                                   dhcdbd_remove_dead_subscriber);
                }
                twalk (subs->subscribers, dhcdbd_signal_subscribers);
        }
}

void
dhcdbd_purge_empty_prospective_subscription (const void *p, const VISIT which,
                                                                                         const int level)
{
        DHCDBD_Subscription *subs, *const *sbpp =
                (DHCDBD_Subscription * const *) p;
        if ((sbpp == 0L) || ((subs = *sbpp) == 0L)
                || ((which != leaf) && (which != postorder))
                )
                return;

        tdelete (subs, &(prospective_subscriptions), dhc_subs_comparator);
        dhcdbd_debug ("free prospective subscription %p", subs);
        free (subs);
}

void
dhcdbd_purge_dead_prospective_subscriber (const void *p, const VISIT which,
                                                                                  const int level)
{
        DHCDBD_Subscription *subs, *const *sbpp =
                (DHCDBD_Subscription * const *) p;
        DHCDBD_Subscriber *subr, *const *sbrpp;

        if ((sbpp == 0L) || ((subs = *sbpp) == 0L)
                || ((which != leaf) && (which != postorder))
                )
                return;

        if (((sbrpp =
                  tfind (subscriber_copy, &(subs->subscribers),
                                 dhc_subr_comparator)) == 0L)
                || ((subr = *sbrpp) == 0L)
                )
                return;

        tdelete (subr, &(subs->subscribers), dhc_subr_comparator);

        if (subs->subscribers == 0L)
                tsearch (subs, &(empty_prospective_subscriptions),
                                 dhc_subs_comparator);

        dhcdbd_debug ("Removed dead p-subr %p %s from option %p %s", subr,
                                  subr->subscriber, subr->subs, subr->subs->opt->name);
        dhcdbd_free_subscriber (subr);
}

void
dhcdbd_purge_dead_prospective_subscribers (const void *p, const VISIT which,
                                                                                   const int level)
{
        DHCDBD_Subscriber *subr, *const *sbrpp = (DHCDBD_Subscriber * const *) p;

        if ((sbrpp == 0L) || ((subr = *sbrpp) == 0L)
                || ((which != leaf) && (which != postorder))
                )
                return;
        subscriber_copy = subr;
        twalk (prospective_subscriptions,
                   dhcdbd_purge_dead_prospective_subscriber);
        subscriber_copy = 0L;
}

void
dhcdbd_inform_subscribers (const void *p, const VISIT which, const int level)
{
        DHCDBD_Option *opt, *const *opp = p;

        if ((opp == 0L) || ((opt = *opp) == 0L)
                || ((which != leaf) && (which != postorder))
                )
                return;
        if ((opt->option->universe == DHC_P_Universe)
                && (opt->option->code == DHCP_DHCP_STATE)
                )
                return;
        dhcdbd_inform_option_subscribers (opt);
}

void
dhcdbd_free_option (void *p)
{
        DHCDBD_Option *opt = (DHCDBD_Option *) p;

        if (opt->output != 0L)
                free (opt->output);
        opt->output = 0L;
        if (opt->value != 0L)
                free (opt->value);
        opt->value = 0L;
        if (opt->qname != 0L)
                free (opt->qname);
        opt->qname = 0L;
        free (opt);
}

void
dhcdbd_remove_option_handler (const void *p, const VISIT which,
                                                          const int level)
{
        DHCDBD_Option *opt, *const *opp = (DHCDBD_Option * const *) p;
        char path[1024];

        if ((opp == 0L) || ((opt = *opp) == 0L)
                || ((which != postorder) && (which != leaf))
                )
                return;

        snprintf (path, 1024, "%s.text.get.%s", opt->d_if->name, opt->qname);
        dbus_svc_remove_handler (opt->d_if->dbus, dhcdbd_object_path, path, 0L);
        snprintf (path, 1024, "%s.binary.get.%s", opt->d_if->name, opt->qname);
        dbus_svc_remove_handler (opt->d_if->dbus, dhcdbd_object_path, path, 0L);
        snprintf (path, 1024, "%s.dbus.get.%s", opt->d_if->name, opt->qname);
        dbus_svc_remove_handler (opt->d_if->dbus, dhcdbd_object_path, path, 0L);
        snprintf (path, 1024, "%s.text.format.get.%s", opt->d_if->name,
                          opt->qname);
        dbus_svc_remove_handler (opt->d_if->dbus, dhcdbd_object_path, path, 0L);
        snprintf (path, 1024, "%s.binary.format.get.%s", opt->d_if->name,
                          opt->qname);
        dbus_svc_remove_handler (opt->d_if->dbus, dhcdbd_object_path, path, 0L);
        snprintf (path, 1024, "%s.dbus.format.get.%s", opt->d_if->name,
                          opt->qname);
        dbus_svc_remove_handler (opt->d_if->dbus, dhcdbd_object_path, path, 0L);
}

void
dhcdbd_remove_old_option_handler (const void *p, const VISIT which,
                                                                  const int level)
{
        DHCDBD_Option *opt, *const *opp = (DHCDBD_Option * const *) p;
        char path[1024];

        if ((opp == 0L) || ((opt = *opp) == 0L)
                || ((which != postorder) && (which != leaf))
                )
                return;

        snprintf (path, 1024, "%s.text.get.old.%s", opt->d_if->name, opt->qname);
        dbus_svc_remove_handler (opt->d_if->dbus, dhcdbd_object_path, path, 0L);
        snprintf (path, 1024, "%s.binary.get.old.%s", opt->d_if->name,
                          opt->qname);
        dbus_svc_remove_handler (opt->d_if->dbus, dhcdbd_object_path, path, 0L);
        snprintf (path, 1024, "%s.dbus.get.old.%s", opt->d_if->name, opt->qname);
        dbus_svc_remove_handler (opt->d_if->dbus, dhcdbd_object_path, path, 0L);
        snprintf (path, 1024, "%s.text.format.old.get.%s", opt->d_if->name,
                          opt->qname);
        dbus_svc_remove_handler (opt->d_if->dbus, dhcdbd_object_path, path, 0L);
        snprintf (path, 1024, "%s.binary.format.old.get.%s", opt->d_if->name,
                          opt->qname);
        dbus_svc_remove_handler (opt->d_if->dbus, dhcdbd_object_path, path, 0L);
        snprintf (path, 1024, "%s.dbus.format.get.old.%s", opt->d_if->name,
                          opt->qname);
        dbus_svc_remove_handler (opt->d_if->dbus, dhcdbd_object_path, path, 0L);
}

void
dhcdbd_make_old_options (const void *p, const VISIT which, const int level)
{
        DHCDBD_Option *opt, *const *opp = (DHCDBD_Option * const *) p;
        char path[1024];

        if ((opp == 0L) || ((opt = *opp) == 0L)
                || ((which != postorder) && (which != leaf))
                )
                return;

        snprintf (path, 1024, "%s.text.get.old.%s", opt->d_if->name, opt->qname);
        dbus_svc_add_handler (opt->d_if->dbus, dhcdbd_object_path, path,
                                                  dhcdbd_text_get_option, opt);
        snprintf (path, 1024, "%s.binary.get.old.%s", opt->d_if->name,
                          opt->qname);
        dbus_svc_add_handler (opt->d_if->dbus, dhcdbd_object_path, path,
                                                  dhcdbd_binary_get_option, opt);
        snprintf (path, 1024, "%s.dbus.get.old.%s", opt->d_if->name, opt->qname);
        dbus_svc_add_handler (opt->d_if->dbus, dhcdbd_object_path, path,
                                                  dhcdbd_dbus_get_option, opt);
        snprintf (path, 1024, "%s.text.format.old.get.%s", opt->d_if->name,
                          opt->qname);
        dbus_svc_add_handler (opt->d_if->dbus, dhcdbd_object_path, path,
                                                  dhcdbd_text_get_format, opt);
        snprintf (path, 1024, "%s.dbus.format.get.old.%s", opt->d_if->name,
                          opt->qname);
        dbus_svc_add_handler (opt->d_if->dbus, dhcdbd_object_path, path,
                                                  dhcdbd_dbus_get_format, opt);
        snprintf (path, 1024, "%s.binary.format.old.get.%s", opt->d_if->name,
                          opt->qname);
        dbus_svc_add_handler (opt->d_if->dbus, dhcdbd_object_path, path,
                                                  dhcdbd_dbus_get_format, opt);
}

void
dhcdbd_dhcp_option_handler (DHCO dhc, DHC_Option * o, void *value, int length,
                                                        char *prefix, void *obj)
{
        char *output = 0L, path[1024], *p;
        DHC_IF *dhc_if = (DHC_IF *) obj;
        DHCDBD_Option *opt = (DHCDBD_Option *) malloc (sizeof (DHCDBD_Option));
        DHC_Universe *du = dhco_find_universe_by_code (dhc_if->dhco, o->universe);
        uint32_t len;
        char pfx[16] = "";

        dhcdbd_debug
                ("Option Handler: opt:%p d_if:%p %s Got option %s prefix:%s code:%d type:%s length:%d v:%p obj:%p\n",
                 opt, dhc_if, dhc_if->name, o->name, prefix, o->code, o->type, length,
                 value, obj);
        dhco_output_option (dhc, o, value, length, &output, 0);
        if (opt == 0L)
                return;
        memset (opt, '\0', sizeof (opt));
        opt->d_if = dhc_if;
        opt->option = o;
        opt->output = output;
        opt->value = value;
        opt->length = length;
        if ((du->code != DHC_O_Universe) && (du->code != DHC_P_Universe))
                len = snprintf (path, 1024, "%s.%s", du->name, o->name) + 1;
        else
                len = snprintf (path, 1024, "%s", o->name) + 1;
        opt->qname = (char *) malloc (len);
        strncpy (opt->qname, path, len);
        for (p = opt->qname; (p != 0L) && (*p != '\0') && (p = strchr (p, '-'));
                 p++)
                *p = '_';                               /* AARGH! D-BUS doesn't think names with '-'s 
                                                                 * are proper member names. */
        if ((prefix != 0L) && (strcmp (prefix, "old") == 0)) {
                tsearch (opt, &(dhc_if->old_options), dhcdbd_option_comparator);
                sprintf (pfx, "%s.", prefix);
        } else {
                pfx[0] = '\0';
                tsearch (opt, &(dhc_if->options), dhcdbd_option_comparator);
        }

        snprintf (path, 1024, "%s.text.get.%s%s", dhc_if->name, pfx, opt->qname);
        if (!dbus_svc_add_handler
                (dhc_if->dbus, dhcdbd_object_path, path, dhcdbd_text_get_option,
                 opt)) {
                dhcdbd_log
                        ("dhcdbd_dhcp_option_handler: failed to register handler: %s %s",
                         dhcdbd_object_path, path);
        }

        snprintf (path, 1024, "%s.binary.get.%s%s", dhc_if->name, pfx,
                          opt->qname);
        if (!dbus_svc_add_handler
                (dhc_if->dbus, dhcdbd_object_path, path, dhcdbd_binary_get_option,
                 opt)) {
                dhcdbd_log
                        ("dhcdbd_dhcp_option_handler: failed to register handler: %s %s",
                         dhcdbd_object_path, path);
                return;
        }

        snprintf (path, 1024, "%s.dbus.get.%s%s", dhc_if->name, pfx, opt->qname);
        if (!dbus_svc_add_handler
                (dhc_if->dbus, dhcdbd_object_path, path, dhcdbd_dbus_get_option,
                 opt)) {
                dhcdbd_log
                        ("dhcdbd_dhcp_option_handler: failed to register handler: %s %s",
                         dhcdbd_object_path, path);
                return;
        }

        snprintf (path, 1024, "%s.text.format.get.%s%s", dhc_if->name, pfx,
                          opt->qname);
        if (!dbus_svc_add_handler
                (dhc_if->dbus, dhcdbd_object_path, path, dhcdbd_text_get_format,
                 opt)) {
                dhcdbd_log
                        ("dhcdbd_dhcp_option_handler: failed to register handler: %s %s",
                         dhcdbd_object_path, path);
                return;
        }

        snprintf (path, 1024, "%s.binary.format.get.%s%s", dhc_if->name, pfx,
                          opt->qname);
        if (!dbus_svc_add_handler
                (dhc_if->dbus, dhcdbd_object_path, path, dhcdbd_dbus_get_format,
                 opt)) {
                dhcdbd_log
                        ("dhcdbd_dhcp_option_handler: failed to register handler: %s %s",
                         dhcdbd_object_path, path);
                return;
        }

        snprintf (path, 1024, "%s.dbus.format.get.%s%s", dhc_if->name, pfx,
                          opt->qname);
        if (!dbus_svc_add_handler
                (dhc_if->dbus, dhcdbd_object_path, path, dhcdbd_dbus_get_format,
                 opt)) {
                dhcdbd_log
                        ("dhcdbd_dhcp_option_handler: failed to register handler: %s %s",
                         dhcdbd_object_path, path);
                return;
        }
}

dbus_svc_HandlerResult
dhcdbd_if_set (DBusMsgHandlerArgs)
{
        DHC_IF *d_if = (DHC_IF *) object;
        DHCDBD_Option *opt = 0L, *const *opp, os;
        char error_name[1024] = "", error_message[1024] = "";
        uint32_t success_response = 0, es = 0, new_serial;
        char *dhcp_option_settings = 0L, est[] = "END_OPTIONS", *ovs;
        void *ov, *old_options = 0L;

        dhcdbd_debug ("dhcdbd_if_set: %p %s", d_if, d_if->name);
        if (strcmp (signature, "s") != 0) {
                return NOT_HANDLED;
        }
        if (!dbus_svc_get_args
                (dbus, msg, TYPE_STRING, &dhcp_option_settings, TYPE_INVALID)) {
                dhcdbd_log ("Get Args failed");
                return NOT_HANDLED;
        }

        if (d_if->options != 0L) {
                old_options = d_if->options;
                twalk (d_if->options, dhcdbd_remove_option_handler);
                d_if->options = 0L;
        }

        if (d_if->old_options != 0L) {
                twalk (d_if->old_options, dhcdbd_remove_old_option_handler);
                tdestroy (d_if->old_options, dhcdbd_free_option);
                d_if->old_options = 0L;
        }

        dhcdbd_debug ("Got option settings: %s - d_if: %p", dhcp_option_settings,
                                  d_if);
        success_response =
                dhco_parse_option_settings (d_if->dhco, dhcp_option_settings,
                                                                        dhcdbd_dhcp_option_handler, d_if);

        if (d_if->old_options == 0L) {  /* REBOOT state - all options are * "new" 
                                                                         * - make last "new" options * new "old"
                                                                         * options: */
                if (old_options != 0L)
                        twalk (old_options, dhcdbd_make_old_options);
        } else if (old_options != 0L)
                tdestroy (old_options, dhcdbd_free_option);

        os.qname = "reason";
        if (((opp =
                  tfind (&os, &(d_if->options), dhcdbd_option_comparator)) != 0L)
                && ((opt = *opp) != 0L)
                ) {
                dhcdbd_if_state_change (d_if, *((uint32_t *) (opt->value)),
                                                                opt->output);
        }

        if (d_if->subscriptions != 0L) {
                if (d_if->dead_subscribers != 0L) {
                        if (prospective_subscriptions != 0L) {
                                empty_prospective_subscriptions = 0L;
                                twalk (d_if->dead_subscribers,
                                           dhcdbd_purge_dead_prospective_subscribers);
                                if (empty_prospective_subscriptions != 0L) {
                                        twalk (empty_prospective_subscriptions,
                                                   dhcdbd_purge_empty_prospective_subscription);
                                        tdestroy (empty_prospective_subscriptions,
                                                          dhcdbd_no_free);
                                        empty_prospective_subscriptions = 0L;
                                }
                        }
                        tdestroy (d_if->dead_subscribers, dhcdbd_free_subscriber);
                        d_if->dead_subscribers = 0L;
                }

                if (d_if->live_subscribers != 0L) {
                        tdestroy (d_if->live_subscribers, dhcdbd_no_free);
                        d_if->live_subscribers = 0L;
                }

                if (opt != 0L)
                        dhcdbd_inform_option_subscribers (opt);

                twalk (d_if->options, dhcdbd_inform_subscribers);

                es = DHC_END_OPTIONS;
                ov = opt->value;
                ovs = opt->output;
                opt->value = (void *) &es;
                opt->output = &(est[0]);
                dhcdbd_inform_option_subscribers (opt);
                opt->value = ov;
                opt->output = ovs;
        }

        if (reply_expected) {
                dhcdbd_debug ("Sending reply  %d to %s", serial, sender);
                if (error_name[0] != '\0')
                        dbus_svc_send (dbus, ERROR, serial, &new_serial, sender, path,
                                                   interface, member, TYPE_STRING, error_name,
                                                   TYPE_STRING, error_message, TYPE_INVALID);
                else
                        dbus_svc_send (dbus, RETURN, serial, &new_serial, sender, path,
                                                   interface, member, TYPE_UINT32, &success_response,
                                                   TYPE_INVALID);
        }
        return HANDLED;
}

void
dhcdbd_if_text_list (const void *p, const VISIT which, const int level)
{
        DHCDBD_Option *dho, *const *dhopp = p;

        if ((dhopp == 0L) || ((dho = *dhopp) == 0L)
                || ((which != leaf) && (which != postorder))
                || (dho->d_if->list_msg->ok != 1)
                )
                return;

        dho->d_if->list_msg->ok =
                dbus_svc_message_append_args
                (dho->d_if->dbus, dho->d_if->list_msg->msg,
                 TYPE_STRING, &(dho->option->name),
                 TYPE_STRING, &(dho->output), TYPE_INVALID);
}

void
dhcdbd_if_binary_list (const void *p, const VISIT which, const int level)
{
        DHCDBD_Option *dho, *const *dhopp = p;
        uint8_t *vptr[1];

        if ((dhopp == 0L) || ((dho = *dhopp) == 0L)
                || ((which != leaf) && (which != postorder))
                || (dho->d_if->list_msg->ok != 1)
                )
                return;
        vptr[0] = (uint8_t *) (dho->value);
        dho->d_if->list_msg->ok =
                dbus_svc_message_append_args
                (dho->d_if->dbus, dho->d_if->list_msg->msg,
                 TYPE_STRING, &(dho->option->name),
                 TYPE_ARRAY, TYPE_BYTE, &vptr, dho->length, TYPE_INVALID);
}

void
dhcdbd_if_dbus_list (const void *p, const VISIT which, const int level)
{
        DHCDBD_Option *dho, *const *dhopp = p;
        DHCDBD_DBus_Option_Message dbo;

        if ((dhopp == 0L) || ((dho = *dhopp) == 0L)
                || ((which != leaf) && (which != postorder))
                || (dho->d_if->list_msg->ok != 1)
                )
                return;
        if ((dho->d_if->list_msg->ok =
                 dbus_svc_message_append_args
                 (dho->d_if->dbus, dho->d_if->list_msg->msg,
                  TYPE_STRING, &(dho->option->name), TYPE_INVALID)
                ) != 1)
                return;
        dbo.dbus = dho->d_if->dbus;
        dbo.ok = 1;
        dbo.msg = dho->d_if->list_msg->msg;
        dhco_map_option (dho->d_if->dhco, dho->option,
                                         dho->value, dho->length,
                                         dho->d_if->mapping, dhcdbd_dbus_option_type, &dbo);
}

dbus_svc_HandlerResult
dhcdbd_if_option_list (DBusMsgHandlerArgs,
                                           void (*lister) (const void *p, const VISIT which,
                                                                           const int level))
{
        DHC_IF *d_if = object;
        char error_name[1024] = "", error_message[1024] = "";
        uint32_t new_serial;

        dhcdbd_debug ("dhcdbd_if_text_get: %s", d_if->name);
        if (reply_expected) {
                dhcdbd_debug ("Sending reply  %d to %s", serial, sender);
                if ((d_if->list_msg->msg =
                         dbus_svc_new_message
                         (d_if->dbus, RETURN, serial, sender, path, interface, member)
                        ) == 0L) {
                        dhcdbd_log ("dbus_svc_new_message failed");
                        sprintf (error_name, "%s.InternalError", dhcdbd_interface_prefix);
                        sprintf (error_message, "dbus_svc_new_message failed");
                        dbus_svc_send (d_if->dbus, ERROR, serial, &new_serial, sender,
                                                   path, interface, member, TYPE_STRING, error_name,
                                                   TYPE_STRING, error_message, TYPE_INVALID);
                        d_if->list_msg->msg = 0L;
                        return HANDLED;
                }
                d_if->list_msg->ok = 1;
                twalk (d_if->options, lister);
                if (d_if->list_msg->ok == 1) {
                        if (!dbus_svc_send_message
                                (d_if->dbus, d_if->list_msg->msg, &new_serial)) {
                                dhcdbd_log ("dbus_svc_send_message failed");
                                sprintf (error_name, "%s.InternalError",
                                                 dhcdbd_interface_prefix);
                                sprintf (error_message, "dhdbd_if_text_list failed");
                                dbus_svc_send (d_if->dbus, ERROR, serial, &new_serial, sender,
                                                           path, interface, member, TYPE_STRING,
                                                           error_name, TYPE_STRING, error_message,
                                                           TYPE_INVALID);
                        }
                        d_if->list_msg->msg = 0L;
                        return HANDLED;
                }
        }
        return HANDLED;
}

dbus_svc_HandlerResult
dhcdbd_if_text_get (DBusMsgHandlerArgs)
{
        return dhcdbd_if_option_list
                (dbus, type, reply_expected, serial,
                 destination, path, member, interface,
                 if_suffix, sender, signature, msg,
                 prefix, suffix, prefixObject, object, dhcdbd_if_text_list);
}

dbus_svc_HandlerResult
dhcdbd_if_binary_get (DBusMsgHandlerArgs)
{
        return dhcdbd_if_option_list
                (dbus, type, reply_expected, serial,
                 destination, path, member, interface,
                 if_suffix, sender, signature, msg,
                 prefix, suffix, prefixObject, object, dhcdbd_if_binary_list);
}

dbus_svc_HandlerResult
dhcdbd_if_dbus_get (DBusMsgHandlerArgs)
{
        return dhcdbd_if_option_list
                (dbus, type, reply_expected, serial,
                 destination, path, member, interface,
                 if_suffix, sender, signature, msg,
                 prefix, suffix, prefixObject, object, dhcdbd_if_dbus_list);
}

void
dhcdbd_list_an_if (const void *p, const VISIT which, const int level)
{
        DHC_IF *d_if, *const *dpp = p;

        if ((dpp == 0L) || ((d_if = *dpp) == 0L)
                || ((which != leaf) && (which != postorder))
                )
                return;

        dbus_svc_message_append_args
                (d_if->dbus, if_list_msg, TYPE_STRING, &(d_if->name), TYPE_INVALID);
}

dbus_svc_HandlerResult
dhcdbd_list_if (DBusMsgHandlerArgs)
{
        char error_name[1024] = "", error_message[1024] = "";
        uint32_t new_serial;

        if (reply_expected) {
                if (if_tree == 0L) {
                        sprintf (error_name, "%s.InvalidOperation",
                                         dhcdbd_interface_prefix);
                        sprintf (error_message, "there are no interfaces configured.");
                        dbus_svc_send (dbus, ERROR, serial, &new_serial, sender, path,
                                                   interface, member, TYPE_STRING, error_name,
                                                   TYPE_STRING, error_message, TYPE_INVALID);
                        if_list_msg = 0L;
                        return HANDLED;
                }

                if ((if_list_msg =
                         dbus_svc_new_message
                         (dbus, RETURN, serial, sender, path, interface, member)
                        ) == 0) {
                        dhcdbd_log ("dbus_svc_new_message failed");
                        sprintf (error_name, "%s.InternalError", dhcdbd_interface_prefix);
                        sprintf (error_message, "dbus_svc_new_message failed");
                        dbus_svc_send (dbus, ERROR, serial, &new_serial, sender, path,
                                                   interface, member, TYPE_STRING, error_name,
                                                   TYPE_STRING, error_message, TYPE_INVALID);
                        if_list_msg = 0L;
                        return HANDLED;
                }

                twalk (if_tree, dhcdbd_list_an_if);

                if (!dbus_svc_send_message (dbus, if_list_msg, &new_serial)) {
                        dhcdbd_log ("dbus_svc_send_message failed");
                        sprintf (error_name, "%s.InternalError", dhcdbd_interface_prefix);
                        sprintf (error_message, "dhdbd_if_text_list failed");
                        dbus_svc_send (dbus, ERROR, serial, &new_serial, sender, path,
                                                   interface, member, TYPE_STRING, error_name,
                                                   TYPE_STRING, error_message, TYPE_INVALID);
                }
                if_list_msg = 0L;
        }
        return HANDLED;
}

static uint8_t
dhcdbd_new_if (DBUS_SVC dbus, DHCO dhco, char *if_name)
{
        DHC_IF *d_if = (DHC_IF *) malloc (sizeof (DHC_IF));
        char path[1024];

        if (d_if == 0L)
                return 0;
        memset (d_if, '\0', sizeof (DHC_IF));
        if ((d_if->name = (char *) malloc (strlen (if_name) + 1)) == 0L)
                return 0;
        strcpy (d_if->name, if_name);
        d_if->dbus = dbus;
        d_if->dhco = dhco;
        d_if->mapping = dhco_new_mapping (dhco, dhc_type_mapping);
        d_if->state = DHC_IF_INACTIVE;
        d_if->list_msg = (DHCDBD_Option_List_Message *)
                malloc (sizeof (DHCDBD_Option_List_Message));
        dhcdbd_copy_subscriptions (d_if);
        if (d_if->list_msg == 0L)
                return 0;
        memset (d_if->list_msg, '\0', sizeof (DHCDBD_Option_List_Message));
        snprintf (path, 1024, "%s.up", if_name);
        if (!dbus_svc_add_handler
                (dbus, dhcdbd_object_path, path, dhcdbd_if_up, d_if))
                return 0;
        snprintf (path, 1024, "%s.down", if_name);
        if (!dbus_svc_add_handler
                (dbus, dhcdbd_object_path, path, dhcdbd_if_down, d_if))
                return 0;
        snprintf (path, 1024, "%s.set", if_name);
        if (!dbus_svc_add_handler
                (dbus, dhcdbd_object_path, path, dhcdbd_if_set, d_if))
                return 0;
        snprintf (path, 1024, "%s.text.get", if_name);
        if (!dbus_svc_add_handler
                (dbus, dhcdbd_object_path, path, dhcdbd_if_text_get, d_if))
                return 0;
        snprintf (path, 1024, "%s.binary.get", if_name);
        if (!dbus_svc_add_handler
                (dbus, dhcdbd_object_path, path, dhcdbd_if_binary_get, d_if))
                return 0;
        snprintf (path, 1024, "%s.dbus.get", if_name);
        if (!dbus_svc_add_handler
                (dbus, dhcdbd_object_path, path, dhcdbd_if_dbus_get, d_if))
                return 0;
        snprintf (path, 1024, "%s.subscribe.dbus", if_name);
        if (!dbus_svc_add_handler
                (dbus, dhcdbd_object_path, path, dhcdbd_if_subscribe, d_if))
                return 0;
        snprintf (path, 1024, "%s.subscribe.binary", if_name);
        if (!dbus_svc_add_handler
                (dbus, dhcdbd_object_path, path, dhcdbd_if_subscribe, d_if))
                return 0;
        snprintf (path, 1024, "%s.subscribe.text", if_name);
        if (!dbus_svc_add_handler
                (dbus, dhcdbd_object_path, path, dhcdbd_if_subscribe, d_if))
                return 0;
        tsearch (d_if, &(if_tree), if_name_comparator);
        dhcdbd_debug ("NEW d_if: %p %s", d_if, d_if->name);
        return (1);
}

static uint8_t
dhcdbd_interface_found (char *name, char **real_name)
{
        struct ifaddrs *ifa = 0, *ifap;
        char *p = 0L;
        uint8_t found = 0;

        if ((getifaddrs (&ifa) != 0) || (ifa == 0L)) {
                dhcdbd_log ("getifaddrs failed");
                return 0;
        }
        dhcdbd_debug ("getifaddrs : OK");

        if ((p = strchr (name, ':')) != 0L)
                *p = '\0';

        for (ifap = ifa; ifap != 0L; ifap = ifap->ifa_next) {
                if ((ifap->ifa_name != 0L)
                        && (strcmp (name, ifap->ifa_name) == 0)
                        ) {
                        found = 1;
                        break;
                }
        }
        dhcdbd_debug ("getifaddrs: found");
        if (found && (p != 0L)) {
                if (real_name == 0L)
                        return 0;
                if ((*real_name = malloc (strlen (p) + 1)) == 0L)
                        return 0L;
                strcpy (*real_name, p);
                *p = ':';
        }
        freeifaddrs (ifa);
        return found;
}

dbus_svc_HandlerResult
dhcdbd_message_handler (DBUS_SVC dbus, dbus_svc_MessageType type,
                                                uint8_t reply_expected, uint32_t serial,
                                                char *destination, char *path, char *member,
                                                char *interface, char *if_suffix, char *sender,
                                                char *signature, dbus_svc_MessageHandle msg,
                                                char *prefix, char *suffix, void *prefixObject,
                                                void *object)
{                                                               /* This * handler * gets * all * messages 
                                                                 * to 
                                                                 * "root" 
                                                                 * path 
                                                                 * that 
                                                                 * have 
                                                                 * no 
                                                                 * sub-path 
                                                                 * handler 
                                                                 */

        uint32_t new_serial, success_response = 0;
        char *real_if_name = 0L;
        char error_name[1024] = "", error_message[1024] = "";
        uint8_t shutdown = 0;
        DHCO dhco = (DHCO) object;

        dhcdbd_debug
                ("Handler: got message %u %u %s %s %s %s %s %s %s %s %s %p %p",
                 reply_expected, serial, destination, path, member, interface,
                 if_suffix, sender, signature, prefix, suffix, prefixObject, object);

        /* handle "fake" commands for which no "handlers" are registered: */
        if ((strcasecmp (member, "quit") == 0)
                || (strcasecmp (member, "exit") == 0)
                || (strcasecmp (member, "shutdown") == 0)
                ) {
                dhcdbd_debug ("Received shutdown message - shutting down.");
                dbus_svc_quit (dbus);
                shutdown = 1;
        }

        if ((shutdown == 0) && (strcmp (path, dhcdbd_object_path) == 0) &&
                (strcasecmp (member, "list") == 0)) {
                dhcdbd_list_if (dbus, type, reply_expected, serial, destination, path,
                                                member, interface, if_suffix, sender, signature, msg,
                                                prefix, suffix, prefixObject, object);
                return HANDLED;
        }

        if (suffix == 0L)
                suffix = member;

        if ((shutdown == 0)
                && (strcmp (suffix, "ping") != 0)
                && (strcmp (suffix, "up") != 0)
                ) {
                /* Handle as yet unhandled command */

                if ((suffix != 0L) && (strchr (suffix, '/') == 0L)) {   /* MUST be an 
                                                                                                                                 * interface */

                        if (dhcdbd_interface_found (suffix, &real_if_name)) {
                                if (real_if_name != 0L)
                                        suffix = real_if_name;
                                dhcdbd_debug ("Found interface: %s", suffix);

                                if (!dhcdbd_new_if (dbus, dhco, suffix)) {      /* handlers * for
                                                                                                                         * * interface *
                                                                                                                         * commands *
                                                                                                                         * registered */
                                        dhcdbd_log ("dhcdbd_new_if failed for %s", suffix);
                                        snprintf (error_name, 1024, "%s.InternalError",
                                                          interface);
                                        snprintf (error_message, 1024,
                                                          "dhcdbd_new_if failed for %s.", suffix);
                                } else
                                        return HANDLED_NOW;
                        } else {
                                snprintf (error_name, 1024, "%s.BadInterface", interface);
                                snprintf (error_message, 1024, "Interface %s not found.",
                                                  suffix);
                                dhcdbd_log ("Failed to find interface: %s", suffix);
                        }
                }

        }

        if (reply_expected) {
                dhcdbd_debug ("Sending reply  %d to %s", serial, sender);
                if (error_name[0] != '\0')
                        dbus_svc_send (dbus, ERROR, serial, &new_serial, sender, path,
                                                   interface, member, TYPE_STRING, error_name,
                                                   TYPE_STRING, error_message, TYPE_INVALID);
                else
                        dbus_svc_send (dbus, RETURN, serial, &new_serial, sender, path,
                                                   interface, member, TYPE_UINT32, &success_response,
                                                   TYPE_INVALID);
        }

        return HANDLED;
}

dbus_svc_HandlerResult
dhcdbd_message_filter (DBUS_SVC dbus, dbus_svc_MessageType type,
                                           uint8_t reply_expected, uint32_t serial,
                                           char *destination, char *path, char *member,
                                           char *interface, char *if_suffix, char *sender,
                                           char *signature, dbus_svc_MessageHandle msg,
                                           char *prefix, char *suffix, void *prefixObject,
                                           void *object)
{                                                               /* Filter * to * shutdown * when *
                                                                 * dbus-daemon * shuts * down */

        dhcdbd_debug ("Filter: got message %u %u %s %s %s %s %s %s",
                                  reply_expected, serial, destination, path, member,
                                  interface, sender, signature);
        if ((type == SIGNAL)
                && (strcmp (path, "/org/freedesktop/DBus/Local") == 0)
                && (strcmp (member, "Disconnected") == 0)
                ) {
                dhcdbd_debug ("Filtered QUIT message OK");
                dbus_svc_quit (dbus);
                return HANDLED;
        }
        return NOT_HANDLED;
}

int
dhcdbd_run (D_BUS_TYPE bus)
{
        DBUS_SVC dbus;
        DHCO dhco;
        struct sigaction sa;
        char path[1024];
        int fd, l;

        if (dhcdbd_daemonize && (daemon (0, 0) == -1))
                return errno;

        openlog ("dhcdbd", LOG_NDELAY | LOG_CONS, LOG_USER);

        dbus = dbus_svc_init (bus, dhcdbd_destination, dhcdbd_log, 0L);

        if (dbus == 0L) {
                dhcdbd_log ("Failed to initialise D-Bus service.\n");
                return (1);
        }

        dhco = dhco_new (dhcdbd_log);
        if (dhco == 0L) {
                dhcdbd_log ("Failed to initialise DHCP options object.\n");
                return (1);
        }

        dhcdbd_debug ("dhco: %p", dhco);

        if (!dbus_svc_add_path_handler
                (dbus, dhcdbd_object_path, dhcdbd_interface_prefix,
                 dhcdbd_message_handler, dhco, TRUE)
                )
                return (1);

        if (!dbus_svc_add_message_filter (dbus, dhcdbd_message_filter, 0)
                )
                return (1);

        snprintf (path, 1024, "subscribe.binary.binary");
        if (!dbus_svc_add_handler
                (dbus, dhcdbd_object_path, path, dhcdbd_if_subscribe, dhco))
                return (1);
        snprintf (path, 1024, "subscribe.text.text");
        if (!dbus_svc_add_handler
                (dbus, dhcdbd_object_path, path, dhcdbd_if_subscribe, dhco))
                return (1);
        snprintf (path, 1024, "subscribe.dbus.dbus");
        if (!dbus_svc_add_handler
                (dbus, dhcdbd_object_path, path, dhcdbd_if_subscribe, dhco))
                return (1);

        memset (&sa, '\0', sizeof (struct sigaction));
        sa.sa_sigaction = dhc_reaper;
        sa.sa_flags = SA_SIGINFO;       /* NO RESTART, NO DEFER, CLDSTOP */
        if (sigaction (SIGCHLD, &sa, 0L) == -1) {
                dhcdbd_log ("sigaction failed: %s\n", strerror (errno));
                return (1);
        }
        unlink (DHCDBD_PID_FILE);
        if ((fd = open (DHCDBD_PID_FILE, O_WRONLY | O_CREAT)) == -1)
                exit (errno);
        l = sprintf (path, "%u", getpid ());
        l = write (fd, path, l);
        fsync (fd);
        close (fd);
        dhcdbd_log ("Started up.");
        dbus_svc_main_loop (dbus, dhcdbd_work);
        dhcdbd_debug ("Main Loop Exited.");
        dbus_svc_shutdown (dbus);
        unlink (DHCDBD_PID_FILE);
        dhcdbd_log ("Shut down.");
        return (0);
}
