#ifndef LIST_H
#define LIST_H

#if _XOPEN_SOURCE < 500
	#error _XOPEN_SOURCE must be >= 500
#endif

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

typedef struct list list_t ;
typedef struct list_iterator list_iterator_t ;

typedef int (* list_compare_f) (const void *, const void *) ;

typedef enum list_iterator_mode {
	LIST_READ,
	LIST_WRITE
} list_iterator_mode_t ;

/**
 * Create a new list.
 *
 * @return the new list
 * @return NULL if there is not enough memory
 */
list_t * list_new (void) ;

/**
 * Destroy a list without freeing its elements.
 */
int list_free (list_t * list) ;

/**
 * Destroy a list and free its elements.
 */
int list_free_all (list_t * list, void (* free_element) (void *)) ;

/**
 * Add an element at the end of the list.
 * An element cannot be added more than once.
 * (based on a comparator function, defaults to pointer comparison if no comparator provided)
 *
 * @warning It may cause a deadlock if an iterator is open on the list.
 *
 * @return 0 if it is added
 * @return a negative value otherwise
 */
int list_add (list_t * list, const void * element, list_compare_f compare) ;

/**
 * Remove an element from the list.
 * (based on a comparator function, defaults to pointer comparison if no comparator provided)
 *
 * @return the pointer of the removed element
 * @return NULL otherwise
 */
void * list_remove (list_t * list, const void * element, list_compare_f compare) ;

/**
 * Remove all elements from the list but the elements are not freed.
 *
 * @return 0 if the elements are removed
 * @return a negative value otherwise
 */
int list_remove_all (list_t * list) ;

/**
 * Check if the list contains an element
 * (based on a comparator function, defaults to pointer comparison if no comparator provided)
 *
 * @return 0 if the element is not in the list
 * @return another value otherwise
 */
int list_contains (list_t * list, const void * element, list_compare_f compare) ;

/**
 * Allocate an iterator that points no element (before the first one).
 * In read mode, no thread can modify the list until the iterator is closed.
 * In write mode, no thread can read or modify the list until the iterator is closed.
 *
 * @return 0 if the iterator is initialized
 * @return another value otherwise
 */
list_iterator_t * list_iterator_new (list_t * list, list_iterator_mode_t mode) ;

/**
 * Move an iterator before the first element.
 */
void list_iterator_reset (list_iterator_t * iterator) ;

/**
 * Destroy an iterator.
 */
int list_iterator_free (list_iterator_t * iterator) ;

/**
 * Move to the next element.
 *
 * @return 0 if the next element is pointed by iterator
 * @return a negative value otherwise
 */
int list_iterator_next (list_iterator_t * iterator) ;

/**
 * Check if there is a next element.
 *
 * @return 0 if there is no next element
 * @return another value otherwise
 */
int list_iterator_has_next (const list_iterator_t * iterator) ;

/**
 * Get an element pointed by an iterator from the list.
 *
 * @return the element
 * @return NULL if there is no element at the current position
 */
void * list_iterator_get (list_iterator_t * iterator) ;

/**
 * Remove an element pointed by an iterator from the list.
 *
 * @return 0 if it is removed
 * @return a negative value otherwise
 */
int list_iterator_remove (list_iterator_t * iterator) ;

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif /* LIST_H */
