#include "monitor.h"

#include "list.h"
#include "debug.h"

#include <errno.h>
#include <limits.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/param.h>
#include <sys/inotify.h>

/*********   events   *********/

/* The directory was modified for one or more of the following reason:
 *   - It is moved or removed
 *   - Files were added into or moved to it
 *   - It was closed after a write operation
 *   - Its attributes have changed
 *   - Some files in it were deleted
 *   - Some files were moved from it
 */
#define MODIFICATION_EVENTS ( \
	REMOVE_EVENTS | \
	ADD_EVENTS | \
	IN_CLOSE_WRITE | \
	IN_ATTRIB | \
	IN_DELETE | \
	IN_MOVED_FROM \
)

/* The directory was moved or deleted. */
#define REMOVE_EVENTS ( \
	IN_DELETE_SELF | \
	IN_MOVE_SELF \
)

/* Files were moved to or create in the directory. */
#define ADD_EVENTS ( \
	IN_CREATE | \
	IN_MOVED_TO \
)

static const char inotify_strings[][19] = {
	"accessed",           /* IN_ACCESS        0x       1 */
	"modified",           /* IN_MODIFY        0x       2 */
	"attributes changed", /* IN_ATTRIB        0x       4 */
	"writed",             /* IN_CLOSE_WRITE   0x       8 */
	"closed",             /* IN_CLOSE_NOWRITE 0x      10 */
	"opened",             /* IN_OPEN          0x      20 */
	"moved from",         /* IN_MOVED_FROM    0x      40 */
	"moved to",           /* IN_MOVED_TO      0x      80 */
	"created",            /* IN_CREATE        0x     100 */
	"subfile deleted",    /* IN_DELETE        0x     200 */
	"deleted",            /* IN_DELETE_SELF   0x     400 */
	"moved",              /* IN_MOVE_SELF     0x     800 */
	"",                   /* nothing          0x    1000 */
	"unmounted",          /* IN_UNMOUNT       0x    2000 */
	"inotify overflow",   /* IN_Q_OVERFLOW    0x    4000 */
	"ignored",            /* IN_IGNORED       0x    8000 */
	"",                   /* nothing          0x   10000 */
	"",                   /* nothing          0x   20000 */
	"",                   /* nothing          0x   40000 */
	"",                   /* nothing          0x   80000 */
	"",                   /* nothing          0x  100000 */
	"",                   /* nothing          0x  200000 */
	"",                   /* nothing          0x  400000 */
	"",                   /* nothing          0x  800000 */
	"only directory",     /* IN_ONLYDIR       0x 1000000 */
	"not follow",         /* IN_DONT_FOLLOW   0x 2000000 */
	"",                   /* nothing          0x 4000000 */
	"",                   /* nothing          0x 8000000 */
	"",                   /* nothing          0x10000000 */
	"mask add",           /* IN_MASK_ADD      0x20000000 */
	"is directory",       /* IN_ISDIR         0x40000000 */
	"only once"           /* IN_ONESHOT       0x80000000 */
} ;

static int stradd (char *dst, size_t *size, size_t *offset, const char *src) {
	size_t length = MIN (strlen (src) + 1, *size) ;
	if (length > 1) {
		strncpy (dst + *offset, src, length) ;
		length -- ;
		*offset += length ;
		*size -= length ;
		dst[*offset] = 0 ;
	}
	return length ;
}

static const char* inotify_i2a (uint32_t event) {
	static char text[100] ;
	size_t size = sizeof text ;
	size_t offset = 0 ;
	int bit ;
	for (bit = 0 ; bit < 32 ; bit ++) {
		if (event & (1 << bit)) {
			if (offset > 0)
				stradd (text, &size, &offset, " / ") ;
			stradd (text, &size, &offset, inotify_strings [bit]) ;
		}
	}
	return text ;
}

/*********   directory list   *********/

typedef struct monitored_directory {
	int wd ;
	char path[] ;
} monitored_directory_t ;
static list_t *directory_list = NULL ;

static int directory_list_add (int wd, const char *directory) {
	monitored_directory_t *saved_directory ;
	saved_directory = malloc (sizeof (monitored_directory_t) + strlen (directory) + 1) ;
	if (saved_directory == NULL) {
		perror ("malloc") ;
		return -1 ;
	}
	saved_directory->wd = wd ;
	strcpy (saved_directory->path, directory) ;
	return list_add (directory_list, saved_directory, NULL) ;
}

static char* directory_list_get (int wd) {
	char *directory = NULL ;
	list_iterator_t *iterator ;
	monitored_directory_t *current ;
	iterator = list_iterator_new (directory_list, LIST_READ) ;
	if (iterator == NULL)
		return NULL ;
	while (list_iterator_next (iterator) == 0) {
		current = list_iterator_get (iterator) ;
		if (current->wd == wd) {
			directory = current->path ;
			break ;
		}
	}
	list_iterator_free (iterator) ;
	return directory ;
}

static int directory_list_remove (int wd) {
	list_iterator_t *iterator ;
	monitored_directory_t *current ;
	int found = 0 ;
	iterator = list_iterator_new (directory_list, LIST_WRITE) ;
	if (iterator == NULL)
		return -1 ;
	while (list_iterator_next (iterator) == 0) {
		current = list_iterator_get (iterator) ;
		if (current->wd == wd) {
			found ++ ;
			free (current) ;
			list_iterator_remove (iterator) ;
			break ;
		}
	}
	list_iterator_free (iterator) ;
	return found > 0 ;
}

/*********   inotify   *********/

static int fd = -1 ;

static char* monitoring_path_get (const struct inotify_event *event) {
	char *parent_directory ;
	char *path ;
	parent_directory = directory_list_get (event->wd) ;
	if (parent_directory == NULL)
		return NULL ;
	path = malloc (strlen (parent_directory) + 1 + event->len) ;
	if (path == NULL) {
		perror ("malloc") ;
		return NULL ;
	}
	sprintf (path, "%s/%s", parent_directory, event->name) ;
	return path ;
}

static int monitoring_add (const char *directory) {
	int wd ;
	dprintf (E_DBG, "ADD    %s \t for events 0x%x", directory, MODIFICATION_EVENTS) ;
	/* add to the kernel's inotify list */
	wd = inotify_add_watch (fd, directory, MODIFICATION_EVENTS) ;
	if (wd < 0) {
		perror ("inotify_add_watch") ;
		return -1 ;
	}

	/* keep a trace of the full path */
	return directory_list_add (wd, directory) ;
}

static int monitoring_recursive_add (const struct inotify_event *event,
				     on_file_evt_delegate on_file_add) {
	int error_code ;
	char *path ;
	path = monitoring_path_get (event) ;
	if (path == NULL)
		return -1 ;
	error_code = directory_recursive_process (path,
						  monitoring_add,
						  on_file_add) ;
	free (path) ;
	return error_code ;
}

static int monitoring_remove (const struct inotify_event *event) {
	int error_code = 0 ;
	dprintf (E_DBG, "REMOVE %d", event->wd) ;
	error_code |= directory_list_remove (event->wd) ;
	if (! (event->mask & IN_IGNORED))
		/* it should be useless because always done by the kernel (flag IN_IGNORED) */
		error_code |= inotify_rm_watch (fd, event->wd) ;
	return error_code ;
}

static int monitoring_update_on_event (const struct inotify_event *event,
				       on_file_evt_delegate on_add,
				       on_file_evt_delegate on_mod,
				       on_file_evt_delegate on_remove) {
	if (event->mask == 0) {
		dprintf (E_LOG, "bad event (0)") ;
		return -1 ;
	}
	/* start monitoring of added directory (and subdirectories) */
	if (event->mask & ADD_EVENTS && event->mask & IN_ISDIR)
	  return monitoring_recursive_add (event, on_add) ;

	/* call delegate on content add */
	if(event->mask & ADD_EVENTS)
	  return on_add (monitoring_path_get (event)) ;

	/* call delegate on content modification */
	if(event->mask & (IN_ATTRIB | IN_CLOSE_WRITE))
	  return on_mod (monitoring_path_get (event)) ;

	/* call delegate on content removed */
	if(event->mask & (IN_DELETE | IN_MOVED_FROM))
	  return on_remove (monitoring_path_get (event)) ;

	/* stop monitoring of removed directory */
	if (event->mask & REMOVE_EVENTS)
	  monitoring_remove (event) ;

	return 0 ;
}

static int monitoring_update_on_event_buffer (const char *event_buffer,
					      int *length,
					      on_file_evt_delegate on_add,
					      on_file_evt_delegate on_mod,
					      on_file_evt_delegate on_remove) {
	int error_code = 0 ;
	int event_offset, event_size ;
	struct inotify_event *event ;

	/* get all events of the buffer */
	event_offset = 0 ;
	while (*length > 0) {
		event = (struct inotify_event *) (event_buffer + event_offset) ;
		event_size = sizeof (struct inotify_event) + event->len ;
		event_offset += event_size ;
		*length -= event_size ;
		if(event->len == 0) continue;
		dprintf (E_DBG, "EVENT 0x%x (%s): %s", event->mask, inotify_i2a (event->mask), event->name) ;
		error_code |= monitoring_update_on_event (event,
							  on_add,
							  on_mod,
							  on_remove) ;
	}

	return error_code ;
}

int monitoring_open (const char *directory, path_process file_filter) {
	int error_code ;

	if (fd < 0) {
		fd = inotify_init () ;
		if (fd < 0) {
			perror ("inotify_init") ;
			return -1 ;
		}
	}

	if (directory_list == NULL) {
		directory_list = list_new () ;
		if (directory_list == NULL)
			return -1 ;
	}

	if (directory != NULL) {
		error_code = directory_recursive_process (directory,
							  monitoring_add,
							  file_filter) ;
		if (error_code != 0)
			return -1 ;
	}

	return fd ;
}

int monitoring_read (on_file_evt_delegate on_add,
		     on_file_evt_delegate on_mod,
		     on_file_evt_delegate on_remove) {
	/* allocate at least 3 structs with a flexible array of size NAME_MAX */
	char event_buffer[(sizeof (struct inotify_event) + NAME_MAX) * 3]
	/* event_buffer must be aligned for field access when casted to inotify_event */
	/* do not remove this option: the field access could fail on ARM */
		__attribute__ ((aligned (__WORDSIZE >> 3))) ;
	int read_bytes ;

	read_bytes = read (fd, event_buffer, sizeof event_buffer) ;
	if (read_bytes < 0) {
		if (errno == EINTR)
			return 0 ;
		perror("read inotify") ;
		return -1 ;
	}

	/* monitor added directories */
	return monitoring_update_on_event_buffer (event_buffer,
						  &read_bytes,
						  on_add,
						  on_mod,
						  on_remove) ;
}

void monitoring_close () {
	if (fd >= 0) {
		close (fd) ;
		fd = -1 ;
	}
	if (directory_list != NULL) {
		list_free_all (directory_list, NULL) ;
		directory_list = NULL ;
	}
}
