/*
 * arch/arm/mach-feroceon-kw/irq.c
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/irq.h>
#include <linux/io.h>
#include <asm/gpio.h>

static void feroceon_kw_irq_mask(u32 irq)
{
	void __iomem *maskaddr = get_irq_chip_data(irq);
	u32 mask;

	mask = readl(maskaddr);
	mask &= ~(1 << (irq & 31));
	writel(mask, maskaddr);
}

static void feroceon_kw_irq_unmask(u32 irq)
{
	void __iomem *maskaddr = get_irq_chip_data(irq);
	u32 mask;

	mask = readl(maskaddr);
	mask |= 1 << (irq & 31);
	writel(mask, maskaddr);
}

static struct irq_chip feroceon_kw_irq_chip = {
	.name		= "feroceon_kw_irq",
	.mask		= feroceon_kw_irq_mask,
	.mask_ack	= feroceon_kw_irq_mask,
	.unmask		= feroceon_kw_irq_unmask,
};

void __init feroceon_kw_irq_init(unsigned int irq_start, void __iomem *maskaddr)
{
	unsigned int i;

	/*
	 * Mask all interrupts initially.
	 */
	writel(0, maskaddr);

	/*
	 * Register IRQ sources.
	 */
	for (i = 0; i < 32; i++) {
		unsigned int irq = irq_start + i;

		set_irq_chip(irq, &feroceon_kw_irq_chip);
		set_irq_chip_data(irq, maskaddr);
		set_irq_handler(irq, handle_level_irq);
		irq_desc[irq].status |= IRQ_LEVEL;
		set_irq_flags(irq, IRQF_VALID);
	}
}

static void gpio_irq_handler(unsigned int irq, struct irq_desc *desc)
{
	BUG_ON(irq < IRQ_GPP_LOW_0_7);
	BUG_ON(irq > IRQ_GPP_HIGH_16_23);

	feroceon_gpio_irq_handler((irq - IRQ_GPP_LOW_0_7) << 3);
}

#ifdef CONFIG_DEBUG_FS

#include <linux/debugfs.h>
#include <linux/seq_file.h>

static int dbg_irq_show(struct seq_file *s, void *unused)
{
	seq_printf(s, "Main irq mask (low)    : 0x%08x\n",
		   readl(MV_ASM_IRQ_MASK_LOW_REG));
	seq_printf(s, "Main irq mask (high)   : 0x%08x\n",
		   readl(MV_ASM_IRQ_MASK_HIGH_REG));
	seq_printf(s, "GPIO level mask (low)  : 0x%08x\n",
		   readl(GPIO_LEVEL_MASK(0)));
	seq_printf(s, "GPIO level mask (high) : 0x%08x\n",
		   readl(GPIO_LEVEL_MASK(32)));
	seq_printf(s, "GPIO edge mask (low)   : 0x%08x\n",
		   readl(GPIO_EDGE_MASK(0)));
	seq_printf(s, "GPIO edge mask (high)  : 0x%08x\n",
		   readl(GPIO_EDGE_MASK(32)));
	return 0;
}

static int dbg_irq_open(struct inode *inode, struct file *file)
{
	return single_open(file, dbg_irq_show, &inode->i_private);
}

static struct file_operations feroceon_kw_irq_debug_operations={
	.open		= dbg_irq_open,
	.read 		= seq_read,
	.llseek 	= seq_lseek,
	.release	= single_release,
};

static int __init feroceon_kw_irq_init_debug(void)
{
	debugfs_create_file("kw_irqs", S_IRUGO, NULL, NULL,
			    &feroceon_kw_irq_debug_operations);
	return 0;
}
late_initcall(feroceon_kw_irq_init_debug);
#endif

void __init feroceon_kw_init_irq(void)
{
	int i;
	u32 u;

	feroceon_kw_irq_init(0, (void __iomem *)(MV_ASM_IRQ_MASK_LOW_REG));
	feroceon_kw_irq_init(32, (void __iomem *)(MV_ASM_IRQ_MASK_HIGH_REG));

	/*
	 * Mask and clear GPIO IRQ interrupts.
	 */
	writel(0, GPIO_LEVEL_MASK(0));
	writel(0, GPIO_EDGE_MASK(0));
	writel(0, GPIO_EDGE_CAUSE(0));
	writel(0, GPIO_LEVEL_MASK(32));
	writel(0, GPIO_EDGE_MASK(32));
	writel(0, GPIO_EDGE_CAUSE(32));

	/* Enable summarize of main interrupt high cause register. */
	u = readl(INTER_REGS_BASE + MV_IRQ_MASK_LOW_REG);
	u |= 1 << IRQ_MAIN_HIGH_SUM;
	writel(u, INTER_REGS_BASE + MV_IRQ_MASK_LOW_REG);

	for (i = IRQ_GPP_START; i < NR_IRQS; i++) {
		set_irq_chip(i, &feroceon_gpio_irq_chip);
		set_irq_handler(i, handle_level_irq);
		irq_desc[i].status |= IRQ_LEVEL;
		set_irq_flags(i, IRQF_VALID);
	}

	set_irq_chained_handler(IRQ_GPP_LOW_0_7, gpio_irq_handler);
	set_irq_chained_handler(IRQ_GPP_LOW_8_15, gpio_irq_handler);
	set_irq_chained_handler(IRQ_GPP_LOW_16_23, gpio_irq_handler);
	set_irq_chained_handler(IRQ_GPP_LOW_24_31, gpio_irq_handler);
	set_irq_chained_handler(IRQ_GPP_HIGH_0_7, gpio_irq_handler);
	set_irq_chained_handler(IRQ_GPP_HIGH_8_15, gpio_irq_handler);
	set_irq_chained_handler(IRQ_GPP_HIGH_16_23, gpio_irq_handler);
}
