/*
 * arch/arm/mach-feroceon-kw/lacie/lacie-v2_common.c
 *
 * Common setup for LaCie boards based on Feroceon-kw SoCs.
 *
 * Copyright (c) 2011 LaCie
 *
 * Author: Simon Guinot <sguinot@lacie.com>
 *
 * This file is licensed under the terms of the GNU General Public
 * License version 2.  This program is licensed "as is" without any
 * warranty of any kind, whether express or implied.
 */

#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/delay.h>
#include <linux/io.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/partitions.h>
#include <linux/spi/spi.h>
#include <linux/spi/flash.h>
#include <mach/feroceon-kw.h>
#include "core.h"

/* Board revision support */

#define PHY_ADDR		0x0000
#define SMI_REG			0x0004
#define  SMI_BUSY		0x10000000
#define  SMI_READ_VALID		0x08000000
#define  SMI_OPCODE_READ	0x04000000
#define  SMI_OPCODE_WRITE	0x00000000

#define MII_PHYSID1		0x02 /* PHY ID 1 */
#define MII_PHYSID2		0x03 /* PHY ID 2 */

/*
 * all the MII management functions are imported for the driver mv643xx_eth.c.
 */
static int smi_is_done(void __iomem *smi_reg)
{
	return !(readl(smi_reg) & SMI_BUSY);
}

static int smi_wait_ready(void __iomem *smi_reg)
{
	int i;

	for (i = 0; !smi_is_done(smi_reg); i++) {
		if (i == 10)
			return -ETIMEDOUT;
		msleep(10);
	}
	return 0;
}

static int __init smi_bus_read(void __iomem *smi_reg, int phy_addr,
			       int reg, u32 *val)
{
        if (smi_wait_ready(smi_reg)) {
                pr_warning("SMI bus busy timeout\n");
                return -ETIMEDOUT;
        }

        writel(SMI_OPCODE_READ | (reg << 21) | (phy_addr << 16), smi_reg);

        if (smi_wait_ready(smi_reg)) {
                pr_warning("SMI bus busy timeout\n");
                return -ETIMEDOUT;
        }

        *val = readl(smi_reg);
        if (!(*val & SMI_READ_VALID)) {
                pr_warning("SMI bus read not valid\n");
                return -ENODEV;
        }

	return 0;
}

int __init boardrev_get_phy_id(int ge_num, int phy_addr, u32 *phy_id)
{
	int ret;
	u32 val;
	void __iomem *smi_reg;

	if (ge_num == 0)
		smi_reg = (void __iomem *) GE00_PHYS_BASE + 0x2000 + SMI_REG;
	else if (ge_num == 1)
		smi_reg = (void __iomem *) GE01_PHYS_BASE + 0x2000 + SMI_REG;
	else
		return -ENODEV;

	ret = smi_bus_read(smi_reg, phy_addr, MII_PHYSID1, &val);
	if (ret < 0)
		return ret;

	*phy_id = (val & 0xffff) << 16;

	ret = smi_bus_read(smi_reg, phy_addr, MII_PHYSID2, &val);
	if (ret < 0)
		return ret;

	*phy_id |= (val & 0xffff);

	pr_debug("%s: phy_id=0x%08x\n", __func__, *phy_id);

	return 0;
}

/* 512KB SPI Flash on Boot Device (MACRONIX MX25L4005) */

static struct mtd_partition lacie_v2_flash_parts[] = {
	{
		/* U-Boot environment: offset=0x7e000 size=0x1000 */
		.name		= "u-boot",
		.size		= MTDPART_SIZ_FULL,
		.offset		= 0,
		.mask_flags	= MTD_POWERUP_LOCK,	/* Locked by U-Boot */
	},
};

static const struct flash_platform_data lacie_v2_flash = {
	.type		= "mx25l4005a",
	.name		= "spi_flash",
	.parts		= lacie_v2_flash_parts,
	.nr_parts	= ARRAY_SIZE(lacie_v2_flash_parts),
};

static struct spi_board_info __initdata lacie_v2_spi_slave_info[] = {
	{
		.modalias	= "m25p80",
		.platform_data	= &lacie_v2_flash,
		.irq		= -1,
		.max_speed_hz	= 20000000,
		.bus_num	= 0,
		.chip_select	= 0,
	},
};

void __init lacie_v2_sf_init(void)
{
	spi_register_board_info(lacie_v2_spi_slave_info,
				ARRAY_SIZE(lacie_v2_spi_slave_info));
	feroceon_kw_spi_init();
}

/* NAND flash */

static struct mtd_partition lacie_v2_nand_parts[] = {
	{
		/* U-Boot environment: offset=0xa0000 size=0x20000 */
		.name = "u-boot",
		.offset = 0,
		.size = SZ_1M
	}, {
		.name = "uImage",
		.offset = MTDPART_OFS_NXTBLK,
		.size = SZ_16M
	}, {
		.name = "root",
		.offset = MTDPART_OFS_NXTBLK,
		.size = MTDPART_SIZ_FULL
	},
};

void __init lacie_v2_nand_init(void)
{
	feroceon_kw_nand_init(lacie_v2_nand_parts,
			      ARRAY_SIZE(lacie_v2_nand_parts), 50);
}
