/*
 * arch/arm/mach-feroceon-kw/lacie/net5big_v2-setup.c
 *
 * LaCie 5Big Network v2 board setup.
 *
 * Copyright (c) 2009 LaCie
 *
 * Author: Simon Guinot <sguinot@lacie.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/kernel.h>
#include <linux/platform_device.h>
#include <linux/i2c.h>
#include <linux/input.h>
#include <linux/gpio_keys.h>
#include <linux/leds.h>
#include <linux/ata_platform.h>
#include <linux/mv643xx_eth.h>
#include <mach/leds-netxbig.h>
#include <asm/gpio.h>
#include <asm/mach-types.h>
#include <asm/mach/arch.h>
#include <scsi/scsi_spinup.h>

#include "core.h"
#include "lacie_v2-common.h"
#include "marvell_boardenv/net5big_v2-boardenv.h"
#include "lacie/common.h"
#include "lacie/drivers/gpio-hd-power.h"
#include "lacie/drivers/gpio-usb.h"
#include "lacie/drivers/g762.h"

extern struct lacie_disk_map *lacie_disk_map;

/* Ethernet */

static struct mv643xx_eth_platform_data net5big_v2_ge00_data = {
	.phy_addr	= MV643XX_ETH_PHY_ADDR(8),
};

static struct mv643xx_eth_platform_data net5big_v2_ge01_data = {
	.phy_addr	= MV643XX_ETH_PHY_ADDR(0),
};

/* GPIO buttons */

#define NET5BIG_V2_SWITCH_POWER_ON	0x1
#define NET5BIG_V2_SWITCH_POWER_OFF	0x2

static struct gpio_keys_button net5big_v2_buttons[] = {
	[0] = {
		.type		= EV_SW,
		.code		= NET5BIG_V2_SWITCH_POWER_ON,
		.gpio		= NET5BIG_V2_GPIO_SWITCH_POWER_ON,
		.desc		= "Back power switch (on|auto)",
	},
	[1] = {
		.type		= EV_SW,
		.code		= NET5BIG_V2_SWITCH_POWER_OFF,
		.gpio		= NET5BIG_V2_GPIO_SWITCH_POWER_OFF,
		.desc		= "Back power switch (auto|off)",
	},
	[2] = {
		.code		= KEY_SLEEP,
		.gpio		= NET5BIG_V2_GPIO_PUSH_BUTTON,
		.desc		= "Front push button",
		.active_low	= 1,
	},
};

static struct gpio_keys_platform_data net5big_v2_button_data = {
	.buttons	= net5big_v2_buttons,
	.nbuttons	= ARRAY_SIZE(net5big_v2_buttons),
};

static struct platform_device net5big_v2_gpio_buttons = {
	.name		= "gpio-keys",
	.id		= -1,
	.dev		= {
		.platform_data 	= &net5big_v2_button_data,
	},
};

/* GPIO extension LEDs */

/*
 * The LEDs are controlled by a CPLD and can be configured through a GPIO
 * extension bus:
 *
 * - address register : bit [0-2] -> GPIO [47-49]
 * - data register    : bit [0-2] -> GPIO [44-46]
 * - enable register  : GPIO 29
 */

static int net5big_v2_gpio_ext_addr[] = { 47, 48, 49 };
static int net5big_v2_gpio_ext_data[] = { 44, 45, 46 };

static struct netxbig_gpio_ext net5big_v2_gpio_ext = {
	.addr		= net5big_v2_gpio_ext_addr,
	.num_addr	= ARRAY_SIZE(net5big_v2_gpio_ext_addr),
	.data		= net5big_v2_gpio_ext_data,
	.num_data	= ARRAY_SIZE(net5big_v2_gpio_ext_data),
	.enable		= 29,
};

/*
 * Address register selection:
 *
 * addr | register
 * ----------------------------
 *   0  | front LED
 *   1  | front LED brightness
 *   2  | SATA LED brightness
 *   3  | SATA0 LED
 *   4  | SATA1 LED
 *   5  | SATA2 LED
 *   6  | SATA3 LED
 *   7  | SATA4 LED
 *
 * Data register configuration:
 *
 * data | LED brightness
 * -------------------------------------------------
 *   0  | min (off)
 *   -  | -
 *   7  | max
 *
 * data | front LED mode
 * -------------------------------------------------
 *   0  | fix off
 *   1  | fix blue on
 *   2  | fix red on
 *   3  | blink blue on=1 sec and blue off=1 sec
 *   4  | blink red on=1 sec and red off=1 sec
 *   5  | blink blue on=2.5 sec and red on=0.5 sec
 *   6  | blink blue on=1 sec and red on=1 sec
 *   7  | blink blue on=0.5 sec and blue off=2.5 sec
 *
 * data | SATA LED mode
 * -------------------------------------------------
 *   0  | fix off
 *   1  | SATA activity blink
 *   2  | fix red on
 *   3  | blink blue on=1 sec and blue off=1 sec
 *   4  | blink red on=1 sec and red off=1 sec
 *   5  | blink blue on=2.5 sec and red on=0.5 sec
 *   6  | blink blue on=1 sec and red on=1 sec
 *   7  | fix blue on
 */

static int net5big_v2_red_mled[NETXBIG_LED_MODE_NUM] = {
	[NETXBIG_LED_OFF]	= 0,
	[NETXBIG_LED_ON]	= 2,
	[NETXBIG_LED_SATA]	= NETXBIG_LED_INVALID_MODE,
	[NETXBIG_LED_TIMER1]	= 4,
	[NETXBIG_LED_TIMER2]	= NETXBIG_LED_INVALID_MODE,
};

static int net5big_v2_blue_pwr_mled[NETXBIG_LED_MODE_NUM] = {
	[NETXBIG_LED_OFF]	= 0,
	[NETXBIG_LED_ON]	= 1,
	[NETXBIG_LED_SATA]	= NETXBIG_LED_INVALID_MODE,
	[NETXBIG_LED_TIMER1]	= 3,
	[NETXBIG_LED_TIMER2]	= 7,
};

static int net5big_v2_blue_sata_mled[NETXBIG_LED_MODE_NUM] = {
	[NETXBIG_LED_OFF]	= 0,
	[NETXBIG_LED_ON]	= 7,
	[NETXBIG_LED_SATA]	= 1,
	[NETXBIG_LED_TIMER1]	= 3,
	[NETXBIG_LED_TIMER2]	= NETXBIG_LED_INVALID_MODE,
};

static struct netxbig_led_timer net5big_v2_led_timer[] = {
	[0] = {
		.delay_on	= 500,
		.delay_off	= 500,
		.mode		= NETXBIG_LED_TIMER1,
	},
	[1] = {
		.delay_on	= 500,
		.delay_off	= 1000,
		.mode		= NETXBIG_LED_TIMER2,
	},
};

#define NET5BIG_LED(_name, maddr, mval, baddr)			\
	{ .name		= _name,				\
	  .mode_addr	= maddr,				\
	  .mode_val	= mval,					\
	  .bright_addr	= baddr }

static struct netxbig_led net5big_v2_leds_ctrl[] = {
	NET5BIG_LED("net5big-v2:blue:power", 0, net5big_v2_blue_pwr_mled,  1),
	NET5BIG_LED("net5big-v2:red:power",  0, net5big_v2_red_mled,       1),
	NET5BIG_LED("net5big-v2:blue:sata0", 3, net5big_v2_blue_sata_mled, 2),
	NET5BIG_LED("net5big-v2:red:sata0",  3, net5big_v2_red_mled,       2),
	NET5BIG_LED("net5big-v2:blue:sata1", 4, net5big_v2_blue_sata_mled, 2),
	NET5BIG_LED("net5big-v2:red:sata1",  4, net5big_v2_red_mled,       2),
	NET5BIG_LED("net5big-v2:blue:sata2", 5, net5big_v2_blue_sata_mled, 2),
	NET5BIG_LED("net5big-v2:red:sata2",  5, net5big_v2_red_mled,       2),
	NET5BIG_LED("net5big-v2:blue:sata3", 6, net5big_v2_blue_sata_mled, 2),
	NET5BIG_LED("net5big-v2:red:sata3",  6, net5big_v2_red_mled,       2),
	NET5BIG_LED("net5big-v2:blue:sata4", 7, net5big_v2_blue_sata_mled, 2),
	NET5BIG_LED("net5big-v2:red:sata4",  7, net5big_v2_red_mled,       2),
};

static struct netxbig_led_platform_data net5big_v2_leds_data = {
	.gpio_ext	= &net5big_v2_gpio_ext,
	.timer		= net5big_v2_led_timer,
	.num_timer	= ARRAY_SIZE(net5big_v2_led_timer),
	.leds		= net5big_v2_leds_ctrl,
	.num_leds	= ARRAY_SIZE(net5big_v2_leds_ctrl),
};

static struct platform_device net5big_v2_leds = {
	.name	= "leds-netxbig",
	.id	= -1,
	.dev	= {
		.platform_data	= &net5big_v2_leds_data,
	},
};

/* GPIO USB (with on the go fakery) */

static struct gpio_usb net5big_v2_gpio_usb_port0[] = {
	[0] = {
		.name	= "fuse",
		.type	= GPIO_USB_FUSE,
		.num	= NET5BIG_V2_GPIO_USB_FUSE,
	},
	[1] = {
		.name	= "vbus-out",
		.type	= GPIO_USB_VBUS_OUT,
		.num	= NET5BIG_V2_GPIO_USB_VBUS_OUT,
	},
};

static struct gpio_usb_port net5big_v2_gpio_usb_ports[] = {
	[0] = {
		.name		= "port0",
		.gpio		= net5big_v2_gpio_usb_port0,
		.num_gpio	= ARRAY_SIZE(net5big_v2_gpio_usb_port0),
		.reload_delay	= 10000,
		.spurious_delay	= 100,
	},
};

struct gpio_usb_platform_data net5big_v2_lacie_gpio_usb_data = {
	.port		= net5big_v2_gpio_usb_ports,
	.num_port	= ARRAY_SIZE(net5big_v2_gpio_usb_ports),
};

static struct platform_device net5big_v2_gpio_usb = {
	.name		= "gpio-usb",
	.id		= -1,
	.dev		= {
		.platform_data	= &net5big_v2_lacie_gpio_usb_data,
	},
};

/* GPIO hard disk power */

static struct gpio_hd_power net5big_v2_hd_power[] = {
	[0] = {
		.name		= "hd0",
		.power_pin	= NET5BIG_V2_GPIO_HD0_POWER,
		.pres_pin	= NET5BIG_V2_GPIO_HD0_PRES,
		.have_power_pin	= 1,
		.have_pres_pin	= 1,
		.pres_act_low	= 1,
	},
	[1] = {
		.name		= "hd1",
		.power_pin	= NET5BIG_V2_GPIO_HD1_POWER,
		.pres_pin	= NET5BIG_V2_GPIO_HD1_PRES,
		.have_power_pin	= 1,
		.have_pres_pin	= 1,
		.pres_act_low	= 1,
	},
	[2] = {
		.name		= "hd2",
		.power_pin	= NET5BIG_V2_GPIO_HD2_POWER,
		.pres_pin	= NET5BIG_V2_GPIO_HD2_PRES,
		.have_power_pin	= 1,
		.have_pres_pin	= 1,
		.pres_act_low	= 1,
	},
	[3] = {
		.name		= "hd3",
		.power_pin	= NET5BIG_V2_GPIO_HD3_POWER,
		.pres_pin	= NET5BIG_V2_GPIO_HD3_PRES,
		.have_power_pin	= 1,
		.have_pres_pin	= 1,
		.pres_act_low	= 1,
	},
	[4] = {
		.name		= "hd4",
		.power_pin	= NET5BIG_V2_GPIO_HD4_POWER,
		.pres_pin	= NET5BIG_V2_GPIO_HD4_PRES,
		.have_power_pin	= 1,
		.have_pres_pin	= 1,
		.pres_act_low	= 1,
	},
	[5] = {
		.name		= "esata0",
		.is_extern	= 1,
	},
	[6] = {
		.name		= "esata1",
		.is_extern	= 1,
	},
};

static struct gpio_hd_power_platform_data net5big_v2_hd_power_data = {
	.hd		= net5big_v2_hd_power,
	.num_hds	= ARRAY_SIZE(net5big_v2_hd_power),
};

static struct platform_device net5big_v2_gpio_hd_power = {
	.name		= "gpio-hd-power",
	.id		= -1,
	.dev		= {
		.platform_data 	= &net5big_v2_hd_power_data,
	},
};

/* I2C fan */

static struct g762_platform_data net5big_v2_g762_data  = {
       .ref_clk = 32768,
};

/* I2C eeprom */

static struct i2c_board_info net5big_v2_i2c_info[] = {
	{
		I2C_BOARD_INFO("at24", 0x50),
		.type = "24c04",
	}, {
		I2C_BOARD_INFO("g762", 0x3e),
		.type = "g762",
		.platform_data = &net5big_v2_g762_data,
	}
};

/* internal disk map */

#if defined(CONFIG_SATA_MV)
static struct lacie_disk net5big_v2_disks[] = {
	{
		.host_no = 0,
		.channel = 3,
		.id 	 = 0,
		.letter  = 'a',
	},
	{
		.host_no = 0,
		.channel = 2,
		.id 	 = 0,
		.letter  = 'b',
	},
	{
		.host_no = 0,
		.channel = 1,
		.id 	 = 0,
		.letter  = 'c',
	},
	{
		.host_no = 1,
		.channel = 2,
		.id 	 = 0,
		.letter  = 'd',
	},
	{
		.host_no = 1,
		.channel = 1,
		.id 	 = 0,
		.letter  = 'e',
	},
};
#else
static struct lacie_disk net5big_v2_disks[] = {
	{
		.host_no = 0,
		.channel = 0,
		.id 	 = 3,
		.letter  = 'a',
	},
	{
		.host_no = 0,
		.channel = 0,
		.id 	 = 2,
		.letter  = 'b',
	},
	{
		.host_no = 0,
		.channel = 0,
		.id 	 = 1,
		.letter  = 'c',
	},
	{
		.host_no = 1,
		.channel = 0,
		.id 	 = 2,
		.letter  = 'd',
	},
	{
		.host_no = 1,
		.channel = 0,
		.id 	 = 1,
		.letter  = 'e',
	},
};
#endif

static struct lacie_disk_map net5big_v2_disk_map = {
	.disks = net5big_v2_disks,
	.num_disks = ARRAY_SIZE(net5big_v2_disks),
};

/* SATA ports */

static struct mv_sata_platform_data net5big_v2_sata_data = {
	.n_ports = 2,
};

/* Registration */

static struct platform_device *net5big_v2_gpio_devices[] __initdata = {
	&net5big_v2_gpio_buttons,
	&net5big_v2_gpio_hd_power,
	&net5big_v2_gpio_usb,
	&net5big_v2_leds,
};

static void __init net5big_v2_gpio_init(void)
{
        platform_add_devices(net5big_v2_gpio_devices,
				ARRAY_SIZE(net5big_v2_gpio_devices));
}

static void __init net5big_v2_i2c_eeprom_init(void)
{
	i2c_register_board_info(0, net5big_v2_i2c_info,
				ARRAY_SIZE(net5big_v2_i2c_info));
}

static void __init net5big_v2_disk_map_init(void)
{
	lacie_disk_map = &net5big_v2_disk_map;
}

static void net5big_v2_power_off(void)
{
	gpio_set_value(NET5BIG_V2_GPIO_POWER_OFF, 1);
}

#define SPINUP_DISK_TIMEOUT	10
#define SPINUP_MAX_DISKS_NUM	1

void __init net5big_v2_init(void)
{
	int err;

	/*
	 * Basic setup. Needs to be called early.
	 */
	net5big_v2_boardenv_early_init();
	mv_init();

	feroceon_kw_sata_init(&net5big_v2_sata_data);
	feroceon_kw_ge00_init(&net5big_v2_ge00_data);
	feroceon_kw_ge01_init(&net5big_v2_ge01_data);
	mv_usb_init();
	if (machine_is_net5big_nand_v2())
		lacie_v2_nand_init();
	else
		lacie_v2_sf_init();

	net5big_v2_gpio_init();
	net5big_v2_i2c_eeprom_init();
	net5big_v2_disk_map_init();

	spinup_set_params(SPINUP_DISK_TIMEOUT, SPINUP_MAX_DISKS_NUM);

	err = gpio_request(NET5BIG_V2_GPIO_POWER_OFF, "power-off");
	if (err == 0) {
		err = gpio_direction_output(NET5BIG_V2_GPIO_POWER_OFF, 0);
		if (err)
			gpio_free(NET5BIG_V2_GPIO_POWER_OFF);
	}
	if (err)
		pr_err("net5big_v2: failed to configure power-off GPIO\n");

	/* Inhibit power-off switch action. */
	err = gpio_request(NET5BIG_V2_GPIO_INHIBIT_POWER_OFF,
			   "inhibit power-off");
	if (err == 0) {
		err = gpio_direction_output(NET5BIG_V2_GPIO_INHIBIT_POWER_OFF,
					    1);
		if (err)
			gpio_free(NET5BIG_V2_GPIO_INHIBIT_POWER_OFF);
	}
	if (err)
		pr_err("net5big_v2: failed to configure inhibit power-off GPIO\n");

	pm_power_off = net5big_v2_power_off;

	return;
}

MACHINE_START(NET5BIG_V2, "net5big_v2")
	.phys_io = 0xf1000000,
	.io_pg_offst = ((0xf1000000) >> 18) & 0xfffc,
	.boot_params = 0x00000100,
	.init_machine = net5big_v2_init,
	.map_io = mv_map_io,
	.init_irq = feroceon_kw_init_irq,
	.timer = &mv_timer,
MACHINE_END

MACHINE_START(NET5BIG_NAND_V2, "net5big_nand_v2")
	.phys_io = 0xf1000000,
	.io_pg_offst = ((0xf1000000) >> 18) & 0xfffc,
	.boot_params = 0x00000100,
	.init_machine = net5big_v2_init,
	.map_io = mv_map_io,
	.init_irq = feroceon_kw_init_irq,
	.timer = &mv_timer,
MACHINE_END
