/*
 * arch/arm/mach-feroceon-kw/lacie/wireless_space-setup.c
 *
 * LaCie Wireless Space board setup.
 *
 * Copyright (C) 2009 LaCie
 *
 * Author: Simon Guinot <sguinot@lacie.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/kernel.h>
#include <linux/platform_device.h>
#include <linux/i2c.h>
#include <linux/input.h>
#include <linux/gpio_keys.h>
#include <linux/leds.h>
#include <linux/ata_platform.h>
#include <linux/mv643xx_eth.h>
#include <linux/gpio.h>
#include <linux/gpio-fan.h>
#include <linux/mtd/partitions.h>
#include <asm/mach-types.h>
#include <asm/mach/arch.h>

#include "core.h"
#include "lacie_v2-common.h"
#include "marvell_boardenv/wireless_space-boardenv.h"
#include "lacie/common.h"
#include "lacie/drivers/gpio-hd-power.h"
#include "lacie/drivers/gpio-usb.h"
#include "lacie/drivers/gpio-ws-leds.h"

extern struct lacie_disk_map *lacie_disk_map;

/* Ethernet */

static struct mv643xx_eth_platform_data wireless_space_ge00_data = {
	.phy_addr	= MV643XX_ETH_PHY_ADDR(0xa),
};

static struct mv643xx_eth_platform_data wireless_space_ge01_data = {
	.phy_addr	= MV643XX_ETH_PHY_ADDR(0x6),
};

/* NAND flash */

static struct mtd_partition wireless_space_nand_parts[] = {
        {
                .name = "u-boot",
                .offset = 0,
                .size = SZ_1M
        }, {
                .name = "uImage",
                .offset = MTDPART_OFS_NXTBLK,
                .size = 0x00300000
        }, {
                .name = "uInitrd_m",
                .offset = MTDPART_OFS_NXTBLK,
                .size = SZ_4M
        }, {
                .name = "root",
                .offset = MTDPART_OFS_NXTBLK,
                .size = MTDPART_SIZ_FULL
        },
};

/* GPIO buttons */

static struct gpio_keys_button wireless_space_buttons[] = {
	[0] = {
		.code		= KEY_POWER,
		.gpio		= WIRELESS_SPACE_GPIO_KEY_POWER,
		.desc		= "Rear power button",
	},
	[1] = {
		.code		= KEY_WLAN,
		.gpio		= WIRELESS_SPACE_GPIO_KEY_WPS,
		.desc		= "Front Wifi button (WPS, activity)",
		.active_low	= 1,
	},
};

static struct gpio_keys_platform_data wireless_space_button_data = {
	.buttons	= wireless_space_buttons,
	.nbuttons	= ARRAY_SIZE(wireless_space_buttons),
};

static struct platform_device wireless_space_gpio_buttons = {
	.name		= "gpio-keys",
	.id		= -1,
	.dev		= {
		.platform_data 	= &wireless_space_button_data,
	},
};

/*
 * Wireless Space GPIO LEDs
 *
 * Hardware blink capabilities:
 *
 * - hardware blink: the red LED can be configured to blink at a fixed hardware
 *   frequency.
 * - HD or SATA blink: the blue and green LEDs can be configured to blink in
 *   relation with the disk ot SATA activity. Note that the SATA blink GPIO is
 *   shared between all the LEDs.
 */


static struct ws_led wireless_space_leds[] = {
	[0] = {
		.name		= "ws:red:front",
		.enable		= {
			.name	= "red_led_enable",
			.gpio	= WIRELESS_SPACE_GPIO_LED_RED,
		},
		.hw_blink	= {
			.name		= "red_led_hw_blink",
			.gpio		= WIRELESS_SPACE_GPIO_LED_RED_BLINK,
			.active_low	= 1,
		},
		.delay_on	= 100,
		.delay_off	= 100,
		.have_hw_blink	= 1,
	},
	[1] = {
		.name		= "ws:green:front",
		.enable		= {
			.name	= "green_led_enable",
			.gpio	= WIRELESS_SPACE_GPIO_LED_GREEN,
		},
		.have_hd_blink	= 1,
	},
	[2] = {
		.name		= "ws:blue:front",
		.enable		= {
			.name	= "blue_led_enable",
			.gpio	= WIRELESS_SPACE_GPIO_LED_BLUE,
		},
		.have_hd_blink	= 1,
	},
};

static struct ws_led_platform_data wireless_space_gpio_leds_data = {
	.num_leds	= ARRAY_SIZE(wireless_space_leds),
	.leds		= wireless_space_leds,
	.clk_ctrl	= {
		.name	= "led_clk_ctrl",
		.gpio	= WIRELESS_SPACE_GPIO_LED_CLK_CTRL,
	},
	.hd_blink	= {
		.name		= "led_hd_blink",
		.gpio		= WIRELESS_SPACE_GPIO_LED_SATA_BLINK,
		.active_low	= 1,
	},
};

static struct platform_device wireless_space_gpio_leds = {
	.name		= "gpio-ws-leds",
	.id		= -1,
	.dev		= {
		.platform_data	= &wireless_space_gpio_leds_data,
	},
};

/* GPIO FAN */

/* Designed for fan 40x40x16: ADDA AD0412LB-D50 6000rpm@12v */
static struct gpio_fan_speed wireless_space_fan_speed[] = {
	{    0,  0 },
	{ 1220, 15 },
	{ 1325, 14 },
	{ 1700, 13 },
	{ 2000, 12 },
	{ 3200, 11 },
	{ 3800, 10 },
	{ 4900,  9 },
	{ 5400,  8 },
};

static unsigned wireless_space_fan_ctrl[] = { 37, 7, 36, 38 };

static struct gpio_fan_alarm wireless_space_fan_alarm = {
	.gpio		= 39,
	.active_low	= 1,
};

static struct gpio_fan_platform_data wireless_space_fan_data = {
	.num_ctrl	= ARRAY_SIZE(wireless_space_fan_ctrl),
	.ctrl		= wireless_space_fan_ctrl,
	.alarm		= &wireless_space_fan_alarm,
	.num_speed	= ARRAY_SIZE(wireless_space_fan_speed),
	.speed		= wireless_space_fan_speed,
};

static struct platform_device wireless_space_gpio_fan = {
	.name	= "gpio-fan",
	.id	= -1,
	.dev	= {
		.platform_data	= &wireless_space_fan_data,
	},
};

/* GPIO USB */

static struct gpio_usb wireless_space_gpio_usb_port0[] = {
	[0] = {
		.name	= "fuse",
		.type	= GPIO_USB_FUSE,
		.num	= WIRELESS_SPACE_GPIO_USB_FUSE,
	},
	[1] = {
		.name	= "vbus-out",
		.type	= GPIO_USB_VBUS_OUT,
		.num	= WIRELESS_SPACE_GPIO_USB_VBUS_OUT,
	},
};

static struct gpio_usb_port wireless_space_gpio_usb_ports[] = {
	[0] = {
		.name		= "port0",
		.gpio		= wireless_space_gpio_usb_port0,
		.num_gpio	= ARRAY_SIZE(wireless_space_gpio_usb_port0),
		.reload_delay	= 10000,
		.spurious_delay	= 100,
	},
};

struct gpio_usb_platform_data wireless_space_lacie_gpio_usb_data = {
	.port		= wireless_space_gpio_usb_ports,
	.num_port	= ARRAY_SIZE(wireless_space_gpio_usb_ports),
};

static struct platform_device wireless_space_gpio_usb = {
	.name		= "gpio-usb",
	.id		= -1,
	.dev		= {
		.platform_data	= &wireless_space_lacie_gpio_usb_data,
	},
};

/* GPIO hard disk power */

static struct gpio_hd_power wireless_space_hd_power[] = {
	[0] = {
		.name		= "hd0",
		.power_pin	= WIRELESS_SPACE_GPIO_HD0_POWER,
		.have_power_pin	= 1,
	},
};

static struct gpio_hd_power_platform_data wireless_space_hd_power_data = {
	.hd		= wireless_space_hd_power,
	.num_hds	= ARRAY_SIZE(wireless_space_hd_power),
};

static struct platform_device wireless_space_gpio_hd_power = {
	.name		= "gpio-hd-power",
	.id		= -1,
	.dev		= {
		.platform_data 	= &wireless_space_hd_power_data,
	},
};

/* I2C eeprom */

static struct i2c_board_info wireless_space_i2c_info[] = {
	{
		I2C_BOARD_INFO("at24", 0x50),
		.type = "24c04",
	},
};

/* internal disk map */

static struct lacie_disk wireless_space_disks[] = {
	{
		.host_no = 0,
		.channel = 0,
		.id 	 = 0,
		.letter  = 'a',
	},
};

static struct lacie_disk_map wireless_space_disk_map = {
	.disks = wireless_space_disks,
	.num_disks = ARRAY_SIZE(wireless_space_disks),
};

/* SATA ports */

static struct mv_sata_platform_data wireless_space_sata_data = {
	.n_ports = 1,
};

/* Registration */

static struct platform_device *wireless_space_gpio_devices[] __initdata = {
	&wireless_space_gpio_buttons,
	&wireless_space_gpio_fan,
	&wireless_space_gpio_hd_power,
	&wireless_space_gpio_leds,
	&wireless_space_gpio_usb,
};

static void __init wireless_space_gpio_init(void)
{
        platform_add_devices(wireless_space_gpio_devices,
				ARRAY_SIZE(wireless_space_gpio_devices));
}

static void __init wireless_space_i2c_eeprom_init(void)
{
	i2c_register_board_info(0, wireless_space_i2c_info,
				ARRAY_SIZE(wireless_space_i2c_info));
}

static void __init wireless_space_disk_map_init(void)
{
	lacie_disk_map = &wireless_space_disk_map;
}


static void wireless_space_power_off(void)
{
	gpio_set_value(WIRELESS_SPACE_GPIO_POWER_OFF, 1);
}

void __init wireless_space_init(void)
{
	int err;

	/*
	 * Basic setup. Needs to be called early.
	 */
	wireless_space_boardenv_early_init();
	mv_init();

	feroceon_kw_sata_init(&wireless_space_sata_data);
	feroceon_kw_ge00_init(&wireless_space_ge00_data);
	feroceon_kw_ge00_init(&wireless_space_ge01_data);
	mv_usb_init();

	feroceon_kw_nand_init(wireless_space_nand_parts,
			      ARRAY_SIZE(wireless_space_nand_parts), 25);

	wireless_space_gpio_init();
	wireless_space_i2c_eeprom_init();
	wireless_space_disk_map_init();

	err = gpio_request(WIRELESS_SPACE_GPIO_POWER_OFF, "power-off");
	if (err == 0) {
		err = gpio_direction_output(WIRELESS_SPACE_GPIO_POWER_OFF, 0);
		if (err)
			gpio_free(WIRELESS_SPACE_GPIO_POWER_OFF);
	}
	if (err)
		pr_err("wireless-space: failed to configure power-off GPIO\n");

	pm_power_off = wireless_space_power_off;

	return;
}

MACHINE_START(WIRELESS_SPACE ,"wireless_space")
	.phys_io = 0xf1000000,
	.io_pg_offst = ((0xf1000000) >> 18) & 0xfffc,
	.boot_params = 0x00000100,
	.init_machine = wireless_space_init,
	.map_io = mv_map_io,
	.init_irq = feroceon_kw_init_irq,
	.timer = &mv_timer,
MACHINE_END
