/*
 * lacie/lacie-common.c
 *
 * Copyright (c) 2009 LaCie
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <scsi/scsi_device.h>
#include <scsi/scsi_host.h>

#include "common.h"

struct lacie_usb_storage_id *lacie_usb_storage_id = NULL;
EXPORT_SYMBOL(lacie_usb_storage_id);

/* contains internal/external drives mapping */
struct lacie_disk_map *lacie_disk_map = NULL;

/**
 *	lacie_get_sd_name - called by the scsi layer sd.c by the
 *	sd_probe method. It is called once for each scsi device
 *	(not just disks) present.
 *	This function permit to set a SCSI disk name accordingly to a board
 *	specific internal disk map. This allow to associate a SATA disk (host,
 *	channel and id) with a fixed device name.
 *	External disk (USB and eSATA) are renamed as /dev/se[x].
 *
 *	@sdp: scsi device
 *	@prefix_letter: scsi prefix letter to return
 *                      'd' for an internal disk
 *                      'e' for an external disk
 *	@index: scsi disk index to return
 *
 *	The input parameters are only modified if the given SCSI device match
 *	a map entry. If no disk map is available, return -ENODEV and 0
 *	otherwise.
 *
 **/
int lacie_get_sd_name(struct scsi_device *sdp, char *prefix_letter, u32 *index)
{
	int i;
	struct lacie_disk *disk;

	/* if the disk map is not declared */
	if (!lacie_disk_map) {
		pr_err("No LaCie disk map available\n");
		return -ENODEV;
	}

	/* Found an USB disk  */
	if (strcmp("usb-storage", sdp->host->hostt->name) == 0)
		goto exit_external;

	/* Walk through the internal disk map and try to match an entry */
	for (i = 0; i < lacie_disk_map->num_disks; i++) {
		disk = &lacie_disk_map->disks[i];
		if (sdp->channel == disk->channel && sdp->id == disk->id &&
		    sdp->host->host_no == disk->host_no) {
			*prefix_letter = 'd';
			*index = disk->letter - 'a';
			return 0;
		}

	}

	/* Found an eSATA disk */

exit_external:
	*prefix_letter = 'e';
	return 0;
}

/**
 *	lacie_get_disk_num - could be called by the SCSI layer sd.c or
 *	by a low level SCSI driver (SATA driver for example).
 *	This function allow to get the case disk number associated with
 *	a SCSI disk.
 *	@sdp: scsi device
 *
 *	Returns "disk number" or -1
 *
 **/
int lacie_get_disk_num(struct scsi_device *sdp)
{
	int i;
	struct lacie_disk *disk;

	if (unlikely(!lacie_disk_map))
		return -1;

	for (i = 0; i < lacie_disk_map->num_disks; i++) {
		disk = &lacie_disk_map->disks[i];
		if (sdp->channel == disk->channel && sdp->id == disk->id &&
			sdp->host->host_no == disk->host_no) {
			return i;
		}
	}
	return -1;
}
EXPORT_SYMBOL(lacie_get_disk_num);
