/*
 * lacie/drivers/gpio-hd-power.c
 *
 * LaCie GPIO hard disk power driver
 *
 * Copyright (c) 2009 LaCie
 * 	Benoit Canet <benoit.canet@gmail.com>
 *	Simon Guinot <sguinot@lacie.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * TODO: Add support for disk power up completion. May be look for SCSI host
 *       initialization.
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/irq.h>
#include <linux/platform_device.h>
#include <linux/gpio.h>

#include "gpio-hd-power.h"

#define POWER_UP_TIMEOUT	15000

struct hd_power {
	const char *name;
	struct kobject kobj;
	unsigned power_pin;
	unsigned pres_pin;
	struct work_struct hd_pres_notify_work;
	struct platform_device *pdev;
	u8 power_up_complete:1;
	u8 have_power_pin:1;
	u8 have_pres_pin:1;
	u8 power_act_low:1;
	u8 pres_act_low:1;
};
#define to_hd_power_obj(x) container_of(x, struct hd_power, kobj)

static struct hd_power *hd_power_array[MAX_DISK_NUM];
static struct mutex hd_power_mutex;
static wait_queue_head_t hd_power_wait;
static int extern_disk_num;

struct hd_power_attribute {
	struct attribute attr;
	size_t (*show)(struct hd_power *hd_power,
		       struct hd_power_attribute *attr, char *buff);
	size_t (*store)(struct hd_power *hd_power,
			struct hd_power_attribute *attr, const char *buff,
			size_t count);
};
#define to_hd_power_attr(x) container_of(x, struct hd_power_attribute, attr)

static void hd_power_release(struct kobject *kobj)
{
	struct hd_power *hd_power = to_hd_power_obj(kobj);

	kfree(hd_power);
}

static ssize_t hd_power_show(struct kobject *kobj, struct attribute *attr,
			     char *buf)
{
	struct hd_power *hd_power = to_hd_power_obj(kobj);
	struct hd_power_attribute *hd_power_attr = to_hd_power_attr(attr);

	if (!hd_power_attr->show)
		return -EIO;

	return hd_power_attr->show(hd_power, hd_power_attr, buf);
}

static ssize_t hd_power_store(struct kobject *kobj, struct attribute *attr,
			      const char *buf, size_t count)
{
	struct hd_power *hd_power = to_hd_power_obj(kobj);
	struct hd_power_attribute *hd_power_attr = to_hd_power_attr(attr);

	if (!hd_power_attr->store)
		return -EIO;

	return hd_power_attr->store(hd_power, hd_power_attr, buf, count);
}

static struct sysfs_ops hd_power_sysfs_ops = {
	.show = hd_power_show,
	.store = hd_power_store,
};

static struct kobj_type hd_power_ktype = {
	.release = hd_power_release,
	.sysfs_ops = &hd_power_sysfs_ops,
	.default_attrs = NULL,
};

static struct kset hd_power_kset = {
	.kobj = {
		.name = "hd-power",
	},
};

static inline int hd_power_set(struct hd_power *hd_power, int enable)
{
	int power, pres;

	if (!hd_power->have_power_pin)
		return -ENODEV;

	if (hd_power->have_pres_pin) {
		pres = gpio_get_value(hd_power->pres_pin);
		pres = hd_power->pres_act_low ? !pres : pres;
		if (!pres)
			return -ENODEV;
	}

	power = gpio_get_value(hd_power->power_pin);
	power = hd_power->power_act_low ? !power : power;
	if (power == enable)
		return 0;

	/* Enable or disable hard disk power. In the case of a power enable
	 * request, wait for completion or timeout. This delay prevent from
	 * power up several disks at the same time. */
	hd_power->power_up_complete = 0;
	gpio_set_value(hd_power->power_pin,
		       hd_power->power_act_low? !enable : enable);
	if (enable)
		wait_event_timeout(hd_power_wait,
				   hd_power->power_up_complete != 0,
				   msecs_to_jiffies(POWER_UP_TIMEOUT));

	return 0;
}

static size_t power_show(struct hd_power *hd_power,
			 struct hd_power_attribute *attr, char *buff)
{
	int enable;

	enable = gpio_get_value(hd_power->power_pin);
	return sprintf(buff, "%i\n",
		       hd_power->power_act_low? !enable : enable);
}

static size_t power_store(struct hd_power *hd_power,
			  struct hd_power_attribute *attr,
			  const char *buff, size_t count)
{
	int err = 0, enable;

	if (sscanf(buff, "%i", &enable) != 1 || (enable != (enable & 1)))
		return -EINVAL;

	mutex_lock(&hd_power_mutex);
	err = hd_power_set(hd_power, enable);
	mutex_unlock(&hd_power_mutex);

	return err ? err : count;
}

static struct hd_power_attribute power_attribute =
	__ATTR(power, S_IRUGO | S_IWUSR, power_show, power_store);

static size_t pres_show(struct hd_power *hd_power,
			struct hd_power_attribute *attr, char *buff)
{
	int enable;

	enable = gpio_get_value(hd_power->pres_pin);
	return sprintf(buff, "%i\n", hd_power->pres_act_low? !enable : enable);
}

static struct hd_power_attribute pres_attribute =
	__ATTR(presence, S_IRUGO, pres_show, NULL);

static ssize_t all_hd_power_store(struct device *dev,
				  struct device_attribute *attr,
				  const char *buff, size_t count)
{
	int hd, enable;

	if (sscanf(buff, "%i", &enable) != 1 || (enable != (enable & 1)))
		return -EINVAL;

	mutex_lock(&hd_power_mutex);

	for (hd = 0; (hd < MAX_DISK_NUM) && hd_power_array[hd]; hd++) {
		hd_power_set(hd_power_array[hd], enable);
	}

	mutex_unlock(&hd_power_mutex);

	return count;
}

static DEVICE_ATTR(power, S_IRUGO | S_IWUSR, NULL, all_hd_power_store);

static ssize_t hd_num_show(struct device *dev,
			   struct device_attribute *attr, char *buff)
{
	struct platform_device *pdev = to_platform_device(dev);
	struct gpio_hd_power_platform_data *pdata = pdev->dev.platform_data;

	return sprintf(buff, "%i\n", pdata->num_hds - extern_disk_num);
}

static DEVICE_ATTR(hd_num, S_IRUGO | S_IWUSR, hd_num_show, NULL);

/* Notify userspace from a disk plug or unplug. */
static void hd_pres_notify(struct work_struct *ws)
{
	struct hd_power *hd_power =
		container_of(ws, struct hd_power, hd_pres_notify_work);

	sysfs_notify(&hd_power->kobj, NULL, "presence");
	kobject_uevent(&hd_power->kobj, KOBJ_CHANGE);
}

static irqreturn_t hd_pres_irq_handler(int irq, void *data)
{
	struct hd_power *hd_power = data;
	int pres, power;

	if (hd_power->have_power_pin) {
		/* Set power pin off when the hard disk is unpluged. */
		pres = gpio_get_value(hd_power->pres_pin);
		pres = hd_power->pres_act_low ? !pres : pres;
		power = gpio_get_value(hd_power->power_pin);
		power = hd_power->power_act_low ? !power : power;
		if (!pres && power) {
			dev_dbg(&hd_power->pdev->dev, "%s: power off\n",
				hd_power->name);
			gpio_set_value(hd_power->power_pin,
				       hd_power->power_act_low? 1 : 0);
		}
	}

	schedule_work(&hd_power->hd_pres_notify_work);

	return IRQ_HANDLED;
}

static inline void free_hd_power(struct hd_power *hd_power)
{
	if (hd_power->have_power_pin) {
		sysfs_remove_file(&hd_power->kobj, &power_attribute.attr);
		gpio_free(hd_power->power_pin);
	}
	if (hd_power->have_pres_pin) {
		free_irq(gpio_to_irq(hd_power->pres_pin), hd_power);
		sysfs_remove_file(&hd_power->kobj, &pres_attribute.attr);
		gpio_free(hd_power->pres_pin);
	}
	kobject_put(&hd_power->kobj);
}

static int gpio_hd_power_probe(struct platform_device *pdev)
{
	struct gpio_hd_power_platform_data *pdata = pdev->dev.platform_data;
	struct gpio_hd_power *gpio_hd_power;
	int hd, err;

	if (unlikely(pdata->num_hds > MAX_DISK_NUM))
		return -EINVAL;

	err = kset_register(&hd_power_kset);
	if (err)
		return err;

	mutex_init(&hd_power_mutex);
	init_waitqueue_head(&hd_power_wait);
	extern_disk_num = 0;

	for (hd = 0; hd < pdata->num_hds; hd++) {
		struct hd_power *hd_power;
		char gpio_name[16];
		int pres_irq;

		gpio_hd_power = &pdata->hd[hd];
		hd_power = kzalloc(sizeof (*hd_power), GFP_KERNEL);
		hd_power_array[hd] = hd_power;

		if (gpio_hd_power->is_extern)
			extern_disk_num += 1;

		hd_power->name = gpio_hd_power->name;

		/*
		 * Create a new hard disk sub-kobject for gpio-hd-power.
		 */
		hd_power->kobj.kset = &hd_power_kset;
		err = kobject_init_and_add(&hd_power->kobj, &hd_power_ktype,
					   &pdev->dev.kobj, "%s",
					   hd_power->name);
		if (err) {
			dev_err(&pdev->dev, "failed to register kobject: %s\n",
				kobject_name(&hd_power->kobj));
			goto exit_free_hd_power;
		}
		kobject_uevent(&hd_power->kobj, KOBJ_ADD);

		hd_power->pdev = pdev;

		/*
		 * Add power attribute.
		 */
		if (gpio_hd_power->have_power_pin) {
			hd_power->power_pin = gpio_hd_power->power_pin;
			hd_power->power_act_low = gpio_hd_power->power_act_low;
			sprintf(gpio_name, "%s_power", hd_power->name);
			err = gpio_request(hd_power->power_pin, gpio_name);
			if (err) {
				dev_err(&pdev->dev,
					"failed to request GPIO %d: %s\n",
					hd_power->power_pin, gpio_name);
				goto exit_free_hd_power;
			}
			err = gpio_direction_output(hd_power->power_pin,
					gpio_get_value(hd_power->power_pin));
			if (err) {
				dev_err(&pdev->dev,
					"failed to configure GPIO %d: %s\n",
					hd_power->power_pin, gpio_name);
				gpio_free(hd_power->power_pin);
				goto exit_free_hd_power;
			}
			err = sysfs_create_file(&hd_power->kobj,
						&power_attribute.attr);
			if (err) {
				dev_err(&pdev->dev,
					"failed to create %s/power\n",
					kobject_name(&hd_power->kobj));
				gpio_free(hd_power->power_pin);
				goto exit_free_hd_power;
			}
			hd_power->have_power_pin = 1;
		}

		/*
		 * Add presence attribute.
		 */
		if (gpio_hd_power->have_pres_pin) {
			hd_power->pres_pin = gpio_hd_power->pres_pin;
			hd_power->pres_act_low = gpio_hd_power->pres_act_low;
			sprintf(gpio_name, "%s_presence", hd_power->name);
			err = gpio_request(hd_power->pres_pin, gpio_name);
			if (err) {
				dev_err(&pdev->dev,
					"failed to request GPIO %d: %s\n",
					hd_power->pres_pin, gpio_name);
				goto exit_free_hd_power;
			}
			err = gpio_direction_input(hd_power->pres_pin);
			if (err) {
				dev_err(&pdev->dev,
					"failed to configure GPIO %d: %s\n",
					hd_power->power_pin, gpio_name);
				gpio_free(hd_power->power_pin);
				goto exit_free_hd_power;
			}
			err = sysfs_create_file(&hd_power->kobj,
						&pres_attribute.attr);
			if (err) {
				dev_err(&pdev->dev,
					"failed to create %s/presence\n",
					kobject_name(&hd_power->kobj));
				gpio_free(hd_power->pres_pin);
				goto exit_free_hd_power;
			}
			INIT_WORK(&hd_power->hd_pres_notify_work,
				  hd_pres_notify);
			pres_irq = gpio_to_irq(hd_power->pres_pin);
			set_irq_type(pres_irq, IRQ_TYPE_EDGE_BOTH);
			err = request_irq(pres_irq, hd_pres_irq_handler,
					  0, hd_power->name, hd_power);
			if (err) {
				dev_err(&pdev->dev,
					"failed to register presence irq %d\n",
					pres_irq);
				sysfs_remove_file(&hd_power->kobj,
						  &pres_attribute.attr);
				gpio_free(hd_power->pres_pin);
				goto exit_free_hd_power;
			}
			hd_power->have_pres_pin = 1;
		}
	}

	/* Attach a power attribute to the main kobject gpio-hd-power. */
	err = device_create_file(&pdev->dev, &dev_attr_power);
	if (err) {
		dev_err(&pdev->dev,
			"failed to create gpio-hd-power/power\n");
		goto exit_free_hd_power;
	}

	/* Attach a hd_num attribute to the main kobject gpio-hd-power. */
	err = device_create_file(&pdev->dev, &dev_attr_hd_num);
	if (err) {
		dev_err(&pdev->dev,
			"failed to create gpio-hd-power/hd_num\n");
		goto exit_free_attr_power;
	}

	dev_info(&pdev->dev, "GPIO Hard Disk power device initialized\n");

	return 0;

exit_free_attr_power:
	device_remove_file(&pdev->dev, &dev_attr_power);
exit_free_hd_power:
	for (; hd >= 0; hd--)
		free_hd_power(hd_power_array[hd]);

	return err;
}

static int __devexit gpio_hd_power_remove(struct platform_device *pdev)
{
	struct gpio_hd_power_platform_data *pdata = pdev->dev.platform_data;
	int hd;

	for (hd = 0; hd < pdata->num_hds; hd++) {
		free_hd_power(hd_power_array[hd]);
	}
	device_remove_file(&pdev->dev, &dev_attr_power);
	device_remove_file(&pdev->dev, &dev_attr_hd_num);
	kset_unregister(&hd_power_kset);

	return 0;
}

static struct platform_driver gpio_hd_power_driver = {
	.probe 	= gpio_hd_power_probe,
	.remove = __devexit_p(gpio_hd_power_remove),
	.driver = {
		.name = "gpio-hd-power",
	},
};

static int __init gpio_hd_power_init(void)
{
	return platform_driver_register(&gpio_hd_power_driver);
}

static void __exit gpio_hd_power_exit(void)
{
	platform_driver_unregister(&gpio_hd_power_driver);
}

module_init(gpio_hd_power_init);
module_exit(gpio_hd_power_exit);

MODULE_AUTHOR("Benoit CANET <benoit.canet@gmail.com>");
MODULE_DESCRIPTION("LaCie GPIO Hard Disk power driver");
MODULE_LICENSE("GPL");
