/*
 * lacie/drivers/gpio-ws-leds.c
 *
 * LaCie Wireless Space GPIO LEDs driver.
 *
 * Copyright (c) 2009 LaCie
 *
 * Author: Simon Guinot <sguinot@lacie.com>
 *
 * Highly based on the gpio-leds driver.
 * 	Copyright (C) 2007 8D Technologies inc.
 * 	Raphael Assenat <raph@8d.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/platform_device.h>
#include <linux/leds.h>
#include <linux/delay.h>
#include <linux/gpio.h>

#include "gpio-ws-leds.h"

struct ws_led_data {
	struct led_classdev cdev;
	struct ws_led_gpio enable;
	struct ws_led_gpio hw_blink;
	u32 delay_on;
	u32 delay_off;
	u8 have_hd_blink:1;
	u8 have_hw_blink:1;
};

static struct ws_led_gpio clk_ctrl;
static struct ws_led_gpio hd_blink;
static spinlock_t lock; /* Clock control lock. */

static inline void ws_led_gpio_set(struct ws_led_gpio *ws_led_gpio, int value)
{
	BUG_ON(ws_led_gpio == NULL);

	gpio_set_value(ws_led_gpio->gpio,
		       ws_led_gpio->active_low ? !value : value);
}

static inline int ws_led_gpio_get(struct ws_led_gpio *ws_led_gpio)
{
	int value;

	BUG_ON(ws_led_gpio == NULL);

	value = gpio_get_value(ws_led_gpio->gpio);
	return ws_led_gpio->active_low ? !value : value;
}

static void ws_led_set(struct led_classdev *led_cdev,
			    enum led_brightness value)
{
	struct ws_led_data *led_dat =
		container_of(led_cdev, struct ws_led_data, cdev);
	struct ws_led_gpio *enable = &led_dat->enable;
	struct ws_led_gpio *hw_blink = &led_dat->hw_blink;

	int level;

	if (value == LED_OFF)
		level = 0;
	else
		level = 1;

	spin_lock(&lock);
	ws_led_gpio_set(&clk_ctrl, 0);

	/* Disable hardware blink modes */
	if (led_dat->have_hw_blink)
		ws_led_gpio_set(hw_blink, 0);
	if (led_dat->have_hd_blink)
		ws_led_gpio_set(&hd_blink, 0);

	ws_led_gpio_set(enable, level);

	ws_led_gpio_set(&clk_ctrl, 1);
	spin_unlock(&lock);
}

static int ws_led_hw_blink_set(struct led_classdev *led_cdev,
			       unsigned long *delay_on,
			       unsigned long *delay_off)
{
	int ret = -EINVAL;
	struct ws_led_data *led_dat =
		container_of(led_cdev, struct ws_led_data, cdev);
	struct ws_led_gpio *enable = &led_dat->enable;
	struct ws_led_gpio *hw_blink = &led_dat->hw_blink;

	/* Use hardware blink if the requested blink frequency is available. */
	if (*delay_on == led_dat->delay_on &&
	    *delay_off == led_dat->delay_off) {
		spin_lock(&lock);
		ws_led_gpio_set(&clk_ctrl, 0);
		ws_led_gpio_set(hw_blink, 1);
		ws_led_gpio_set(enable, 1);
		ws_led_gpio_set(&clk_ctrl, 1);
		spin_unlock(&lock);
		ret = 0;
	}
	return ret;
}

static ssize_t ws_led_hd_blink_store(struct device *dev,
				     struct device_attribute *attr,
				     const char *buf, size_t count)
{
	int value;
	struct ws_led_data *led_dat = dev_get_drvdata(dev);
	struct ws_led_gpio *enable = &led_dat->enable;

	value = !!simple_strtol(buf, NULL, 10);

	spin_lock(&lock);
	ws_led_gpio_set(&clk_ctrl, 0);
	ws_led_gpio_set(&hd_blink, value);
	ws_led_gpio_set(enable, value);
	ws_led_gpio_set(&clk_ctrl, 1);
	spin_unlock(&lock);

	return count;
}

static ssize_t ws_led_hd_blink_show(struct device *dev,
				    struct device_attribute *attr, char *buf)
{
	return sprintf(buf, "%i\n", ws_led_gpio_get(&hd_blink));
}

static DEVICE_ATTR(hd_blink, 0644, ws_led_hd_blink_show, ws_led_hd_blink_store);

static inline int configure_ws_led_gpio(struct ws_led_gpio *ws_led_gpio)
{
	int ret = 0;

	ret = gpio_request(ws_led_gpio->gpio, ws_led_gpio->name);
	if (ret == 0) {
		ret = gpio_direction_output(ws_led_gpio->gpio,
					    ws_led_gpio->active_low);
		if (ret)
			gpio_free(ws_led_gpio->gpio);
	}
	return ret;
}

static inline void free_ws_led_data(struct ws_led_data *led_dat)
{
	gpio_free(led_dat->enable.gpio);
	if (led_dat->have_hw_blink)
		gpio_free(led_dat->hw_blink.gpio);
}

static int gpio_ws_led_probe(struct platform_device *pdev)
{
	struct ws_led_platform_data *pdata = pdev->dev.platform_data;
	struct ws_led *cur_led;
	struct ws_led_data *leds_data, *led_dat;
	int led, ret = 0;

	if (!pdata)
		return -EBUSY;

	leds_data = kzalloc(sizeof(struct ws_led_data) * pdata->num_leds,
			    GFP_KERNEL);
	if (!leds_data)
		return -ENOMEM;

	ret = configure_ws_led_gpio(&pdata->clk_ctrl);
	if (ret) {
		dev_err(&pdev->dev, "failed to configure GPIO %s\n",
			pdata->clk_ctrl.name);
		goto err_free_leds_data;
	}

	clk_ctrl = pdata->clk_ctrl;

	ret = configure_ws_led_gpio(&pdata->hd_blink);
	if (ret) {
		dev_err(&pdev->dev, "failed to configure GPIO %s\n",
			pdata->hd_blink.name);
		goto err_free_clk_ctrl;
	}

	hd_blink = pdata->hd_blink;

	spin_lock_init(&lock);

	for (led = 0; led < pdata->num_leds; led++) {

		cur_led = &pdata->leds[led];
		led_dat = &leds_data[led];

		ret = configure_ws_led_gpio(&cur_led->enable);
		if (ret) {
			dev_err(&pdev->dev, "failed to configure GPIO %s\n",
				cur_led->enable.name);
			goto err_free_led;
		}

		led_dat->cdev.name = cur_led->name;
		led_dat->cdev.default_trigger = cur_led->default_trigger;
		led_dat->enable = cur_led->enable;
		led_dat->cdev.brightness_set = ws_led_set;
		led_dat->cdev.brightness = LED_OFF;
		led_dat->cdev.flags |= LED_CORE_SUSPENDRESUME;

		if (cur_led->have_hw_blink) {
			ret = configure_ws_led_gpio(&cur_led->hw_blink);
			if (ret) {
				gpio_free(cur_led->enable.gpio);
				dev_err(&pdev->dev,
					"failed to configure GPIO %s\n",
					cur_led->hw_blink.name);
				goto err_free_led;
			}

			led_dat->hw_blink = cur_led->hw_blink;
			led_dat->delay_on = cur_led->delay_on;
			led_dat->delay_off = cur_led->delay_off;
			led_dat->cdev.blink_set = ws_led_hw_blink_set;
			led_dat->have_hw_blink = cur_led->have_hw_blink;
		}

		ret = led_classdev_register(&pdev->dev, &led_dat->cdev);
		if (ret < 0) {
			free_ws_led_data(led_dat);
			goto err_free_led;
		}

		dev_set_drvdata(led_dat->cdev.dev, led_dat);

		if (cur_led->have_hd_blink) {
			ret = device_create_file(led_dat->cdev.dev,
						 &dev_attr_hd_blink);
			if (ret < 0) {
				led_classdev_unregister(&led_dat->cdev);
				free_ws_led_data(led_dat);
				goto err_free_led;
			}
			led_dat->have_hd_blink = cur_led->have_hd_blink;
		}

	}

	platform_set_drvdata(pdev, leds_data);

	return 0;

err_free_led:
	if (led > 0) {
		for (led = led - 1; led >= 0; led--) {
			device_remove_file(leds_data[led].cdev.dev,
					   &dev_attr_hd_blink);
			led_classdev_unregister(&leds_data[led].cdev);
			free_ws_led_data(&leds_data[led]);
		}
	}
	gpio_free(hd_blink.gpio);
err_free_clk_ctrl:
	gpio_free(clk_ctrl.gpio);
err_free_leds_data:
	kfree(leds_data);

	return ret;
}

static int __devexit gpio_ws_led_remove(struct platform_device *pdev)
{
	int led;
	struct ws_led_platform_data *pdata = pdev->dev.platform_data;
	struct ws_led_data *leds_data;

	leds_data = platform_get_drvdata(pdev);

	for (led = 0; led < pdata->num_leds; led++) {
		device_remove_file(leds_data[led].cdev.dev, &dev_attr_hd_blink);
		led_classdev_unregister(&leds_data[led].cdev);
		free_ws_led_data(&leds_data[led]);
	}

	gpio_free(hd_blink.gpio);
	gpio_free(clk_ctrl.gpio);
	kfree(leds_data);

	return 0;
}

static struct platform_driver gpio_ws_led_driver = {
	.probe		= gpio_ws_led_probe,
	.remove		= __devexit_p(gpio_ws_led_remove),
	.driver		= {
		.name	= "gpio-ws-leds",
		.owner	= THIS_MODULE,
	},
};

static int __init gpio_ws_led_init(void)
{
	return platform_driver_register(&gpio_ws_led_driver);
}

static void __exit gpio_ws_led_exit(void)
{
	platform_driver_unregister(&gpio_ws_led_driver);
}

module_init(gpio_ws_led_init);
module_exit(gpio_ws_led_exit);

MODULE_AUTHOR("Simon Guinot <sguinot@lacie.com>");
MODULE_DESCRIPTION("LaCie Wireless Space GPIO LED driver");
MODULE_LICENSE("GPL");
MODULE_ALIAS("platform:gpio-ws-leds");
