/*
 * lacie/drivers/input-watchdog.c
 *
 * Input watchdog - reboot the machine on long press on the POWER key
 * This module is based on evbug.c Copyright (c) 1999-2001 Vojtech Pavlik
 *
 *  Copyright (c) 2009 LaCie
 *  		Benoît CANET <benoit.canet@gmail.com>
 */

/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#include <linux/input.h>
#include <linux/device.h>
#include <linux/jiffies.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/slab.h>
#include <linux/sched.h>
#include <linux/reboot.h>
#include <linux/wait.h>
#include <linux/workqueue.h>

#define LONG_PRESS_TIMEOUT 15000 /* 15 seconds timeout */

/* used to wait for press timeout or key release */
DECLARE_WAIT_QUEUE_HEAD(long_press_or_release_wait);

/* used to store the POWER key state */
static atomic_t power_key_pressed = ATOMIC_INIT(0);

static void detect_long_press_work_handler(struct work_struct *work);
/* work job detecting long key presses */
static DECLARE_WORK(detect_long_press_work, detect_long_press_work_handler);
/* our own workqueue (can't take the kernel global workqueue because we will block in it */
static struct workqueue_struct *long_press_workqueue;

static void detect_long_press_work_handler(struct work_struct *work)
{
	int result;

	/* wait until button released or timeout */
	result = wait_event_interruptible_timeout(long_press_or_release_wait,
						  atomic_read(&power_key_pressed) == 0,
						  msecs_to_jiffies(LONG_PRESS_TIMEOUT));
	/* if we reached the timeout reboot */
	if (result == 0)
		kernel_restart(NULL);
	/* else the button was released (result > 0) or we where interrupted by a signal (result == -ERESTARTSYS)
	 * => do nothing */
	return;
}


static void input_watchdog_event(struct input_handle *handle, unsigned int type, unsigned int code, int value)
{
	/* watch KEY_POWER events */
	if (type == EV_KEY && code == KEY_POWER) {
		atomic_set(&power_key_pressed, value);
		if (value == 1)
			/* on key press launch the long key press detection job
			 * only if it is not already launched */
			queue_work(long_press_workqueue, &detect_long_press_work);
		else if (value == 0)
			/* on release cancel previous jobs */
			wake_up_interruptible(&long_press_or_release_wait);
	}
	return;
}

static int input_watchdog_connect(struct input_handler *handler, struct input_dev *dev, const struct input_device_id *id)
{
	struct input_handle *handle;
	int error;

	handle = kzalloc(sizeof(struct input_handle), GFP_KERNEL);
	if (!handle)
		return -ENOMEM;

	handle->dev = dev;
	handle->handler = handler;
	handle->name = "input-watchdog";

	error = input_register_handle(handle);
	if (error)
		goto err_free_handle;

	error = input_open_device(handle);
	if (error)
		goto err_unregister_handle;

	return 0;

 err_unregister_handle:
	input_unregister_handle(handle);
 err_free_handle:
	kfree(handle);
	return error;
}

static void input_watchdog_disconnect(struct input_handle *handle)
{
	input_close_device(handle);
	input_unregister_handle(handle);
	kfree(handle);
}

static const struct input_device_id input_watchdog_ids[] = {
	{ .driver_info = 1 },	/* Matches all devices */
	{ },			/* Terminating zero entry */
};

MODULE_DEVICE_TABLE(input, input_watchdog_ids);

static struct input_handler input_watchdog_handler = {
	.event =	input_watchdog_event,
	.connect =	input_watchdog_connect,
	.disconnect =	input_watchdog_disconnect,
	.name =		"inputwatchdog",
	.id_table =	input_watchdog_ids,
};

static int __init input_watchdog_init(void)
{
	/* Set up our workqueue. */
	long_press_workqueue = create_singlethread_workqueue("input-watchdog");

	return input_register_handler(&input_watchdog_handler);
}

static void __exit input_watchdog_exit(void)
{
	/* cleanup the workqueue */
	flush_workqueue(long_press_workqueue);
	destroy_workqueue(long_press_workqueue);

	input_unregister_handler(&input_watchdog_handler);
}

module_init(input_watchdog_init);
module_exit(input_watchdog_exit);

MODULE_AUTHOR("Benoît CANET <benoit.canet@gmail.com>");
MODULE_DESCRIPTION("Input watchdog module");
MODULE_LICENSE("GPL");
