/*
 * lacie/drivers/ledtrig-hdd.c
 *
 * Hard disk activity LED trigger
 *
 * Copyright (c) 2009 LaCie
 *
 * Author: Simon Guinot <sguinot@lacie.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/module.h>
#include <linux/jiffies.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/list.h>
#include <linux/spinlock.h>
#include <linux/device.h>
#include <linux/sysdev.h>
#include <linux/timer.h>
#include <linux/ctype.h>
#include <linux/leds.h>
#include "leds.h"

#define HDD_MAX_NUM	16

struct hdd_led_group {
	struct list_head	list;
	rwlock_t		rw_lock; /* HDD LED group lock */
	struct timer_list	timer;
};

struct hdd_led {
	int			brightness_on;
	int			hdd_num;
	unsigned long		delay_off; /* disk activity timer in msecs */
	struct led_classdev	*led_cdev;
	struct list_head	node;
};

static struct hdd_led_group hdd_led_group[HDD_MAX_NUM];

static inline void remove_hdd_led_from_group(struct hdd_led *hdd_led)
{
	struct hdd_led_group *group;

	if (hdd_led->hdd_num < 0)
		return;

	group = &hdd_led_group[hdd_led->hdd_num];
	write_lock_irq(&group->rw_lock);
	list_del(&hdd_led->node);
	write_unlock_irq(&group->rw_lock);
}

static inline void add_hdd_led_to_group(struct hdd_led *hdd_led)
{
	struct hdd_led_group *group;

	if (hdd_led->hdd_num < 0)
		return;

	group = &hdd_led_group[hdd_led->hdd_num];
	write_lock_irq(&group->rw_lock);
	list_add(&hdd_led->node, &group->list);
	write_unlock_irq(&group->rw_lock);
}

/**
 * ledtrig_hdd_activaty() - Turn off LEDs during hard disk activity
 * with an active hard disk.
 * @hdd_num: active hard disk number
 * Context: unknown (irq or user)
 */
void ledtrig_hdd_activity(int hdd_num)
{
	struct hdd_led_group *group;
	struct hdd_led *hdd_led;
	unsigned long flags;

	if (hdd_num < 0 || hdd_num > HDD_MAX_NUM)
		return;

	group = &hdd_led_group[hdd_num];

	/* Turn off all LEDs for this group. */
	read_lock_irqsave(&group->rw_lock, flags);
	list_for_each_entry(hdd_led, &group->list, node) {
		if (led_get_brightness(hdd_led->led_cdev) != LED_OFF) {
			led_set_brightness(hdd_led->led_cdev, LED_OFF);
			dev_dbg(hdd_led->led_cdev->dev,
				"Turn off (HDD activity)\n");
		}
		mod_timer(&group->timer,
			  jiffies + msecs_to_jiffies(hdd_led->delay_off));
	}
	read_unlock_irqrestore(&group->rw_lock, flags);
}
EXPORT_SYMBOL(ledtrig_hdd_activity);

static void hdd_led_timer_handler(unsigned long data)
{
        struct hdd_led_group *group = (struct hdd_led_group *) data;
	struct hdd_led *hdd_led;

	/* Turn on all LEDs for this group. */
	read_lock_irq(&group->rw_lock);
	list_for_each_entry(hdd_led, &group->list, node) {
		if (led_get_brightness(hdd_led->led_cdev) !=
		    hdd_led->brightness_on) {
			led_set_brightness(hdd_led->led_cdev,
					   hdd_led->brightness_on);
			dev_dbg(hdd_led->led_cdev->dev,
				"Turn on (timer expire)\n");
		}
	}
	read_lock_irq(&group->rw_lock);
}

static ssize_t led_hdd_num_show(struct device *dev,
				struct device_attribute *attr, char *buf)
{
	struct led_classdev *led_cdev = dev_get_drvdata(dev);
	struct hdd_led *hdd_led = led_cdev->trigger_data;

	return sprintf(buf, "%d\n", hdd_led->hdd_num);
}

static ssize_t led_hdd_num_store(struct device *dev,
				 struct device_attribute *attr,
				 const char *buf, size_t count)
{
	struct led_classdev *led_cdev = dev_get_drvdata(dev);
	struct hdd_led *hdd_led = led_cdev->trigger_data;
	int hdd_num = simple_strtol(buf, NULL, 10);

	if (hdd_num > HDD_MAX_NUM || hdd_num < -1)
		return -EINVAL;

	remove_hdd_led_from_group(hdd_led);
	hdd_led->hdd_num = hdd_num;
	if (hdd_num != -1)
		led_set_brightness(hdd_led->led_cdev, hdd_led->brightness_on);

	add_hdd_led_to_group(hdd_led);

	return count;
}

static DEVICE_ATTR(hdd_num, 0644, led_hdd_num_show, led_hdd_num_store);

static ssize_t led_delay_off_show(struct device *dev,
				  struct device_attribute *attr, char *buf)
{
	struct led_classdev *led_cdev = dev_get_drvdata(dev);
	struct hdd_led *hdd_led = led_cdev->trigger_data;

	return sprintf(buf, "%lu\n", hdd_led->delay_off);
}

static ssize_t led_delay_off_store(struct device *dev,
				   struct device_attribute *attr,
				   const char *buf, size_t count)
{
	struct led_classdev *led_cdev = dev_get_drvdata(dev);
	struct hdd_led *hdd_led = led_cdev->trigger_data;

	hdd_led->delay_off = simple_strtoul(buf, NULL, 10);

	return count;
}

static DEVICE_ATTR(delay_off, 0644, led_delay_off_show, led_delay_off_store);

static void hdd_trig_activate(struct led_classdev *led_cdev)
{
	int ret;
	struct hdd_led *hdd_led;

	hdd_led = kzalloc(sizeof(struct hdd_led), GFP_KERNEL);
	if (!hdd_led)
		return;

	hdd_led->brightness_on = LED_FULL;
	hdd_led->delay_off = 10;
	hdd_led->hdd_num = -1;
	hdd_led->led_cdev = led_cdev;
	led_cdev->trigger_data = hdd_led;

	ret = device_create_file(led_cdev->dev, &dev_attr_hdd_num);
	if (ret)
		goto exit_kfree;

	ret = device_create_file(led_cdev->dev, &dev_attr_delay_off);
	if (ret)
		goto exit_hdd_num;

	return;

exit_hdd_num:
	device_remove_file(led_cdev->dev, &dev_attr_hdd_num);
exit_kfree:
	led_cdev->trigger_data = NULL;
	kfree(hdd_led);
}

static void hdd_trig_deactivate(struct led_classdev *led_cdev)
{
	struct hdd_led *hdd_led = led_cdev->trigger_data;

	if (unlikely(!hdd_led))
		return;

	remove_hdd_led_from_group(hdd_led);
	device_remove_file(led_cdev->dev, &dev_attr_delay_off);
	device_remove_file(led_cdev->dev, &dev_attr_hdd_num);
	led_cdev->trigger_data = NULL;
	kfree(hdd_led);
}

static struct led_trigger hdd_led_trigger = {
	.name     = "hdd",
	.activate = hdd_trig_activate,
	.deactivate = hdd_trig_deactivate,
};

static int __init hdd_trig_init(void)
{
	int i;

	for (i = 0; i < HDD_MAX_NUM; i++) {
		init_timer(&hdd_led_group[i].timer);
		hdd_led_group[i].timer.function = hdd_led_timer_handler;
		hdd_led_group[i].timer.data = (unsigned long) &hdd_led_group[i];
		INIT_LIST_HEAD(&hdd_led_group[i].list);
		rwlock_init(&hdd_led_group[i].rw_lock);
	}

	return led_trigger_register(&hdd_led_trigger);
}

static void __exit hdd_trig_exit(void)
{
	int i;

	for (i = 0; i < HDD_MAX_NUM; i++)
		del_timer_sync(&hdd_led_group[i].timer);

	led_trigger_unregister(&hdd_led_trigger);
}

module_init(hdd_trig_init);
module_exit(hdd_trig_exit);

MODULE_AUTHOR("Simon Guinot <sguinot@lacie.com>");
MODULE_DESCRIPTION("Hard disk activity LED trigger");
MODULE_LICENSE("GPL");
