/*
 * lacie/drivers/ledtrig-run-light.c
 *
 * LED kernel running light trigger
 *
 * Copyright (c) 2009 LaCie
 *
 * Author: Simon Guinot <sguinot@lacie.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/module.h>
#include <linux/jiffies.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/list.h>
#include <linux/spinlock.h>
#include <linux/device.h>
#include <linux/sysdev.h>
#include <linux/timer.h>
#include <linux/ctype.h>
#include <linux/leds.h>
#include "leds.h"

#define RUN_LIGHT_MAX_GROUP 8

struct run_light_group {
	struct list_head	run_light_led_list;
	spinlock_t		run_light_led_list_lock;
	struct run_light_led	*active_led;
	struct timer_list	timer;
};

struct run_light_led {
	int			brightness_on;
	unsigned int		group_num;
	unsigned long		delay_on;
	struct led_classdev	*led_cdev;
	struct list_head	node;
};

static struct run_light_group run_light_group[RUN_LIGHT_MAX_GROUP];

/*
 * Check if a LED running light group is active.
 * Must be called with the running light LED list locked.
 */
static inline int is_run_light_group_active(struct run_light_group *group)
{
	struct run_light_led *run_light_led;

	list_for_each_entry(run_light_led, &group->run_light_led_list, node) {
		if (run_light_led->delay_on)
			return 1;
	}
	return 0;
}

/*
 * Return the next light LED.
 * Must be called with the running light LED list locked.
 */
static inline struct run_light_led *get_next_light_led(struct run_light_led *run_light_led)
{
	struct run_light_led *next;
	struct run_light_group *group =
		&run_light_group[run_light_led->group_num];

	if (list_is_last(&run_light_led->node,
			&group->run_light_led_list))
		next = list_first_entry(&group->run_light_led_list,
					struct run_light_led, node);
	else
		next = list_entry(run_light_led->node.next,
				struct run_light_led, node);

	return next;
}

/*
 * Add a LED to a running light LED's group.
 */
static inline void add_led_to_group(struct run_light_led *run_light_led)
{
	int group_activity;
	struct run_light_group *group =
		&run_light_group[run_light_led->group_num];

	spin_lock_bh(&group->run_light_led_list_lock);

	/* Only add active LED's */
	if (run_light_led->delay_on == 0)
		goto exit_unlock;

	dev_dbg(run_light_led->led_cdev->dev,
		"Add to group %d\n", run_light_led->group_num);

	group_activity = is_run_light_group_active(group);
	list_add_tail(&run_light_led->node, &group->run_light_led_list);

	/* Start the runing light timer if the first active LED has
	 * been added */
	if (!group_activity) {
		group->active_led = run_light_led;
		mod_timer(&group->timer, jiffies + 1);
	}

exit_unlock:
	spin_unlock_bh(&group->run_light_led_list_lock);
}

/*
 * Remove a LED from a running light LED's group.
 */
static inline void del_led_from_group(struct run_light_led *run_light_led)
{
	struct run_light_group *group =
		&run_light_group[run_light_led->group_num];

	spin_lock_bh(&group->run_light_led_list_lock);

	if (run_light_led->delay_on == 0)
		goto exit_unlock;

	dev_dbg(run_light_led->led_cdev->dev,
		"Remove from group %d\n", run_light_led->group_num);

	/* Reset the active light LED if needed */
	if (group->active_led == run_light_led) {
		group->active_led = get_next_light_led(run_light_led);
		if (group->active_led == run_light_led)
			group->active_led = NULL;
	}

	list_del(&run_light_led->node);

	/* Stop the running light timer if the last active LED has
	 * been removed */
	if (!is_run_light_group_active(group))
		del_timer_sync(&group->timer);

exit_unlock:
	spin_unlock_bh(&group->run_light_led_list_lock);
}

static void led_run_light_timer_handler(unsigned long data)
{
	struct run_light_group *group = (struct run_light_group *) data;
	struct run_light_led *run_light_led;
	struct led_classdev *led_cdev;

	spin_lock_bh(&group->run_light_led_list_lock);

	run_light_led = group->active_led;

	if (unlikely(!run_light_led ||
			list_empty(&group->run_light_led_list)))
		goto exit_unlock;

	led_cdev = run_light_led->led_cdev;

	/* Turn current LED off */
	led_set_brightness(led_cdev, LED_OFF);

	/* Get next light LED */
	run_light_led = get_next_light_led(run_light_led);

	if (unlikely(!run_light_led || !run_light_led->delay_on)) {
		dev_err(led_cdev->dev, "Can't find next running light LED\n");
		group->active_led = NULL;
		goto exit_unlock;
	}

	/* Turn on next light LED */
	led_cdev = run_light_led->led_cdev;
	led_set_brightness(led_cdev, run_light_led->brightness_on);
	group->active_led = run_light_led;
	dev_dbg(led_cdev->dev, "Turn on [%ld ms]\n",
		run_light_led->delay_on);
	mod_timer(&group->timer, jiffies +
		msecs_to_jiffies(run_light_led->delay_on));

exit_unlock:
	spin_unlock_bh(&group->run_light_led_list_lock);
}

static ssize_t led_delay_on_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct led_classdev *led_cdev = dev_get_drvdata(dev);
	struct run_light_led *run_light_led = led_cdev->trigger_data;

	return sprintf(buf, "%lu\n", run_light_led->delay_on);
}

static ssize_t led_delay_on_store(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t count)
{
	struct led_classdev *led_cdev = dev_get_drvdata(dev);
	struct run_light_led *run_light_led = led_cdev->trigger_data;
	unsigned long delay_on;
	char *end;

	if (count >= PAGE_SIZE)
		return -EINVAL;

	delay_on = simple_strtoul(buf, &end, 10);
	if (!*buf || (*end && *end != '\n'))
		return -EINVAL;

	if (unlikely(delay_on == run_light_led->delay_on))
		return count;

	if (delay_on == 0) {
		/* Remove inactive LED from the running light LED's group */
		del_led_from_group(run_light_led);
		run_light_led->delay_on = delay_on;
		led_set_brightness(run_light_led->led_cdev, LED_OFF);
	} else if (run_light_led->delay_on == 0) {
		/* Add the new active LED to the running light LED's group */
		run_light_led->delay_on = delay_on;
		add_led_to_group(run_light_led);
	} else
		run_light_led->delay_on = delay_on;

	return count;
}

static DEVICE_ATTR(delay_on, 0644, led_delay_on_show, led_delay_on_store);

static ssize_t led_group_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct led_classdev *led_cdev = dev_get_drvdata(dev);
	struct run_light_led *run_light_led = led_cdev->trigger_data;

	return sprintf(buf, "%i\n", run_light_led->group_num);
}

static ssize_t led_group_store(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t count)
{
	struct led_classdev *led_cdev = dev_get_drvdata(dev);
	struct run_light_led *run_light_led = led_cdev->trigger_data;
	unsigned int group_num;
	char *end;

	if (count >= PAGE_SIZE)
		return -EINVAL;

	group_num = simple_strtoul(buf, &end, 10);
	if (!*buf || (*end && *end != '\n'))
		return -EINVAL;

	if (group_num > RUN_LIGHT_MAX_GROUP)
		return -EINVAL;

	if (unlikely(run_light_led->group_num == group_num))
		return count;

	del_led_from_group(run_light_led);
	run_light_led->group_num = group_num;
	add_led_to_group(run_light_led);

	return count;
}

static DEVICE_ATTR(group, 0644, led_group_show, led_group_store);

static void run_light_trig_activate(struct led_classdev *led_cdev)
{
	int ret;
	struct run_light_led *run_light_led;

	run_light_led = kzalloc(sizeof(struct run_light_led), GFP_KERNEL);
	if (!run_light_led)
		return;

	run_light_led->brightness_on = LED_FULL;
	run_light_led->delay_on = 0;
	run_light_led->group_num = 0;
	run_light_led->led_cdev = led_cdev;
	led_cdev->trigger_data = run_light_led;

	ret = device_create_file(led_cdev->dev, &dev_attr_delay_on);
	if (ret)
		goto exit_kfree;
	ret = device_create_file(led_cdev->dev, &dev_attr_group);
	if (ret)
		goto exit_delay_on;

	return;

exit_delay_on:
	device_remove_file(led_cdev->dev, &dev_attr_delay_on);
exit_kfree:
	led_cdev->trigger_data = NULL;
	kfree(run_light_led);
}

static void run_light_trig_deactivate(struct led_classdev *led_cdev)
{
	struct run_light_led *run_light_led = led_cdev->trigger_data;

	if (unlikely(!run_light_led))
		return;

	/* Remove LED from the running light list */
	del_led_from_group(run_light_led);

	device_remove_file(led_cdev->dev, &dev_attr_delay_on);
	device_remove_file(led_cdev->dev, &dev_attr_group);
}

static struct led_trigger run_light_led_trigger = {
	.name     = "run-light",
	.activate = run_light_trig_activate,
	.deactivate = run_light_trig_deactivate,
};

static int __init run_light_trig_init(void)
{
	int i;

	for (i = 0; i < RUN_LIGHT_MAX_GROUP; i++) {
		init_timer(&run_light_group[i].timer);
		run_light_group[i].timer.function =
			led_run_light_timer_handler;
		run_light_group[i].timer.data =
			(unsigned long) &run_light_group[i];
		INIT_LIST_HEAD(&run_light_group[i].run_light_led_list);
		spin_lock_init(&run_light_group[i].run_light_led_list_lock);
	}
	return led_trigger_register(&run_light_led_trigger);
}

static void __exit run_light_trig_exit(void)
{
	int i;

	for (i = 0; i < RUN_LIGHT_MAX_GROUP; i++)
		del_timer_sync(&run_light_group[i].timer);

	led_trigger_unregister(&run_light_led_trigger);
}

module_init(run_light_trig_init);
module_exit(run_light_trig_exit);

MODULE_AUTHOR("Simon Guinot <sguinot@lacie.com>");
MODULE_DESCRIPTION("Run light LED trigger");
MODULE_LICENSE("GPL");
