#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <time.h>
#include <getopt.h>
#include <sys/stat.h>

#include "lacfg.h"

#define msg(x)				printf(x)
#define msgfmt(format, args ...)  	printf(format, args)
#define err(format, ...)  		msg("Error: "), msgfmt( format, __VA_ARGS__)
#define warn(format, ...)  		msg("Warn: "), msgfmt( format, __VA_ARGS__)

#ifdef DEBUG
#define debug(format, args ...)  	printf(format, args)
#else
#define debug(format, args ...)
#endif

#define MAJOR(dev)      	((dev & 0xfff00) >> 8)
#define MINOR(dev)      	((dev & 0xff) | ((dev >> 12) & 0xfff00))
#define RAID_MAJOR		9
#define MAX_RAID_MINOR  	5
#define UUID_LENGTH		35

int isDeviceValid(const char *dev)
{
	struct stat sb;
	int major, minor;
	if(strlen(dev) != 8)
		return 0;
	if (stat(dev, &sb)) {
		err("stat of %s failed: %s\n", dev,
				strerror(errno));
		return 0;
	}
	if(!S_ISBLK(sb.st_mode)) 
		return 0;
	major = MAJOR(sb.st_rdev);
	minor = MINOR(sb.st_rdev);
	if ((major != RAID_MAJOR) || !(minor < MAX_RAID_MINOR) )
		return 0;
	return 1;
}

// FIXME Add proper checks
int checkUUIDFormat(const char *uuid)
{
	if(strlen(uuid) != UUID_LENGTH)
		return 0;
	return 1;
}
int writeUUID(const char *dev, const char *uuid)
{
	int minor = atoi(&dev[7]);
	char key[20];

	if(!dev)
		return 0;

	if(minor > MAX_RAID_MINOR)
	{
		err("Device %s not in list\n", dev);
		return 0;
	}

	sprintf(key, LACFG_RAID_FORMAT, minor);

	debug("Writing UUID %s for device %s\n", uuid, dev);
	lacfg_set(key, (char *) uuid, UUID_LENGTH);
	lacfg_save();

	return 1;
}

static int readUUID (int minor) {
	char key[20];
	char result[UUID_LENGTH + 10];

	if(minor > MAX_RAID_MINOR)
	{
		err("Device /dev/md%d not in list\n", minor);
		return 0;
	}

	sprintf(key, LACFG_RAID_FORMAT, minor);

	debug("Reading UUID for device /dev/md%d. Offset is %x\n", minor, offset);
	lacfg_load_key(key);

	/* get uuid and null terminate it */
	memcpy(result , lacfg_get(key), UUID_LENGTH);
	result[UUID_LENGTH] = 0;
	printf("%s\n", result);

	return 0;
}

void usage()
{
	printf("Usage: md_eeprog <OPTION> [UUID] <MD_DEV>\n");
	printf("\t-s,   --setuuid   write uuid to eeprom (requires UUID)\n");
	printf("\t-g,   --getuuid   read uuid from eeprom\n");
	printf("\t-G                read uuid from eeprom by specifying a minor instead of a dev\n");
	printf("\nExamples: \n\tmd_eeprog -s  6b8b4567:327b23c6:643c9869:66334873 /dev/md0\n");
	printf("\tmd_eeprog -g /dev/md0\n");
}
int main(int argc, char **argv)
{
	char *uuid = NULL;
	char *raid_dev = NULL;
	int isWriteUUID = 0;
	int get_from_minor = 0;

	int c;
	int digit_optind = 0;

	if (argc < 3)
	{
		usage();
		exit(1);
	}
	while (1) {
		int this_option_optind = optind ? optind : 1;
		int option_index = 0;
		static struct option long_options[] = {
			{"getuuid", 0, 0, 'g'},
			{"setuuid", 1, 0, 's'},
			{0, 0, 0, 0}
		};

		c = getopt_long (argc, argv, "gGs:",
				long_options, &option_index);
		if (c == -1)
			break;	
		switch (c) {
			case 'G':
				get_from_minor = 1;
			case 'g':
				break;
			case 's':
				uuid = optarg;
				if(!checkUUIDFormat(uuid))
					exit(1);
				isWriteUUID=1;
				break;
			default:
				err("Option not recognized %c\n", c);
				exit(1);
		}
	}
	if (optind < argc) {
		raid_dev=argv[optind++];		
		if(!get_from_minor && !isDeviceValid(raid_dev))
		{
			err("Device \"%s\" not valid\n", raid_dev);
			exit(1);
		}
	}
	else {
		usage();
		exit(1);
	}

	lacfg_init(STORAGE_PATH);

	if(isWriteUUID && uuid)
		writeUUID(raid_dev, uuid);
	else if (get_from_minor)
		readUUID(atoi(raid_dev));
	else
		readUUID(atoi(raid_dev + 7));

	lacfg_cleanup();

	exit(0);
}

