#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define DEBUG_NO_OVERRIDE

#include <libshred.h>

//#define DEBUG_TRACE

/*
	Features:
		- handle and report shifted memory
		- handle bad arguments, like NULL pointers
		- report unfreed memory
		! double free trace (not nicely handled)
*/

typedef struct
{
	uint8 *	Mem;
	int		szMem;
	
	// creator reference
	char *	Filename;
	int		Line;
	char *	Function;
	
}TMemSegment;


static void *CS = NULL;
static int InitInProgress = 0;
static TMemSegment *MemSegments = NULL;
static int lMemSegments = 0;
static char ContextString[256] = {0};


static void _Debug_Cleanup(void);
static void _Debug_Init(void);
static void _Debug_Segment_DeleteAt(int ID);
void Debug_Dump(void);


static void _Debug_AtExit(void)
{
	Debug_Dump();
	_Debug_Cleanup();
}


static void _Debug_Init(void)
{
	if(!CS)
	{
		InitInProgress = 1;
		CS = CriticalSection_Open();
		atexit(_Debug_AtExit);
		InitInProgress = 0;
	}
}


void Debug_Init(void)
{
	_Debug_Init();
}


static void _Debug_Cleanup(void)
{
	int i;
	
	InitInProgress = 1;
	CriticalSection_Close(CS);
	InitInProgress = 0;
	
	for(i=0;i<lMemSegments;i++)
		_Debug_Segment_DeleteAt(0);
	
	// more cleanup
	CS = NULL;
	InitInProgress = 0;
	MemSegments = NULL;
	lMemSegments = 0;
	ContextString[0] = 0;
}


static char *_Debug_UpdateContextString(char *AllocContext, char *Filename, int Line, char *Function)
{
	snprintf(ContextString, sizeof(ContextString) - 1, "%s in file %s, line %d, function %s()", AllocContext, Filename, Line, Function);
	return ContextString;
}


static int _Debug_Segment_Create(int Size, char *Filename, int Line, char *Function)
{
	TMemSegment *pSegment;
	int ID = lMemSegments;
	
	MemSegments = realloc(MemSegments, (lMemSegments + 1) * sizeof(TMemSegment));
	pSegment = &MemSegments[lMemSegments++];
	
	pSegment->Mem = malloc(Size);
	pSegment->szMem = Size;
	pSegment->Filename = strdup(Filename);
	pSegment->Line = Line;
	pSegment->Function = strdup(Function);
	
	return ID;
}


static int _Debug_Segment_FindStrict(uint8 *Mem)
{
	int i;
	
	for(i=0;i<lMemSegments;i++)
	{
		TMemSegment *pSegment = &MemSegments[i];
		if(pSegment->Mem == Mem)
			return i;
	}
	return -1;
}


static int _Debug_Segment_FindLoose(uint8 *Mem)
{
	int i;
	
	for(i=0;i<lMemSegments;i++)
	{
		TMemSegment *pSegment = &MemSegments[i];
		if(Mem >= pSegment->Mem && Mem < pSegment->Mem + pSegment->szMem)
			return i;
	}
	return -1;
}


static int _Debug_Segment_Find(uint8 *Mem)
{
	int ID;
	
	ID = _Debug_Segment_FindStrict(Mem);
	if(ID < 0)
	{
		ID = _Debug_Segment_FindLoose(Mem);
		if(ID < 0)
			LOG_DEBUG("DEBUG: %s: error: access to unknown segment", ContextString);
		else
			LOG_DEBUG("DEBUG: %s: warning: access to shifted address: %p / %p", ContextString, Mem, MemSegments[ID].Mem);
	}
	return ID;
}


static void _Debug_Segment_DeleteAt(int ID)
{
	TMemSegment *pSegment = NULL;
	
	// fixme: we should keep trace of this ?
	
	if(ID < 0 || ID >= lMemSegments)
		return;
	
	pSegment = &MemSegments[ID];
	
	if(pSegment->Mem)
		free(pSegment->Mem);
	if(pSegment->Filename)
		free(pSegment->Filename);
	if(pSegment->Function)
		free(pSegment->Function);
	
	memmove(&MemSegments[ID], &MemSegments[ID + 1], (lMemSegments - ID - 1) * sizeof(TMemSegment));
	lMemSegments--;
	MemSegments = realloc(MemSegments, lMemSegments * sizeof(TMemSegment));
}


void Debug_Free(void *Segment, char *Filename, int Line, char *Function)
{
	char AllocContext[64];
	int ID;
	
	
	if(InitInProgress)
	{
#ifdef DEBUG_TRACE
		LOG_DEBUG(">>> free(init)");
#endif
		free(Segment);
#ifdef DEBUG_TRACE
		LOG_DEBUG("<<< free(init)");
#endif
		return;
	}
	
#ifdef DEBUG_TRACE
	LOG_DEBUG(">>> free(*)");
#endif
	
	_Debug_Init();
	
	CriticalSection_Enter(CS);
	
	snprintf(AllocContext, sizeof(AllocContext) - 1, "free(%p)", Segment);
	_Debug_UpdateContextString(AllocContext, Filename, Line, Function);
	//LOG_DEBUG("DEBUG: %s", ContextString);
	
	ID = _Debug_Segment_Find((uint8*)Segment);
	if(ID >= 0)
		_Debug_Segment_DeleteAt(ID);
	
	CriticalSection_Leave(CS);
	
#ifdef DEBUG_TRACE
	LOG_DEBUG("<<< free(*)");
#endif
}


void * Debug_Calloc(int Number, int Size, char *Filename, int Line, char *Function)
{
	char AllocContext[64];
	void *Segment = NULL;
	
	
	if(InitInProgress)
	{
#ifdef DEBUG_TRACE
		LOG_DEBUG(">>> calloc(init)");
#endif
		Segment = calloc(Number, Size);
#ifdef DEBUG_TRACE
		LOG_DEBUG("<<< calloc(init)");
#endif
		return Segment;
	}
	
#ifdef DEBUG_TRACE
	LOG_DEBUG(">>> calloc(*)");
#endif
	
	_Debug_Init();
	
	CriticalSection_Enter(CS);
	
	snprintf(AllocContext, sizeof(AllocContext) - 1, "malloc(%d)", Size);
	_Debug_UpdateContextString(AllocContext, Filename, Line, Function);
	//LOG_DEBUG("DEBUG: %s", ContextString);
	
	{
		int ID = _Debug_Segment_Create(Number * Size, Filename, Line, Function);
		Segment = MemSegments[ID].Mem;
	}
	
	CriticalSection_Leave(CS);
	
#ifdef DEBUG_TRACE
	LOG_DEBUG("<<< calloc(*)");
#endif
	
	return Segment;
}


void * Debug_Malloc(int Size, char *Filename, int Line, char *Function)
{
	char AllocContext[64];
	void *Segment = NULL;
	
	
	if(InitInProgress)
	{
#ifdef DEBUG_TRACE
		LOG_DEBUG(">>> malloc(init)");
#endif
		Segment = malloc(Size);
#ifdef DEBUG_TRACE
		LOG_DEBUG("<<< malloc(init)");
#endif
		return Segment;
	}
	
#ifdef DEBUG_TRACE
	LOG_DEBUG(">>> malloc(*)");
#endif
	
	_Debug_Init();
	
	CriticalSection_Enter(CS);
	
	snprintf(AllocContext, sizeof(AllocContext) - 1, "malloc(%d)", Size);
	_Debug_UpdateContextString(AllocContext, Filename, Line, Function);
	//LOG_DEBUG("DEBUG: %s", ContextString);
	
	{
		int ID = _Debug_Segment_Create(Size, Filename, Line, Function);
		Segment = MemSegments[ID].Mem;
	}
	
	CriticalSection_Leave(CS);
	
#ifdef DEBUG_TRACE
	LOG_DEBUG("<<< malloc(*)");
#endif
	
	return Segment;
}


void * Debug_Realloc(void *Segment, int Size, char *Filename, int Line, char *Function)
{
	char AllocContext[64];
	
	
	if(InitInProgress)
	{
#ifdef DEBUG_TRACE
		LOG_DEBUG(">>> realloc(init)");
#endif
		Segment = realloc(Segment, Size);
#ifdef DEBUG_TRACE
		LOG_DEBUG("<<< realloc(init)");
#endif
		return Segment;
	}
	
#ifdef DEBUG_TRACE
	LOG_DEBUG(">>> realloc(*)");
#endif
	
	_Debug_Init();
	
	CriticalSection_Enter(CS);
	
	snprintf(AllocContext, sizeof(AllocContext) - 1, "realloc(%p,%d)", Segment, Size);
	_Debug_UpdateContextString(AllocContext, Filename, Line, Function);
	//LOG_DEBUG("DEBUG: %s", ContextString);
	
	if(!Segment)
	{
		int ID = _Debug_Segment_Create(Size, Filename, Line, Function);
		Segment = MemSegments[ID].Mem;
	}
	else
	{
		int ID = _Debug_Segment_Find((uint8*)Segment);
		if(ID >= 0)
		{
			if(Size > 0)
			{
				Segment = MemSegments[ID].Mem;
				Segment = realloc(Segment, Size);
				MemSegments[ID].Mem = Segment;
				MemSegments[ID].szMem = Size;
				String_SafeReplace(MemSegments[ID].Filename, Filename);
				MemSegments[ID].Line = Line;
				String_SafeReplace(MemSegments[ID].Function, Function);
			}
			else
			{
				Segment = NULL;
				_Debug_Segment_DeleteAt(ID);
			}
		}
	}
	
	CriticalSection_Leave(CS);
	
#ifdef DEBUG_TRACE
	LOG_DEBUG("<<< realloc(*)");
#endif
	
	return Segment;
}


char * Debug_Strdup(const char *Source, char *Filename, int Line, char *Function)
{
	char AllocContext[64];
	char *Segment = NULL;
	
	
	if(InitInProgress)
	{
#ifdef DEBUG_TRACE
		LOG_DEBUG(">>> strdup(init)");
#endif
		Segment = strdup(Source);
#ifdef DEBUG_TRACE
		LOG_DEBUG("<<< strdup(init)");
#endif
		return Segment;
	}
	
#ifdef DEBUG_TRACE
	LOG_DEBUG(">>> strdup(*)");
#endif
	
	_Debug_Init();
	
	CriticalSection_Enter(CS);
	
	snprintf(AllocContext, sizeof(AllocContext) - 1, "strdup(%s)", Source);
	_Debug_UpdateContextString(AllocContext, Filename, Line, Function);
	//LOG_DEBUG("DEBUG: %s", ContextString);
	
	if(Source)
	{
		int Size = strlen(Source) + 1;
		int ID = _Debug_Segment_Create(Size, Filename, Line, Function);
		Segment = (char*)MemSegments[ID].Mem;
		memcpy(Segment, Source, Size);
	}
	else
	{
		LOG_DEBUG("DEBUG: %s: error: null pointer", ContextString);
	}
	
	CriticalSection_Leave(CS);
	
#ifdef DEBUG_TRACE
	LOG_DEBUG("<<< strdup(*)");
#endif
	
	return Segment;
}


void Debug_Dump(void)
{
	int i;
	
	if(InitInProgress)
		return;
	
	_Debug_Init();
	
	CriticalSection_Enter(CS);
	
	if(lMemSegments > 0)
	{
		LOG_DEBUG("DEBUG: >>> dump >>>");
		
		for(i=0;i<lMemSegments;i++)
		{
			TMemSegment *pSegment = &MemSegments[i];
			if(pSegment->Mem)
			{
				char AllocContext[256];
				char MemPreview[16];
				int j;
				
				memset(MemPreview, 0, sizeof(MemPreview));
				memcpy(MemPreview, pSegment->Mem, MIN(pSegment->szMem, sizeof(MemPreview)));
				for(j=0;j<sizeof(MemPreview);j++)
				{
					uint8 c = (uint8)MemPreview[j];
					if((c < 32 && c != '\r' && c != '\n' && c != '\t') || (c > 126))
						MemPreview[j] = '.';
				}
				
				snprintf(AllocContext, sizeof(AllocContext) - 1, "unfreed memory at [%p]:[%.*s], size %d", pSegment->Mem, sizeof(MemPreview), MemPreview, pSegment->szMem);
				_Debug_UpdateContextString(AllocContext, pSegment->Filename, pSegment->Line, pSegment->Function);
				LOG_DEBUG("DEBUG: %s", ContextString);
			}
			else
			{
				char AllocContext[64];
				snprintf(AllocContext, sizeof(AllocContext) - 1, "??? memory at [%p], size %d", pSegment->Mem, pSegment->szMem);
				_Debug_UpdateContextString(AllocContext, pSegment->Filename, pSegment->Line, pSegment->Function);
				LOG_DEBUG("DEBUG: %s", ContextString);
			}
		}
		LOG_DEBUG("DEBUG: <<< dump <<<");
	}
	
	CriticalSection_Leave(CS);
}
