#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <sys/stat.h>
#include <unistd.h>
#include <dirent.h>
#ifdef WIN32
#include <io.h>
#endif

#include <libshred.h>

#ifndef _S_ISREG
#define	_S_ISREG(m)	(((m) & _S_IFMT) == _S_IFREG)
#endif

#ifndef _S_ISDIR
#define	_S_ISDIR(m)	(((m) & _S_IFMT) == _S_IFDIR)
#endif


int FileSystem_ReadFileFirstLine(char *Filename, char *Line, int szLine)
{
	FILE *File;
	char Buffer[64];
	int Success = 0;
	
	
	if(!Filename || !Line || szLine <= 0)
		return 0;
	
	File = fopen(Filename, "r");
	if(File)
	{
		Buffer[0] = 0;
		fgets(Buffer, sizeof(Buffer) - 1, File);
		if(String_Trim(Buffer) > 0)
		{
			snprintf(Line, szLine - 1, "%s", Buffer);
			Success = 1;
		}
		fclose(File);
	}
	return Success;
}


int FileSystem_ReadTextFile(char *Filename, char ***pLines, int *plLines)
{
	FILE *File;
	char *Line, **Lines;
	int szLine, lLines;
	
	
	if(!Filename || !pLines || !plLines)
		return 0;
	
	*pLines = NULL;
	*plLines = 0;
	
	File = fopen(Filename, "r");
	if(!File)
		return 0;
	
	szLine = 64 * 1024;
	Line = malloc(szLine);
	
	Lines = NULL;
	lLines = 0;
	
	while(!feof(File))
	{
		Line[0] = 0;
		fgets(Line, szLine - 1, File);
		
		Lines = realloc(Lines, (lLines + 1) * sizeof(char*));
		Lines[lLines++] = strdup(Line);
	}
	
	free(Line);
	
	fclose(File);
	
	*pLines = Lines;
	*plLines = lLines;
	
	return 1;
}


int FileSystem_FileToBuffer(char *Filename, uint8 *Buffer, int *pszBuffer)
{
	FILE *File;
	int szRead, szToRead, FileSize;
	int64 FileSize64, FileSize64Max;
	
	
	if(!Filename || !Buffer || !pszBuffer)
	{
		LOG_DEBUG("error: invalid arguments");
		return 0;
	}
	
	FileSize64 = FileSystem_FileSize(Filename);
	if(FileSize64 < 0)
	{
		LOG_DEBUG("error: getting file size");
		return 0;
	}
	
	FileSize64Max = 32*1024*1024;
	if(FileSize64 > FileSize64Max)
	{
#ifdef WIN32
		LOG_DEBUG("error: file size too big: %I64d / %I64d", FileSize64, FileSize64Max);
#else
		LOG_DEBUG("error: file size too big: %lld / %lld", FileSize64, FileSize64Max);
#endif
		return 0;
	}
	FileSize = (int)FileSize64;
	
	File = fopen(Filename, "rb");
	if(!File)
	{
		LOG_DEBUG("error: opening file '%s'", Filename);
		return 0;
	}
	
	if(FileSize > *pszBuffer)
	{
		LOG_DEBUG("error: buffer too small (%d/%d bytes)", *pszBuffer, FileSize);
		fclose(File);
		return 0;
	}
	
	szToRead = FileSize;
	while(szToRead > 0)
	{
		szRead = fread(Buffer + FileSize - szToRead, 1, szToRead, File);
		if(szRead <= 0)
		{
			LOG_DEBUG("error: fread()");
			fclose(File);
			return 0;
		}
		szToRead -= szRead;
	}
	fclose(File);
	
	*pszBuffer = FileSize;
	
	return 1;
}


int FileSystem_BufferToFile(uint8 *Buffer, int szBuffer, char *Filename)
{
	FILE *File;
	int szWrite, szToWrite;
	
	
	if(!Filename || (szBuffer && !Buffer))
	{
		LOG_DEBUG("error: invalid arguments");
		return 0;
	}
	
	File = fopen(Filename, "wb");
	if(!File)
	{
		LOG_DEBUG("error: opening file '%s'", Filename);
		return 0;
	}
	
	szToWrite = szBuffer;
	while(szToWrite > 0)
	{
		szWrite = fwrite(Buffer + szBuffer - szToWrite, 1, szToWrite, File);
		if(szWrite <= 0)
		{
			LOG_DEBUG("error: fread()");
			fclose(File);
			return 0;
		}
		szToWrite -= szWrite;
	}
	fclose(File);
	
	return 1;
}


int FileSystem_FileExists(char *Path)
{
	struct stat Status;
	
	if(stat(Path, &Status) == 0)
	{
#ifdef WIN32
		if(_S_ISREG(Status.st_mode))
#else
		if(S_ISREG(Status.st_mode))
#endif
			return 1;
		return 0;
	}
	
	// large file support
#ifdef LINUX
	if(errno == EOVERFLOW)
	{
		struct stat64 Status64;
		
		if(stat64(Path, &Status64) == 0)
		{
			if(S_ISREG(Status.st_mode))
				return 1;
		}
	}
#endif
	
	return 0;
}


int FileSystem_FolderExists(char *Path)
{
	struct stat Status;
	
	if(stat(Path, &Status) < 0)
		return 0;
	
#ifdef WIN32
	if(_S_ISDIR(Status.st_mode))
#else
	if(S_ISDIR(Status.st_mode))
#endif
		return 1;
	
	return 0;
}


int64 FileSystem_FileSize(char *Filename)
{
	struct stat Status;
	
	if(stat(Filename, &Status) < 0)
		return -1;
	
#ifdef WIN32
	if(_S_ISREG(Status.st_mode))
#else
	if(S_ISREG(Status.st_mode))
#endif
		return Status.st_size;
	
	return -1;
}


int FileSystem_FileDelete(char *Filename)
{
	return !unlink(Filename);
}


int FileSystem_FileCopy(char *FilenameSrc, char *FilenameDst)
{
	FILE *FileSrc, *FileDst;
	int szCopy;
	uint8 *Buffer;
	int szBuffer = 1024 * 32;
	
	
	if(!FilenameSrc || !FilenameDst)
		return 0;
	
	// fixme: maybe a bug: we should touch the file
	if(String_Equal(FilenameSrc, FilenameDst))
		return 1;
	
	szCopy = FileSystem_FileSize(FilenameSrc);
	
	FileSrc = NULL;
	FileDst = NULL;
	
	FileSrc = fopen(FilenameSrc, "rb");
	if(!FileSrc)
		return 0;
	
	FileDst = fopen(FilenameDst, "wb");
	if(!FileDst)
	{
		fclose(FileSrc);
		return 0;
	}
	
	Buffer = malloc(szBuffer);
	while(szCopy > 0)
	{
		int szRead;
		
		szRead = fread(Buffer, 1, szBuffer, FileSrc);
		if(szRead <= 0)
		{
			free(Buffer);
			fclose(FileSrc);
			fclose(FileDst);
			FileSystem_FileDelete(FilenameDst);
			return 0;
		}
		
		szCopy -= szRead;
		
		fwrite(Buffer, 1, szRead, FileDst);
	}
	free(Buffer);
	
	fclose(FileSrc);
	fclose(FileDst);
	
	return 1;
}


int FileSystem_FileMove(char *FilenameSrc, char *FilenameDst)
{
	if(!FileSystem_FileCopy(FilenameSrc, FilenameDst))
		return 0;
	return FileSystem_FileDelete(FilenameSrc);
}


#ifdef WIN32
#define DEFAULT_SLASH					'/'
#define DEFAULT_DIR_SEPARATOR			';'
#define IS_SLASH(c)						((c) == '/' || (c) == '\\')
#define IS_ABSOLUTE_PATH(path, len)		(len >= 2 && ((isalpha(path[0]) && path[1] == ':')))
#endif

#if defined(LINUX) || defined(APPLE) || defined(FREEBSD)
#define DEFAULT_SLASH					'/'
#define DEFAULT_DIR_SEPARATOR			':'
#define IS_SLASH(c)						((c) == '/')
#define IS_ABSOLUTE_PATH(path, len)		(len > 0 && IS_SLASH(path[0]))
#endif

static int _Path_Cleanup(char *AbsPath)
{
	char *pBufWrite, *pBuf;
	int SlashFound = 0;
	
	
	pBuf = &AbsPath[0];
	pBufWrite = pBuf;
	while(*pBuf)
	{
		if(IS_SLASH(*pBuf))
		{
			if(SlashFound)
			{
				pBuf++;
				continue;
			}
			SlashFound = 1;
			*pBufWrite++ = DEFAULT_SLASH;
		}
		else
		{
			SlashFound = 0;
			
			if(pBufWrite != pBuf)
				*pBufWrite = *pBuf;
			pBufWrite++;
		}
		
		pBuf++;
	}
	*pBufWrite = 0;
	
	return (int)(pBufWrite - &AbsPath[0]);
}


int FileSystem_RealPath(char *Path, char *ResolvedPath, int *pszResolvedPath)
{
	char AbsPath[512], *Tokens[512], *pBuf;
	int i, lPath, lAbsPath, lTokens, lResolvedPath, szResolvedPath;
	
	
	if(!Path || !ResolvedPath || !pszResolvedPath)
		return 0;
	
	szResolvedPath = *pszResolvedPath;
	lPath = strlen(Path);
	
	if(IS_ABSOLUTE_PATH(Path, lPath))
	{
		// copy string
		
		if(lPath >= szResolvedPath - 1)
			return 0;
		
		memcpy(AbsPath, Path, lPath);
		AbsPath[lPath] = 0;
	}
	else
	{
		//	copy cwd then concatenate string
		if(!getcwd(AbsPath, sizeof(AbsPath)))
			return 0;
		
		lAbsPath = strlen(AbsPath);
		if(lAbsPath + lPath + 1 >= sizeof(AbsPath))
			return 0;
		
		AbsPath[lAbsPath] = DEFAULT_SLASH;
		memcpy(&AbsPath[lAbsPath + 1], Path, lPath + 1);
		lAbsPath += 1 + lPath;
	}
	
	// override default slash, and drop duplicate
	lAbsPath = _Path_Cleanup(AbsPath);
	
	if(lAbsPath >= szResolvedPath - 1)
		return 0;
	
	// build tree
	{
		lTokens = 0;
		pBuf = &AbsPath[0];
		memset(Tokens, 0, sizeof(Tokens));
		for(i=0;i<lAbsPath;i++)
		{
			if(IS_SLASH(AbsPath[i]))
			{
				Tokens[lTokens++] = pBuf;
				AbsPath[i] = 0;
				pBuf = &AbsPath[i + 1];
			}
		}
		Tokens[lTokens++] = pBuf;
	}
	
	// simplify tree
	// skip first token (root), should check for error
	for(i=1;i<lTokens;i++)
	{
		// throw "."
		if(strcmp(Tokens[i], ".") == 0)
		{
			// shift down
			memmove(&Tokens[i], &Tokens[i + 1], (lTokens - i - 1) * sizeof(char*));
			lTokens--;
			i--;
			continue;
		}
		
		if(strcmp(Tokens[i], "..") == 0)
		{
			if(i <= 1)
			{
				// .. conducts to /, just shift all > 1 to 0 offset
				// shift down
				memmove(&Tokens[i], &Tokens[i + 1], (lTokens - i - 1) * sizeof(char*));
				lTokens--;
				i--;
				continue;
			}
			else
			{
				// shift all > i to i-1 offset
				memmove(&Tokens[i - 1], &Tokens[i + 1], (lTokens - i - 1) * sizeof(char*));
				lTokens -= 2;
				i -= 2;
				continue;
			}
		}
	}
	
	// rebuild string
	{
		int lToken;
		
		
		pBuf = &ResolvedPath[0];
		for(i=0;i<lTokens;i++)
		{
			lToken = strlen(Tokens[i]);
			
			memcpy(pBuf, Tokens[i], lToken);
			pBuf += lToken;
			*pBuf++ = DEFAULT_SLASH;
		}
		// remove trailing '/', only if Tokens[0] is not empty (linux) and lTokens > 0 (at least 1 folder)
		if(lTokens && Tokens[0][0])
			pBuf[-1] = 0;
		else
			*pBuf = 0;
	}
	
	// cleanup again
	lResolvedPath = _Path_Cleanup(ResolvedPath);
	*pszResolvedPath = lResolvedPath;
	
	return 1;
}


char *FileSystem_RealPathEx(char *Path, char *RelativePathRoot, char *TmpBuf, int *pszTmpBuf, int IsFile)
{
	int lPath, lTmpBuf;
	char TmpRootPath[256];
	
	if(!Path || !TmpBuf || !pszTmpBuf)
		return NULL;
	
	lTmpBuf = *pszTmpBuf;
	lPath = strlen(Path);
	
	// change root for later resolution
	if(!IS_ABSOLUTE_PATH(Path, lPath))
	{
		if(RelativePathRoot)
			snprintf(TmpRootPath, sizeof(TmpRootPath) - 1, "%s/%s", RelativePathRoot, Path);
		else
			snprintf(TmpRootPath, sizeof(TmpRootPath) - 1, "%s", Path);
		Path = TmpRootPath;
	}
	
	if(!FileSystem_RealPath(Path, TmpBuf, &lTmpBuf))
		return NULL;
	
	if(IS_SLASH(TmpBuf[lTmpBuf - 1]))
		TmpBuf[--lTmpBuf] = 0;
	
	if(IsFile)
	{
		if(!FileSystem_FileExists(TmpBuf))
		{
			//LOG_DEBUG("error: file not found: %s", TmpBuf);
			return NULL;
		}
	}
	else
	{
		if(!FileSystem_FolderExists(TmpBuf))
		{
			//LOG_DEBUG("error: folder not found: %s", TmpBuf);
			return NULL;
		}
	}
	
	*pszTmpBuf = lTmpBuf;
	return TmpBuf;
}


int FileSystem_SplitDirFileName(char *Path, char *Dirname, int *pszDirname, char *Filename, int *pszFilename)
{
	char TmpBuf[256];
	int i, lTmpBuf;
	
	if(!Path || (!Dirname && !Filename) || (Dirname && !pszDirname) || (Filename && !pszFilename))
	{
		LOG_DEBUG("error: invalid arguments");
		return 0;
	}
	
	if((pszDirname && *pszDirname <= 2) || (pszFilename && *pszFilename <= 2))
	{
		LOG_DEBUG("error: invalid arguments");
		return 0;
	}
	
	lTmpBuf = sizeof(TmpBuf);
	if(!FileSystem_RealPath(Path, TmpBuf, &lTmpBuf))
		return 0;
	
	if(IS_SLASH(TmpBuf[lTmpBuf - 1]))
		TmpBuf[--lTmpBuf] = 0;
	
	for(i=lTmpBuf-1;i>=0;i--)
	{
		if(IS_SLASH(TmpBuf[i]))
		{
			if(Dirname)
				*pszDirname = snprintf(Dirname, *pszDirname - 1, "%.*s", i, TmpBuf);
			if(Filename)
				*pszFilename = snprintf(Filename, *pszFilename - 1, "%s", TmpBuf + i + 1);
			return 1;
		}
	}
	
	return 0;
}


int FileSystem_GetTempPath(char *Path, int szPath)
{
	char TmpPath[256];
	int lTmpPath;
	
	
	if(!Path || szPath <= 0)
		return 0;
	
	memset(TmpPath, 0, sizeof(TmpPath));
	
#ifdef WIN32
	lTmpPath = GetTempPath(sizeof(TmpPath), TmpPath);
	if(TmpPath[lTmpPath - 1] == '\\' || TmpPath[lTmpPath - 1] == '/')
	{
		TmpPath[lTmpPath - 1] = 0;
		lTmpPath--;
	}
#else
	lTmpPath = snprintf(TmpPath, sizeof(TmpPath) - 1, "/tmp");
#endif
	
	if(lTmpPath >= szPath)
		return 0;
	
	memcpy(Path, TmpPath, lTmpPath + 1);
	
	return lTmpPath;
}


int FileSystem_GetTempFilename(char *Filename, int szFilename, char *Prefix)
{
	char TmpFilename[512];
	int lTmpFilename, lPrefix, lSuffix;
	uint32 uRand;
	
	
	if(!Filename || szFilename <= 0)
		return 0;
	
	//memset(TmpFilename, 0, sizeof(TmpFilename));
	lPrefix = 0;
	if(Prefix)
		lPrefix = strlen(Prefix);
	
	lTmpFilename = FileSystem_GetTempPath(TmpFilename, sizeof(TmpFilename));
	if(lTmpFilename <= 0)
		return 0;
	
	uRand = (uint32)getpid() ^ Random_PRNG();
	
	if(lPrefix)
		lSuffix = snprintf(TmpFilename + lTmpFilename, sizeof(TmpFilename) - lTmpFilename - 1, "%c%s.%08x.tmp", DEFAULT_SLASH, Prefix, uRand);
	else
		lSuffix = snprintf(TmpFilename + lTmpFilename, sizeof(TmpFilename) - lTmpFilename - 1, "%c%08x.tmp", DEFAULT_SLASH, uRand);
	
	if(FileSystem_FileExists(TmpFilename))
	{
		int i;
		
		for(i=0;i<128;i++)
		{
			if(lPrefix)
				lSuffix = snprintf(TmpFilename + lTmpFilename, sizeof(TmpFilename) - lTmpFilename - 1, "%c%s.%08x.%08x.tmp", DEFAULT_SLASH, Prefix, uRand, i);
			else
				lSuffix = snprintf(TmpFilename + lTmpFilename, sizeof(TmpFilename) - lTmpFilename - 1, "%c%08x.%08x.tmp", DEFAULT_SLASH, uRand, i);
			
			if(!FileSystem_FileExists(TmpFilename))
				goto _ok_;
		}
		
		return 0;
	}
	
_ok_:
	lTmpFilename += lSuffix;
	if(lTmpFilename >= szFilename)
		return 0;
	
	memcpy(Filename, TmpFilename, lTmpFilename + 1);
	
	return lTmpFilename;
}


static int _Sort_Clbk(const void *Arg0, const void *Arg1)
{
	char *Str0 = *(char**)Arg0;
	char *Str1 = *(char**)Arg1;
	return String_Compare_Natural(Str0, Str1, 1);
}

int FileSystem_GetFilesInDir(char *BasePath, char *MatchString, char ***pFiles, int *plFiles)
{
	DIR *hFolder;
	struct dirent *FolderItem;
	char **Files = NULL;
	int lFiles = 0;
	
	if(!BasePath || !pFiles || !plFiles)
		return 0;
	
	*pFiles = NULL;
	*plFiles = 0;
	
	hFolder = opendir(BasePath);
	if(!hFolder)
		return 0;
	
	while(1)
	{
		FolderItem = readdir(hFolder);
		if(!FolderItem)
			break;
		
		if(!MatchString || String_Match(FolderItem->d_name, MatchString))
		{
			Files = realloc(Files, (lFiles + 1) * sizeof(char*));
			Files[lFiles] = strdup(FolderItem->d_name);
			lFiles++;
		}
	}
	closedir(hFolder);
	
	qsort(Files, lFiles, sizeof(char*), _Sort_Clbk);
	
	*pFiles = Files;
	*plFiles = lFiles;
	
	return 1;
}


int FileSystem_ChangeDir(char *Path)
{
	if(!Path)
		return 0;
	return (chdir(Path) == 0);
}
