#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#if defined(LINUX)
#include <linux/sockios.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <linux/if.h>
#include <errno.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#elif defined(FREEBSD) || defined(APPLE)
#include <sys/param.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/sysctl.h>
#include <net/if.h>
#include <net/if_dl.h>
#include <net/route.h>
#include <netinet/in.h>
#include <netinet/if_ether.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/sockio.h>
#include <ifaddrs.h>
#include <sys/ioctl.h>
#include <netdb.h>
#endif

#include <libshred.h>



static int _GetHexValue(char Hex)
{
	int iHex = tolower(Hex);
	
	if(iHex >= '0' && iHex <= '9')
		return iHex - '0';
	
	if(iHex >= 'a' && iHex <= 'f')
		return 10 + iHex - 'a';
	
	return -1;
}


#ifdef LINUX
static int _HexStringToUINT32(char *String, uint32 *pdwValue)
{
	uint32 dwValue;
	int i, b;
	
	
	if(!String || !pdwValue)
		return 0;
	
	if(strlen(String) != 8)
		return 0;
	
	dwValue = 0;
	for(i=0;i<8;i++)
	{
		b = _GetHexValue(tolower(String[i]));
		if(b < 0)
			return 0;
		dwValue |= ((uint32)b) << (28 - 4 * i);
	}
	
	*pdwValue = dwValue;
	
	return 1;
}
#endif


int IP_MACFromString(char *MAC, uint8 bMAC[MAC_ADDR_RAW_SIZE])
{
	int i, bh, bl;
	
	
	if(!MAC || !bMAC)
		return 0;
	
	if(strlen(MAC) != 17)
		return 0;
	
	for(i=0;i<MAC_ADDR_RAW_SIZE;i++)
	{
		bh = _GetHexValue(MAC[3 * i]);
		bl = _GetHexValue(MAC[3 * i + 1]);
		
		if(bh < 0 || bl < 0)
			return 0;
		
		bMAC[i] = bh * 16 + bl;
		
		if(i < 5)
		{
			char Sep = MAC[3 * i + 2];
			
			if(Sep != ':' && Sep != '-')
				return 0;
		}
	}
	
	return 1;
}


char * IP_MACToString(uint8 MAC[MAC_ADDR_RAW_SIZE], char *TmpBuf, int szTmpBuf)
{
	char *HexCharset = "0123456789abcdef";
	char Buf[18];
	
	if(!MAC || !TmpBuf || szTmpBuf < 18)
		return NULL;
	
	Buf[ 0] = HexCharset[MAC[0] / 16];
	Buf[ 1] = HexCharset[MAC[0] % 16];
	Buf[ 2] = ':';
	Buf[ 3] = HexCharset[MAC[1] / 16];
	Buf[ 4] = HexCharset[MAC[1] % 16];
	Buf[ 5] = ':';
	Buf[ 6] = HexCharset[MAC[2] / 16];
	Buf[ 7] = HexCharset[MAC[2] % 16];
	Buf[ 8] = ':';
	Buf[ 9] = HexCharset[MAC[3] / 16];
	Buf[10] = HexCharset[MAC[3] % 16];
	Buf[11] = ':';
	Buf[12] = HexCharset[MAC[4] / 16];
	Buf[13] = HexCharset[MAC[4] % 16];
	Buf[14] = ':';
	Buf[15] = HexCharset[MAC[5] / 16];
	Buf[16] = HexCharset[MAC[5] % 16];
	Buf[17] = 0;
	
	String_Copy(Buf, TmpBuf, szTmpBuf);
	
	return TmpBuf;
}


uint32 IP_IPToUint32(char *IP)
{
	uint32 dwIP;
	if(Net_IPToUint32(IP, &dwIP))
		return Net_ToHost32(dwIP);
	return 0;
}


char *IP_Uint32ToIP(uint32 dwIP, char *TmpBuf, int szTmpBuf)
{
	return Net_Uint32ToIP(Net_ToNet32(dwIP), TmpBuf, szTmpBuf);
}


uint32 IP_GetNetworkAddr(uint32 dwIP, uint32 dwNetmask)
{
	return (dwIP & dwNetmask);
}


uint32 IP_GetBroadcastAddr(uint32 dwIP, uint32 dwNetmask)
{
	return (dwIP | (~dwNetmask));
}


uint32 IP_GetFirstAddr(uint32 dwIP, uint32 dwNetmask)
{
	return IP_GetNetworkAddr(dwIP, dwNetmask) + 1;
}


uint32 IP_GetLastAddr(uint32 dwIP, uint32 dwNetmask)
{
	return IP_GetBroadcastAddr(dwIP, dwNetmask) - 1;
}


int IP_NetmaskToCidr(uint32 dwNetmask)
{
	int i;
	
	if(!dwNetmask)
		return 0;
	
	//dwNetmask = Net_ToHost32(dwNetmask);
	i = 0;
	while(dwNetmask)
	{
		dwNetmask <<= 1;
		i++;
	}
	
	return i;
}


char *IP_CidrToNetmask(int Cidr, char *TmpBuf, int szTmpBuf)
{
	uint32 dwNetmask;
	int i;
	
	dwNetmask = 0;
	for(i=0;i<Cidr;i++)
		dwNetmask = (dwNetmask >> 1) | 0x80000000;
	
	return IP_Uint32ToIP(dwNetmask, TmpBuf, szTmpBuf);
}


TInterfaceInfo *IP_GetInterface(TInterfaceInfo *IPInfo, int szIPInfo, char *Name)
{
	int i;
	
	if(!IPInfo || szIPInfo <= 0 || !Name)
		return NULL;
	
	for(i=0;i<szIPInfo;i++)
	{
		TInterfaceInfo *pInfo = &IPInfo[i];
		if(strcmp(pInfo->Name, Name) == 0)
			return pInfo;
	}
	return NULL;
}


void IP_FreeInterfaceTable(TInterfaceInfo *IPInfo, int szIPInfo)
{
	int i;
	
	
	if(!IPInfo || szIPInfo <= 0)
		return;
	
	for(i=0;i<szIPInfo;i++)
	{
		if(IPInfo[i].Name)
			free(IPInfo[i].Name);
		
		if(IPInfo[i].Description)
			free(IPInfo[i].Description);
	}
	
	free(IPInfo);
}


#ifdef WIN32
int IP_GetInterfaceTable_WIN32(TInterfaceInfo **pIPInfo, int *pszIPInfo)
{
	PIP_ADAPTER_INFO pAdapterInfo;
	PIP_ADAPTER_INFO pAdapter = NULL;
	ULONG ulOutBufLen;
	DWORD dwRetVal = 0;
	TInterfaceInfo *IPInfo = NULL;
	int szIPInfo = 0;
	
	
	if(!pIPInfo || !pszIPInfo)
		return 0;
	
	*pszIPInfo = 0;
	
	pAdapterInfo = (IP_ADAPTER_INFO*)malloc(sizeof(IP_ADAPTER_INFO));
	memset(pAdapterInfo, 0, sizeof(IP_ADAPTER_INFO));
	
	// make an initial call to GetAdaptersInfo to get
	// the necessary size into the ulOutBufLen variable
	ulOutBufLen = sizeof(IP_ADAPTER_INFO);
	if(GetAdaptersInfo(pAdapterInfo, &ulOutBufLen) == ERROR_BUFFER_OVERFLOW)
	{
		free(pAdapterInfo);
		pAdapterInfo = (IP_ADAPTER_INFO*)malloc(ulOutBufLen);
		memset(pAdapterInfo, 0, ulOutBufLen);
	}
	
	dwRetVal = GetAdaptersInfo(pAdapterInfo, &ulOutBufLen);
	if(dwRetVal == ERROR_SUCCESS)
	{
		IP_ADDR_STRING *IPList;
		char *IP, *Netmask;
		
		
		pAdapter = pAdapterInfo;
		
		while(pAdapter)
		{
			//DumpAdapter(pAdapter);
			
			// skip non ethernet adapter
			if(pAdapter->Type != MIB_IF_TYPE_ETHERNET)
			{
				pAdapter = pAdapter->Next;
				continue;
			}
			
			IPList = &pAdapter->IpAddressList;
			
			while(IPList)
			{
				IP = IPList->IpAddress.String;
				Netmask = IPList->IpMask.String;
				
				IPInfo = realloc(IPInfo, (szIPInfo + 1) * sizeof(TInterfaceInfo));
				memset(&IPInfo[szIPInfo], 0, sizeof(TInterfaceInfo));
				IPInfo[szIPInfo].Name = strdup(pAdapter->AdapterName);
				IPInfo[szIPInfo].Description = strdup(pAdapter->Description);
				IPInfo[szIPInfo].dwIP = IP_IPToUint32(IP);
				IPInfo[szIPInfo].dwNetmask = IP_IPToUint32(Netmask);
				if(pAdapter->AddressLength == MAC_ADDR_RAW_SIZE)
					memcpy(&IPInfo[szIPInfo].bMAC, pAdapter->Address, MAC_ADDR_RAW_SIZE);
				szIPInfo++;
				
				IPList = IPList->Next;
			}
			
			pAdapter = pAdapter->Next;
		}
	}
	else
	{
		LPVOID lpMsgBuf;
		
		if(FormatMessage(FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM | FORMAT_MESSAGE_IGNORE_INSERTS, NULL, dwRetVal, MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), (LPTSTR)&lpMsgBuf, 0, NULL))
			LOG_DEBUG("error: GetAdaptersInfo(): %s", (char*)lpMsgBuf);
		LocalFree(lpMsgBuf);
	}
	
	free(pAdapterInfo);
	
	*pIPInfo = IPInfo;
	*pszIPInfo = szIPInfo;
	
	return 1;
}
#endif


#ifdef LINUX
int IP_GetInterfaceTable_LINUX(TInterfaceInfo **pIPInfo, int *pszIPInfo)
{
	struct ifreq IfReq;
	struct sockaddr_in *pAddr;
	int i, Success = 0;
	uint32 dwIP, dwNetmask;
	uint8 bMAC[MAC_ADDR_RAW_SIZE];
	TInterfaceInfo *IPInfo = NULL;
	int szIPInfo = 0;
	char *DeviceName;
	char **DeviceNames = NULL;
	int lDeviceNames = 0;
	int Socket = -1;
	
	
	if(!pIPInfo || !pszIPInfo)
		return 0;
	
	*pszIPInfo = 0;
	
	//Socket = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
	Socket = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
	if(Socket == -1)
	{
		LOG_DEBUG("error: socket()");
		return 0;
	}
	
	// build interface name list
	
	// try to use /proc/net/dev to populate DeviceNames
	{
		FILE *File;
		char Line[256];
		int iPos;
		
		File = fopen("/proc/net/dev", "r");
		if(!File)
		{
			LOG_DEBUG("warning: cannot open /proc/net/dev");
		}
		else
		{
			fgets(Line, sizeof(Line), File);	// skip first 2 lines
			fgets(Line, sizeof(Line), File);
			
			while(!feof(File))
			{
				Line[0] = 0;
				fgets(Line, sizeof(Line), File);
				String_Trim(Line);
				iPos = String_IndexOf(Line, ':');
				if(iPos > 0)
				{
					Line[iPos] = 0;
					String_Trim(Line);
					
					DeviceNames = realloc(DeviceNames, (lDeviceNames + 1) * sizeof(char*));
					DeviceNames[lDeviceNames++] = strdup(Line);
				}
			}
			fclose(File);
		}
	}
	
	// use SIOCGIFCONF method instead
	if(!lDeviceNames)
	{
		struct ifconf IfConf;
		int nInterfaces;
		
		memset(&IfConf, 0, sizeof(struct ifconf));
		IfConf.ifc_len = 4096;
		IfConf.ifc_buf = (char*)malloc(IfConf.ifc_len);
		memset(IfConf.ifc_buf, 0, IfConf.ifc_len);
		
		if(ioctl(Socket, SIOCGIFCONF, &IfConf) == -1)
		{
			LOG_DEBUG("error: ioctl(SIOCGIFCONF)");
			goto _end_;
		}
		
		nInterfaces = IfConf.ifc_len / sizeof(struct ifreq);
		LOG_DEBUG("nInterfaces: %d", nInterfaces);
		LOG_DEBUG("IfConf.ifc_len: %d", IfConf.ifc_len);
		
		for(i=0;i<nInterfaces;i++)
		{
			DeviceName = IfConf.ifc_req[i].ifr_name;
			LOG_DEBUG("interface name: [%s]", DeviceName);
			String_Trim(DeviceName);
			if(DeviceName[0] < 32)
			{
				LOG_DEBUG("warning: strange interface name: [%s]", DeviceName);
				continue;
			}
			
			DeviceNames = realloc(DeviceNames, (lDeviceNames + 1) * sizeof(char*));
			DeviceNames[lDeviceNames++] = strdup(DeviceName);
		}
		free(IfConf.ifc_buf);
	}
	DeviceName = NULL;
	
	
	// now request info on found devices
	memset(&IfReq, 0, sizeof(struct ifreq));
	for(i=0;i<lDeviceNames;i++)
	{
		DeviceName = DeviceNames[i];
		
		strcpy(IfReq.ifr_name, DeviceName);
		
		// ip address
		dwIP = 0;
		if(ioctl(Socket, SIOCGIFADDR, &IfReq) == -1)
		{
			//LOG_DEBUG("error: ioctl(SIOCGIFADDR) on device [%s]", DeviceName);
		}
		else
		{
			pAddr = (struct sockaddr_in*)&IfReq.ifr_addr;
			dwIP = pAddr->sin_addr.s_addr;
		}
		
		// skip broadcast: SIOCGIFBRDADDR
		
		// netmask
		dwNetmask = 0;
		if(ioctl(Socket, SIOCGIFNETMASK, &IfReq) == -1)
		{
			//LOG_DEBUG("error: ioctl(SIOCGIFNETMASK)");
		}
		else
		{
#if defined(ifr_netmask)
			pAddr = (struct sockaddr_in *)&IfReq.ifr_netmask;
			dwNetmask = pAddr->sin_addr.s_addr;
#else
			// for FreeBSD
			pAddr = (struct sockaddr_in *)&IfReq.ifr_addr;
			dwNetmask = pAddr->sin_addr.s_addr;
#endif
		}
		
		memset(bMAC, 0, MAC_ADDR_RAW_SIZE);
#if defined(SIOCGIFHWADDR)
		// hw_addr, does not work on FreeBSD
		if(ioctl(Socket, SIOCGIFHWADDR, &IfReq) == -1)
		{
			//LOG_DEBUG("error: ioctl(SIOCGIFHWADDR)");
		}
		else
			memcpy(bMAC, IfReq.ifr_hwaddr.sa_data, MAC_ADDR_RAW_SIZE);
#endif
		
		IPInfo = realloc(IPInfo, (szIPInfo + 1) * sizeof(TInterfaceInfo));
		memset(&IPInfo[szIPInfo], 0, sizeof(TInterfaceInfo));
		IPInfo[szIPInfo].Name = strdup(IfReq.ifr_name);
		IPInfo[szIPInfo].Description = NULL;		// not for linux
		IPInfo[szIPInfo].dwIP = Net_ToHost32(dwIP);
		IPInfo[szIPInfo].dwNetmask = Net_ToHost32(dwNetmask);
		memcpy(&IPInfo[szIPInfo].bMAC, bMAC, MAC_ADDR_RAW_SIZE);
		szIPInfo++;
	}
	
	*pIPInfo = IPInfo;
	*pszIPInfo = szIPInfo;
	Success = 1;
	
_end_:
	Net_CloseSocket(&Socket);
	
	if(DeviceNames)
	{
		for(i=0;i<lDeviceNames;i++)
		{
			if(DeviceNames[i])
				free(DeviceNames[i]);
		}
		free(DeviceNames);
	}
	
	return Success;
}
#endif


#if defined(FREEBSD) || defined(APPLE)
int IP_GetInterfaceTable_BSD(TInterfaceInfo **pIPInfo, int *pszIPInfo)
{
	TInterfaceInfo *IPInfo = NULL;
	int szIPInfo = 0;
	struct ifaddrs *IfAddrHead = NULL, *pIfAddr;
	
	if(!pIPInfo || !pszIPInfo)
		return 0;
	
	*pszIPInfo = 0;
	
	if(getifaddrs(&IfAddrHead) < 0)
	{
		LOG_DEBUG("error: getifaddrs()");
		return 0;
	}
	
	for(pIfAddr=IfAddrHead;pIfAddr;pIfAddr=pIfAddr->ifa_next)
	{
		char *Name = pIfAddr->ifa_name;
		TInterfaceInfo *pInfo = NULL;
		
		if(!Name)
			continue;
		
		pInfo = IP_GetInterface(IPInfo, szIPInfo, Name);
		if(!pInfo)
		{
			// if interface not found, create it
			IPInfo = realloc(IPInfo, (szIPInfo + 1) * sizeof(TInterfaceInfo));
			pInfo = &IPInfo[szIPInfo++];
			memset(pInfo, 0, sizeof(TInterfaceInfo));
			pInfo->Name = strdup(Name);
		}
		
		switch(pIfAddr->ifa_addr->sa_family)
		{
			case AF_LINK:
			{
				// in this case, only MAC address is available
				memcpy(pInfo->bMAC, LLADDR((struct sockaddr_dl*)(pIfAddr->ifa_addr)), 6);
				break;
			}
			
			case AF_INET:
			{
				// in this case, only IP address and netmask are available
				pInfo->dwIP = 0;
				if(pIfAddr->ifa_addr)
					pInfo->dwIP = Net_ToHost32(((struct sockaddr_in*)(pIfAddr->ifa_addr))->sin_addr.s_addr);
				
				pInfo->dwNetmask = 0;
				if(pIfAddr->ifa_netmask)
					pInfo->dwNetmask = Net_ToHost32(((struct sockaddr_in*)(pIfAddr->ifa_netmask))->sin_addr.s_addr);
				
				break;
			}
			
			case AF_INET6:
			default:
				// unhandled cases
				break;
		}
	}
	
	freeifaddrs(IfAddrHead);
	
	*pIPInfo = IPInfo;
	*pszIPInfo = szIPInfo;
	
	return 1;
}
#endif


int IP_GetInterfaceTable(TInterfaceInfo **pIPInfo, int *pszIPInfo)
{
#ifdef WIN32
	return IP_GetInterfaceTable_WIN32(pIPInfo, pszIPInfo);
#endif
	
#ifdef LINUX
	return IP_GetInterfaceTable_LINUX(pIPInfo, pszIPInfo);
#endif
	
#if defined(FREEBSD) || defined(APPLE)
	return IP_GetInterfaceTable_BSD(pIPInfo, pszIPInfo);
#endif
	LOG_DEBUG("error: method not implemented");
	return 0;
}


int IP_GetRouteTable(TRouteInfo **pRouteInfo, int *pszRouteInfo)
{
#if defined(WIN32)
	PMIB_IPFORWARDTABLE pIpForwardTable;
	DWORD dwSize = 0, dwRetVal = 0;
	TRouteInfo *RouteInfo = NULL;
	int i, szRouteInfo = 0;
	
	
	if(!pRouteInfo || !pszRouteInfo)
		return 0;
	
	pIpForwardTable = (MIB_IPFORWARDTABLE*)malloc(sizeof(MIB_IPFORWARDTABLE));
	if(GetIpForwardTable(pIpForwardTable, &dwSize, 0) == ERROR_INSUFFICIENT_BUFFER)
	{
		free(pIpForwardTable);
		pIpForwardTable = (MIB_IPFORWARDTABLE*)malloc(dwSize);
	}
	
	dwRetVal = GetIpForwardTable(pIpForwardTable, &dwSize, 0);
	if(dwRetVal == NO_ERROR)
	{
		//char TmpBuf[20];
		
		//LOG_DEBUG("dwNumEntries: %d", (int)pIpForwardTable->dwNumEntries);
		for(i=0;i<pIpForwardTable->dwNumEntries;i++)
		{
			RouteInfo = realloc(RouteInfo, (szRouteInfo + 1) * sizeof(TRouteInfo));
			memset(&RouteInfo[szRouteInfo], 0, sizeof(TRouteInfo));
			RouteInfo[szRouteInfo].dwDestination = Net_ToHost32(pIpForwardTable->table[i].dwForwardDest);
			RouteInfo[szRouteInfo].dwGateway = Net_ToHost32(pIpForwardTable->table[i].dwForwardNextHop);
			RouteInfo[szRouteInfo].dwNetmask = Net_ToHost32(pIpForwardTable->table[i].dwForwardMask);
			szRouteInfo++;
		}
	}
	else
	{
		LPVOID lpMsgBuf;
		
		if(FormatMessage(FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM | FORMAT_MESSAGE_IGNORE_INSERTS, NULL, dwRetVal, MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), (LPTSTR)&lpMsgBuf, 0, NULL))
			LOG_DEBUG("error: GetIpForwardTable(): %s", (char*)lpMsgBuf);
		LocalFree(lpMsgBuf);
	}
	
	if(pIpForwardTable)
		free(pIpForwardTable);
	
	// on windows, we must reverse the table
	{
		TRouteInfo Tmp;
		
		for(i=0;i<szRouteInfo/2;i++)
		{
			memcpy(&Tmp, &RouteInfo[i], sizeof(TRouteInfo));
			memcpy(&RouteInfo[i], &RouteInfo[szRouteInfo - i - 1], sizeof(TRouteInfo));
			memcpy(&RouteInfo[szRouteInfo - i - 1], &Tmp, sizeof(TRouteInfo));
		}
	}
	
	*pRouteInfo = RouteInfo;
	*pszRouteInfo = szRouteInfo;
	
	return 1;
	
#elif defined(LINUX)
#define ROUTE_FILE	"/proc/net/route"
	FILE *File = NULL;
	char Line[256], Destination[12], Gateway[12], Netmask[12];
	int Result;
	uint32 dwDestination, dwGateway, dwNetmask;
	TRouteInfo *RouteInfo = NULL;
	int szRouteInfo = 0;
	
	
	if(!pRouteInfo || !pszRouteInfo)
		return 0;
	
	File = fopen(ROUTE_FILE, "r");
	if(!File)
	{
		LOG_DEBUG("error: opening file '%s'", ROUTE_FILE);
		return 0;
	}
	
	// skip first line
	fgets(Line, sizeof(Line) - 1, File);
	
	while(!feof(File))
	{
		Line[0] = 0;
		fgets(Line, sizeof(Line) - 1, File);
		String_Trim(Line);
		
		if(!Line[0])
			continue;
		
		//	eth0    00000A0A        00000000        0001    0       0       0       0000FFFF        0       0       0
		Result = sscanf(Line, "%*s %s %s %*s %*s %*s %*s %s %*s %*s %*s", Destination, Gateway, Netmask);
		if(Result != 3)
			continue;
		
		if(!_HexStringToUINT32(Destination, &dwDestination))
			continue;
		
		if(!_HexStringToUINT32(Gateway, &dwGateway))
			continue;
		
		if(!_HexStringToUINT32(Netmask, &dwNetmask))
			continue;
		
		RouteInfo = realloc(RouteInfo, (szRouteInfo + 1) * sizeof(TRouteInfo));
		memset(&RouteInfo[szRouteInfo], 0, sizeof(TRouteInfo));
		RouteInfo[szRouteInfo].dwDestination = Net_ToHost32(dwDestination);
		RouteInfo[szRouteInfo].dwGateway = Net_ToHost32(dwGateway);
		RouteInfo[szRouteInfo].dwNetmask = Net_ToHost32(dwNetmask);
		szRouteInfo++;
	}
	fclose(File);
	
	*pRouteInfo = RouteInfo;
	*pszRouteInfo = szRouteInfo;
	
	return 1;
	
#elif defined(FREEBSD) || defined(APPLE)
	TRouteInfo *RouteInfo = NULL;
	int szRouteInfo = 0;
	int Socket = -1;
	int MIB[6] = { CTL_NET, PF_ROUTE, 0, 0, NET_RT_DUMP, 0 };
	size_t lBuffer;
	
	
	if(!pRouteInfo || !pszRouteInfo)
		return 0;
	
	Socket = socket(PF_ROUTE, SOCK_RAW, AF_INET);
	if(Socket < 0)
	{
		LOG_DEBUG("error: socket()");
		return 0;
	}
	
	if(sysctl(MIB, 6, NULL, &lBuffer, NULL, 0) < 0)
	{
		LOG_DEBUG("error: sysctl()");
	}
	else
	{
		if(lBuffer > 0)
		{
			char *Buffer;
			
			Buffer = malloc(lBuffer);
			if (sysctl(MIB, 6, Buffer, &lBuffer, NULL, 0) < 0)
			{
				LOG_DEBUG("error: sysctl()");
			}
			else
			{
				struct rt_msghdr *rtm;
				struct sockaddr_in *sin;
				uint32 dwDestination, dwGateway, dwNetmask;
				char *pBufMax, *pBufNext;
				
				pBufMax = Buffer + lBuffer;
				pBufNext = Buffer;
				
				for(;pBufNext<pBufMax;pBufNext+=rtm->rtm_msglen)
				{
					rtm = (struct rt_msghdr *)pBufNext;
					sin = (struct sockaddr_in *)(rtm + 1);
					
					if((rtm->rtm_addrs & RTA_GATEWAY) == 0 || sin->sin_family != AF_INET)
						continue;
					
#define ROUNDUP(a)	((a) > 0 ? (1 + (((a) - 1) | (sizeof(long) - 1))) : sizeof(long))
#define NEXT_SIN(s)	((struct sockaddr_in*)((u_char*)(s) + ROUNDUP((s)->sin_len)))
					
					dwDestination = sin->sin_addr.s_addr;
					sin = NEXT_SIN(sin);
					
					if(sin->sin_family != AF_INET)
						continue;
					
					dwGateway = sin->sin_addr.s_addr;
					
					dwNetmask = 0;
					if(rtm->rtm_addrs & RTA_NETMASK)
					{
						sin = NEXT_SIN(sin);
						if(sin->sin_len >= 4)
							dwNetmask = sin->sin_addr.s_addr;
					}
					
					RouteInfo = realloc(RouteInfo, (szRouteInfo + 1) * sizeof(TRouteInfo));
					memset(&RouteInfo[szRouteInfo], 0, sizeof(TRouteInfo));
					RouteInfo[szRouteInfo].dwDestination = Net_ToHost32(dwDestination);
					RouteInfo[szRouteInfo].dwGateway = Net_ToHost32(dwGateway);
					RouteInfo[szRouteInfo].dwNetmask = Net_ToHost32(dwNetmask);
					szRouteInfo++;
				 }
			}
			free(Buffer);
		}
	}
	
	close(Socket);
	
	*pRouteInfo = RouteInfo;
	*pszRouteInfo = szRouteInfo;
	
	return 1;
#endif
	
	return 0;
}


int IP_GetARPTable(TARPInfo **pARPInfo, int *pszARPInfo)
{
#if defined(LINUX)
#define ARP_FILE	"/proc/net/arp"
	FILE *File = NULL;
	char Line[256], IP[16], MAC[18];
	int Result;
	uint32 dwIP;
	uint8 bMAC[MAC_ADDR_RAW_SIZE];
	TARPInfo *ARPInfo = NULL;
	int szARPInfo = 0;
	
	
	if(!pARPInfo || !pszARPInfo)
		return 0;
	
	File = fopen(ARP_FILE, "r");
	if(!File)
	{
		LOG_DEBUG("error: opening file '%s'", ARP_FILE);
		return 0;
	}
	
	// skip first line
	fgets(Line, sizeof(Line) - 1, File);
	
	while(!feof(File))
	{
		Line[0] = 0;
		fgets(Line, sizeof(Line) - 1, File);
		String_Trim(Line);
		
		if(!Line[0])
			continue;
		
		//	10.10.2.1        0x1         0x2         00:17:31:DB:A9:9F     *        eth0
		Result = sscanf(Line, "%s %*s %*s %s %*s %*s", IP, MAC);
		if(Result != 2)
			continue;
		
		dwIP = IP_IPToUint32(IP);
		IP_MACFromString(MAC, bMAC);
		
		ARPInfo = realloc(ARPInfo, (szARPInfo + 1) * sizeof(TARPInfo));
		memset(&ARPInfo[szARPInfo], 0, sizeof(TARPInfo));
		ARPInfo[szARPInfo].dwIP = dwIP;
		memcpy(&ARPInfo[szARPInfo].bMAC, bMAC, MAC_ADDR_RAW_SIZE);
		szARPInfo++;
	}
	fclose(File);
	
	*pARPInfo = ARPInfo;
	*pszARPInfo = szARPInfo;
	
	return 1;
	
#elif defined(WIN32)
	
	PMIB_IPNETTABLE IpNetTable = NULL;
	DWORD dwSize = 0, dwRetVal = 0, dwResult;
	TARPInfo *ARPInfo = NULL;
	int i, szARPInfo = 0;
	
	
	if(!pARPInfo || !pszARPInfo)
		return 0;
	
	dwResult = GetIpNetTable(NULL, &dwSize, 0);
	if(dwResult == ERROR_INSUFFICIENT_BUFFER)
		IpNetTable = (MIB_IPNETTABLE *)malloc(dwSize);
	
	dwRetVal = GetIpNetTable(IpNetTable, &dwSize, 0);
	if(dwRetVal == NO_ERROR)
	{
		for(i=0;i<IpNetTable->dwNumEntries;i++)
		{
			// we keep only mac address
			if(IpNetTable->table[i].dwPhysAddrLen != MAC_ADDR_RAW_SIZE)
				continue;
			
			ARPInfo = realloc(ARPInfo, (szARPInfo + 1) * sizeof(TARPInfo));
			memset(&ARPInfo[szARPInfo], 0, sizeof(TARPInfo));
			ARPInfo[szARPInfo].dwIP = Net_ToHost32(IpNetTable->table[i].dwAddr);
			memcpy(&ARPInfo[szARPInfo].bMAC, IpNetTable->table[i].bPhysAddr, MAC_ADDR_RAW_SIZE);
			szARPInfo++;
		}
	}
	else
	{
		LPVOID lpMsgBuf;
		
		if(FormatMessage(FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM | FORMAT_MESSAGE_IGNORE_INSERTS, NULL, dwRetVal, MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), (LPTSTR)&lpMsgBuf, 0, NULL))
			LOG_DEBUG("error: GetIpNetTable(): %s", (char*)lpMsgBuf);
		LocalFree(lpMsgBuf);
	}
	
	if(IpNetTable)
		free(IpNetTable);
	
	*pARPInfo = ARPInfo;
	*pszARPInfo = szARPInfo;
	
	return 1;
	
#elif defined(FREEBSD) || defined(APPLE)
	int Socket = -1;
	size_t lBuffer;
	int MIB[6] = { CTL_NET, PF_ROUTE, 0, AF_INET, NET_RT_FLAGS, RTF_LLINFO };
	TARPInfo *ARPInfo = NULL;
	int szARPInfo = 0;
	
	
	if(!pARPInfo || !pszARPInfo)
		return 0;
	
	Socket = socket(PF_ROUTE, SOCK_RAW, 0);
	if(Socket < 0)
	{
		LOG_DEBUG("error: socket()");
		return 0;
	}
	
	if(sysctl(MIB, 6, NULL, &lBuffer, NULL, 0) < 0)
	{
		LOG_DEBUG("error: sysctl()");
	}
	else
	{
		if(lBuffer != 0)
		{
			char *Buffer;
			
			Buffer = malloc(lBuffer);
			
			if(sysctl(MIB, 6, Buffer, &lBuffer, NULL, 0) < 0)
			{
				LOG_DEBUG("error: sysctl()");
			}
			else
			{
				struct rt_msghdr *rtm;
				struct sockaddr_dl *sdl;
				struct sockaddr_in *sin;
				char *pBufMax, *pBufNext;
				
				
				pBufMax = Buffer + lBuffer;
				for(pBufNext=Buffer;pBufNext<pBufMax;pBufNext+=rtm->rtm_msglen)
				{
					rtm = (struct rt_msghdr*)pBufNext;
					sin = (struct sockaddr_in*)(rtm + 1);
					sdl = (struct sockaddr_dl*)(sin + 1);
					
					if(sin->sin_family == AF_INET && sdl->sdl_family == AF_LINK && sdl->sdl_alen == MAC_ADDR_RAW_SIZE)
					{
						ARPInfo = realloc(ARPInfo, (szARPInfo + 1) * sizeof(TARPInfo));
						memset(&ARPInfo[szARPInfo], 0, sizeof(TARPInfo));
						ARPInfo[szARPInfo].dwIP = Net_ToHost32(sin->sin_addr.s_addr);
						memcpy(&ARPInfo[szARPInfo].bMAC, LLADDR(sdl), MAC_ADDR_RAW_SIZE);
						szARPInfo++;
					}
				}
			}
			free(Buffer);
		}
	}
	
	close(Socket);
	
	*pARPInfo = ARPInfo;
	*pszARPInfo = szARPInfo;
	
	return (szARPInfo != 0);
#endif
	
	return 0;
}


static void IP_DestroyInterface(TInterfaceInfo *Interface)
{
	if(Interface)
	{
		if(Interface->Name)
			free(Interface->Name);
		if(Interface->Description)
			free(Interface->Description);
		free(Interface);
	}
}


static TInterfaceInfo *IP_GetDefaultInterface(void)
{
	TInterfaceInfo *IPInfo, *pIPInfo;
	int i, lIPInfo;
	uint32 dwGatewayIP = 0;
	TInterfaceInfo *DefaultInterface = NULL;
	
	
	if(!IP_GetInterfaceTable(&IPInfo, &lIPInfo))
	{
		LOG_DEBUG("error: IP_GetInterfaceTable()");
		return NULL;
	}
	
	dwGatewayIP = IP_GetDefaultGateway();
	if(!dwGatewayIP)
	{
		LOG_DEBUG("error: IP_GetDefaultGateway()");
		IP_FreeInterfaceTable(IPInfo, lIPInfo);
		return NULL;
	}
	
	for(i=0;i<lIPInfo;i++)
	{
		pIPInfo = &IPInfo[i];
		
		if(pIPInfo->dwIP && pIPInfo->dwNetmask && (pIPInfo->dwIP & pIPInfo->dwNetmask) == (dwGatewayIP & pIPInfo->dwNetmask))
		{
			//dwIP = pIPInfo->dwIP;
			DefaultInterface = malloc(sizeof(TInterfaceInfo));
			memcpy(DefaultInterface, pIPInfo, sizeof(TInterfaceInfo));
			DefaultInterface->Name = strdup(pIPInfo->Name);
			if(pIPInfo->Description)
				DefaultInterface->Description = strdup(pIPInfo->Description);
			
			//LOG_DEBUG("default interface: %s (desc:%s), ip: %s", pIPInfo->Name, (pIPInfo->Description)?pIPInfo->Description:"", Net_Uint32ToIP(dwIP, TmpBuf));
			break;
		}
	}
	
	IP_FreeInterfaceTable(IPInfo, lIPInfo);
	
	return DefaultInterface;
}


uint32 IP_GetDefaultIPAddress(void)
{
	TInterfaceInfo *Interface;
	uint32 dwIP = 0;
	
	Interface = IP_GetDefaultInterface();
	if(Interface)
	{
		dwIP = Interface->dwIP;
		IP_DestroyInterface(Interface);
	}
	
	return dwIP;
}


int IP_GetDefaultInterfaceParams(char *Name, int szName, uint32 *pdwIP, uint32 *pdwNetmask, uint8 bMAC[MAC_ADDR_RAW_SIZE])
{
	TInterfaceInfo *Interface;
	int Result = 0;
	
	Interface = IP_GetDefaultInterface();
	if(Interface)
	{
		if(Name)
			snprintf(Name, szName - 1, "%s", Interface->Name);
		if(pdwIP)
			*pdwIP = Interface->dwIP;
		if(pdwNetmask)
			*pdwNetmask = Interface->dwNetmask;
		if(bMAC)
			memcpy(bMAC, Interface->bMAC, MAC_ADDR_RAW_SIZE);
		
		IP_DestroyInterface(Interface);
		
		Result = 1;
	}
	
	return Result;
}


uint32 IP_GetDefaultGateway(void)
{
	TRouteInfo *RouteInfo, *pRouteInfo;
	int i, lRouteInfo;
	uint32 dwIP = 0;
	
	
	if(!IP_GetRouteTable(&RouteInfo, &lRouteInfo))
	{
		LOG_DEBUG("error: IP_GetRouteTable()");
		return 0;
	}
	
	for(i=0;i<lRouteInfo;i++)
	{
		pRouteInfo = &RouteInfo[i];
		
		if(!pRouteInfo->dwDestination && !pRouteInfo->dwNetmask && pRouteInfo->dwGateway)
		{
			// find corresponding ip address
			dwIP = pRouteInfo->dwGateway;
			break;
		}
	}
	
	free(RouteInfo);
	
	return dwIP;
}


int IP_GetDnsHostname(char *Hostname, int szHostname)
{
	if(!Hostname)
		return 0;
	
	if(szHostname <= 0)
		return 0;
	
	Hostname[0] = 0;
	
#ifdef WIN32
	{
		DWORD Size = szHostname;
		return GetComputerNameExA(ComputerNameDnsHostname, Hostname, &Size);
	}
#endif

#ifdef APPLE
	{
		char **Lines = NULL;
		int lLines = 0;
		
		if(System_ExecuteGrabOutput("hostname 2> /dev/null", &Lines, &lLines))
		{
			if(lLines > 0)
			{
				char *Line = Lines[0];
				int iPos = String_IndexOf(Line, '.');
				if(iPos > 0)
					String_CopyEx(Line, Hostname, szHostname, iPos);
			}
			StringArray_Free(&Lines, &lLines);
		}
		
		return (Hostname[0] != 0);
	}
#else
	return FileSystem_ReadFileFirstLine("/proc/sys/kernel/hostname", Hostname, szHostname);
#endif
}


#ifdef WIN32
static FIXED_INFO *_GetNetworkParams(void)
{
	FIXED_INFO *pInfo;
    ULONG szInfo;
    
    
    pInfo = malloc(sizeof(FIXED_INFO));
    szInfo = sizeof(FIXED_INFO);
    
    if(GetNetworkParams(pInfo, &szInfo) == ERROR_BUFFER_OVERFLOW)
    {
    	free(pInfo);
    	
        pInfo = malloc(szInfo);
		
        // try again
        if(GetNetworkParams(pInfo, &szInfo) != ERROR_SUCCESS)
        {
            free(pInfo);
            return NULL;   // error
        }
    }
	
    return pInfo;
}
#endif


int IP_GetDnsDomain(char *Domain, int szDomain)
{
	Domain[0] = 0;
	
#ifdef WIN32
	{
		FIXED_INFO *pInfo = _GetNetworkParams();
		if(pInfo)
		{
			snprintf(Domain, szDomain - 1, "%s", pInfo->DomainName);
			free(pInfo);
			return 1;
		}
	}
#else
	// this is somewhat wrong, as it is the NIS domain, not the DNS domain
	// should look into /etc/resolv.conf
	if(FileSystem_ReadFileFirstLine("/proc/sys/kernel/domainname", Domain, szDomain))
	{
		if(strcmp(Domain, "(none)") == 0)
			Domain[0] = 0;
		return 1;
	}
#endif
	return 0;
}


int IP_GetDnsServers(uint32 **pdwServers, int *pszServers)
{
	uint32 *dwServers = NULL;
	int lServers = 0;
	
	*pdwServers = NULL;
	*pszServers = 0;
	
#ifdef WIN32
	{
		FIXED_INFO *pInfo = _GetNetworkParams();
		if(pInfo)
		{
			IP_ADDR_STRING *pIPAddr;
			
			
			pIPAddr = pInfo->DnsServerList.Next;
			while(pIPAddr)
			{
				uint32 dwIP;
				
				dwIP = IP_IPToUint32(pIPAddr->IpAddress.String);
				if(dwIP)
				{
					dwServers = realloc(dwServers, (lServers + 1) * sizeof(uint32));
					dwServers[lServers++] = dwIP;
				}
				pIPAddr = pIPAddr->Next;
			}
			free(pInfo);
		}
	}
#else
	{
		char **Lines;
		int i, lLines;
		
		if(FileSystem_ReadTextFile("/etc/resolv.conf", &Lines, &lLines))
		{
			
			for(i=0;i<lLines;i++)
			{
				char *Line = Lines[i];
				char *Trail;
				
				String_Trim(Line);
				
				if(String_StartsWithEx(Line, "nameserver", &Trail))
				{
					uint32 dwIP;
					
					String_Trim(Trail);
					dwIP = IP_IPToUint32(Trail);
					if(dwIP)
					{
						dwServers = realloc(dwServers, (lServers + 1) * sizeof(uint32));
						dwServers[lServers++] = dwIP;
					}
				}
			}
			
			if(Lines)
			{
				for(i=0;i<lLines;i++)
				{
					if(Lines[i])
						free(Lines[i]);
				}
				free(Lines);
			}
		}
	}
#endif
	
	if(lServers <= 0)
		return 0;
	
	*pdwServers = dwServers;
	*pszServers = lServers;
	
	return 1;
}


static int _IP_GetHostIPEx(char *Hostname, uint32 **pdwIPList, int *plIPList)
{
	uint32 dwIP;
	int i;
	uint32 *dwIPList = NULL;
	
	
	// try parse IP address first
	dwIP = inet_addr(Hostname);
	if(dwIP != 0 && dwIP != INADDR_NONE)
	{
		if(plIPList)
		{
			dwIPList = malloc(sizeof(uint32));
			*plIPList = 1;
		}
		else
		{
			dwIPList = malloc(2 * sizeof(uint32));
			dwIPList[1] = 0;
		}
		dwIPList[0] = Net_ToHost32(dwIP);
		*pdwIPList = dwIPList;
		
		return 1;
	}
	
	// then try resolv name
	{
		struct addrinfo *AddrInfoResult = NULL;
		struct addrinfo *pInfo = NULL;
		struct sockaddr_in *pAddrIn = NULL;
		int n;
		//char TmpBuf[20];
		
		
		if(getaddrinfo(Hostname, NULL, NULL, &AddrInfoResult) != 0)
		{
			//LOG_DEBUG("error: getaddrinfo()");
			return 0;
		}
		
		// count results
		n = 0;
		pInfo = AddrInfoResult;
		while(pInfo)
		{
			pAddrIn = (struct sockaddr_in*)pInfo->ai_addr;
			//dwIP = pAddrIn->sin_addr.s_addr;
			//LOG_DEBUG("%s => %s", Hostname, Net_Uint32ToIP(dwIP, TmpBuf));
			pInfo = pInfo->ai_next;
			n++;
		}
		
		if(n == 0)
		{
			if(AddrInfoResult)
				freeaddrinfo(AddrInfoResult);
			return 0;
		}
		
		if(plIPList)
		{
			dwIPList = malloc(n * sizeof(uint32));
			*plIPList = n;
		}
		else
		{
			dwIPList = malloc((n + 1) * sizeof(uint32));
			dwIPList[n] = 0;
		}
		
		pInfo = AddrInfoResult;
		for(i=0;i<n;i++)
		{
			pAddrIn = (struct sockaddr_in*)pInfo->ai_addr;
			dwIPList[i] = Net_ToHost32(pAddrIn->sin_addr.s_addr);
			
			//dwIP = pAddrIn->sin_addr.s_addr;
			//LOG_DEBUG("%s => %s", Hostname, Net_Uint32ToIP(dwIP, TmpBuf));
			
			pInfo = pInfo->ai_next;
		}
		
		if(AddrInfoResult)
			freeaddrinfo(AddrInfoResult);
		
		*pdwIPList = dwIPList;
		
		return 1;
	}
}


int IP_GetHostIPEx(char *Hostname, uint32 **pdwIPList, int *plIPList)
{
	// override previous function, discard duplicate
	uint32 *dwIPList1 = NULL, *dwIPList2 = NULL;
	int i, j, Result, lIPList1 = 0, lIPList2 = 0;
	
	Result = _IP_GetHostIPEx(Hostname, &dwIPList1, &lIPList1);
	if(!Result)
		return 0;
	
	// allocate same space than previous results, this will be enough
	dwIPList2 = malloc(lIPList1 * sizeof(uint32));
	lIPList2 = 0;
	
	for(i=0;i<lIPList1;i++)
	{
		uint32 dwIP1 = dwIPList1[i];
		
		for(j=0;j<lIPList2;j++)
		{
			uint32 dwIP2 = dwIPList2[j];
			
			if(dwIP2 == dwIP1)
				break;
		}
		
		// not duplicate
		if(j >= lIPList2)
			dwIPList2[lIPList2++] = dwIP1;
	}
	
	// resize to fit our needs
	dwIPList2 = realloc(dwIPList2, lIPList2 * sizeof(uint32));
	
	*pdwIPList = dwIPList2;
	*plIPList = lIPList2;
	
	free(dwIPList1);
	
	return 1;
}


int IP_GetHostIP(char *Hostname, uint32 *pdwIP)
{
	struct in_addr Addr;
	uint32 dwIP;
	int i;
	
	
	// try IP first
	dwIP = inet_addr(Hostname);
	
	// then resolv name
	if(dwIP == 0 || dwIP == INADDR_NONE)
	{
		struct hostent *Host;
		
		Host = gethostbyname(Hostname);
		if(Host)
		{
			if(Host->h_addrtype == 2 && Host->h_length == 4)
			{
				for(i=0;Host->h_addr_list[i]!=0;i++)
				{
					memcpy(&Addr, Host->h_addr_list[i], sizeof(struct in_addr));
					dwIP = Addr.s_addr;
					break;
				}
			}
		}
	}
	if(dwIP == 0 || dwIP == INADDR_NONE)
		return 0;
	
	if(pdwIP)
		*pdwIP = Net_ToHost32(dwIP);
	
	return 1;
}
