#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

// getpid()
#include <sys/types.h>
#include <unistd.h>

#ifdef WIN32
#include <process.h>
#define getpid _getpid
#endif

#include <libshred.h>


static TLog DefaultLog = {0, NULL, NULL, 0, 0, 0, 0, 0, 0, 0, 0, NULL};
static int InitDone = 0;
static void *UncleanMutex = NULL;


static void Log_Init(void)
{
	if(!InitDone)
	{
		Log_Reset(&DefaultLog);
		InitDone = 1;
	}
}


static void _Log_Cleanup(void)
{
	if(UncleanMutex)
		CriticalSection_Close(UncleanMutex);
}


void Log_SetDefault(TLog *Log)
{
	if(Log)
		memcpy(&DefaultLog, Log, sizeof(TLog));
	
	InitDone = 1;
}


TLog *Log_GetDefault(void)
{
	return &DefaultLog;
}


void Log_EnableDefault(int Enable)
{
	Log_Init();
	DefaultLog.Disabled = !Enable;
}


void Log_Reset(TLog *Log)
{
	if(Log)
	{
		memset(Log, 0, sizeof(TLog));
		Log->Mode = LOG_AUTO_FLUSH;
		Log->File = stdout;
		Log->Filename = NULL;
	}
}


int Log_SetFile(TLog *Log, FILE *File)
{
	if(!Log || !File)
		return 0;
	
	Log->Filename = NULL;
	Log->File = File;
	
	return 1;
}


void Log_SetCallback(TLog *Log, TLogCallback *Proc, void *Data)
{
	if(!Log)
		return;
	
	Log->ClbkProc = Proc;
	Log->ClbkData = Data;
}


int Log_Print(TLog *Log, int Level, char *Filename, int iLine, char *Function, char *Format, ...)
{
	char Prefix[256], Line[2048];
	va_list arg;
	int Done, lPrefix, lLine;
	
	
	if(!Log)
	{
		if(!InitDone)
			Log_Init();
		Log = &DefaultLog;
		
		if((Log->Mode & LOG_THREAD_SAFE) && !Log->Mutex)
		{
			//Debug_Init();
			Log->Mutex = CriticalSection_Open();
			UncleanMutex = Log->Mutex;
			atexit(_Log_Cleanup);
		}
	}
	
	if(!Log->File || Log->Disabled)
	{
		// log is disabled
		return 0;
	}
	if(Log->Mask && !(Log->Mask & Level))
	{
		// discard unwanted log level, errors should always be matched
		return 0;
	}
	
	if((Log->nLinesMax >= 100 && Log->nLines >= Log->nLinesMax) || (Log->szOutMax >= 1000 && Log->szOut >= Log->szOutMax))
	{
		if(Log->Filename && Log->File)
		{
			int DisabledOld = Log->Disabled;
			
			// do rotation now
			
			Log->Disabled = 1;
			if(Log->Mutex)
				CriticalSection_Enter(Log->Mutex);
			
			// close file
			fclose(Log->File);
			
			// rotate
			if(Log->nRotateFile >= 1)
			{
				int i;
				char FileSrc[256], FileDst[256];
				
				i = 1;
				while(i < Log->nRotateFile)
				{
					snprintf(FileSrc, sizeof(FileSrc) - 1, "%s.%d", Log->Filename, Log->nRotateFile - i - 1);
					snprintf(FileDst, sizeof(FileDst) - 1, "%s.%d", Log->Filename, Log->nRotateFile - i);
					FileSystem_FileMove(FileSrc, FileDst);
					i++;
				}
				snprintf(FileSrc, sizeof(FileSrc) - 1, "%s", Log->Filename);
				snprintf(FileDst, sizeof(FileDst) - 1, "%s.0", Log->Filename);
				FileSystem_FileMove(FileSrc, FileDst);
			}
			
			// reopen file
			Log->File = fopen(Log->Filename, "w");
			
			// reset counters
			Log->nLines = 0;
			Log->szOut = 0;
			
			if(Log->Mutex)
				CriticalSection_Leave(Log->Mutex);
			Log->Disabled = DisabledOld;
		}
	}
	
	// write prefix
	lPrefix = 0;
	Prefix[0] = 0;
	
	if(Log->Mode & LOG_SHOW_DATE)
	{
		char TimeString[24];
		TTime Time;
		
		// format: 2011/10/07 09:58:51.718
		
		Time_GetLocalTime(&Time);
		snprintf(TimeString, sizeof(TimeString), "%4d/%2d/%2d %2d:%2d:%2d.%03d", Time.Year, Time.Month, Time.Day, Time.Hour, Time.Minute, Time.Second, Time.Milliseconds);
		
		if(TimeString[5] == ' ')
			TimeString[5] = '0';
		if(TimeString[8] == ' ')
			TimeString[8] = '0';
		
		if(TimeString[11] == ' ')
			TimeString[11] = '0';
		if(TimeString[14] == ' ')
			TimeString[14] = '0';
		if(TimeString[17] == ' ')
			TimeString[17] = '0';
		
		lPrefix += String_CopyEx(TimeString, Prefix + lPrefix, sizeof(Prefix) - lPrefix, 23);
		lPrefix += String_CopyEx(":", Prefix + lPrefix, sizeof(Prefix) - lPrefix, 1);
	}
	
	if(Log->Mode & LOG_SHOW_PID)
		lPrefix += snprintf(Prefix + lPrefix, sizeof(Prefix) - lPrefix - 1, "pid#%08x:", getpid());
	
	if(Log->Mode & LOG_SHOW_THREADID)
#ifdef APPLE
		lPrefix += snprintf(Prefix + lPrefix, sizeof(Prefix) - lPrefix - 1, "thread#%p:", Thread_GetCurrentThreadId());
#else
		lPrefix += snprintf(Prefix + lPrefix, sizeof(Prefix) - lPrefix - 1, "thread#%08x:", Thread_GetCurrentThreadId());
#endif
	
	if(Log->Mode & LOG_SHOW_FILE)
	{
		if(Filename)
		{
			lPrefix += String_Copy(Filename, Prefix + lPrefix, sizeof(Prefix) - lPrefix);
			lPrefix += String_CopyEx(":", Prefix + lPrefix, sizeof(Prefix) - lPrefix, 1);
		}
		else
			lPrefix += String_CopyEx("-:", Prefix + lPrefix, sizeof(Prefix) - lPrefix, 2);
	}
	
	if(Log->Mode & LOG_SHOW_LINE)
	{
		lPrefix += snprintf(Prefix + lPrefix, sizeof(Prefix) - lPrefix - 1, "%d:", iLine);
	}
	
	if(Log->Mode & LOG_SHOW_FUNCTION)
	{
		if(Function)
		{
			lPrefix += String_Copy(Function, Prefix + lPrefix, sizeof(Prefix) - lPrefix);
			lPrefix += String_CopyEx("():", Prefix + lPrefix, sizeof(Prefix) - lPrefix, 3);
		}
		else
			lPrefix += String_CopyEx("-:", Prefix + lPrefix, sizeof(Prefix) - lPrefix, 2);
		
	}
	
	// add padding
	if(Log->Padding > lPrefix)
	{
		int lPadding = Log->Padding - lPrefix;
		
		if(lPadding >= sizeof(Prefix) - lPrefix)
			lPadding = sizeof(Prefix) - lPrefix - 1;
		
		memset(Prefix + lPrefix, ' ', lPadding);
		lPrefix += lPadding;
		Prefix[lPrefix] = 0;
	}
	
	if(Log->Mutex)
		CriticalSection_Enter(Log->Mutex);
	
	// write line
	Line[0] = 0;
	lLine = 0;
	
	if(lPrefix)
	{
		Done = String_CopyEx(Prefix, Line + lLine, sizeof(Line) - lLine, lPrefix);
		lLine += (Done > 0) ? Done : 0;
		Done = String_CopyEx("  ", Line + lLine, sizeof(Line) - lLine, 2);
		lLine += (Done > 0) ? Done : 0;
	}
	
#ifndef WIN32
	if(!(Log->Mode & LOG_NO_COLOR) && Level == LOG_LEVEL_ERROR)
	{
		Done = String_CopyEx("\e[41m", Line + lLine, sizeof(Line) - lLine, 5);
		lLine += (Done > 0) ? Done : 0;
	}
#endif
	
	// vsnprintf behaviour is quite fuzzy on linux, we must take care of the input space and the output length ourself
	if(lLine + 1 < sizeof(Line))
	{
		va_start(arg, Format);
		Done = vsnprintf(Line + lLine, sizeof(Line) - lLine - 1, Format, arg);
		lLine += (Done > 0) ? Done : 0;
		va_end(arg);
		
		if(lLine > sizeof(Line) - 1)
			lLine = sizeof(Line) - 1;
	}
	
#ifndef WIN32
	if(!(Log->Mode & LOG_NO_COLOR) && Level == LOG_LEVEL_ERROR)
	{
		Done = String_CopyEx("\e[m", Line + lLine, sizeof(Line) - lLine, 3);
		lLine += (Done > 0) ? Done : 0;
	}
#endif
	
	// in case we overflow, force new line
	if(lLine >= sizeof(Line) - 1)
		lLine = sizeof(Line) - 2;
	Line[lLine++] = '\n';
	Line[lLine] = 0;
	
	if(Log->ClbkProc)
	{
		Log->ClbkProc(Line, Log->ClbkData);
	}
	else
	{
		fwrite(Line, 1, lLine, Log->File);
		if(Log->Mode & LOG_AUTO_FLUSH)
			fflush(Log->File);
	}
	
	Log->szOut += lLine;
	Log->nLines++;
	
	if(Log->Mutex)
		CriticalSection_Leave(Log->Mutex);
	
	return lLine;
}
