#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#if defined(LINUX) || defined(APPLE) || defined(FREEBSD)
#include <errno.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <net/if.h>
#include <netinet/tcp.h>
#include <sys/stat.h>
#endif

#ifdef WIN32
#include <winsock2.h>
#include <windows.h>
#endif

#include <libshred.h>

#define LISTEN_BACKLOG	8	/* default on BSD-derived systems: 5 */

#ifdef WIN32
#define BUFFER_CAST		char *
#else
#define BUFFER_CAST		void *
#endif


void Net_Init(void)
{
#ifdef WIN32
	WSADATA Wsadata;
	WSAStartup(MAKEWORD(2,2), &Wsadata);
#endif
}


void Net_Cleanup(void)
{
#ifdef WIN32
	WSACleanup();
#endif
}


char *Net_Uint32ToIP(uint32 dwIP, char *TmpBuf, int szTmpBuf)
{
	struct in_addr Addr;
	
	Addr.s_addr = dwIP;
	TmpBuf[0] = 0;
	snprintf(TmpBuf, szTmpBuf - 1, "%s", inet_ntoa(Addr));
	
	return TmpBuf;
}


int Net_IPToUint32(char *IP, uint32 *pdwIP)
{
	if(!IP || !pdwIP)
		return 0;
	
#ifdef WIN32
	{
		uint32 dwIP = 0;
		
		if(strcmp(IP, "255.255.255.255") == 0
		|| strcmp(IP, "0377.0377.0377.0377") == 0
		|| strcasecmp(IP, "0xff.0xff.0xff.0xff") == 0)
		{
			*pdwIP = (uint32)-1;
			return 1;
		}
		
		dwIP = inet_addr(IP);
		if(dwIP == ((uint32)-1))
			return 0;
		
		*pdwIP = dwIP;
		return 1;
	}
#else
	{
		struct in_addr Addr;
		if(!inet_aton(IP, &Addr))
			return 0;
		
		*pdwIP = Addr.s_addr;
		return 1;
	}
#endif
	
	return 0;
}


char *Net_AddrToString(uint32 dwIP, uint16 wPort, char *TmpBuf, int szTmpBuf)
{
	struct in_addr Addr;
	
	Addr.s_addr = dwIP;
	TmpBuf[0] = 0;
	snprintf(TmpBuf, szTmpBuf - 1, "%s:%d", inet_ntoa(Addr), Net_ToHost16(wPort));
	
	return TmpBuf;
}


void Net_StringToAddr(char *Host, uint32 *pdwIP, uint16 *pwPort)
{
	char Hostname[256];
	int iPos, Port;
	uint32 dwIP = 0;
	
	
	iPos = String_IndexOf(Host, ':');
	if(iPos > 0)
	{
		snprintf(Hostname, sizeof(Hostname), "%.*s", iPos, Host);
		Port = atoi(&Host[iPos + 1]);
	}
	else
	{
		snprintf(Hostname, sizeof(Hostname), "%s", Host);
		Port = 0;
	}
	IP_GetHostIP(Hostname, &dwIP);
	
	if(pdwIP)
		*pdwIP = Net_ToNet32(dwIP);
	if(pwPort)
		*pwPort = Net_ToNet16(Port);
}


int Net_Option_SetBlock(int Socket, int Enable)
{
#ifdef WIN32
	u_long ioctl_arg = (Enable != 0);
	ioctlsocket(Socket, FIONBIO, (u_long*)&ioctl_arg);
#endif
	
#if defined(LINUX) || defined(APPLE) || defined(FREEBSD)
	int Flags = fcntl(Socket, F_GETFL);
	if(Flags < 0)
	{
		LOG_DEBUG("error: fcntl(F_GETFL): %d", Net_GetLastError());
		return 0;
	}
	
	if(Enable)
		Flags &= ~O_NONBLOCK;
	else
		Flags |= O_NONBLOCK;
	
	if(fcntl(Socket, F_SETFL, Flags) < 0)
	{
		LOG_DEBUG("error: fcntl(F_SETFL): %d", Net_GetLastError());
		return 0;
	}
#endif
	return 1;
}


int Net_Option_SetTimeout(int Socket, int RecvTimeout, int SendTimeout)
{
	if(RecvTimeout >= 0)
	{
#if defined(LINUX) || defined(APPLE) || defined(FREEBSD)
		struct timeval TV;
		
		TV.tv_sec = RecvTimeout / 1000;
		TV.tv_usec = (RecvTimeout % 1000) * 1000;
		
		if(setsockopt(Socket, SOL_SOCKET, SO_RCVTIMEO, (void*)&TV, sizeof(TV)) < 0)
		{
			LOG_DEBUG("error: setsockopt(SO_RCVTIMEO): %d", Net_GetLastError());
			return 0;
		}
#else
		if(setsockopt(Socket, SOL_SOCKET, SO_RCVTIMEO, (void*)&RecvTimeout, sizeof(RecvTimeout)) < 0)
		{
			LOG_DEBUG("error: setsockopt(SO_RCVTIMEO): %d", Net_GetLastError());
			return 0;
		}
#endif
	}
	
	if(SendTimeout >= 0)
	{
#if defined(LINUX) || defined(APPLE) || defined(FREEBSD)
		struct timeval TV;
		
		TV.tv_sec = SendTimeout / 1000;
		TV.tv_usec = (SendTimeout % 1000) * 1000;
		
		if(setsockopt(Socket, SOL_SOCKET, SO_SNDTIMEO, (void*)&TV, sizeof(TV)) < 0)
		{
			LOG_DEBUG("error: setsockopt(SO_SNDTIMEO): %d", Net_GetLastError());
			return 0;
		}
#else
		if(setsockopt(Socket, SOL_SOCKET, SO_RCVTIMEO, (void*)&SendTimeout, sizeof(SendTimeout)) < 0)
		{
			LOG_DEBUG("error: setsockopt(SO_SNDTIMEO): %d", Net_GetLastError());
			return 0;
		}
#endif
	}
	
	return 1;
}


int Net_Option_BindToDevice(int Socket, char *Device)
{
#ifndef LINUX
	return 0;
#else
	struct ifreq Interface;
	
	memset(&Interface, 0, sizeof(struct ifreq));
	Interface.ifr_ifrn.ifrn_name[0] = 0;
	if(Device)
		snprintf(Interface.ifr_ifrn.ifrn_name, IFNAMSIZ - 1, "%s", Device);
	if(setsockopt(Socket, SOL_SOCKET, SO_BINDTODEVICE,(void*)&Interface, sizeof(Interface)) < 0)
	{
		LOG_DEBUG("error: setsockopt(SO_BINDTODEVICE): %d", Net_GetLastError());
		return 0;
	}
	return 1;
#endif
}


int Net_Option_SetDoNotRoute(int Socket, int Enable)
{
	if(setsockopt(Socket, SOL_SOCKET, SO_DONTROUTE, (void*)&Enable, sizeof(Enable)) < 0)
	{
		LOG_DEBUG("error: setsockopt(SO_DONTROUTE): %d", Net_GetLastError());
		return 0;
	}
	return 1;
}


int Net_Option_SetBroadcast(int Socket, int Enable)
{
	if(setsockopt(Socket, SOL_SOCKET, SO_BROADCAST, (void*)&Enable, sizeof(Enable)) < 0)
	{
		LOG_DEBUG("error: setsockopt(SO_BROADCAST): %d", Net_GetLastError());
		return 0;
	}
	return 1;
}


int Net_Option_SetKeepAlive(int Socket, int Enable)
{
	if(setsockopt(Socket, SOL_SOCKET, SO_KEEPALIVE, (void*)&Enable, sizeof(Enable)) < 0)
	{
		LOG_DEBUG("error: setsockopt(SO_KEEPALIVE): %d", Net_GetLastError());
		return 0;
	}
	return 1;
}


int Net_Option_SetTcpNoDelay(int Socket, int Enable)
{
	if(setsockopt(Socket, IPPROTO_TCP, TCP_NODELAY, (void*)&Enable, sizeof(Enable)) < 0)
	{
		LOG_DEBUG("error: setsockopt(TCP_NODELAY): %d", Net_GetLastError());
		return 0;
	}
	return 1;
}


int Net_Option_SetMulticastTTL(int Socket, int TTL)
{
	if(setsockopt(Socket, IPPROTO_IP, IP_MULTICAST_TTL, (void*)&TTL, sizeof(TTL)) < 0)
	{
		LOG_DEBUG("error: setsockopt(IP_MULTICAST_TTL): %d", Net_GetLastError());
		return 0;
	}
	return 1;
}


int Net_Option_SetMulticastIF(int Socket, uint32 dwIP)
{
	if(setsockopt(Socket, IPPROTO_IP, IP_MULTICAST_IF, (void*)&dwIP, sizeof(dwIP)) < 0)
	{
		LOG_DEBUG("error: setsockopt(IP_MULTICAST_IF): %d", Net_GetLastError());
		return 0;
	}
	return 1;
}


int Net_Option_AddMulticastMembership(int Socket, uint32 dwIFaceIP, uint32 dwMulticastIP)
{
	struct ip_mreq McastAddr;
	
	memset(&McastAddr, 0, sizeof(struct ip_mreq));
	McastAddr.imr_interface.s_addr = dwIFaceIP;
	McastAddr.imr_multiaddr.s_addr = dwMulticastIP;
	
	if(setsockopt(Socket, IPPROTO_IP, IP_ADD_MEMBERSHIP, (void*)&McastAddr, sizeof(struct ip_mreq)) < 0)
	{
		LOG_DEBUG("error: setsockopt(IP_ADD_MEMBERSHIP): %d", Net_GetLastError());
		return 0;
	}
	return 1;
}


int Net_Option_DropMulticastMembership(int Socket, uint32 dwIFaceIP, uint32 dwMulticastIP)
{
	struct ip_mreq McastAddr;
	
	memset(&McastAddr, 0, sizeof(struct ip_mreq));
	McastAddr.imr_interface.s_addr = dwIFaceIP;
	McastAddr.imr_multiaddr.s_addr = dwMulticastIP;
	
	if(setsockopt(Socket, IPPROTO_IP, IP_DROP_MEMBERSHIP, (void*)&McastAddr, sizeof(struct ip_mreq)) < 0)
	{
		LOG_DEBUG("error: setsockopt(IP_DROP_MEMBERSHIP): %d", Net_GetLastError());
		return 0;
	}
	return 1;
}

/*
fixme: left to implement:
	IP_MULTICAST_LOOP
	IP_BLOCK_SOURCE
	IP_UNBLOCK_SOURCE
	IP_ADD_SOURCE_MEMBERSHIP
	IP_DROP_SOURCE_MEMBERSHIP
*/

int Net_Option_SetRecvBufferSize(int Socket, int Size)
{
	if(setsockopt(Socket, SOL_SOCKET, SO_RCVBUF, (void*)&Size, sizeof(Size)) < 0)
	{
		LOG_DEBUG("error: setsockopt(SO_RCVBUF): %d", Net_GetLastError());
		return 0;
	}
	return 1;
}


int Net_Option_SetSendBufferSize(int Socket, int Size)
{
	if(setsockopt(Socket, SOL_SOCKET, SO_SNDBUF, (void*)&Size, sizeof(Size)) < 0)
	{
		LOG_DEBUG("error: setsockopt(SO_SNDBUF): %d", Net_GetLastError());
		return 0;
	}
	return 1;
}


#ifdef WIN32

#undef FD_ZERO2
#define FD_ZERO2(set)\
do\
{\
	TFDSet *Set = (TFDSet*)set;\
	Set->Count = 0;\
}while(0)

#undef FD_ISSET2
#define FD_ISSET2(fd, set) __WSAFDIsSet((SOCKET)(fd), (fd_set*)(set))

#undef FD_SET2
#define FD_SET2(fd, set)\
do\
{\
	uint32 i;\
	TFDSet *Set = (TFDSet*)set;\
	for(i=0;i<Set->Count;i++)\
	{\
		if(Set->Socket[i] == (fd))\
		{\
			break;\
		}\
	}\
	if(i >= Set->Count)\
	{\
		if(Set->Count < FD_SETSIZE)\
		{\
			Set->Socket[i] = (fd);\
			Set->Count++;\
		}\
	}\
}while(0)

#else

#undef FD_ZERO2
#define FD_ZERO2(set) FD_ZERO(set)

#undef FD_ISSET2
#define FD_ISSET2(fd, set) FD_ISSET(fd, set)

#undef FD_SET2
#define FD_SET2(fd, set) FD_SET(fd, set)

#endif


void Net_Select_Reset(TNetSelect *NetSelect)
{
	if(!NetSelect)
		return;
	
	memset(NetSelect, 0, sizeof(TNetSelect));
}


void Net_Select_Add(TNetSelect *NetSelect, int FD, uint8 OpFlag)
{
	TNetSelectFD *pFD;
	
	
	if(!NetSelect || FD < 0 || OpFlag == NET_OP_NONE)
		return;
	
	if(NetSelect->lFD >= FD_SETSIZE)
	{
		LOG_DEBUG("error: fd set is full");
		return;
	}
	
	pFD = &NetSelect->FD[NetSelect->lFD];
	NetSelect->lFD++;
	
	pFD->FD = FD;
	pFD->OpFlag = OpFlag;
}


int Net_Select(TNetSelect *NetSelect, int Timeout)
{
	TNetSelectFD *pFD;
	int i, MaxFD, Result;
	struct timeval TV, *pTV = NULL;
	
	
	if(!NetSelect)
		return 0;
	
	MaxFD = 0;
	
	// wait for incoming packet
	FD_ZERO2(&NetSelect->ReadFD);
	FD_ZERO2(&NetSelect->WriteFD);
	FD_ZERO2(&NetSelect->ExceptFD);
	
	if(NetSelect->lFD <= 0)
		return 0;
	
	for(i=0;i<NetSelect->lFD;i++)
	{
		pFD = &NetSelect->FD[i];
		
		switch(pFD->OpFlag)
		{
			case NET_OP_ACCEPT:
				FD_SET2(pFD->FD, &NetSelect->ReadFD);
				MaxFD = MAX(pFD->FD, MaxFD);
				break;
			
			case NET_OP_CONNECT:
				FD_SET2(pFD->FD, &NetSelect->WriteFD);
#ifdef WIN32
				FD_SET2(pFD->FD, &NetSelect->ExceptFD);
#endif
				MaxFD = MAX(pFD->FD, MaxFD);
				break;
			
			case NET_OP_READ:
				FD_SET2(pFD->FD, &NetSelect->ReadFD);
				MaxFD = MAX(pFD->FD, MaxFD);
				break;
			
			case NET_OP_WRITE:
				FD_SET2(pFD->FD, &NetSelect->WriteFD);
				MaxFD = MAX(pFD->FD, MaxFD);
				break;
		}
	}
	
	if(Timeout >= 0)  // a negative value for Timeout makes select blocking
	{
		TV.tv_sec = Timeout / 1000;
		TV.tv_usec = (Timeout % 1000) * 1000;
		pTV = &TV;
	}
	
	Result = select(MaxFD + 1, (fd_set*)&NetSelect->ReadFD, (fd_set*)&NetSelect->WriteFD, (fd_set*)&NetSelect->ExceptFD, pTV);
	//LOG_DEBUG("select result: %d", Result);
	if(Result < 0)
	{
		int ErrorID = Net_GetLastError();
#if defined(LINUX) || defined(APPLE) || defined(FREEBSD)
		if(ErrorID == EINTR || ErrorID == EAGAIN)
#else
		if(ErrorID == WSAEINTR)
#endif
		{
			// nothing to do, but reset FDSETs, in order to avoid strange behaviour on LINUX
			FD_ZERO2(&NetSelect->ReadFD);
			FD_ZERO2(&NetSelect->WriteFD);
			FD_ZERO2(&NetSelect->ExceptFD);
			return 0;
		}
		//LOG_DEBUG("error: select(): %d", ErrorID);
	}
	
	return Result;
}


int Net_Select_Check(TNetSelect *NetSelect, int FD)
{
	TNetSelectFD *pFD;
	int i;
	
	
	if(!NetSelect)
		return 0;
	
	if(FD < 0)
		return 0;
	
	for(i=0;i<NetSelect->lFD;i++)
	{
		pFD = &NetSelect->FD[i];
		
		if(pFD->FD < 0)
			continue;
		
		if(pFD->FD != FD)
			continue;
		
		switch(pFD->OpFlag)
		{
			case NET_OP_ACCEPT:
			case NET_OP_READ:
			{
				if(FD_ISSET(pFD->FD, &NetSelect->ReadFD))
					return 1;
				break;
			}
			
			case NET_OP_CONNECT:
			{
				if(FD_ISSET(pFD->FD, &NetSelect->WriteFD))
				{
#if defined(LINUX) || defined(APPLE) || defined(FREEBSD)
					int Error, k;
					k = sizeof(Error);
					
					if(getsockopt(pFD->FD, SOL_SOCKET, SO_ERROR, (char*)&Error, (socklen_t*)&k) < 0)
						return -1;
					
					// could not establish remote connection
					if(Error)
						return -1;
#endif
					return 1;
				}
				
#ifdef WIN32
				if(FD_ISSET(pFD->FD, &NetSelect->ExceptFD))
				{
					int Error, k;
					k = sizeof(Error);
					
					if(getsockopt(pFD->FD, SOL_SOCKET, SO_ERROR, (char*)&Error, (socklen_t*)&k) < 0)
						return -1;
					
					// could not establish remote connection
					if(Error)
						return -1;
				}
#endif
				break;
			}
			
			case NET_OP_WRITE:
			{
				if(FD_ISSET(pFD->FD, &NetSelect->WriteFD))
					return 1;
				break;
			}
		}
		break;
	}
	return 0;
}


int Net_OpenUDPSocket(void)
{
	int Socket;
	
	Socket = socket(PF_INET, SOCK_DGRAM, IPPROTO_UDP);
	if(Socket < 0)
		LOG_DEBUG("error: socket(): %d", Net_GetLastError());
	
	return Socket;
}


int Net_OpenTCPSocket(void)
{
	int Socket;
	
	Socket = socket(AF_INET, SOCK_STREAM, 0);
	if(Socket < 0)
		LOG_DEBUG("error: socket(): %d", Net_GetLastError());
	
	return Socket;
}


int Net_Connect(int Socket, uint32 dwIP, uint16 wPort)
{
	struct sockaddr_in Addr;
	
	
	memset(&Addr, 0, sizeof(struct sockaddr_in));
	Addr.sin_family = AF_INET;
	Addr.sin_port = wPort;
	Addr.sin_addr.s_addr = dwIP;
	
	if(connect(Socket, (struct sockaddr*)&Addr, sizeof(struct sockaddr_in)) < 0)
	{
		int Result = Net_GetLastError();
		switch(Result)
		{
#if defined(LINUX) || defined(APPLE) || defined(FREEBSD)
 			case EINPROGRESS:
 			case EALREADY:
#else
 			case WSAEINPROGRESS:
#endif
				break;
			
			default:
				LOG_DEBUG("error: connect(): %d", Net_GetLastError());
				return 0;
		}
	}
	
	return 1;
}


int Net_ConnectNonBlock(int Socket, uint32 dwIP, uint16 wPort)
{
	int Result;
	struct sockaddr_in Addr;
	
	
	Net_Option_SetBlock(Socket, 0);
	
	memset(&Addr, 0, sizeof(struct sockaddr_in));
	Addr.sin_family = AF_INET;
	Addr.sin_port = wPort;
	Addr.sin_addr.s_addr = dwIP;
	
	if(connect(Socket, (struct sockaddr*)&Addr, sizeof(struct sockaddr_in)) < 0)
	{
		Result = Net_GetLastError();
		switch(Result)
		{
#if defined(LINUX) || defined(APPLE) || defined(FREEBSD)
 			case EINPROGRESS:
 			case EALREADY:
#else
 			case WSAEWOULDBLOCK:
#endif
				break;
			
			default:
				LOG_DEBUG("error: connect(): %d", Net_GetLastError());
				return 0;
		}
	}
	
	return 1;
}


int Net_CheckNonBlockConnect(int Socket)
{
	int MaxFD, Result;
	struct timeval TV;
	fd_set WriteFD;
#ifdef WIN32
	fd_set ExceptFD;
#endif
	
	MaxFD = 0;
	
	FD_ZERO(&WriteFD);
	
	FD_SET(Socket, &WriteFD);
	MaxFD = MAX(Socket, MaxFD);
	
#ifdef WIN32
	memcpy(&ExceptFD, &WriteFD, sizeof(fd_set));
#endif
	
	TV.tv_sec = 0;
	TV.tv_usec = 0;
	
#ifdef WIN32
	Result = select(MaxFD + 1, NULL, &WriteFD, &ExceptFD, &TV);
#else
	Result = select(MaxFD + 1, NULL, &WriteFD, NULL, &TV);
#endif
	
	if(Result == 0)
		return 0;
	
	if(Result < 0)
	{
		Result = Net_GetLastError();
		return -1;
	}
	
	if(FD_ISSET(Socket, &WriteFD))
	{
#if defined(LINUX) || defined(APPLE) || defined(FREEBSD)
		int Error, k;
		k = sizeof(Error);
		
		if(getsockopt(Socket, SOL_SOCKET, SO_ERROR, (char*)&Error, (socklen_t*)&k) < 0)
			return -1;
		
		if(Error)
			return -1;
#endif
		
		// reset block state
		Net_Option_SetBlock(Socket, 1);
		
		return 1;
	}
	
#ifdef WIN32
	if(FD_ISSET(Socket, &ExceptFD))
	{
		int Error, k;
		k = sizeof(Error);
		
		if(getsockopt(Socket, SOL_SOCKET, SO_ERROR, (char*)&Error, (socklen_t*)&k) < 0)
			return -1;
		
		if(Error)
			return -1;
	}
#endif
	
	return 0;
}


// like connect(), but with an explicit timeout
int Net_ConnectWithTimeout(int Socket, uint32 dwIP, uint16 wPort, int Timeout)
{
	struct sockaddr_in Addr;
	
	
	memset(&Addr, 0, sizeof(struct sockaddr_in));
	Addr.sin_family = AF_INET;
	Addr.sin_port = wPort;
	Addr.sin_addr.s_addr = dwIP;
	
	if(!Timeout)
	{
		return (connect(Socket, (struct sockaddr*)&Addr, sizeof(struct sockaddr_in)) == 0);
	}
	else
	{
		int Result;
		
		if(!Net_Option_SetTimeout(Socket, Timeout, Timeout))
			return 0;
		
		if(connect(Socket, (struct sockaddr *)&Addr, sizeof(struct sockaddr_in)) < 0)
		{
			Result = Net_GetLastError();
			switch(Result)
			{
#if defined(LINUX) || defined(APPLE) || defined(FREEBSD)
 				case EINPROGRESS:
 				case EALREADY:
#else
 				case WSAEINPROGRESS:
#endif
					break;
				
				default:
					LOG_DEBUG("error: connect(): %d", Net_GetLastError());
					return 0;
			}
			// timeout
			return 0;
		}
		return 1;
	}
}


int Net_BindSocketEx(int Socket, uint32 dwIP, uint16 wPort, int ReuseAddr)
{
	struct sockaddr_in Addr;
	
	
#if defined(LINUX) || defined(APPLE) || defined(FREEBSD)
	// set the close-on-exec flag to 1
	fcntl(Socket, F_SETFD, 1);
#endif
	
	if(ReuseAddr)
	{
		int k;
		
		k = 1;
		if(setsockopt(Socket, SOL_SOCKET, SO_REUSEADDR, (char*)&k, sizeof(k)) == -1)
		{
			LOG_DEBUG("error: setsockopt(SO_REUSEADDR): %d", Net_GetLastError());
			return 0;
		}
		
#if defined(APPLE) || defined(FREEBSD)
		k = 1;
		if(setsockopt(Socket, SOL_SOCKET, SO_REUSEPORT, (char*)&k, sizeof(k)) == -1)
		{
			LOG_DEBUG("error: setsockopt(SO_REUSEPORT): %d", Net_GetLastError());
			return 0;
		}
#endif
	}
	
	memset(&Addr, 0, sizeof(struct sockaddr_in));
	Addr.sin_family = AF_INET;
	Addr.sin_port = wPort;
	Addr.sin_addr.s_addr = dwIP;
	
	if(bind(Socket, (struct sockaddr*)&Addr, sizeof(struct sockaddr_in)) < 0)
	{
		LOG_DEBUG("error: bind(): %d", Net_GetLastError());
		return 0;
	}
	
	return 1;
}


int Net_Listen(int Socket, int Backlog)
{
	if(Backlog <= 0)
		Backlog = LISTEN_BACKLOG;
	
	if(listen(Socket, Backlog) < 0)
	{
		LOG_DEBUG("error: listen(): %d", Net_GetLastError());
		return 0;
	}
	
	return 1;
}


int Net_Accept(int ListenSocket, int *pConnSocket, uint32 *pdwIP, uint16 *pwPort)
{
	int k, ConnSocket;
	struct sockaddr_in Addr;
	
	
	k = sizeof(struct sockaddr_in);
	ConnSocket = accept(ListenSocket, (struct sockaddr*)&Addr, (socklen_t*)&k);
	if(ConnSocket < 0)
	{
		int ErrorID = Net_GetLastError();
#if defined(LINUX) || defined(APPLE) || defined(FREEBSD)
		if(ErrorID == EINTR || ErrorID == EAGAIN)
#else
		if(ErrorID == WSAEINTR)
#endif
		{
			*pConnSocket = -1;
			return 0;
		}
		else
		{
			LOG_DEBUG("error: accept(): %d", ErrorID);
			return 0;
		}
	}
	
	*pConnSocket = ConnSocket;
	if(pdwIP)
		*pdwIP = Addr.sin_addr.s_addr;
	if(pwPort)
		*pwPort = Addr.sin_port;
	
	return 1;
}


int Net_AcceptWithTimeout(int ListenSocket, int *pConnSocket, int Timeout, uint32 *pdwIP, uint16 *pwPort)
{
	int MaxFD, Result, k;
	fd_set ReadFD;
	struct timeval TV;
	struct sockaddr_in Addr;
	
	
	if(Timeout < 0)
		return Net_Accept(ListenSocket, pConnSocket, pdwIP, pwPort);
	
	while(1)
	{
		MaxFD = 0;
		
		// wait for incoming packet
		FD_ZERO(&ReadFD);
		
		// add udp socket
		FD_SET(ListenSocket, &ReadFD);
		MaxFD = MAX(ListenSocket, MaxFD);
		
		TV.tv_sec = Timeout / 1000;
		TV.tv_usec = (Timeout % 1000) * 1000;
		Result = select(MaxFD + 1, &ReadFD, NULL, NULL, &TV);
		
		if(Result == 0)
		{
			return 0;
		}
		else if(Result < 0)
		{
			int ErrorID = Net_GetLastError();
#if defined(LINUX) || defined(APPLE) || defined(FREEBSD)
			if(ErrorID == EINTR || ErrorID == EAGAIN)
#else
			if(ErrorID == WSAEINTR)
#endif
				continue;
			
			LOG_DEBUG("error: select(): %d", ErrorID);
			break;
		}
		
		// handles tcp accept here
		if(FD_ISSET(ListenSocket, &ReadFD))
		{
			int ConnSocket;
			
			
			k = sizeof(struct sockaddr_in);
_handle_tcp_accept_:
			ConnSocket = accept(ListenSocket, (struct sockaddr*)&Addr, (socklen_t*)&k);
			if(ConnSocket < 0)
			{
				int ErrorID = Net_GetLastError();
#if defined(LINUX) || defined(APPLE) || defined(FREEBSD)
				if(ErrorID == EINTR || ErrorID == EAGAIN)
#else
				if(ErrorID == WSAEINTR)
#endif
					goto _handle_tcp_accept_;
		
				LOG_DEBUG("error: accept(): %d", ErrorID);
				// fixme
				return 0;
			}
			
			*pConnSocket = ConnSocket;
			if(pdwIP)
				*pdwIP = Addr.sin_addr.s_addr;
			if(pwPort)
				*pwPort = Addr.sin_port;
			
			return 1;
		}
	}
	
	return 0;
}


void Net_CloseSocket(int *pSocket)
{
	if(pSocket)
	{
		if(*pSocket >= 0)
		{
#if defined(LINUX) || defined(APPLE) || defined(FREEBSD)
			close(*pSocket);
#else
			closesocket(*pSocket);
#endif
			*pSocket = -1;
		}
	}
}


int Net_GetSockName(int Socket, uint32 *pdwIP, uint16 *pwPort)
{
	struct sockaddr_in Addr;
	int k;
	
	
	k = sizeof(Addr);
	if(getsockname(Socket, (struct sockaddr*)&Addr, (socklen_t*)&k) != -1)
	{
		if(pdwIP)
			*pdwIP = Addr.sin_addr.s_addr;
		if(pwPort)
			*pwPort = Addr.sin_port;
		return 1;
	}
	return 0;
}


int Net_GetPeerName(int Socket, uint32 *pdwIP, uint16 *pwPort)
{
	struct sockaddr_in Addr;
	int k;
	
	k = sizeof(Addr);
	if(getpeername(Socket, (struct sockaddr*)&Addr, (socklen_t*)&k) != -1)
	{
		if(pdwIP)
			*pdwIP = Addr.sin_addr.s_addr;
		if(pwPort)
			*pwPort = Addr.sin_port;
		return 1;
	}
	
	return 0;
}


int Net_Write(int Socket, uint8 *Buffer, int szBuffer)
{
	int Result;
	
_again_:
#if defined(LINUX) || defined(APPLE) || defined(FREEBSD)
	Result = write(Socket, (char*)Buffer, szBuffer);
#else
	Result = send(Socket, (char*)Buffer, szBuffer, 0);
#endif
	if(Result < 0)
	{
		int ErrorID = Net_GetLastError();
#if defined(LINUX) || defined(APPLE) || defined(FREEBSD)
		if(ErrorID == EINTR)
#else
		if(ErrorID == WSAEINTR)
#endif
			goto _again_;
		LOG_DEBUG("error: write(): %d", ErrorID);
		return -1;
	}
	return Result;
}


int Net_WriteBlock(int Socket, uint8 *Buffer, int szBuffer)
{
	int szToWrite, szWrite;
	
	// will never work on datagram, but let user use another type of fd (pipe, unix...)
	if(Net_GetSocketType(Socket) == SOCKET_TYPE_UDP)
		return 0;
	
	szToWrite = szBuffer;
	while(szToWrite > 0)
	{
		szWrite = Net_Write(Socket, Buffer + szBuffer - szToWrite, szToWrite);
		if(szWrite <= 0)
			return 0;
		szToWrite -= szWrite;
	}
	
	return 1;
}


int Net_Read(int Socket, uint8 *Buffer, int szBuffer)
{
	int Result;
	
_again_:
#if defined(LINUX) || defined(APPLE) || defined(FREEBSD)
	Result = read(Socket, (char*)Buffer, szBuffer);
#else
	Result = recv(Socket, (char*)Buffer, szBuffer, 0);
#endif
	if(Result < 0)
	{
		int ErrorID = Net_GetLastError();
#if defined(LINUX) || defined(APPLE) || defined(FREEBSD)
		if(ErrorID == EINTR)
#else
		if(ErrorID == WSAEINTR)
#endif
			goto _again_;
		LOG_DEBUG("error: read(): %d", ErrorID);
		return -1;
	}
	return Result;
}


int Net_ReadBlock(int Socket, uint8 *Buffer, int szBuffer)
{
	int szToRead, szRead;
	
	// will never work on datagram, but let user use another type of fd (pipe, unix...)
	if(Net_GetSocketType(Socket) == SOCKET_TYPE_UDP)
		return 0;
	
	szToRead = szBuffer;
	while(szToRead > 0)
	{
		szRead = Net_Read(Socket, Buffer + szBuffer - szToRead, szToRead);
		if(szRead <= 0)
			return 0;
		
		szToRead -= szRead;
	}
	
	return 1;
}


int Net_SendTo(int Socket, uint8 *Buffer, int szBuffer, uint32 dwIP, uint16 wPort)
{
	struct sockaddr_in Addr;
	int Result;
	
	memset(&Addr, 0, sizeof(Addr));
	Addr.sin_family = AF_INET;
	Addr.sin_addr.s_addr = dwIP;
	Addr.sin_port = wPort;
	
_again_:
	Result = sendto(Socket, (BUFFER_CAST)Buffer, szBuffer, 0, (struct sockaddr *)&Addr, sizeof(Addr));
	if(Result < 0)
	{
		int ErrorID = Net_GetLastError();
#if defined(LINUX) || defined(APPLE) || defined(FREEBSD)
		if(ErrorID == EINTR)
#else
		if(ErrorID == WSAEINTR)
#endif
			goto _again_;
		LOG_DEBUG("error: sendto(): %d", ErrorID);
		return -1;
	}
	return Result;
}


int Net_RecvFrom(int Socket, uint8 *Buffer, int szBuffer, uint32 *pdwIP, uint16 *pwPort)
{
	struct sockaddr_in Addr;
	int Result, k;
	
	
	memset(&Addr, 0, sizeof(Addr));
	k = sizeof(Addr);
	
_again_:
	Result = recvfrom(Socket, (BUFFER_CAST)Buffer, szBuffer, 0, (struct sockaddr *)&Addr, (socklen_t*)&k);
	if(Result < 0)
	{
		int ErrorID = Net_GetLastError();
#if defined(LINUX) || defined(APPLE) || defined(FREEBSD)
		if(ErrorID == EINTR)
#else
		if(ErrorID == WSAEINTR)
#endif
			goto _again_;
		LOG_DEBUG("error: recvfrom(): %d", ErrorID);
		return -1;
	}
	
	if(pdwIP)
		*pdwIP = Addr.sin_addr.s_addr;
	if(pwPort)
		*pwPort = Addr.sin_port;
	
	return Result;
}


int Net_GetSocketType(int Socket)
{
	int Type, k;
	
	k = sizeof(int);
	if(getsockopt(Socket, SOL_SOCKET, SO_TYPE, (void*)&Type, (void*)&k) < 0)
		return SOCKET_TYPE_NONE;
	
	if(Type == SOCK_STREAM)
		return SOCKET_TYPE_TCP;
	
	if(Type == SOCK_DGRAM)
		return SOCKET_TYPE_UDP;
	
	return SOCKET_TYPE_NONE;
}


int Net_GetSocketError(int Socket, int *pError)
{
	int Error = 0, k;
	k = sizeof(Error);
	
	if(!pError)
		return 0;
	*pError = 0;
	
	if(Socket < 0)
		return 0;
	
	if(getsockopt(Socket, SOL_SOCKET, SO_ERROR, (char*)&Error, (socklen_t*)&k) < 0)
		return 0;
	
	*pError = Error;
	
	return 1;
}


#if defined(LINUX) || defined(APPLE) || defined(FREEBSD)

#define MAX_PATH_LENGTH		(108 - 1)

int Net_OpenUnixSocket(void)
{
	int Socket;
	
	Socket = socket(AF_UNIX, SOCK_STREAM, 0);
	if(Socket < 0)
		LOG_DEBUG("error: socket(): %d", Net_GetLastError());
	Process_SetCloseOnExec(Socket);
	return Socket;
}

int Net_BindUnix(int Socket, char *Path)
{
	struct sockaddr_un Addr;
	int lAddr, lPath;
	
	
	lPath = strlen(Path);
	if(lPath > MAX_PATH_LENGTH)
	{
		LOG_DEBUG("error: path too long: %d/%d", lPath, MAX_PATH_LENGTH);
		return 0;
	}
	
	lAddr = sizeof(Addr.sun_family) + lPath;
	Addr.sun_family = AF_UNIX;
	strcpy(Addr.sun_path, Path);
	
	unlink(Path);
	
	if(bind(Socket, (struct sockaddr *)&Addr, (socklen_t)lAddr) < 0)
	{
		LOG_DEBUG("error: bind(): %d: %s", Net_GetLastError(), Path);
		return 0;
	}
	
	chmod(Path, DEFFILEMODE);
	
	return 1;
}

int Net_AcceptUnix(int ListenSocket, int *pConnSocket)
{
	struct sockaddr_un Addr;
	int lAddr, Socket;
	
	
	lAddr = sizeof(struct sockaddr_un);
	Socket = accept(ListenSocket, (struct sockaddr*)&Addr, (socklen_t*)&lAddr);
	if(Socket < 0)
	{
		LOG_DEBUG("error: accept(): %d", Net_GetLastError());
		return 0;
	}
	*pConnSocket = Socket;
	
	return 1;
}

int Net_ConnectUnix(int Socket, char *Path)
{
	struct sockaddr_un Addr;
	int lAddr, lPath;
	
	
	lPath = strlen(Path);
	if(lPath > MAX_PATH_LENGTH)
	{
		LOG_DEBUG("error: path too long: %d/%d", lPath, MAX_PATH_LENGTH);
		return 0;
	}
	Addr.sun_family = AF_UNIX;
	strcpy(Addr.sun_path, Path);
	lAddr = lPath + sizeof(Addr.sun_family);
	if(connect(Socket, (struct sockaddr *)&Addr, (socklen_t)lAddr) < 0)
	{
		LOG_DEBUG("error: connect(): %d", Net_GetLastError());
		return 0;
	}
	return 1;
}

#endif
