#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>
#include <unistd.h>

#include <libshred.h>


/*
	Base 64 functions
*/

/* table originale */
static char Base64_Table[] =
{
	'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M',
	'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z',
	'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm',
	'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z',
	'0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '+', '/', '\0'
};
static char Base64_Pad = '=';
static short Base64_rTable[256];
static int Base64_InitOK = 0;

static void _String_Base64Init(void)
{
	uint8 c;
	int i;
	
	for(i=0;i<256;i++)
		Base64_rTable[i] = -1;
	
	for(i=0;i<64;i++)
	{
		c = (uint8)Base64_Table[i];
		Base64_rTable[c] = i;
	}
}


int String_Base64EncodeSize(int Size)
{
	return ((Size + 2) / 3) * 4;
}


int String_Base64EncodeInputSize(int OutSize)
{
	return (OutSize / 4) * 3;
}


int String_Base64Encode(uint8 *String, int lString, char **pBufferOut, int *pszBufferOut)
{
	uint8 *StrIn = String;
	uint8 *pBuf, *BufferOut;
	
	
	if(lString <= 0 || ((lString + 2) / 3) >= (1 << (sizeof(int) * 8 - 2)))
		return 0;
	
	if(!pBufferOut || !pszBufferOut)
		return 0;
	
	if(!*pBufferOut)
	{
		BufferOut = (uint8*)malloc(((lString + 2) / 3) * 4 + 1);
		*pBufferOut = (char*)BufferOut;
	}
	else
	{
		int szBufferOut = *pszBufferOut;
		
		if(szBufferOut < lString * 4 / 3 + 1)
			return 0;
		
		BufferOut = (uint8*)*pBufferOut;
	}
	
	pBuf = BufferOut;
	while(lString > 2)
	{
		/* keep going until we have less than 24 bits */
		*pBuf++ = Base64_Table[StrIn[0] >> 2];
		*pBuf++ = Base64_Table[((StrIn[0] & 0x03) << 4) + (StrIn[1] >> 4)];
		*pBuf++ = Base64_Table[((StrIn[1] & 0x0f) << 2) + (StrIn[2] >> 6)];
		*pBuf++ = Base64_Table[StrIn[2] & 0x3f];

		StrIn += 3;
		lString -= 3; /* we just handle 3 octets of data */
	}
	
	/* now deal with the tail end of things */
	if(lString != 0)
	{
		*pBuf++ = Base64_Table[StrIn[0] >> 2];
		if(lString > 1)
		{
			*pBuf++ = Base64_Table[((StrIn[0] & 0x03) << 4) + (StrIn[1] >> 4)];
			*pBuf++ = Base64_Table[(StrIn[1] & 0x0f) << 2];
			*pBuf++ = Base64_Pad;
		}
		else
		{
			*pBuf++ = Base64_Table[(StrIn[0] & 0x03) << 4];
			*pBuf++ = Base64_Pad;
			*pBuf++ = Base64_Pad;
		}
	}
	
	*pBuf = 0;
	*pszBufferOut = (int)(pBuf - BufferOut);
	
	return 1;
}

/* as above, but backwards. :) */
int String_Base64Decode(char *String, int lString, uint8 **pBufferOut, int *pszBufferOut)
{
	char *StrIn = String;
	int ch, i = 0, j = 0, k;
	uint8 *BufferOut;
	
	
	if(!Base64_InitOK)
	{
		_String_Base64Init();
		Base64_InitOK = 1;
	}
	
	if(!pBufferOut || !pszBufferOut)
		return 0;
	
	if(!*pBufferOut)
	{
		BufferOut = (uint8*)malloc(lString);	// this should be enough
		*pBufferOut = BufferOut;
	}
	else
	{
		int szBufferOut = *pszBufferOut;
		
		if(szBufferOut < lString * 3 / 4)
			return 0;
		
		BufferOut = *pBufferOut;
	}
	
	/* run through the whole string, converting as we go */
	while((ch = *StrIn++) != '\0' && lString-- > 0)
	{
		if (ch == Base64_Pad)
			break;
		
	    /* When Base64 gets POSTed, all pluses are interpreted as spaces.
		   This line changes them back.  It's not exactly the Base64 spec,
		   but it is completely compatible with it (the spec says that
		   spaces are invalid).  This will also save many people considerable
		   headache.  - Turadg Aleahmad <turadg@wise.berkeley.edu>
	    */

		if(ch == ' ')
			ch = '+'; 
		
		ch = Base64_rTable[ch];
		if(ch < 0)
			continue;
		
		switch(i % 4)
		{
			case 0:
				BufferOut[j] = ch << 2;
				break;
			case 1:
				BufferOut[j++] |= ch >> 4;
				BufferOut[j] = (ch & 0x0f) << 4;
				break;
			case 2:
				BufferOut[j++] |= ch >>2;
				BufferOut[j] = (ch & 0x03) << 6;
				break;
			case 3:
				BufferOut[j++] |= ch;
				break;
		}
		i++;
	}
	
	k = j;
	/* mop things up if we ended on a boundary */
	if(ch == Base64_Pad)
	{
		switch(i % 4)
		{
			case 1:
				free(BufferOut);
				return 0;
			case 2:
				k++;
			case 3:
				BufferOut[k++] = 0;
		}
	}
	
	*pszBufferOut = j;
	
	return 1;
}


/*
	URL operations
*/

#define HEXDIGIT(x) (((x) <= '9') ? (x) - '0' : ((x) & 7) + 9)
#define HEX_ESC_CHAR	'%'

/* same as String_URLEncode used with HEX_ESC_CHAR */
int String_URLEncode(char *From, char *To, int Length)
{
	char c;
	int x, x1, x2, iTo;
	char *Char2Hex = "0123456789ABCDEF";
	
	iTo = 0;
	while((c = *From++) != 0)
	{
		if(isalnum(c))
		{
			if(iTo+1 >= Length)
			{
				*To = 0;
				return 0;
			}
			*To++ = c;
			iTo++;
		}
		else
		{
			if(iTo+3 >= Length)
			{
				*To = 0;
				return 0;
			}
			*To++ = HEX_ESC_CHAR;
			x = (int)(uint8)c;
			x1 = (x>>4);
			x2 = x & 0x0000000f;
			
			*To++ = Char2Hex[x1];
			*To++ = Char2Hex[x2];
			iTo+=3;
		}
	}
	*To = 0;
	return 1;
}

int String_URLDecode(char *From, char *To, int Length)
{
	char c;
	int x1, x2, iTo;
	
	iTo = 0;
	while((c = *From++) != 0)
	{
		if(iTo+1 >= Length)
		{
			*To = 0;
			return 0;
		}
		if(c == HEX_ESC_CHAR)
		{
			x1 = *From++;
			if(!isxdigit(x1))
				return 0;
			x2 = *From++;
			if(!isxdigit(x2))
				return 0;
			*To++ = (HEXDIGIT(x1) << 4) + HEXDIGIT(x2);
		}
		else
		{
			*To++ = c;
		}
		iTo++;
	}
	*To = 0;
	return 1;
}

/*
	split functions
*/

int String_Split(char *String, char Separator, char ***pArray, int *plArray)
{
	int iRead, iStart, lString, szArray;
	char **Array;
	
	if(!String || !Separator || !pArray || !plArray)
		return 0;
	
	lString = strlen(String);
	iRead = 0;
	iStart = 0;
	Array = NULL;
	szArray = 0;
	while(1)
	{
		if(iRead >= lString)
		{
			/* assure length > 0 */
			Array = realloc(Array, (szArray + 1) * sizeof(char*));
			Array[szArray] = strdup(String + iStart);
			szArray++;
			break;
		}
		if(String[iRead] == Separator)
		{
			char *StringTo;
			int Length = iRead - iStart;
			
			Array = realloc(Array, (szArray + 1) * sizeof(char*));
			
			StringTo = malloc(Length + 1);
			memcpy(StringTo, String + iStart, Length);
			StringTo[Length] = 0;
			
			Array[szArray] = StringTo;
			szArray++;
			iStart = iRead+1;
		}
		iRead++;
	}
	
	*pArray = Array;
	*plArray = szArray;
	
	return 1;
}

/*
	append functions
*/

void String_Append_Init(TAppendInfo *Info, int MaxSize)
{
	memset(Info, 0, sizeof(TAppendInfo));
	Info->MaxSize = MaxSize;
}


static void _String_Append_StringBlock(TAppendInfo *Info, char *String, int lString)
{
	int lCopy;
	
	
	lCopy = lString;
	if(lString + Info->lBuffer >= Info->szBuffer)
	{
		int szBuffer = ((lString + Info->lBuffer) / 2 + 1) * 2;
		if(Info->MaxSize > 0 && szBuffer > Info->MaxSize)
		{
			szBuffer = Info->MaxSize;
			lCopy = szBuffer - Info->lBuffer - 1;
		}
		if(Info->szBuffer != szBuffer)
		{
			Info->szBuffer = szBuffer;
			Info->Buffer = realloc(Info->Buffer, Info->szBuffer);
		}
	}
	
	if(lCopy > 0)
	{
		memcpy(Info->Buffer + Info->lBuffer, String, lCopy);
		Info->lBuffer += lCopy;
		Info->Buffer[Info->lBuffer] = 0;
	}
}


void String_Append_String(TAppendInfo *Info, char *String)
{
	int lString;
	
	lString = strlen(String);
	_String_Append_StringBlock(Info, String, lString);
}


void String_Append_StringArgs(TAppendInfo *Info, char *Format, ...)
{
	char String[1024];
	int lString;
	va_list arg;
	
	
	va_start(arg, Format);
	lString = vsnprintf(String, sizeof(String) - 1, Format, arg);
	va_end(arg);
	
	_String_Append_StringBlock(Info, String, lString);
}


char *String_Append_Finalize(TAppendInfo *Info, int *pLength)
{
	if(pLength)
		*pLength = Info->lBuffer;
	return Info->Buffer;
}

/*
	array functions
*/

void StringArray_Init(char ***pArray, int *plArray)
{
	*pArray = NULL;
	*plArray = 0;
}


void StringArray_Free(char ***pArray, int *plArray)
{
	char **Array = *pArray;
	int i, lArray = *plArray;
	
	if(Array)
	{
		for(i=0;i<lArray;i++)
			String_SafeFree(Array[i]);
		free(Array);
	}
	
	*pArray = NULL;
	*plArray = 0;
}


static int _StringArray_Expand(char ***pArray, int *plArray)
{
	char **Array = *pArray;
	int ID, lArray = *plArray;
	
	ID = lArray;
	Array = realloc(Array, (lArray + 1) * sizeof(char*));
	Array[lArray++] = NULL;
	
	*pArray = Array;
	*plArray = lArray;
	
	return ID;
}


void StringArray_AppendStringArgs(char ***pArray, int *plArray, char *Format, ...)
{
	char String[1024];
	int ID;
	va_list arg;
	
	
	va_start(arg, Format);
	vsnprintf(String, sizeof(String) - 1, Format, arg);
	va_end(arg);
	
	ID = _StringArray_Expand(pArray, plArray);
	(*pArray)[ID] = strdup(String);
}


void StringArray_AppendString(char ***pArray, int *plArray, char *String)
{
	int ID;
	
	ID = _StringArray_Expand(pArray, plArray);
	(*pArray)[ID] = String_Duplicate(String);
}


/*
	check functions
*/

int String_CheckIP(char *String)
{
	int n, i0, i1, i2, i3, lArray;
	char **Array;
	
	
	if(!String)
		return 0;
	
	if(!String_Split(String, '.', &Array, &lArray))
		return 0;
	
	if(lArray != 4)
	{
		StringArray_Free(&Array, &lArray);
		return 0;
	}
	
	i0 = -1;
	i1 = -1;
	i2 = -1;
	i3 = -1;
	
	n = 0;
	n += 1 - String_GetNumber(Array[0], &i0, NULL);
	n += 1 - String_GetNumber(Array[1], &i1, NULL);
	n += 1 - String_GetNumber(Array[2], &i2, NULL);
	n += 1 - String_GetNumber(Array[3], &i3, NULL);
	
	n += 1 - String_IsDigit(Array[0]);
	n += 1 - String_IsDigit(Array[1]);
	n += 1 - String_IsDigit(Array[2]);
	n += 1 - String_IsDigit(Array[3]);
	
	StringArray_Free(&Array, &lArray);
	
	if(n > 0)
		return 0;
	
	if(i0 < 0 || i0 >= 256
	|| i1 < 0 || i1 >= 256
	|| i2 < 0 || i2 >= 256
	|| i3 < 0 || i3 >= 256)
	{
		return 0;
	}
	
	return 1;
}

int String_IsDNSNameEx(char *String, int lString)
{
	char *pString;
	
	if(!String)
		return -1;
	
	if(lString < 0)
		lString = strlen(String);
	
	if(lString < 1)
		return 0;
	
	if(String[0] == '-' || String[lString - 1] == '-')
		return 0;
	
	pString = String;
	while(*pString)
	{
		if(pString - String >= lString)
			break;
		
		if(!isalnum(*pString) && *pString != '-')
			return 0;
		
		pString++;
	}
	return 1;
}

int String_IsAlphaNumEx(char *String, int lString)
{
	char *pString;
	
	if(!String)
		return -1;
	
	if(lString < 0)
		lString = strlen(String);
	
	pString = String;
	while(*pString)
	{
		if(pString - String >= lString)
			break;
		
		if(!isalnum(*pString))
			return 0;
		
		pString++;
	}
	return 1;
}

int String_IsAlphaEx(char *String, int lString)
{
	char *pString;
	
	if(!String)
		return -1;
	
	if(lString < 0)
		lString = strlen(String);
	
	pString = String;
	while(*pString)
	{
		if(pString - String >= lString)
			break;
		
		if(!isalpha(*pString))
			return 0;
		
		pString++;
	}
	return 1;
}

int String_IsDigitEx(char *String, int lString)
{
	char *pString;
	
	if(!String)
		return -1;
	
	if(lString < 0)
		lString = strlen(String);
	
	pString = String;
	while(*pString)
	{
		if(pString - String >= lString)
			break;
		
		if(!isdigit(*pString))
			return 0;
		
		pString++;
	}
	return 1;
}

int String_CheckFQDN(char *String)
{
	char **Array = NULL;
	int i, lArray = 0, iLastItem, Success = 0;
	
	
	if(!String)
		return 0;
	
	if(strlen(String) >= 128)
		return 0;
	
	if(!String_Split(String, '.', &Array, &lArray))
		return 0;
	
	iLastItem = lArray - 1;
	for(i=0;i<iLastItem;i++)
	{
		if(!Array[i][0])
			goto _end_;
		if(!String_IsDNSName(Array[i]))
			goto _end_;
	}
	
	if(!Array[iLastItem][0])
		goto _end_;
	
	if(!String_IsAlpha(Array[iLastItem]))
		goto _end_;
	
	if(strlen(Array[iLastItem]) > 4)
		goto _end_;
	
	Success = 1;
_end_:
	StringArray_Free(&Array, &lArray);
	
	return Success;
}

int String_IsNumber(char *String)
{
	int c;
	
	if(!String)
		return 0;
	
	while(1)
	{
		c = *String;
		if(!c)
			break;
		
		if(isdigit(c))
		{
			String++;
			continue;
		}
		return 0;
	}
	return 1;
}


/* nDigits can be NULL */
int String_GetNumber(char *String, int *Number, int *nDigits)
{
	int Sign, n, c, _nDigits;
	
	if(!String || !Number)
		return 0;
	
	Sign = 1;
	if(String[0] == '-')
	{
		Sign = -1;
		String++;
	}
	
	_nDigits = 0;
	n = 0;
	while(1)
	{
		c = *String;
		if(!c)
			break;
		
		if(isdigit(c))
		{
			n = n*10 + (c - '0');
			String++;
			_nDigits++;
			continue;
		}
		if(_nDigits > 0)
			*Number = Sign*n;
		if(nDigits)
			*nDigits = _nDigits;
		return (_nDigits > 0);
	}
	if(_nDigits <= 0)
	{
		if(nDigits)
			*nDigits = 0;
		return 0;
	}
	if(nDigits)
		*nDigits = _nDigits;
	*Number = Sign*n;
	return 1;
}


// this code is 10.5 x faster than snprintf(Buffer, szBuffer - 1, "%d", n)
char * String_IntToString(int n, char *Buffer, int szBuffer)
{
	int Remainder, Negative;
	char *pBuf, *pBufStart;
	
	
	pBufStart = Buffer;
	Negative = 0;
	if(n < 0)
	{
		if(n < -2147483647)
			return NULL;
		
		n = -n;
		Negative = 1;
		pBufStart++;
	}
	
	// check minimum buffer size (2 for positive number, 3 for negative)
	if(szBuffer < 2 + Negative)
		return NULL;
	
	pBuf = Buffer + szBuffer - 1;
	*pBuf-- = 0;
	while(1)
	{
		if(pBuf < pBufStart)
		{
			// not enough space
			return NULL;
		}
		
		Remainder = n % 10;
		n /= 10;
		
		*pBuf = '0' + Remainder;
		
		if(!n)
		{
			if(Negative)
			{
				pBuf--;
				*pBuf = '-';
				return pBuf;
			}
			else
			{
				return pBuf;
			}
		}
		
		pBuf--;
	}
	
	return NULL;
}

/*
	search/match functions
*/

#if 1

/* Boyer-Moore algorithm implementation */
uint8 * String_BinarySearch(uint8 *Buffer, int lBuffer, uint8 *Pattern, int lPattern)
{
	int i, j, k, Skip[256];
	
	
	if(lPattern < 0)
		lPattern = strlen((char*)Pattern);
	
	if(!lPattern)
		return Buffer;
	
	for(k = 0; k < 255; k++)
		Skip[k] = lPattern;
	for(k = 0; k < lPattern - 1; k++)
		Skip[(uint32)Pattern[k]] = lPattern - k - 1;
	
	for(k = lPattern - 1; k < lBuffer; k += Skip[(uint32)(Buffer[k] & (255))])
	{
		for(j = lPattern - 1, i = k; j >= 0 && Buffer[i] == Pattern[j]; j--)
			i--;
		if(j == (-1))
			return Buffer + i + 1;
	}
	
	// not found
	return NULL;
}

#else

/* naive binary search */
uint8 * String_BinarySearch(uint8 *Buffer, int lBuffer, uint8 *Pattern, int lPattern)
{
	uint8 *pSearch, *pSearch2, *pPattern;
	uint8 *pBufferEnd, *pPatternEnd;
	
	
	if(lPattern < 0)
		lPattern = strlen((char*)Pattern);
	
	pBufferEnd = Buffer + lBuffer;
	pPatternEnd = Pattern + lPattern;
	
	pSearch = Buffer;
	pPattern = Pattern;
	while(1)
	{
		if(*pSearch != *pPattern)
		{
			pSearch++;
			if(pSearch >= pBufferEnd)
				return NULL;
		}
		else
		{
			pSearch2 = pSearch;
			
_again_:
			pSearch2++;
			pPattern++;
			
			if(pSearch2 >= pBufferEnd)
				return NULL;
			
			if(pPattern >= pPatternEnd)
				return pSearch;
			
			if(*pSearch2 != *pPattern)
			{
				pPattern = Pattern;
				pSearch++;
				continue;
			}
			else
			{
				goto _again_;
			}
		}
	}
	return NULL;
}

#endif


static inline int _String_Match_Asterisk(char **pWildcard, char **pString)
{
	int Match = 1;
	
	
	// skip leading asterisk
	(*pWildcard)++; 
	while((**pString) && ((**pWildcard == '?') || (**pWildcard == '*')))
	{
		if(**pWildcard == '?') 
			(*pString)++;
		(*pWildcard)++;
	}
	while((**pWildcard) == '*')
		(*pWildcard)++;
	
	if(!(**pString) && (**pWildcard))
		return 0;
	if(!(**pString) && !(**pWildcard))
		return 1; 
	
	if(!String_Match((*pString), *pWildcard))
	{
		do 
		{
			(*pString)++;
			while(((**pWildcard) != (**pString)) && (**pString))
				(*pString)++;
		} while((**pString) ? (!String_Match((*pString), *pWildcard)) : (0 != (Match = 0)));
	}
	if(!(**pString) && !(**pWildcard))
		Match = 1;
	return Match;
}


int String_Match(char *String, char *Wildcard)
{
	int Match = 1;
	
	while(1)
	{
		if(!*Wildcard || !*String || !Match)
			break;
		
		switch(*Wildcard)
		{
			case '?':
				String++;
				Wildcard++;
				break;
		
			case '*':
				Match = _String_Match_Asterisk(&Wildcard, &String);
				break;
			
			default:
				Match = (*Wildcard == *String);
				String++;
				Wildcard++;
				break;
		}
	}
	while((*Wildcard == '*') && Match) 
		Wildcard++;
	
	return (int)(Match && (!*String) && (!*Wildcard));
}


int String_IndexOf(char *String, char Character)
{
	char *pString;
	
	if(!String)
		return -1;
	
	pString = String;
	while(*pString)
	{
		if(*pString == Character)
			return (int)(pString - String);
		pString++;
	}
	return -1;
}


int String_LastIndexOf(char *String, char Character)
{
	char *pString;
	int lString;
	
	if(!String)
		return -1;
	
	lString = strlen(String);
	pString = String + lString - 1;
	while(pString >= String)
	{
		if(*pString == Character)
			return (int)(pString - String);
		pString--;
	}
	return -1;
}


int String_TrimEx(char *Line, int MaxLength)
{
	int iStart, iEnd;
	char c;
	
	if(!Line)
		return 0;
	
	if(MaxLength == 0)
		MaxLength = strlen(Line);
	
	for(iStart=0;iStart<MaxLength;iStart++)
	{
		c = Line[iStart];
		if(c!=' ' && c!='\t')
			break;
	}
	
	for(iEnd=iStart+1;iEnd<MaxLength;iEnd++)
	{
		c = Line[iEnd];
		if(c==0)
			break;
	}
	iEnd--;
	while(iEnd >= iStart)
	{
		c = Line[iEnd];
		if(c!=0 && c!=' ' && c!='\t' && c!=10 && c!=13)
			break;
		iEnd--;
	}
	iEnd++;
	
	memmove(Line, Line+iStart, iEnd-iStart);
	Line[iEnd-iStart] = 0;
	
	return iEnd-iStart;
}


static char _ToLowerArray[256] = {
	  0,   1,   2,   3,   4,   5,   6,   7,   8,   9,  10,  11,  12,  13,  14,  15,
	 16,  17,  18,  19,  20,  21,  22,  23,  24,  25,  26,  27,  28,  29,  30,  31,
	 32,  33,  34,  35,  36,  37,  38,  39,  40,  41,  42,  43,  44,  45,  46,  47,
	 48,  49,  50,  51,  52,  53,  54,  55,  56,  57,  58,  59,  60,  61,  62,  63,
	 64,  97,  98,  99, 100, 101, 102, 103, 104, 105, 106, 107, 108, 109, 110, 111,
	112, 113, 114, 115, 116, 117, 118, 119, 120, 121, 122,  91,  92,  93,  94,  95,
	 96,  97,  98,  99, 100, 101, 102, 103, 104, 105, 106, 107, 108, 109, 110, 111,
	112, 113, 114, 115, 116, 117, 118, 119, 120, 121, 122, 123, 124, 125, 126, 127,
	128, 129, 130, 131, 132, 133, 134, 135, 136, 137, 138, 139, 140, 141, 142, 143,
	144, 145, 146, 147, 148, 149, 150, 151, 152, 153, 154, 155, 156, 157, 158, 159,
	160, 161, 162, 163, 164, 165, 166, 167, 168, 169, 170, 171, 172, 173, 174, 175,
	176, 177, 178, 179, 180, 181, 182, 183, 184, 185, 186, 187, 188, 189, 190, 191,
	192, 193, 194, 195, 196, 197, 198, 199, 200, 201, 202, 203, 204, 205, 206, 207,
	208, 209, 210, 211, 212, 213, 214, 215, 216, 217, 218, 219, 220, 221, 222, 223,
	224, 225, 226, 227, 228, 229, 230, 231, 232, 233, 234, 235, 236, 237, 238, 239,
	240, 241, 242, 243, 244, 245, 246, 247, 248, 249, 250, 251, 252, 253, 254, 255
};

static char _ToUpperArray[256] = {
	  0,   1,   2,   3,   4,   5,   6,   7,   8,   9,  10,  11,  12,  13,  14,  15,
	 16,  17,  18,  19,  20,  21,  22,  23,  24,  25,  26,  27,  28,  29,  30,  31,
	 32,  33,  34,  35,  36,  37,  38,  39,  40,  41,  42,  43,  44,  45,  46,  47,
	 48,  49,  50,  51,  52,  53,  54,  55,  56,  57,  58,  59,  60,  61,  62,  63,
	 64,  65,  66,  67,  68,  69,  70,  71,  72,  73,  74,  75,  76,  77,  78,  79,
	 80,  81,  82,  83,  84,  85,  86,  87,  88,  89,  90,  91,  92,  93,  94,  95,
	 96,  65,  66,  67,  68,  69,  70,  71,  72,  73,  74,  75,  76,  77,  78,  79,
	 80,  81,  82,  83,  84,  85,  86,  87,  88,  89,  90, 123, 124, 125, 126, 127,
	128, 129, 130, 131, 132, 133, 134, 135, 136, 137, 138, 139, 140, 141, 142, 143,
	144, 145, 146, 147, 148, 149, 150, 151, 152, 153, 154, 155, 156, 157, 158, 159,
	160, 161, 162, 163, 164, 165, 166, 167, 168, 169, 170, 171, 172, 173, 174, 175,
	176, 177, 178, 179, 180, 181, 182, 183, 184, 185, 186, 187, 188, 189, 190, 191,
	192, 193, 194, 195, 196, 197, 198, 199, 200, 201, 202, 203, 204, 205, 206, 207,
	208, 209, 210, 211, 212, 213, 214, 215, 216, 217, 218, 219, 220, 221, 222, 223,
	224, 225, 226, 227, 228, 229, 230, 231, 232, 233, 234, 235, 236, 237, 238, 239,
	240, 241, 242, 243, 244, 245, 246, 247, 248, 249, 250, 251, 252, 253, 254, 255
};


void String_ToLower(char *String)
{
	uint32 c;
	
	while(1)
	{
		c = *String;
		if(!c)
			break;
		
		*String++ = _ToLowerArray[c];
	}
}


void String_ToUpper(char *String)
{
	uint32 c;
	
	while(1)
	{
		c = *String;
		if(!c)
			break;
		
		*String++ = _ToUpperArray[c];
	}
}


static int _FromHexIndex[256] = {
	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,
	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,
	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,
	0,	1,	2,	3,	4,	5,	6,	7,	8,	9,	-1,	-1,	-1,	-1,	-1,	-1,
	-1,	10,	11,	12,	13,	14,	15,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,
	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,
	-1,	10,	11,	12,	13,	14,	15,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,
	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,
	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,
	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,
	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,
	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,
	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,
	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,
	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,
	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1
};

static char _ToHexIndex[] = "0123456789abcdef";


int String_HexToBuffer(char *HexString, int lHexString, uint8 **pBuffer, int *plBuffer)
{
	int i;
	int kh, kl;
	uint8 *Buffer = NULL;
	int lBuffer = 0;
	
	
	if(!HexString || !pBuffer)
		return 0;
	
	if(lHexString < 0)
		lHexString = strlen(HexString);
	
	if(lHexString % 2 != 0)
	{
		//LOG_DEBUG("error: invalid hexadecimal string: bad length: %s", HexString);
		return 0;
	}
	
	if(!*pBuffer)
	{
		Buffer = malloc(lHexString / 2);
	}
	else
	{
		if(*plBuffer < lHexString / 2)
			return 0;
		Buffer = *pBuffer;
	}
	
	i = 0;
	while(i<lHexString)
	{
		kh = _FromHexIndex[(uint8)HexString[i++]];
		kl = _FromHexIndex[(uint8)HexString[i++]];
		
		if(kh < 0 || kl < 0)
		{
			//LOG_DEBUG("error: invalid hexadecimal string: %s", HexString);
			if(!*pBuffer)
				free(Buffer);
			return 0;
		}
		Buffer[lBuffer++] = (uint8)(kh * 16 + kl);
	}
	
	if(!*pBuffer)
		*pBuffer = Buffer;
	*plBuffer = lBuffer;
	
	return 1;
}


int String_BufferToHex(uint8 *Buffer, int lBuffer, char **pHexString, int *plHexString)
{
	uint32 kh, kl, k;
	char *HexString = NULL, *pHex;
	int lHexString = 0;
	uint8 *pBufMax;
	
	
	if(!Buffer || lBuffer < 0 || !pHexString)
	{
		LOG_DEBUG("error: invalid arguments");
		return 0;
	}
	
	if(!*pHexString)
	{
		HexString = malloc(1 + lBuffer * 2);
	}
	else
	{
		int lHexStringMin = 1 + lBuffer * 2;
		
		*pHexString[0] = 0;
		if(!plHexString)
		{
			LOG_DEBUG("error: missing argument");
			return 0;
		}
		if(*plHexString < lHexStringMin)
		{
			LOG_DEBUG("error: string too short: %d/%d", *plHexString, lHexStringMin);
			return 0;
		}
		HexString = *pHexString;
	}
	
	lHexString = 0;
	pHex = &HexString[0];
	pBufMax = Buffer + lBuffer;
	while(Buffer < pBufMax)
	{
		k = *Buffer++;
		kh = k >> 4;
		kl = k & 0x0f;
		*pHex++ = _ToHexIndex[kh];
		*pHex++ = _ToHexIndex[kl];
	}
	*pHex = 0;
	
	if(!*pHexString)
		*pHexString = HexString;
	if(plHexString)
		*plHexString = lHexString;
	
	return 1;
}


/*
	"toto", "tot"	->	1
	"toto", "to"	->	1
	"toto", "ta"	->	0
	"tot", "toto"	->	0
*/
int String_StartsWithEx(char *String, char *Needle, char **pTrail)
{
	int Skew;
	
	if(!String || !Needle)
		return 0;
	
	while(1)
	{
		if(!*Needle)
		{
			if(pTrail)
				*pTrail = String;
			return 1;
		}
		
		Skew = (int)(*String - *Needle);
		if(Skew)
			return 0;
		
		String++;
		Needle++;
	}
}


int String_EndsWith(char *String, char *Needle)
{
	int lString, lNeedle;
	
	if(!String || !Needle)
		return 0;
	
	lString = strlen(String);
	lNeedle = strlen(Needle);
	
	if(lNeedle > lString)
		return 0;
	
	return (memcmp(String + lString - lNeedle, Needle, lNeedle) == 0);
}


int String_GetKeyValueEx(char *String, char **pKey, char **pValue, char Separator)
{
	int iPos, lKey, lValue;
	char *Key, *Value;
	
	
	if(!String || !Separator || !pKey || !pValue)
		return 0;
	
	iPos = String_IndexOf(String, Separator);
	if(iPos <= 0)
		return 0;
	
	lKey = iPos;
	Key =(char*)malloc(lKey + 1);
	memcpy(Key, String, lKey);
	Key[lKey] = 0;
	lKey = String_Trim(Key);
	
	lValue = strlen(String + iPos + 1);
	Value = (char*)malloc(lValue + 1);
	memcpy(Value, String + iPos + 1, lValue);
	Value[lValue] = 0;
	lValue = String_Trim(Value);
	
	*pKey = Key;
	*pValue = Value;
	
	return 1;
}

// we don't use strndup because it is evil
char * String_DuplicateEx(char *String, int lString)
{
	char *NewString;
	
	if(!String)
		return NULL;
	
	if(lString < 0)
		return strdup(String);
	
	NewString = malloc(lString + 1);
	memcpy(NewString, String, lString);
	NewString[lString] = 0;
	
	return NewString;
}


int String_CopyEx(char *In, char *Out, int szOut, int lCopy)
{
	if(!Out || szOut <= 0 || !In)
		return -1;
	
	int lOutMax = szOut - 1;
	int lIn = strlen(In);
	//int szIn = lIn + 1;
	if(lCopy < 0)
		lCopy = 0x7fffffff;
	lCopy = MIN(lCopy, lOutMax);
	lCopy = MIN(lCopy, lIn);
	
	memcpy(Out, In, lCopy);
	Out[lCopy] = 0;
	
	return lCopy;
}


int String_IsBinary(char *String, int lString)
{
	int i;
	uint8 c;
	
	for(i=0;i<lString;i++)
	{
		c = (uint8)String[i];
		if((c < 32 && c != '\r' && c != '\n' && c != '\t') || (c > 126))
			return 1;
	}
	return 0;
}


int String_Escape(uint8 *In, int lIn, char *Out, int lOut, int Options)
{
	uint8 *pInEnd, *pIn, *pOut, Byte;
	
	
	if(!In || lIn < 0 || !Out || lOut <= 0)
		return 0;
	
	if(!(Options & ESCAPE_OPT_ALL) && (Options & ESCAPE_OPT_NOESCAPECHAR))
		return 0;
	
	if(!(Options & ESCAPE_OPT_NOESCAPECHAR))
	{
		if(lOut < 3 * lIn + 1)
			return 0;
	}
	else
	{
		if(lOut < 2 * lIn + 1)
			return 0;
	}
	
	pIn = In;
	pInEnd = In + lIn;
	pOut = (uint8*)Out;
	
	if(!(Options & ESCAPE_OPT_ALL))
	{
		while(pIn < pInEnd)
		{
			Byte = *pIn++;
			if(Byte >= 32 && Byte <= 126 && Byte != '\\')	// we must escape '\\'
			{
				*pOut++ = (char)Byte;
			}
			else
			{
				*pOut++ = '\\';
				*pOut++ = _ToHexIndex[Byte / 16];
				*pOut++ = _ToHexIndex[Byte % 16];
			}
		}
	}
	else
	{
		if(!(Options & ESCAPE_OPT_NOESCAPECHAR))
		{
			while(pIn < pInEnd)
			{
				Byte = *pIn++;
				*pOut++ = '\\';
				*pOut++ = _ToHexIndex[Byte / 16];
				*pOut++ = _ToHexIndex[Byte % 16];
			}
		}
		else
		{
			while(pIn < pInEnd)
			{
				Byte = *pIn++;
				*pOut++ = _ToHexIndex[Byte / 16];
				*pOut++ = _ToHexIndex[Byte % 16];
			}
		}
	}
	*pOut = 0;
	
	return 1;
}


int String_Equal(char *String1, char *String2)
{
	if(!String1 && !String2)
		return 1;
	
	if(!String1 || !String2)
		return 0;
	
	if(String1 == String2)
		return 1;
	
	return !strcmp(String1, String2);
}


/*
	Natural string comparison functions
*/
#define NATCMP_IS_DIGIT(c)	(isdigit((uint8)c))
#define NATCMP_IS_SPACE(c)	(isspace((uint8)c))
#define NATCMP_TOUPPER(c)	(toupper((uint8)c))


static int _String_Compare_Natural_Right(char *String1, char *String2)
{
	int Bias = 0;
	
	/*
		The longest run of digits wins.  That aside, the greatest
		value wins, but we can't know that it will until we've scanned
		both numbers to know that they have the same magnitude, so we
		remember it in BIAS.
	*/
	for(;; String1++, String2++)
	{
		if(!NATCMP_IS_DIGIT(*String1) && !NATCMP_IS_DIGIT(*String2))
			return Bias;
		else if(!NATCMP_IS_DIGIT(*String1))
			return -1;
		else if(!NATCMP_IS_DIGIT(*String2))
			return 1;
		else if(*String1 < *String2)
		{
			if(!Bias)
				Bias = -1;
		}
		else if(*String1 > *String2)
		{
			if(!Bias)
				Bias = 1;
		}
		else if(!*String1 && !*String2)
			return Bias;
	}
	return 0;
}


static int _String_Compare_Natural_Left(char *String1, char *String2)
{
	// Compare two left-aligned numbers: the first to have a different value wins.
	
	for(;; String1++, String2++)
	{
		if(!NATCMP_IS_DIGIT(*String1) && !NATCMP_IS_DIGIT(*String2))
			return 0;
		else if(!NATCMP_IS_DIGIT(*String1))
			return -1;
		else if(!NATCMP_IS_DIGIT(*String2))
			return 1;
		else if(*String1 < *String2)
			return -1;
		else if(*String1 > *String2)
			return 1;
	}
	return 0;
}


int String_Compare_Natural(char *String1, char *String2, int CaseSensitive)
{
	int iString1, iString2;
	char c1, c2;
	
	
	if(!String1 && !String2)
		return 1;
	
	if(!String1 || !String2)
		return 0;
	
	if(String1 == String2)
		return 1;
	
	iString1 = 0;
	iString2 = 0;
	while(1)
	{
		c1 = String1[iString1];
		c2 = String2[iString2];
		
		/* skip over leading spaces or zeros */
		while(NATCMP_IS_SPACE(c1))
			c1 = String1[++iString1];
		
		while(NATCMP_IS_SPACE(c2))
			c2 = String2[++iString2];
		
		/* process run of digits */
		if(NATCMP_IS_DIGIT(c1) && NATCMP_IS_DIGIT(c2))
		{
			int Result;
			
			if(c1 == '0' || c2 == '0')
			{
				Result = _String_Compare_Natural_Left(String1 + iString1, String2 + iString2);
				if(Result)
					return Result;
			}
			else
			{
				Result = _String_Compare_Natural_Right(String1 + iString1, String2 + iString2);
				if(Result)
					return Result;
			}
		}
		
		if(!c1 && !c2)
		{
			/*
				The strings compare the same.  Perhaps the caller
				will want to call strcmp to break the tie.
			*/
			return 0;
		}
		
		if(!CaseSensitive)
		{
			c1 = NATCMP_TOUPPER(c1);
			c2 = NATCMP_TOUPPER(c2);
		}
		
		if(c1 < c2)
			return -1;
		else if(c1 > c2)
			return 1;
		
		iString1++;
		iString2++;
	}
}
