#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifndef WIN32
#include <errno.h>
#include <sys/utsname.h>
#include <signal.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#endif

#include <libshred.h>

static int _Debug = 0;


void System_SetDebug(int Debug)
{
	_Debug = Debug;
}


int System_ExecuteGrabOutput(char *Cmd, char ***pLines, int *plLines)
{
	FILE *Pipe;
	char Line[1024];
	int Result = 0;
	char **Lines = NULL;
	int lLines = 0;
	
	
	if(!Cmd || !pLines || !plLines)
	{
		LOG_DEBUG("invalid arguments");
		return 0;
	}
	
	if(_Debug)
		LOG_DEBUG("pipe: [%s]", Cmd);
	Pipe = popen(Cmd, "r");
	if(!Pipe)
		return 0;
	
	while(!feof(Pipe))
	{
		if(!fgets(Line, sizeof(Line), Pipe))
			break;
		String_Trim(Line);
		Lines = (char**)realloc(Lines, (lLines + 1) * sizeof(char*));
		Lines[lLines++] = strdup(Line);
	}
	Result = pclose(Pipe);
	
	*pLines = Lines;
	*plLines = lLines;
	
	return !Result;
}


int System_ExecuteGrabOutput_FirstLine(char *Cmd, char *Line, int szLine)
{
	char **Lines = NULL;
	int lLines = 0;
	int Done = 0;
	
	
	if(!Cmd || !Line || szLine <= 0)
	{
		LOG_DEBUG("invalid arguments");
		return 0;
	}
	
	// default: empty line
	Line[0] = 0;
	
	if(System_ExecuteGrabOutput(Cmd, &Lines, &lLines))
	{
		if(lLines > 0)
		{
			String_Copy(Lines[0], Line, szLine);
			Done = 1;
		}
		StringArray_Free(&Lines, &lLines);
	}
	
	return Done;
}


int System_Execute(char *Cmd)
{
	int Result;
	
	
	if(!Cmd)
	{
		LOG_DEBUG("invalid arguments");
		return 0;
	}
	
	if(_Debug)
		LOG_DEBUG("exec: [%s]", Cmd);
	Result = system(Cmd);
	if(_Debug && Result)
		LOG_DEBUG("error: command failed (%d): [%s]", Result, Cmd);
	return !Result;
}


int System_ExecuteArgs(char *Format, ...)
{
	char Cmd[512];
	va_list Args;
	int Result;
	
	va_start(Args, Format);
	vsnprintf(Cmd, sizeof(Cmd) - 1, Format, Args);
	va_end(Args);
	
	Result = System_Execute(Cmd);
	
	return Result;
}


#ifndef WIN32
int System_Daemonize(void)
{
	int i, NullFD;
	
	signal(SIGHUP, SIG_IGN);
	signal(SIGUSR1, SIG_IGN);
	signal(SIGUSR2, SIG_IGN);
	signal(SIGPIPE, SIG_IGN);
	
	/* Make ourselves a daemon. */
	switch(fork())
	{
		case 0:
 			break;
		
		case -1:
		{
			LOG_DEBUG("error: fork()");
  			exit(0);
 		}
		
 		default:
  			exit(0);
	}
	
	setsid();
	
	/* If we're running as a daemon, close the default file descriptors, AFTER forking. */
	
	NullFD = open("/dev/null", O_RDWR);
	if(NullFD < 0)
		exit(1);
	
	dup2(NullFD, STDIN_FILENO);
	dup2(NullFD, STDOUT_FILENO);
	dup2(NullFD, STDERR_FILENO);
	close(NullFD);
	
	// all file descriptors >= 3 will be closed
	for(i=3;i<1024;i++)
		close(i);
	
 	return 1;
}


int System_DupOutputToFile(char *Filename, int DupStdOut, int DupStdErr)
{
	int LogFD, Mode;
	
	if(!DupStdOut && !DupStdErr)
		return 1;
	
	Mode = S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH;
	LogFD = open(Filename, O_WRONLY | O_CREAT | O_TRUNC, Mode);
	if(LogFD < 0)
		return 0;
	
	if(DupStdOut)
		dup2(LogFD, STDOUT_FILENO);
	if(DupStdErr)
		dup2(LogFD, STDERR_FILENO);
	close(LogFD);
	
	if(!String_StartsWith(Filename, "/dev/"))
		chmod(Filename, Mode);
	
	return 1;
}
#endif


/*
	Error functions
*/
uint32 System_GetLastErrorID(void)
{
	uint32 ErrorID;
	
#ifndef WIN32
	ErrorID = errno;
#else
	ErrorID = GetLastError();
#endif
	
	return ErrorID;
}


char *System_GetLastErrorString(uint32 ErrorID, char *TmpBuf, int szTmpBuf)
{
	TmpBuf[0] = 0;
	
#if defined(LINUX) || defined(APPLE) || defined(FREEBSD)
	strerror_r((int)ErrorID, TmpBuf, szTmpBuf - 1);
#else
	FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM, NULL, ErrorID, MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), (LPTSTR)TmpBuf, szTmpBuf - 1, NULL);
	String_Trim(TmpBuf);
#endif
	return TmpBuf;
}


char *System_GetCurrentUser(void)
{
#ifndef WIN32
	return getenv("USER");
#else
	return getenv("USERNAME");
#endif
}


#ifndef WIN32
int System_IsRoot(void)
{
	if(getuid() != 0 || geteuid() != 0)
		return 0;
	return 1;
}
#endif


#ifdef WIN32
#define __environ environ
#endif

#ifdef APPLE
#include <crt_externs.h>
#define __environ (*_NSGetEnviron())
#endif

void System_GetEnvironmentVars(TEnvVar **pEnvVars, int *pszEnvVars)
{
	char **pEnvString, **EnvRoot = __environ;
	TEnvVar *EnvVars = NULL;
	int i, szEnvVars = 0;
	
	
	// first count variables
	pEnvString = EnvRoot;
	while(*pEnvString)
	{
		pEnvString++;
		szEnvVars++;
	}
	
	// then store them in new array
	EnvVars = malloc(szEnvVars * sizeof(TEnvVar));
	memset(EnvVars, 0, szEnvVars * sizeof(TEnvVar));
	
	pEnvString = EnvRoot;
	i = 0;
	while(*pEnvString)
	{
		char *EnvString = *pEnvString;
		TEnvVar *pEnvVar = &EnvVars[i];
		int iPos;
		
		iPos = String_IndexOf(EnvString, '=');
		if(iPos > 0)
		{
			pEnvVar->Name = String_DuplicateEx(EnvString, iPos);
			pEnvVar->Value = String_Duplicate(EnvString + iPos + 1);
		}
		else
		{
			pEnvVar->Name = String_Duplicate(EnvString);
			pEnvVar->Value = NULL;
		}
		
		pEnvString++;
		i++;
	}
	
	// set out args
	*pEnvVars = EnvVars;
	*pszEnvVars = szEnvVars;
}


void System_DestroyEnvironmentVars(TEnvVar *EnvVars, int szEnvVars)
{
	if(EnvVars)
	{
		int i;
		
		for(i=0;i<szEnvVars;i++)
		{
			TEnvVar *pEnvVar = &EnvVars[i];
			String_SafeFree(pEnvVar->Name);
			String_SafeFree(pEnvVar->Value);
		}
		
		free(EnvVars);
	}
}


#ifdef LINUX
int System_GetProcessList_LINUX(TProcessInfo **pProcessList, int *plProcessList)
{
	char **Files = NULL;
	int i, lFiles = 0;
	TProcessInfo *ProcessList = NULL;
	int lProcessList = 0;
	
	
	if(!FileSystem_GetFilesInDir("/proc", NULL, &Files, &lFiles))
		return 0;
	
	for(i=0;i<lFiles;i++)
	{
		char *Filename = Files[i];
		char PPath[32], Path[64];
		char **Lines = NULL;
		int j, lLines = 0;
		int PID, ParentPID = 0, nThreads = 0;
		char *Name = NULL, *State = NULL;
		char **Args = NULL;
		int lArgs = 0;
		
		if(!String_IsNumber(Filename))
			continue;
		
		PID = atoi(Filename);
		snprintf(PPath, sizeof(PPath) - 1, "/proc/%s", Filename);
		
		// test not needed, it will be included just after
		//if(!FileSystem_FolderExists(PPath))
		//	continue;
		
		/*
		{
			struct stat Status;
			
			snprintf(Path, sizeof(Path) - 1, "%s/exe", PPath);
			if(stat(Path, &Status) == 0)
			{
				LOG_DEBUG("S_ISREG: %d", S_ISREG(Status.st_mode));
				LOG_DEBUG("S_ISLNK: %d", S_ISLNK(Status.st_mode));
			}
			IsKernelModule = !FileSystem_FileExists(Path);
		}
		*/
		
		snprintf(Path, sizeof(Path) - 1, "%s/status", PPath);
		if(!FileSystem_FileExists(Path))
			continue;
		
		if(!FileSystem_ReadTextFile(Path, &Lines, &lLines))
			continue;
		
		for(j=0;j<lLines;j++)
		{
			char *Line = Lines[j];
			char *Trail;
			
			if(String_StartsWithEx(Line, "PPid:", &Trail))
			{
				String_Trim(Trail);
				ParentPID = atoi(Trail);
				continue;
			}
			if(String_StartsWithEx(Line, "Name:", &Trail))
			{
				String_Trim(Trail);
				Name = Trail;
				continue;
			}
			if(String_StartsWithEx(Line, "Threads:", &Trail))
			{
				String_Trim(Trail);
				nThreads = atoi(Trail);
				continue;
			}
			if(String_StartsWithEx(Line, "State:", &Trail))
			{
				String_Trim(Trail);
				State = Trail;
				continue;
			}
		}
		
		// read cmdline
		{
			snprintf(Path, sizeof(Path) - 1, "%s/cmdline", PPath);
			if(FileSystem_FileExists(Path))
			{
				char CmdLine[512], *pEnd, *pStart;
				FILE *File;
				int lCmdLine = 0;
				
				// file does not have size, and contains '\0': we must read is from 1 block
				memset(CmdLine, 0, sizeof(CmdLine));
				File = fopen(Path, "r");
				if(File)
				{
					lCmdLine = fread(CmdLine, 1, sizeof(CmdLine) - 2, File);	// -2 because '\0' of last string + '\0' as final character
					fclose(File);
				}
				
				if(lCmdLine > 0)
				{
					StringArray_Init(&Args, &lArgs);
					
					pEnd = CmdLine + lCmdLine;
					pStart = &CmdLine[0];
					
					while(pStart < pEnd)
					{
						int lTrunk = strlen(pStart);
						StringArray_AppendString(&Args, &lArgs, pStart);
						pStart += lTrunk + 1;
					}
				}
			}
			else
			{
				LOG_DEBUG("error: cannot read cmdline");
			}
		}
		
		if(Name && nThreads && State)
		{
			TProcessInfo *pProcess;
			
			ProcessList = realloc(ProcessList, (lProcessList + 1) * sizeof(TProcessInfo));
			pProcess = &ProcessList[lProcessList++];
			
			memset(pProcess, 0, sizeof(TProcessInfo));
			pProcess->Name = strdup(Name);
			pProcess->State = strdup(State);
			pProcess->PID = PID;
			pProcess->ParentPID = ParentPID;
			pProcess->nThreads = nThreads;
			pProcess->CmdLine.Args = Args;
			pProcess->CmdLine.lArgs = lArgs;
			pProcess->IsKernelModule = (pProcess->CmdLine.lArgs == 0);
		}
		else
		{
			StringArray_Free(&Args, &lArgs);
			LOG_DEBUG("error: parsing info for process with pid %d", PID);
		}
		
		StringArray_Free(&Lines, &lLines);
	}
	
	StringArray_Free(&Files, &lFiles);
	
	*pProcessList = ProcessList;
	*plProcessList = lProcessList;
	
	return 1;
}
#endif


int System_GetProcessList(TProcessInfo **pProcessList, int *pszProcessList)
{
#ifdef LINUX
	return System_GetProcessList_LINUX(pProcessList, pszProcessList);
#endif
	
	LOG_DEBUG("error: System_GetProcessList() not implemented");
	return 0;
}


void System_DestroyProcessList(TProcessInfo *ProcessList, int lProcessList)
{
	if(ProcessList)
	{
		int i;
		
		for(i=0;i<lProcessList;i++)
		{
			TProcessInfo *pProcess = &ProcessList[i];
			String_SafeFree(pProcess->Name);
			String_SafeFree(pProcess->State);
			StringArray_Free(&pProcess->CmdLine.Args, &pProcess->CmdLine.lArgs);
		}
		free(ProcessList);
	}
}


int System_GetPID(void)
{
#ifdef WIN32
	return _getpid();
#else
	return getpid();
#endif
}


int System_IsLittleEndian(void)
{
	uint32 k32;
	uint8 *pk;
	
	k32 = (1 << 24) | (2 << 16) | (3 << 8) | (4 << 0);
	pk = (uint8*)&k32;
	if(pk[0] == 4 && pk[1] == 3 && pk[2] == 2 && pk[3] == 1)
		return 1;
	return 0;
}


int System_IsBigEndian(void)
{
	uint32 k32;
	uint8 *pk;
	
	k32 = (1 << 24) | (2 << 16) | (3 << 8) | (4 << 0);
	pk = (uint8*)&k32;
	if(pk[0] == 1 && pk[1] == 2 && pk[2] == 3 && pk[3] == 4)
		return 1;
	return 0;
}


void System_AssertEndianness(void)
{
	int Match = 0;
	char *CompilerTarget = NULL;
	
#ifdef SYS_BIG_ENDIAN
	Match = System_IsBigEndian();
	CompilerTarget = "BIG_ENDIAN";
#else
	Match = System_IsLittleEndian();
	CompilerTarget = "LITTLE_ENDIAN";
#endif
	if(!Match)
	{
		LOG_DEBUG("error: endianness mismatch: compiler target %s mismatch", CompilerTarget);
		exit(0);
	}
}


static char RuntimeVersion[64] = {0};
static char BuildDate[16] = {0};
static char KernelVersion[64] = {0};
static char SystemVersion[128] = {0};
static char SystemArch[32] = {0};


#ifdef WIN32
// WIN32 functions

/*
	Name can be:
		CompanyName
		FileDescription
		FileVersion
		InternalName
		LegalCopyright
		OriginalFilename
		ProductName
		ProductVersion
*/
int System_WIN32_GetFileInfo(char *Filename, char *Name, char *Value, int szValue)
{
	char Query[128], *pValue;
	uint8 *Buffer = NULL;
	int szBuffer, lValue, Result;
	DWORD dwHandle;
	VS_FIXEDFILEINFO *pVerInfo;
	
	
	Result = 0;
	szBuffer = GetFileVersionInfoSize(Filename, &dwHandle);
	if(!szBuffer)
		return 0;
	
	Buffer = malloc(szBuffer + 32);
	
	// get version buffer filled
	if(!GetFileVersionInfo(Filename, dwHandle, szBuffer, Buffer))
		goto _end_;
	
	if(VerQueryValue(Buffer, "\\", (void*)&pVerInfo, (UINT*)&lValue))
	{
		if(pVerInfo)
		{
			WORD *pwVerInfo;
			
			if(VerQueryValue(Buffer, "\\VarFileInfo\\Translation",  (void*)&pwVerInfo, (UINT*)&lValue))
			{
				snprintf(Query, sizeof(Query) - 1, "\\StringFileInfo\\%04x%04x\\%s", pwVerInfo[0], pwVerInfo[1], Name);
				if(VerQueryValue(Buffer, Query, (void*)&pValue, (UINT*)&lValue))
				{
					snprintf(Value, szValue - 1, "%.*s", lValue, pValue);
					Result = 1;
				}
			}
		}
	}
	
_end_:
	if(Buffer)
		free(Buffer);
	
	return Result;
}

#else

static int _LINUX_GetMemory(int *pFree, int *pTotal)
{
	char Buffer[256];
	FILE *File;
	int Result = 0;
	
	File = popen("free -k 2> /dev/null | grep \"buffers/cache\"", "r");
	if(File)
	{
		Buffer[0] = 0;
		fgets(Buffer, sizeof(Buffer) - 1, File);
		if(String_Trim(Buffer) > 0)
		{
			// format: -/+ buffers/cache:      17700     239272
			int iPos = String_IndexOf(Buffer, ':');
			if(iPos > 0)
			{
				int Used, Free;
				char *Values = &Buffer[iPos + 1];
				
				String_Trim(Values);
				Used = 0;
				Free = 0;
				sscanf(Values, "%d %d", &Used, &Free);
				if(pFree)
					*pFree = Free;
				if(pTotal)
					*pTotal = Used + Free;
				
				Result = 1;
			}
		}
		pclose(File);
	}
	
	return Result;
}

#endif


char *System_GetBuildDateEx(char *Date)
{
	// english months: January, February, March, April, May, June, July, August, September, October, November, December
	char Buffer[64];
	int iMonth, iDay, iYear;
	
	
	if(BuildDate[0])
		return BuildDate;
	
	if(!Date)
		Date = __DATE__;
	
	// __DATE__ format: Feb 12 1996
	
	snprintf(Buffer, sizeof(Buffer) - 1, "%s", __DATE__);
	if(strncmp(Buffer, "Jan", 3) == 0)
		iMonth = 1;
	else if(strncmp(Buffer, "Feb", 3) == 0)
		iMonth = 2;
	else if(strncmp(Buffer, "Mar", 3) == 0)
		iMonth = 3;
	else if(strncmp(Buffer, "Apr", 3) == 0)
		iMonth = 4;
	else if(strncmp(Buffer, "May", 3) == 0)
		iMonth = 5;
	else if(strncmp(Buffer, "Jun", 3) == 0)
		iMonth = 6;
	else if(strncmp(Buffer, "Jul", 3) == 0)
		iMonth = 7;
	else if(strncmp(Buffer, "Aug", 3) == 0)
		iMonth = 8;
	else if(strncmp(Buffer, "Sep", 3) == 0)
		iMonth = 9;
	else if(strncmp(Buffer, "Oct", 3) == 0)
		iMonth = 10;
	else if(strncmp(Buffer, "Nov", 3) == 0)
		iMonth = 11;
	else if(strncmp(Buffer, "Dec", 3) == 0)
		iMonth = 12;
	else
		iMonth = 0;
	
	Buffer[6] = 0;
	iDay = atoi(&Buffer[4]);
	iYear = atoi(&Buffer[7]);
	
	snprintf(BuildDate, sizeof(BuildDate) - 1, "%04d-%02d-%02d", iYear, iMonth, iDay);
	
	return BuildDate;
}


char *System_GetRuntimeVersion(void)
{
	char Buffer[256];
	
	if(RuntimeVersion[0])
		return RuntimeVersion;
	
#ifdef WIN32
	/*
		FileVersion:	7.0.2600.2180 (xpsp_sp2_rtm.040803-2158)
		ProductVersion:	7.0.2600.2180
	*/
	if(System_WIN32_GetFileInfo("msvcrt.dll", "ProductVersion", Buffer, sizeof(Buffer)))
		snprintf(RuntimeVersion, sizeof(RuntimeVersion) - 1, "Microsoft C Runtime Library %s", Buffer);
#else
	{
		FILE *File;
		
		File = popen("/lib/libc.so.6 2> /dev/null | grep version", "r");
		if(File)
		{
			// format: GNU C Library stable release version 2.3.6, by Roland McGrath et al.
			Buffer[0] = 0;
			fgets(Buffer, sizeof(Buffer) - 1, File);
			if(String_Trim(Buffer) > 0)
			{
				int iPos = String_IndexOf(Buffer, ',');
				if(iPos > 0)
					Buffer[iPos] = 0;
				snprintf(RuntimeVersion, sizeof(RuntimeVersion) - 1, "%s", Buffer);
			}
			pclose(File);
		}
	}
#endif
	
	return RuntimeVersion;
}


int System_GetProcessorInfo(TProcessorInfo *Info)
{
	memset(Info, 0, sizeof(TProcessorInfo));
	
#ifdef WIN32
	{
		SYSTEM_INFO SystemInfo;
		GetSystemInfo(&SystemInfo);
		Info->NumberOfProcessors = (int)SystemInfo.dwNumberOfProcessors;
	}
	{
		HKEY ProcessorKey;
		DWORD Type, dwBuffer, dwFrequency = 0;
		char Buffer[256];
		
		
		if(RegOpenKeyEx(HKEY_LOCAL_MACHINE, "HARDWARE\\DESCRIPTION\\System\\CentralProcessor\\0", 0, KEY_READ, &ProcessorKey) == ERROR_SUCCESS)
		{
			dwBuffer = sizeof(Buffer) - 1;
			if(RegQueryValueExA(ProcessorKey, "Identifier", NULL, &Type, (BYTE*)Buffer, &dwBuffer) == ERROR_SUCCESS)
			{
				// format: x86 Family 6 Model 10 Stepping 0
				sscanf(Buffer, "%*s Family %d Model %d Stepping %d", &Info->Family, &Info->Model, &Info->Stepping);
			}
			
			dwBuffer = sizeof(Buffer) - 1;
			if(RegQueryValueExA(ProcessorKey, "ProcessorNameString", NULL, &Type, (BYTE*)Buffer, &dwBuffer) == ERROR_SUCCESS)
				snprintf(Info->Name, sizeof(Info->Name) - 1, "%s", Buffer);
			
			dwBuffer = sizeof(Buffer) - 1;
			if(RegQueryValueExA(ProcessorKey, "VendorIdentifier", NULL, &Type, (BYTE*)Buffer, &dwBuffer) == ERROR_SUCCESS)
				snprintf(Info->VendorID, sizeof(Info->VendorID) - 1, "%s", Buffer);
			
			dwBuffer = sizeof(dwFrequency);
			if(RegQueryValueExA(ProcessorKey, "~MHz", NULL, &Type, (BYTE*)&dwFrequency, &dwBuffer) == ERROR_SUCCESS)
				Info->FrequencyMHz = dwFrequency;
			
			RegCloseKey(ProcessorKey);
		}
		else
		{
			LOG_DEBUG("error: RegOpenKeyEx(): %u", (uint32)GetLastError());
		}
	}
#else
	{
		char Buffer[256], *Value;
		FILE *File;
		int ProcID = -1;
		
		File = fopen("/proc/cpuinfo", "r");
		if(File)
		{
			while(!feof(File))
			{
				Buffer[0] = 0;
				fgets(Buffer, sizeof(Buffer) - 1, File);
				if(String_Trim(Buffer) > 0)
				{
					if(String_StartsWith(Buffer, "processor"))
					{
						ProcID++;
						continue;
					}
					
					if(ProcID == 0)
					{
						char *pTrail;
						int iPos;
						
						if(String_StartsWithEx(Buffer, "vendor_id", &pTrail))
						{
							iPos = String_IndexOf(pTrail, ':');
							if(iPos > 0)
							{
								Value = &pTrail[iPos + 1];
								String_Trim(Value);
								if(Value[0])
									snprintf(Info->VendorID, sizeof(Info->VendorID) - 1, "%s", Value);
							}
						}
						else if(String_StartsWithEx(Buffer, "model name", &pTrail))
						{
							iPos = String_IndexOf(pTrail, ':');
							if(iPos > 0)
							{
								Value = &pTrail[iPos + 1];
								String_Trim(Value);
								if(Value[0])
									snprintf(Info->Name, sizeof(Info->Name) - 1, "%s", Value);
							}
						}
						else if(String_StartsWithEx(Buffer, "cpu family", &pTrail))
						{
							iPos = String_IndexOf(pTrail, ':');
							if(iPos > 0)
							{
								Value = &pTrail[iPos + 1];
								String_Trim(Value);
								if(Value[0])
									Info->Family = atoi(Value);
							}
						}
						else if(String_StartsWithEx(Buffer, "model", &pTrail))
						{
							iPos = String_IndexOf(pTrail, ':');
							if(iPos > 0)
							{
								Value = &pTrail[iPos + 1];
								String_Trim(Value);
								if(Value[0])
									Info->Model = atoi(Value);
							}
						}
						else if(String_StartsWithEx(Buffer, "stepping", &pTrail))
						{
							iPos = String_IndexOf(pTrail, ':');
							if(iPos > 0)
							{
								Value = &pTrail[iPos + 1];
								String_Trim(Value);
								if(Value[0])
									Info->Stepping = atoi(Value);
							}
						}
						else if(String_StartsWithEx(Buffer, "cpu MHz", &pTrail))
						{
							iPos = String_IndexOf(pTrail, ':');
							if(iPos > 0)
							{
								Value = &pTrail[iPos + 1];
								String_Trim(Value);
								if(Value[0])
									Info->FrequencyMHz = atoi(Value);
							}
						}
					}
				}
			}
			
			if(ProcID >= 0)
				Info->NumberOfProcessors = ProcID + 1;
			
			fclose(File);
		}
	}
#endif
	
	return 1;
}


int System_GetProcessorFrequency(void)
{
	TProcessorInfo Info;
	
	if(System_GetProcessorInfo(&Info))
		return Info.FrequencyMHz;
	return 0;
}


int System_GetTotalPhysicalMemory(void)
{
	int Size = 0;
	
#ifdef WIN32
	{
		MEMORYSTATUSEX MemoryStatus;
		uint64 Divisor = 1024;
		
		MemoryStatus.dwLength = sizeof(MemoryStatus);
		GlobalMemoryStatusEx(&MemoryStatus);
		Size = (int)(MemoryStatus.ullTotalPhys / Divisor);
	}
#else
	_LINUX_GetMemory(NULL, &Size);
#endif
	
	return Size;
}


int System_GetFreePhysicalMemory(void)
{
	int Size = 0;
	
#ifdef WIN32
	{
		MEMORYSTATUSEX MemoryStatus;
		uint64 Divisor = 1024;
		
		MemoryStatus.dwLength = sizeof(MemoryStatus);
		GlobalMemoryStatusEx(&MemoryStatus);
		Size = (int)(MemoryStatus.ullAvailPhys / Divisor);
	}
#else
	_LINUX_GetMemory(&Size, NULL);
#endif
	
	return Size;
}


char *System_GetKernelVersion(void)
{
	if(KernelVersion[0])
		return KernelVersion;
	
#ifdef WIN32
	System_WIN32_GetFileInfo("kernel32.dll", "FileVersion", KernelVersion, sizeof(KernelVersion));
#else
	FileSystem_ReadFileFirstLine("/proc/sys/kernel/osrelease", KernelVersion, sizeof(KernelVersion));
#endif
	
	return KernelVersion;
}


char *System_GetSystemVersion(void)
{
	if(SystemVersion[0])
		return SystemVersion;
	
#ifdef WIN32
	{
		OSVERSIONINFOEX OSVersionInfo;
		//char Version[256];
		
		OSVersionInfo.dwOSVersionInfoSize = sizeof(OSVERSIONINFOEX);
		if(GetVersionEx((OSVERSIONINFO*)&OSVersionInfo))
		{
			/*
			Log_Debug("  dwMajorVersion: %u", (uint32)OSVersionInfo.dwMajorVersion);
			Log_Debug("  dwMinorVersion: %u", (uint32)OSVersionInfo.dwMinorVersion);
			Log_Debug("  dwBuildNumber: %u", (uint32)OSVersionInfo.dwBuildNumber);
			Log_Debug("  dwPlatformId: %u", (uint32)OSVersionInfo.dwPlatformId);
			Log_Debug("  szCSDVersion: %s", OSVersionInfo.szCSDVersion);
			Log_Debug("  wServicePackMajor: %u", (uint32)OSVersionInfo.wServicePackMajor);
			Log_Debug("  wServicePackMinor: %u", (uint32)OSVersionInfo.wServicePackMinor);
			Log_Debug("  wSuiteMask: %u", (uint32)OSVersionInfo.wSuiteMask);
			Log_Debug("  wProductType: %u", (uint32)OSVersionInfo.wProductType);
			*/
			snprintf(SystemVersion, sizeof(SystemVersion) - 1, "Windows %u.%u (Windows %s build %u) [%s]", (uint32)OSVersionInfo.dwMajorVersion, (uint32)OSVersionInfo.dwMinorVersion, OSVersionInfo.dwPlatformId == VER_PLATFORM_WIN32_WINDOWS ? ((OSVersionInfo.dwMinorVersion > 0) ? "98" : "95") : OSVersionInfo.dwPlatformId == VER_PLATFORM_WIN32_NT ? (OSVersionInfo.dwMajorVersion >= 6 ? (OSVersionInfo.dwMinorVersion > 0 ? "7" : "Vista") : OSVersionInfo.dwMajorVersion >= 5 ? OSVersionInfo.dwMinorVersion > 0 ? "XP" : "2000" : "NT") : "?", (uint32)(OSVersionInfo.dwBuildNumber & 0xffff), OSVersionInfo.szCSDVersion);
			//Log_Debug("Version: [%s]", Version);
		}
	}
#else
	{
		FILE *File;
		int lSystemVersion = 0;
		
		
		{
			char OSType[32];
			
			OSType[0] = 0;
			if(FileSystem_ReadFileFirstLine("/proc/sys/kernel/ostype", OSType, sizeof(OSType)))
				lSystemVersion += snprintf(SystemVersion + lSystemVersion, sizeof(SystemVersion) - lSystemVersion - 1, "%s ", OSType);
			else
				lSystemVersion += snprintf(SystemVersion + lSystemVersion, sizeof(SystemVersion) - lSystemVersion - 1, "Linux ");
		}
		
		File = popen("lsb_release -a 2> /dev/null", "r");
		if(File)
		{
			char Line[128];
			
			while(!feof(File))
			{
				Line[0] = 0;
				fgets(Line, sizeof(Line) - 1, File);
				if(String_Trim(Line) > 0)
				{
					char *pTrail;
					
					if(String_StartsWithEx(Line, "Distributor ID:", &pTrail))
					{
						if(String_Trim(pTrail) > 0)
							lSystemVersion += snprintf(SystemVersion + lSystemVersion, sizeof(SystemVersion) - lSystemVersion - 1, "%s ", pTrail);
					}
					else if(String_StartsWithEx(Line, "Release:", &pTrail))
					{
						if(String_Trim(pTrail) > 0)
							lSystemVersion += snprintf(SystemVersion + lSystemVersion, sizeof(SystemVersion) - lSystemVersion - 1, "%s ", pTrail);
					}
					else if(String_StartsWithEx(Line, "Codename:", &pTrail))
					{
						if(String_Trim(pTrail) > 0)
							lSystemVersion += snprintf(SystemVersion + lSystemVersion, sizeof(SystemVersion) - lSystemVersion - 1, "(%s) ", pTrail);
					}
				}
			}
			pclose(File);
		}
		
		String_Trim(SystemVersion);
	}
#endif
	
	return SystemVersion;
}


char *System_GetSystemArch(void)
{
	if(SystemArch[0])
		return SystemArch;
	
#ifdef WIN32
	{
		char Buffer32[64], Buffer64[64];
		
		Buffer64[0] = 0;
		GetEnvironmentVariable("PROCESSOR_ARCHITEW6432", Buffer64, sizeof(Buffer64) - 1);
		String_Trim(Buffer64);
		
		Buffer32[0] = 0;
		GetEnvironmentVariable("PROCESSOR_ARCHITECTURE", Buffer32, sizeof(Buffer32) - 1);
		String_Trim(Buffer32);
		
		if(!Buffer64[0])
			snprintf(SystemArch, sizeof(SystemArch) - 1, "%s", Buffer32);
		else
			snprintf(SystemArch, sizeof(SystemArch) - 1, "%s/%s", Buffer32, Buffer64);
	}
#else
	{
		struct utsname Data;
		if(uname(&Data) == 0)
			snprintf(SystemArch, sizeof(SystemArch) - 1, "%s", Data.machine);
	}
#endif
	
	return SystemArch;
}


#ifdef WIN32
static inline uint64 _FILETIMEToUint64(FILETIME *pTime)
{
	uint64 uTime = 0;
	if(pTime)
	{
		uTime += (uint64)pTime->dwLowDateTime;
		uTime += ((uint64)pTime->dwHighDateTime) << 32;
	}
	return uTime;
}
#endif

int System_GetUptime(uint32 *pUpTime)
{
	int Success = 0;
	
	if(!pUpTime)
		return 0;
	
	*pUpTime = 0;
	
#ifdef LINUX
	{
		char Line[256];
		uint32 RunTime, IdleTime;
		
		FileSystem_ReadFileFirstLine("/proc/uptime", Line, sizeof(Line));
		if(sscanf(Line, "%u.%*d %u.%*d", &RunTime, &IdleTime) == 2)
		{
			// do not keep IdleTime as it is broken (at least on multi-cpu hosts)
			*pUpTime = RunTime;
			Success = 1;
		}
	}
#endif

#ifdef WIN32
	{
		HANDLE hProcessSnap = NULL;
		HANDLE hProcess = NULL;
		PROCESSENTRY32 PE32;
		
		hProcessSnap = CreateToolhelp32Snapshot(TH32CS_SNAPPROCESS, 0);
		if(hProcessSnap == INVALID_HANDLE_VALUE)
		{
			LOG_DEBUG("error: CreateToolhelp32Snapshot()");
		}
		else
		{
			PE32.dwSize = sizeof(PROCESSENTRY32);
			
			// retrieve information about the first process, and exit if unsuccessful
			if(!Process32First(hProcessSnap, &PE32))
			{
				LOG_DEBUG("error: Process32First()");
			}
			else
			{
				do
				{
					if(strcasecmp(PE32.szExeFile, "winlogon.exe") != 0)
						continue;
					
#ifdef PROCESS_QUERY_LIMITED_INFORMATION
					hProcess = OpenProcess(PROCESS_QUERY_LIMITED_INFORMATION, FALSE, PE32.th32ProcessID);
#else
					hProcess = OpenProcess(PROCESS_QUERY_INFORMATION, FALSE, PE32.th32ProcessID);
#endif
					if(!hProcess)
					{
						LOG_DEBUG("error: OpenProcess()");
					}
					else
					{
						FILETIME CreationTime, ExitTime, KernelTime, UserTime;
						
						if(GetProcessTimes(hProcess, &CreationTime, &ExitTime, &KernelTime, &UserTime))
						{
							SYSTEMTIME SystemTime;
							FILETIME SystemFileTime;
							uint64 uSystemFileTime, uCreationTime, uRunTime;
							uint32 iRunTime;
							
							GetSystemTime(&SystemTime);
							SystemTimeToFileTime(&SystemTime, &SystemFileTime);
							
							uSystemFileTime = _FILETIMEToUint64(&SystemFileTime);
							uCreationTime = _FILETIMEToUint64(&CreationTime);
							
							uRunTime = uSystemFileTime - uCreationTime;
							iRunTime = (int)(uRunTime / (1000 * 1000 * 10));
							
							*pUpTime = iRunTime;
							Success = 1;
						}
						
						CloseHandle(hProcess);
					}
				
				} while(Process32Next(hProcessSnap, &PE32));
			}
			CloseHandle(hProcessSnap);
		}
	}
#endif
	return Success;
}


void System_Dump(void)
{
	TProcessorInfo Info;
	char Buffer[256];
	TEnvVar *EnvVars = NULL;
	int i, szEnvVars = 0;
	
	
	System_AssertEndianness();
	
	//---//
	LOG_DEBUG("build info");
	LOG_DEBUG("  compiler version:         %s", System_GetCompilerVersion());
	LOG_DEBUG("  compiler version major:   %d", System_GetCompilerVersionMajor());
	LOG_DEBUG("  compiler version minor:   %d", System_GetCompilerVersionMinor());
	LOG_DEBUG("  compiler version patch:   %d", System_GetCompilerVersionPatch());
	LOG_DEBUG("  build date:               %s", System_GetBuildDate());
	LOG_DEBUG("  build time:               %s", System_GetBuildTime());
#ifdef LIBSHRED_DEBUG
	LOG_DEBUG("  libshred debug:           yes");
#else
	LOG_DEBUG("  libshred debug:           no");
#endif
	
	//---//
	System_GetProcessorInfo(&Info);
	LOG_DEBUG("processor info:");
	LOG_DEBUG("  name:                     %s", Info.Name);
	LOG_DEBUG("  vendor id:                %s", Info.VendorID);
	LOG_DEBUG("  number of processors:     %d", Info.NumberOfProcessors);
	LOG_DEBUG("  frequency:                %d MHz", Info.FrequencyMHz);
	LOG_DEBUG("  family:                   %d", Info.Family);
	LOG_DEBUG("  model:                    %d", Info.Model);
	LOG_DEBUG("  stepping:                 %d", Info.Stepping);
	
	//---//
	LOG_DEBUG("memory info:");
	LOG_DEBUG("  total physical memory:    %d kB", System_GetTotalPhysicalMemory());
	LOG_DEBUG("  free physical memory:     %d kB", System_GetFreePhysicalMemory());
	
	//---//
	LOG_DEBUG("software info:");
	LOG_DEBUG("  kernel version:           %s", System_GetKernelVersion());
	LOG_DEBUG("  runtime version:          %s", System_GetRuntimeVersion());
	LOG_DEBUG("  system version:           %s", System_GetSystemVersion());
	LOG_DEBUG("  system arch:              %s", System_GetSystemArch());
	LOG_DEBUG("  memory address bits:      %d bits", System_GetMemoryAddressBits());
	
	if(System_IsBigEndian())
		LOG_DEBUG("  endianness:               big endian");
	else if(System_IsLittleEndian())
		LOG_DEBUG("  endianness:               little endian");
	else
		LOG_DEBUG("  endianness:               ?");
	
	//---//
	LOG_DEBUG("host info:");
	IP_GetDnsHostname(Buffer, sizeof(Buffer));
	LOG_DEBUG("  dns hostname:             %s", Buffer);
	IP_GetDnsDomain(Buffer, sizeof(Buffer));
	LOG_DEBUG("  dns domain:               %s", Buffer);
	
	{
		uint32 UpTime;
		int iUpTime, iHours, iMinutes, iSeconds;
		
		System_GetUptime(&UpTime);
		
		iUpTime = UpTime;
		iSeconds = iUpTime % 60;
		iUpTime /= 60;
		iMinutes = iUpTime % 60;
		iUpTime /= 60;
		iHours = iUpTime % 60;
		
		LOG_DEBUG("  uptime:                   %02d:%02d:%02d (%u s)", iHours, iMinutes, iSeconds, UpTime);
	}
	
	//---//
	LOG_DEBUG("console info:");
	{
		int Width, Height;
		Console_GetSize(&Width, &Height);
		LOG_DEBUG("  width:                    %d", Width);
		LOG_DEBUG("  height:                   %d", Height);
	}
	
	//---//
	System_GetEnvironmentVars(&EnvVars, &szEnvVars);
	LOG_DEBUG("environment variables:");
	for(i=0;i<szEnvVars;i++)
	{
		TEnvVar *pEnvVar = &EnvVars[i];
		LOG_DEBUG("  [%s] = [%s]", pEnvVar->Name, pEnvVar->Value);
	}
	System_DestroyEnvironmentVars(EnvVars, szEnvVars);
}
