#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef WIN32
#include <winsock2.h>
#include <windows.h>
#else
#include <time.h>
#include <sys/time.h>
#include <unistd.h>
#include <errno.h>
#endif

#include <libshred.h>

#ifdef WIN32
static uint32 dwStartTick = 0;
static uint64 qwTicksPerMusec = 0;
static uint64 qwStartSmallTick = 0;
#else
static struct timeval StartTV = {0,0};
static struct timeval StartMicroTV = {0,0};
#endif

#define EPOCHFILETIME	(116444736000000000LL)

void Time_GetTimestamp(uint32 *pSec, uint32 *pMicroSec)
{
#ifdef WIN32
	FILETIME FileTime;
	LARGE_INTEGER LargeInt;
	int64 i64;
	
	GetSystemTimeAsFileTime(&FileTime);
	LargeInt.LowPart = FileTime.dwLowDateTime;
	LargeInt.HighPart = FileTime.dwHighDateTime;
	i64 = LargeInt.QuadPart;		// In 100-nanosecond intervals
	i64 -= EPOCHFILETIME;	// Offset to the Epoch time
	i64 /= 10;				// In microseconds
	if(pSec)
		*pSec = (uint32)(i64 / 1000000);
	if(pSec)
		*pMicroSec = (uint32)(i64 % 1000000);
#else
	struct timeval LocalTimeTV;
	
	gettimeofday(&LocalTimeTV, NULL);
	if(pSec)
		*pSec = LocalTimeTV.tv_sec;
	if(pSec)
		*pMicroSec = LocalTimeTV.tv_usec;
#endif
}


void Time_GetLocalTime(TTime *Time)
{
#ifdef WIN32
	SYSTEMTIME LocalTime;
	
	if(!Time)
		return;
	
	GetLocalTime(&LocalTime);
	Time->Year = LocalTime.wYear;
	Time->Month = LocalTime.wMonth;
	Time->Day = LocalTime.wDay;
	Time->Hour = LocalTime.wHour;
	Time->Minute = LocalTime.wMinute;
	Time->Second = LocalTime.wSecond;
	Time->Milliseconds = LocalTime.wMilliseconds;
#else
	struct timeval LocalTimeTV;
	struct tm LocalTimeTM;
	
	if(!Time)
		return;
	
	gettimeofday(&LocalTimeTV, NULL);
	localtime_r(&LocalTimeTV.tv_sec, &LocalTimeTM);
	
	Time->Year = LocalTimeTM.tm_year + 1900;
	Time->Month = LocalTimeTM.tm_mon + 1;
	Time->Day = LocalTimeTM.tm_mday;
	Time->Hour = LocalTimeTM.tm_hour;
	Time->Minute = LocalTimeTM.tm_min;
	Time->Second = LocalTimeTM.tm_sec;
	Time->Milliseconds = LocalTimeTV.tv_usec / 1000;
#endif
}


void Time_GetUTCTime(TTime *Time)
{
#ifdef WIN32
	SYSTEMTIME LocalTime;
	
	if(!Time)
		return;
	
	GetSystemTime(&LocalTime);
	Time->Year = LocalTime.wYear;
	Time->Month = LocalTime.wMonth;
	Time->Day = LocalTime.wDay;
	Time->Hour = LocalTime.wHour;
	Time->Minute = LocalTime.wMinute;
	Time->Second = LocalTime.wSecond;
	Time->Milliseconds = LocalTime.wMilliseconds;
#else
	struct timeval LocalTimeTV;
	struct tm LocalTimeTM;
	
	if(!Time)
		return;
	
	gettimeofday(&LocalTimeTV, NULL);
	gmtime_r(&LocalTimeTV.tv_sec, &LocalTimeTM);
	
	Time->Year = LocalTimeTM.tm_year + 1900;
	Time->Month = LocalTimeTM.tm_mon + 1;
	Time->Day = LocalTimeTM.tm_mday;
	Time->Hour = LocalTimeTM.tm_hour;
	Time->Minute = LocalTimeTM.tm_min;
	Time->Second = LocalTimeTM.tm_sec;
	Time->Milliseconds = LocalTimeTV.tv_usec / 1000;
#endif
}


uint64 Time_GetTime64(TTime *Time)
{
	uint64 Time64 = 0;
	
#ifdef WIN32
	// MSDN: The FILETIME structure is a 64-bit value representing the number of 100-nanosecond intervals since January 1, 1601 (UTC).
	SYSTEMTIME LocalTime;
	FILETIME FileTime;
	
	if(!Time)
	{
		GetSystemTime(&LocalTime);
	}
	else
	{
		LocalTime.wYear = Time->Year;
		LocalTime.wMonth = Time->Month;
		LocalTime.wDay = Time->Day;
		LocalTime.wHour = Time->Hour;
		LocalTime.wMinute = Time->Minute;
		LocalTime.wSecond = Time->Second;
		LocalTime.wMilliseconds = Time->Milliseconds;
	}
	
	if(SystemTimeToFileTime(&LocalTime, &FileTime))
	{
		memcpy(&Time64, &FileTime, sizeof(uint64));
		Time64 = (Time64 / 10000) - 11644473600000ULL;	// substract the Epoch time
	}
#else
	struct timeval LocalTimeTV;
	struct tm LocalTimeTM;
	int Milliseconds = 0;
	uint32 Time32 = 0;
	
	if(!Time)
	{
		gettimeofday(&LocalTimeTV, NULL);
		gmtime_r(&LocalTimeTV.tv_sec, &LocalTimeTM);
		Milliseconds = LocalTimeTV.tv_usec / 1000;
	}
	else
	{
		memset(&LocalTimeTM, 0, sizeof(struct tm));
		LocalTimeTM.tm_year = Time->Year - 1900;
		LocalTimeTM.tm_mon = Time->Month - 1;
		LocalTimeTM.tm_mday = Time->Day;
		LocalTimeTM.tm_hour = Time->Hour;
		LocalTimeTM.tm_min = Time->Minute;
		LocalTimeTM.tm_sec = Time->Second;
		Milliseconds = Time->Milliseconds;
	}
	
	Time32 = (uint32)timegm(&LocalTimeTM);
	if(Time32 != (uint32)-1)
		Time64 = ((uint64)Time32) * 1000 + Milliseconds;
#endif
	
	return Time64;
}


uint32 Time_GetTick(void)
{
#ifdef WIN32
	if(!dwStartTick)
	{
		dwStartTick = GetTickCount();
		return 0;
	}
	return Time_Diff32(GetTickCount(), dwStartTick);
#else
	if(!StartTV.tv_sec)
	{
		gettimeofday(&StartTV, NULL);
		return 0;
	}
	else
	{
		struct timeval NowTV;
		uint32 dwDelay;
		
		gettimeofday(&NowTV, NULL);
		dwDelay = 1000 * (NowTV.tv_sec - StartTV.tv_sec) + (uint32)(NowTV.tv_usec / 1000) - (uint32)(StartTV.tv_usec / 1000);
		return dwDelay;
	}
#endif
}


uint32 Time_GetMicroTick(void)
{
#ifdef WIN32
	if(!qwStartSmallTick)
	{
		//LARGE_INTEGER Tmp;
		uint64 Tmp = 0;
		
		//	receives the current performance-counter frequency, in counts per second
		if(!QueryPerformanceFrequency((LARGE_INTEGER*)&Tmp))
		{
			LOG_DEBUG("error: QueryPerformanceFrequency(): %u", (uint32)GetLastError());
			return 0;
		}
		
		qwTicksPerMusec = Tmp / 1000000;
		QueryPerformanceCounter((LARGE_INTEGER*)&Tmp);
		//qwStartSmallTick = Tmp.QuadPart / TicksPerMusec;
		qwStartSmallTick = Tmp / qwTicksPerMusec;
		
		return 1;
	}
	else
	{
		uint64 Tmp = 0;
		uint64 qwTick = 0;
		
		QueryPerformanceCounter((LARGE_INTEGER*)&Tmp);
		qwTick = Tmp / qwTicksPerMusec;
		
		if(qwTick < qwStartSmallTick)
			return (uint32)((((uint64)-1) - qwStartSmallTick) + qwTick);
		return (uint32)(qwTick - qwStartSmallTick);
	}
	
#else
	if(!StartMicroTV.tv_sec)
	{
		gettimeofday(&StartMicroTV, NULL);
		return 0;
	}
	else
	{
		struct timeval NowTV;
		uint32 dwDelay;
		
		gettimeofday(&NowTV, NULL);
		dwDelay = 1000000 * (NowTV.tv_sec - StartMicroTV.tv_sec) + NowTV.tv_usec - StartMicroTV.tv_usec;
		return dwDelay;
	}
#endif
}


void Time_SleepEx(uint32 Milliseconds, int Wait)
{
#ifdef WIN32
	Sleep(Milliseconds);
#else
	//usleep(Milliseconds * 1000);
	
	struct timespec TV, TVLeft;
	
	TV.tv_sec = (time_t)(Milliseconds / 1000);
	TV.tv_nsec = (Milliseconds % 1000) * 1000 * 1000;
	
_again_:
	if(nanosleep(&TV, &TVLeft) == -1)
	{
		if(Wait && errno == EINTR)
		{
			memcpy(&TV, &TVLeft, sizeof(struct timespec));
			goto _again_;
		}
	}
#endif
}
