#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <libshred.h>

#include <UnitTest.h>


static void TestSuite_PrintSuiteName(char *Name)
{
	printf("Running test suite %s\n", Name);
}


static void TestSuite_PrintTestCaseResult(int Success, char *Info)
{
#ifndef WIN32
	if(Success)
		printf("[\033[1;32m   ok   \033[0;39m] %s\n", Info ? Info : "");
	else
		printf("[\033[1;31m failed \033[0;39m] %s\n", Info ? Info : "");
#else
	HANDLE hConsole = GetStdHandle(STD_OUTPUT_HANDLE);
	CONSOLE_SCREEN_BUFFER_INFO SBInfo;
	
	GetConsoleScreenBufferInfo(hConsole, &SBInfo);
	if(Success)
	{
		printf("[");
		SetConsoleTextAttribute(hConsole, FOREGROUND_GREEN | FOREGROUND_INTENSITY);
		printf("   ok   ");
		SetConsoleTextAttribute(hConsole, SBInfo.wAttributes);
		printf("] %s\n", Info ? Info : "");
	}
	else
	{
		printf("[");
		SetConsoleTextAttribute(hConsole, FOREGROUND_RED | FOREGROUND_INTENSITY);
		printf("   ok   ");
		SetConsoleTextAttribute(hConsole, SBInfo.wAttributes);
		printf("] %s\n", Info ? Info : "");
	}
#endif
}


static void TestSuite_PrintSummary(int nSuccess, int nTotal)
{
	printf("Summary: %d success, %d failed, %d total\n", nSuccess, nTotal - nSuccess, nTotal);
	printf("\n");
}


static void _TestSuite_DestroyTestCase(void *Element)
{
	TTestCase *TestCase = (TTestCase*)Element;
	
	if(TestCase->Info)
		Runtime_Free(TestCase->Info);
	Runtime_Free(TestCase);
}


TTestSuite *TestSuite_Create(char *Name)
{
	TTestSuite *TestSuite;
	
	if(!Name)
	{
		printf("error: missing test suite name\n");
		exit(0);
	}
	
	TestSuite = Runtime_Malloc(sizeof(TTestSuite));
	memset(TestSuite, 0, sizeof(TTestSuite));
	
	TestSuite->Name = String_Duplicate(Name);
	TestSuite->TestCases = Array_CreateEx(_TestSuite_DestroyTestCase);
	
	return TestSuite;
}


void TestSuite_Destroy(TTestSuite *TestSuite)
{
	if(TestSuite)
	{
		Array_Destroy(TestSuite->TestCases);
		
		if(TestSuite->Name)
			Runtime_Free(TestSuite->Name);
		
		Runtime_Free(TestSuite);
	}
}


void TestSuite_AddTestCase(TTestSuite *TestSuite, char *Info, TTestCaseGetInfoClbk GetInfoClbk, TTestCaseRunClbk RunClbk, void *Param, int ExpectedResult)
{
	TTestCase *TestCase;
	
	if(!TestSuite || (!Info && !GetInfoClbk) || !RunClbk)
	{
		printf("error: invalid arguments\n");
		exit(0);
	}
	
	TestCase = Runtime_Malloc(sizeof(TTestCase));
	memset(TestCase, 0, sizeof(TTestCase));
	
	TestCase->Info = String_Duplicate(Info);
	TestCase->GetInfo = GetInfoClbk;
	TestCase->Run = RunClbk;
	TestCase->Param = Param;
	TestCase->ExpectedResult = ExpectedResult;
	
	Array_Add(TestSuite->TestCases, TestCase);
}


void TestSuite_SetTestCase(TTestSuite *TestSuite, char *Info, int Success)
{
	TTestCase *TestCase = NULL;
	int i, n;
	
	if(!TestSuite || !Info)
	{
		printf("error: invalid arguments\n");
		exit(0);
	}
	
	n = Array_Count(TestSuite->TestCases);
	for(i=0;i<n;i++)
	{
		TTestCase *pTestCase = (TTestCase*)Array_GetAt(TestSuite->TestCases, i);
		if(String_Equal(pTestCase->Info, Info))
		{
			TestCase = pTestCase;
			break;
		}
	}
	
	if(!TestCase)
	{
		TestCase = Runtime_Malloc(sizeof(TTestCase));
		memset(TestCase, 0, sizeof(TTestCase));
		
		TestCase->Info = String_Duplicate(Info);
		
		Array_Add(TestSuite->TestCases, TestCase);
	}
	
	TestCase->ExpectedResult = 1;
	TestCase->Result = Success ? 1 : 0;
}


int TestSuite_Run(TTestSuite *TestSuite)
{
	int i, n, nSuccess;
	
	TestSuite_PrintSuiteName(TestSuite->Name);
	nSuccess = 0;
	n = Array_Count(TestSuite->TestCases);
	for(i=0;i<n;i++)
	{
		TTestCase *TestCase;
		int Result, Success;
		char Info[256];
		
		TestCase = (TTestCase*)Array_GetAt(TestSuite->TestCases, i);
		
		Info[0] = 0;
		if(TestCase->Info)
			snprintf(Info, sizeof(Info) - 1, "%s", TestCase->Info);
		else if(TestCase->GetInfo)
			TestCase->GetInfo(TestCase->Param, Info, sizeof(Info));
		
		if(TestCase->Run)
			Result = TestCase->Run(TestCase->Param);
		else
			Result = TestCase->Result;
		
		Success = (Result == TestCase->ExpectedResult);
		nSuccess += Success;
		
		TestSuite_PrintTestCaseResult(Success, Info);
	}
	TestSuite_PrintSummary(nSuccess, n);
	
	return (nSuccess == n);
}
