#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <libshred.h>

#include <UnitTest.h>



int Test_BadKeys(void)
{
	TDictionary *Dict;
	int Success = 0;
	
	Dict = Dictionary_Create();
	
	Dictionary_Set(Dict, "Net.Interface.Name", "ra555");	// valid
	Dictionary_Set(Dict, "Net.Interface.-.Name", "ra555");	// valid, '-' not followed by a number
	Dictionary_Set(Dict, "Net.Interface.-3.Name", "ra555");
	Dictionary_Set(Dict, "@Net.Interface.Name", "ra555");
	if(Dictionary_Count(Dict, NULL) != 2)
		LOG_DEBUG("error: expected only 2 valid keys");
	else
		Success = 1;
	Dictionary_Destroy(Dict);
	
	return Success;
}


int Test_SetDeleteGet(int Modulus)
{
	char TmpKey[256], TmpValue[256], TmpBuf[256];
	int i, n;
	char **Keys = NULL;
	int lKeys = 0, Count = 0;
	TDictionary *Dict = NULL;
	int Success = 1;
	
	
	Dict = Dictionary_CreateEx(Modulus);
	n = 20;
	
	// fill with various data
	for(i=0;i<n;i++)
	{
		snprintf(TmpKey, sizeof(TmpKey) - 1, "String.%d", i);
		snprintf(TmpValue, sizeof(TmpValue) - 1, "Value.%d", i);
		Dictionary_Set(Dict, TmpKey, TmpValue);
	}
	for(i=0;i<n;i++)
	{
		snprintf(TmpKey, sizeof(TmpKey) - 1, "Number.%d", i);
		Dictionary_Set_Int32(Dict, TmpKey, i);
	}
	
	// remove non existing item
	Dictionary_Delete(Dict, "NotFound");
	
	Count = Dictionary_Count(Dict, NULL);
	//LOG_DEBUG("dictionary item count: %d", Count);
	
	// remove existing items
	Dictionary_Delete(Dict, "String.7");
	Dictionary_Delete(Dict, "Number.17");
	
	//Dictionary_Dump(Dict);
	
	// keys number
	Dictionary_GetKeys(Dict, &Keys, &lKeys);
	Count = Dictionary_Count(Dict, NULL);
	
	if(lKeys != Count)
	{
		LOG_DEBUG("error: size mismatch: %d / %d", lKeys, Count);
		Success = 0;
	}
	
	StringArray_Free(&Keys, &lKeys);
	
	
	// re-set deleted entries
	Dictionary_Set(Dict, "String.7", "Value.7");
	Dictionary_Set_Int32(Dict, "Number.17", 17);
	
	//Dictionary_Dump(Dict);
	
	Dictionary_GetKeys(Dict, &Keys, &lKeys);
	for(i=0;i<lKeys/2;i++)
	{
		char *Key = Keys[i];
		int v;
		
		v = Dictionary_Get_Int32(Dict, Key);
		if(v != i)
		{
			LOG_DEBUG("error: value mismatch: [%d/%d]", v, i);
			Success = 0;
		}
	}
	for(i=lKeys/2;i<lKeys;i++)
	{
		char *Key = Keys[i];
		
		snprintf(TmpValue, sizeof(TmpValue) - 1, "Value.%d", i - lKeys/2);
		Dictionary_GetValue(Dict, Key, TmpBuf, sizeof(TmpBuf));
		if(strcmp(TmpBuf, TmpValue) != 0)
		{
			LOG_DEBUG("error: value mismatch: [%s/%s]", TmpBuf, TmpValue);
			Success = 0;
		}
	}
	
	StringArray_Free(&Keys, &lKeys);
	
	//Dictionary_Dump(Dict);
	
	Dictionary_Destroy(Dict);
	
	return Success;
}


int Test_NestedArray(void)
{
	TDictionary *Dict;
	int Success = 1;
	
	
	Dict = Dictionary_Create();
	
	// fill with various data
	Dictionary_Set_Int32(Dict, "Net.Interface.Default", 0);
	
	Dictionary_Set(Dict, "Net.Interface.0.Name", "eth0");
	Dictionary_Set(Dict, "Net.Interface.0.IP.0.Addr", "192.168.0.1");
	Dictionary_Set(Dict, "Net.Interface.0.IP.0.Netmask", "255.255.255.0");
	Dictionary_Set(Dict, "Net.Interface.0.IP.1.Addr", "192.168.1.1");
	Dictionary_Set(Dict, "Net.Interface.0.IP.1.Netmask", "255.255.255.0");
	
	Dictionary_Set(Dict, "Net.Interface.1.Name", "eth1");
	Dictionary_Set(Dict, "Net.Interface.1.IP.0.Addr", "192.168.2.1");
	Dictionary_Set(Dict, "Net.Interface.1.IP.0.Netmask", "255.255.255.0");
	
	Dictionary_Set(Dict, "Net.Interface.3.Name", "ra0");
	
	Dictionary_Set(Dict, "Net.Interface.33Name", "ra555");
	Dictionary_Set(Dict, "Net.Interface.33N.a.m.e", "ra555");
	Dictionary_Set(Dict, "Net.Interface.33", "ra555");
	Dictionary_Set(Dict, "Net.Interface.Other.0.Name", "error");
	
	// should be 4, as holes count
	if(Dictionary_Count(Dict, "Net.Interface") != 4)
	{
		LOG_DEBUG("error: expected 4 interfaces");
		Success = 0;
	}
	if(Dictionary_Count(Dict, "Net.Interface.0.IP") != 2)
	{
		LOG_DEBUG("error: expected 2 ip addresses");
		Success = 0;
	}
	if(Dictionary_Count(Dict, "Net.Interface.1.IP") != 1)
	{
		LOG_DEBUG("error: expected 1 ip addresses");
		Success = 0;
	}
	if(Dictionary_Count(Dict, "Net.Interface.2.IP") != 0)
	{
		LOG_DEBUG("error: expected 0 ip addresses");
		Success = 0;
	}
	if(Dictionary_Count(Dict, "Net.Interface.3.IP") != 0)
	{
		LOG_DEBUG("error: expected 0 ip addresses");
		Success = 0;
	}
	
	Dictionary_Destroy(Dict);
	Dict = NULL;
	
	return Success;
}


int Test_EqualDuplicate(void)
{
	TDictionary *Dict1, *Dict2, *Dict3;
	int Success = 1;
	
	
	Dict1 = Dictionary_Create();
	Dictionary_Set(Dict1, "Root.SubDict.Name", "foo1");
	Dictionary_Set(Dict1, "Root.SubDict.ID", "foo1-id");
	
	Dict2 = Dictionary_Create();
	Dictionary_Set(Dict2, "Root.SubDict.Name", "foo1");
	Dictionary_Set(Dict2, "Root.SubDict.ID", "foo1-id");
	
	Dict3 = Dictionary_Duplicate(Dict1);
	
	if(!Dictionary_Equal(Dict1, Dict1))
	{
		LOG_DEBUG("error: dictionary not equal to itself");
		Success = 0;
	}
	if(!Dictionary_Equal(Dict1, Dict2))
	{
		LOG_DEBUG("error: dictionary not equal to another");
		Success = 0;
	}
	if(!Dictionary_Equal(Dict1, Dict3))
	{
		LOG_DEBUG("error: dictionary not equal to its duplicate");
		Success = 0;
	}
	
	Dictionary_Destroy(Dict1);
	Dict1 = NULL;
	Dictionary_Destroy(Dict2);
	Dict2 = NULL;
	Dictionary_Destroy(Dict3);
	Dict3 = NULL;
	
	return Success;
}


int Test_NestedDict(void)
{
	TDictionary *Dict1, *Dict2, *Dict3;
	int Success = 1;
	
	
	Dict1 = Dictionary_Create();
	Dictionary_Set(Dict1, "Root.SubDict.Name", "foo1");
	Dictionary_Set(Dict1, "Root.SubDict.ID", "foo1-id");
	
	Dict2 = Dictionary_Create();
	Dictionary_Set(Dict2, "Name", "foo2");
	Dictionary_Set(Dict2, "Property", "foo2-prop");
	
	Dictionary_Set_Dict(Dict1, "Root.SubDict", Dict2);
	Dictionary_Set_Dict(Dict1, "Root.SubDict.", Dict2);
	Dictionary_Set_Dict(Dict1, "Root.0", Dict2);
	Dictionary_Set_Dict(Dict1, "Root.0.", Dict2);
	
	Dict3 = Dictionary_Create();
	Dictionary_Set(Dict3, "Root.SubDict.Name", "foo2");
	Dictionary_Set(Dict3, "Root.SubDict.ID", "foo1-id");
	Dictionary_Set(Dict3, "Root.SubDict.Property", "foo2-prop");
	Dictionary_Set(Dict3, "Root.0.Name", "foo2");
	Dictionary_Set(Dict3, "Root.0.Property", "foo2-prop");
	
	//Dictionary_Dump(Dict1);
	//Dictionary_Dump(Dict3);
	
	if(!Dictionary_Equal(Dict1, Dict3))
		Success = 0;
	
	Dictionary_Destroy(Dict1);
	Dict1 = NULL;
	Dictionary_Destroy(Dict2);
	Dict2 = NULL;
	Dictionary_Destroy(Dict3);
	Dict3 = NULL;
	
	return Success;
}


int Test_ReadWriteDict(void)
{
	TDictionary *Dict1, *Dict2;
	char *TmpFile = "dict1.tmp";
	int Success = 1;
	
	Dict1 = Dictionary_Create();
	Dictionary_Set(Dict1, "Root.SubDict.Name", "foo1");
	Dictionary_Set(Dict1, "Root.SubDict.ID", "foo1-id");
	Dictionary_Set(Dict1, "Name", "foo2");
	Dictionary_Set(Dict1, "Property", "foo2-prop");
	
	Dictionary_WriteToFile(Dict1, TmpFile);
	Dict2 = Dictionary_ReadFromFile(TmpFile);
	
	if(!Dictionary_Equal(Dict1, Dict2))
		Success = 0;
	
	Dictionary_Destroy(Dict1);
	Dict1 = NULL;
	Dictionary_Destroy(Dict2);
	Dict2 = NULL;
	
	return Success;
}


int main(void)
{
	int Success = 1;
	
	// test bad keys
	{
		TTestSuite *TestSuite = TestSuite_Create("Bad Keys");
		TestSuite_SetTestCase(TestSuite, "", Test_BadKeys());
		Success &= TestSuite_Run(TestSuite);
		TestSuite_Destroy(TestSuite);
	}
	
	// test set, delete and get operations
	{
		TTestSuite *TestSuite = TestSuite_Create("Set, Delete, Get [%197]");
		TestSuite_SetTestCase(TestSuite, "", Test_SetDeleteGet(197));
		Success &= TestSuite_Run(TestSuite);
		TestSuite_Destroy(TestSuite);
	}
	
	// test set, delete and get operations
	{
		TTestSuite *TestSuite = TestSuite_Create("Set, Delete, Get [%default]");
		TestSuite_SetTestCase(TestSuite, "", Test_SetDeleteGet(0));
		Success &= TestSuite_Run(TestSuite);
		TestSuite_Destroy(TestSuite);
	}
	
	// test netsted array
	{
		TTestSuite *TestSuite = TestSuite_Create("Nested Array");
		TestSuite_SetTestCase(TestSuite, "", Test_NestedArray());
		Success &= TestSuite_Run(TestSuite);
		TestSuite_Destroy(TestSuite);
	}
	
	// test equal, duplicate operations
	{
		TTestSuite *TestSuite = TestSuite_Create("Equal, Duplicate");
		TestSuite_SetTestCase(TestSuite, "", Test_EqualDuplicate());
		Success &= TestSuite_Run(TestSuite);
		TestSuite_Destroy(TestSuite);
	}
	
	// test netsted dict
	{
		TTestSuite *TestSuite = TestSuite_Create("Nested Dict");
		TestSuite_SetTestCase(TestSuite, "", Test_NestedDict());
		Success &= TestSuite_Run(TestSuite);
		TestSuite_Destroy(TestSuite);
	}
	
	// test read/write dict
	{
		TTestSuite *TestSuite = TestSuite_Create("Read/Write Dict");
		TestSuite_SetTestCase(TestSuite, "", Test_ReadWriteDict());
		Success &= TestSuite_Run(TestSuite);
		TestSuite_Destroy(TestSuite);
	}
	
	return (Success == 0);
}
