#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <libshred.h>

#include <UnitTest.h>



typedef struct
{
	char *	String0;
	char *	String1;
	int		ExpectedResult;
	
}T2StringsArg;


typedef struct
{
	uint8 *	Buf0;
	int		szBuf0;
	uint8 *	Buf1;
	int		szBuf1;
	int		ExpectedResult;
	
}T2BufsArg;


void TestSingle_2StringsArg_GetInfo(void *Param, char *Info, int szInfo)
{
	T2StringsArg *Arg = (T2StringsArg*)Param;
	char *String = Arg->String0;
	char *Needle = Arg->String1;
	
	snprintf(Info, szInfo, "[%s] => [%s]", String, Needle);
}


void TestSingle_2BufsArg_GetInfo(void *Param, char *Info, int szInfo)
{
	T2BufsArg *Arg = (T2BufsArg*)Param;
	char *Buf0 = (char*)Arg->Buf0;
	char *Buf1 = (char*)Arg->Buf1;
	
	if(String_IsBinary(Buf0, Arg->szBuf0))
		Buf0 = "<bin>";
	if(String_IsBinary(Buf1, Arg->szBuf1))
		Buf1 = "<bin>";
	
	snprintf(Info, szInfo, "[%s] => [%s]", Buf0, Buf1);
}


int TestSingle_Match_Core(void *Param)
{
	T2StringsArg *Arg = (T2StringsArg*)Param;
	char *String = Arg->String0;
	char *Wildcard = Arg->String1;
	
	return String_Match(String, Wildcard);
}


int TestSingle_EndsWith_Core(void *Param)
{
	T2StringsArg *Arg = (T2StringsArg*)Param;
	char *String = Arg->String0;
	char *Needle = Arg->String1;
	
	return String_EndsWith(String, Needle);
}


int TestSingle_StartsWith_Core(void *Param)
{
	T2StringsArg *Arg = (T2StringsArg*)Param;
	char *String = Arg->String0;
	char *Needle = Arg->String1;
	
	return String_StartsWith(String, Needle);
}


int TestSingle_Base64Encode_Core(void *Param)
{
	T2BufsArg *Arg = (T2BufsArg*)Param;
	char *B64 = NULL;
	int szB64, Result = 0;
	
	String_Base64Encode(Arg->Buf0, Arg->szBuf0, &B64, &szB64);
	if(B64 && szB64 == Arg->szBuf1 && memcmp(B64, Arg->Buf1, Arg->szBuf1) == 0)
		Result = 1;
	Runtime_Free(B64);
	return Result;
}

int TestSingle_Base64Decode_Core(void *Param)
{
	T2BufsArg *Arg = (T2BufsArg*)Param;
	uint8 *B64 = NULL;
	int szB64, Result = 0;
	
	String_Base64Decode((char*)Arg->Buf0, Arg->szBuf0, &B64, &szB64);
	if(B64 && szB64 == Arg->szBuf1 && memcmp(B64, Arg->Buf1, Arg->szBuf1) == 0)
		Result = 1;
	Runtime_Free(B64);
	return Result;
}


int main(void)
{
	int Success = 1;
	
	// StartsWith()
	{
		TTestSuite *TestSuite = TestSuite_Create("StartsWith");
		T2StringsArg Arg[] = {
			{"toto", "", 1},
			{"toto", "t", 1},
			{"toto", "to", 1},
			{"toto", "tot", 1},
			{"toto", "toto", 1},
			{"toto", "totot", 0},
			{"toto", "ta", 0},
			{"toto", "titi", 0},
			{NULL, NULL}
		};
		T2StringsArg *pArg = NULL;
		int i;
		
		i = 0;
		while(1)
		{
			pArg = &Arg[i++];
			if(!pArg->String0 && !pArg->String1)
				break;
			
			TestSuite_AddTestCase(TestSuite, NULL, TestSingle_2StringsArg_GetInfo, TestSingle_StartsWith_Core, pArg, pArg->ExpectedResult);
		}
		
		Success &= TestSuite_Run(TestSuite);
		
		TestSuite_Destroy(TestSuite);
	}
	
	// EndsWith
	{
		TTestSuite *TestSuite = TestSuite_Create("EndsWith");
		T2StringsArg Arg[] = {
			{"toto", "", 1},
			{"toto", "o", 1},
			{"toto", "to", 1},
			{"toto", "oto", 1},
			{"toto", "toto", 1},
			{"toto", "ototo", 0},
			{"toto", "a", 0},
			{"toto", "otot", 0},
			{NULL, NULL}
		};
		T2StringsArg *pArg = NULL;
		int i;
		
		i = 0;
		while(1)
		{
			pArg = &Arg[i++];
			if(!pArg->String0 && !pArg->String1)
				break;
			
			TestSuite_AddTestCase(TestSuite, NULL, TestSingle_2StringsArg_GetInfo, TestSingle_EndsWith_Core, pArg, pArg->ExpectedResult);
		}
		
		Success &= TestSuite_Run(TestSuite);
		
		TestSuite_Destroy(TestSuite);
	}
	
	
	// Base64Encode
	{
		TTestSuite *TestSuite = TestSuite_Create("Base64Encode");
		T2BufsArg Arg[] = {
			{(uint8*)"0123456789", 10,									(uint8*)"MDEyMzQ1Njc4OQ==", 16, 1},
			{(uint8*)"0000111122223333444455556666777788889999", 40,	(uint8*)"MDAwMDExMTEyMjIyMzMzMzQ0NDQ1NTU1NjY2Njc3Nzc4ODg4OTk5OQ==", 56, 1},
			{(uint8*)"abcdefghijklmnopqrstuvwxyz0123456789-+/*", 40,	(uint8*)"YWJjZGVmZ2hpamtsbW5vcHFyc3R1dnd4eXowMTIzNDU2Nzg5LSsvKg==", 56, 1},
			{(uint8*)"\015\036\377\325\033\002", 6,						(uint8*)"DR7/1RsC", 8, 1},
			{NULL, 0, NULL, 0}
		};
		T2BufsArg *pArg = NULL;
		int i;
		
		i = 0;
		while(1)
		{
			pArg = &Arg[i++];
			if(!pArg->Buf0 && !pArg->Buf1)
				break;
			
			TestSuite_AddTestCase(TestSuite, NULL, TestSingle_2BufsArg_GetInfo, TestSingle_Base64Encode_Core, pArg, pArg->ExpectedResult);
		}
		
		Success &= TestSuite_Run(TestSuite);
		
		TestSuite_Destroy(TestSuite);
	}
	
	
	// Base64Decode
	{
		TTestSuite *TestSuite = TestSuite_Create("Base64Decode");
		T2BufsArg Arg[] = {
			{(uint8*)"MDEyMzQ1Njc4OQ==", 16,											(uint8*)"0123456789", 10, 1},
			{(uint8*)"MDAwMDExMTEyMjIyMzMzMzQ0NDQ1NTU1NjY2Njc3Nzc4ODg4OTk5OQ==", 56,	(uint8*)"0000111122223333444455556666777788889999", 40, 1},
			{(uint8*)"YWJjZGVmZ2hpamtsbW5vcHFyc3R1dnd4eXowMTIzNDU2Nzg5LSsvKg==", 56,	(uint8*)"abcdefghijklmnopqrstuvwxyz0123456789-+/*", 40, 1},
			{(uint8*)"DR7/1RsC", 8,														(uint8*)"\015\036\377\325\033\002", 6, 1},
			{NULL, 0, NULL, 0}
		};
		T2BufsArg *pArg = NULL;
		int i;
		
		i = 0;
		while(1)
		{
			pArg = &Arg[i++];
			if(!pArg->Buf0 && !pArg->Buf1)
				break;
			
			TestSuite_AddTestCase(TestSuite, NULL, TestSingle_2BufsArg_GetInfo, TestSingle_Base64Decode_Core, pArg, pArg->ExpectedResult);
		}
		
		Success &= TestSuite_Run(TestSuite);
		
		TestSuite_Destroy(TestSuite);
	}
	
	
	
	// String_HexToBuffer
	{
		TTestSuite *TestSuite = TestSuite_Create("HexToBuffer");
		{
			char HexString[32], ExpectedResult[32];
			char HexValue[32] = "0123456789abcdef";
			int i, j, Result;
			uint8 *Buffer = NULL, Byte;
			int nSuccess, lBuffer = 0;
			
			
			HexString[2] = 0;
			
			
			// test with allocated data
			nSuccess = 0;
			for(i=0;i<16;i++)
			{
				for(j=0;j<16;j++)
				{
					HexString[0] = HexValue[i];
					HexString[1] = HexValue[j];
					Buffer = NULL;
					lBuffer = 0;
					Result = String_HexToBuffer(HexString, -1, &Buffer, &lBuffer);
					//printf("result: %d, length: %d, values: [%s]=[%02x]\n", Result, lBuffer, HexString, Buffer[0]);
					if(Result)
					{
						snprintf(ExpectedResult, sizeof(ExpectedResult) - 1, "%02x", Buffer[0]);
						Runtime_Free(Buffer);
						if(strcmp(ExpectedResult, HexString) == 0)
							nSuccess++;
					}
				}
			}
			TestSuite_SetTestCase(TestSuite, "convert all 256 single hex pair to buffer (alloc)", nSuccess == 256);
			
			
			// test without allocated data
			nSuccess = 0;
			for(i=0;i<16;i++)
			{
				for(j=0;j<16;j++)
				{
					HexString[0] = HexValue[i];
					HexString[1] = HexValue[j];
					Buffer = &Byte;
					lBuffer = 1;
					Result = String_HexToBuffer(HexString, -1, &Buffer, &lBuffer);
					//printf("result: %d, length: %d, values: [%s]=[%02x]\n", Result, lBuffer, HexString, Buffer[0]);
					if(Result)
					{
						snprintf(ExpectedResult, sizeof(ExpectedResult) - 1, "%02x", Buffer[0]);
						if(strcmp(ExpectedResult, HexString) == 0)
							nSuccess++;
					}
				}
			}
			TestSuite_SetTestCase(TestSuite, "convert all 256 single hex pair to buffer (no-alloc)", nSuccess == 256);
			
		}
		
		Success &= TestSuite_Run(TestSuite);
		
		TestSuite_Destroy(TestSuite);
	}
	
	
	// String_BufferToHex
	{
#define UINT8_OVERFLOW_NR	256
		char ExpectedOutput[2 * UINT8_OVERFLOW_NR + 1], *pHexString, HexString[2 * UINT8_OVERFLOW_NR + 1];
		uint8 BufferRef[UINT8_OVERFLOW_NR];
		int i, Result, lHexString;
		
		TTestSuite *TestSuite = TestSuite_Create("BufferToHex");
		
		for(i=0;i<UINT8_OVERFLOW_NR;i++)
		{
			BufferRef[i] = (uint8)i;
			sprintf(&ExpectedOutput[2 * i], "%02x", i);
		}
		
		pHexString = NULL;
		Result = String_BufferToHex(BufferRef, UINT8_OVERFLOW_NR, &pHexString, &lHexString);
		TestSuite_SetTestCase(TestSuite, "convert all 256 byte values to single hex pair (alloc)", Result && strcmp(pHexString, ExpectedOutput) == 0);
		Runtime_Free(pHexString);
		
		pHexString = &HexString[0];
		HexString[0] = 0;
		lHexString = sizeof(HexString);
		Result = String_BufferToHex(BufferRef, UINT8_OVERFLOW_NR, &pHexString, &lHexString);
		TestSuite_SetTestCase(TestSuite, "convert all 256 byte values to single hex pair (no-alloc)", Result && strcmp(HexString, ExpectedOutput) == 0);
		
		// should fail: buffer to short
		pHexString = &HexString[0];
		HexString[0] = 0;
		lHexString = 2 * UINT8_OVERFLOW_NR;
		Result = String_BufferToHex(BufferRef, UINT8_OVERFLOW_NR, &pHexString, &lHexString);
		TestSuite_SetTestCase(TestSuite, "convert all 256 byte values to single hex pair in too short buffer (no-alloc)", !Result);
		
		Success &= TestSuite_Run(TestSuite);
		
		TestSuite_Destroy(TestSuite);
	}
	
	// String_URLEncode
	{
		char *StrRef, *StrExpected;
		char Buffer[1024], Info[256];
		int n;
		
		TTestSuite *TestSuite = TestSuite_Create("URLEncode");
		
		StrRef = "http-+/*:!%.=32&<>()[]{}#~&_@$!?,;http";
		StrExpected = "http%2D%2B%2F%2A%3A%21%25%2E%3D32%26%3C%3E%28%29%5B%5D%7B%7D%23%7E%26%5F%40%24%21%3F%2C%3Bhttp";
		
		n = String_URLEncode(StrRef, Buffer, sizeof(Buffer));
		
		snprintf(Info, sizeof(Info) - 1, "[%s] => [%s]", StrRef, StrExpected);
		TestSuite_SetTestCase(TestSuite, Info, strcmp(Buffer, StrExpected) == 0 && n);
		
		Success &= TestSuite_Run(TestSuite);
		
		TestSuite_Destroy(TestSuite);
	}
	
	// String_URLDecode
	{
		char *StrRef, *StrExpected;
		char Buffer[1024], Info[256];
		int n;
		
		TTestSuite *TestSuite = TestSuite_Create("URLDecode");
		
		StrRef = "http%2D%2B%2F%2A%3A%21%25%2E%3D32%26%3C%3E%28%29%5B%5D%7B%7D%23%7E%26%5F%40%24%21%3F%2C%3Bhttp";
		StrExpected = "http-+/*:!%.=32&<>()[]{}#~&_@$!?,;http";
		
		n = String_URLDecode(StrRef, Buffer, sizeof(Buffer));
		
		snprintf(Info, sizeof(Info) - 1, "[%s] => [%s]", StrRef, StrExpected);
		TestSuite_SetTestCase(TestSuite, Info, strcmp(Buffer, StrExpected) == 0 && n);
		
		Success &= TestSuite_Run(TestSuite);
		
		TestSuite_Destroy(TestSuite);
	}
	
	// String_Match
	{
		TTestSuite *TestSuite = TestSuite_Create("Match");
		T2StringsArg Arg[] = {
			// match tests
			{"", "", 1},
			{"a", "a", 1},
			{"ab", "ab", 1},
			{"a", "?", 1},
			{"ab", "??", 1},
			{"ab", "?b", 1},
			{"ab", "a?", 1},
			{"", "*", 1},
			{"a", "*", 1},
			{"a", "**", 1},
			{"abcdefgh", "*bcd*fg*", 1},
			{"abc", "abc*", 1},
			{"abc", "abc**", 1},
			{"abc", "*abc", 1},
			{"abc", "**abc", 1},
			{"abc", "*???", 1},
			{"abcd", "*???", 1},
			{"abcd", "*?*", 1},
			{"abc", "*bc", 1},
			{"123", "*???*", 1},
			{"abcdefghijklmnopqrstuvwxyz", "*?*d?f*k*op?rs*??z*", 1},
			
			// mismatch tests
			{"a", "", 0},
			{"", "?", 0},
			{"a", "b", 0},
			{"abc", "ab", 0},
			{"abc", "bc", 0},
			{"ab", "abc", 0},
			{"bc", "abc", 0},
			{"b", "*b*a*", 0},
			{"abcdefghi", "*bc*efh", 0},
			{"ab", "*???", 0},
			{"abcd", "*dd", 0},
			{"de", "*a*", 0},
			
			{NULL, NULL}
		};
		T2StringsArg *pArg = NULL;
		int i;
		
		i = 0;
		while(1)
		{
			pArg = &Arg[i++];
			if(!pArg->String0 && !pArg->String1)
				break;
			
			TestSuite_AddTestCase(TestSuite, NULL, TestSingle_2StringsArg_GetInfo, TestSingle_Match_Core, pArg, pArg->ExpectedResult);
		}
		
		Success &= TestSuite_Run(TestSuite);
		
		TestSuite_Destroy(TestSuite);
	}
	
	// String_IndexOf
	// String_LastIndexOf
	
	// String_IsBinary
	
	// String_CopyEx
	
	// String_DuplicateEx
	
	// String_ToLower
	// String_ToUpper
	
	// String_IsNumber
	// String_GetNumber
	// String_IntToString
	
	
	return (Success == 0);
}
