#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <math.h>

#include <libshred.h>


typedef uint32 (*TISqrtProc)(uint32 n);
typedef uint32 (*THash32Proc)(uint8 *Key, int lKey);
typedef void *(*THashInitProc)(void );
typedef void (*THashUpdateProc)(void *Handle, uint8 *Buffer, int lBuffer);
typedef void (*THashFinalProc)(void *Handle, uint8 *Digest);


void PrintResult(char *Name, uint32 dwTick)
{
	printf("      %-24s  %6u ms\n", Name, dwTick);
}


uint32 Benchmark_Integers_SQRT_Process(TISqrtProc ISqrt)
{
	uint32 i, n, n0, n1, r;
	uint32 dwStart, dwDelay;
	
	
	dwStart = Time_GetTick();
	
	// sqrt(2^32) = 2^16
	for(i=0;i<65535;i++)
	{
		n0 = i * i;
		n1 = (i + 1) * (i + 1);
		
		for(n=n0+1;n<n1;n+=7)
		{
			r = ISqrt(n);
			//if(r != i)
			//	printf("error: isqrt result for %u (found %u, should be %u)\n", n, r, i);
		}
		
		//if(i % 512 == 0)
		//	printf("progress: %d%%\r", (int)(n1 / (0xffffffff / 100)));
	}
	n0 = i * i;
	n1 = 0xffffffff;
	for(n=n0;n<n1;n++)
	{
		r = ISqrt(n);
		//if(r != i)
		//	printf("error: isqrt result for %u (found %u, should be %u)\n", n, r, i);
	}
	
	// ugly: discard warning from compiler
	r = r;
	
	dwDelay = Time_GetElapsedTick(dwStart);
	
	return dwDelay;
}


void Benchmark_Integers_SQRT(TISqrtProc ISqrt, char *Name)
{
	uint32 dwTick;
	dwTick = Benchmark_Integers_SQRT_Process(ISqrt);
	PrintResult(Name, dwTick);
}


uint32 Benchmark_Integers_Prime32_Process(void)
{
	uint32 i, dwStart, dwDelay;
	
	
	dwStart = Time_GetTick();
	
	for(i=1;i<12000000;i+=2)
		Math_IsPrime(i);
	
	dwDelay = Time_GetElapsedTick(dwStart);
	
	return dwDelay;
}


void Benchmark_Integers_Prime32(void)
{
	uint32 dwTick;
	dwTick = Benchmark_Integers_Prime32_Process();
	PrintResult("Math_IsPrime()[32bits]", dwTick);
}



uint32 Benchmark_Integers_Prime64_Process(void)
{
	uint64 i;
	uint32 dwStart, dwDelay;
	
	
	dwStart = Time_GetTick();
	
	for(i=8580000001ULL;i<8580330000ULL;i+=2)
		Math_IsPrime_64(i);
	
	dwDelay = Time_GetElapsedTick(dwStart);
	
	return dwDelay;
}


void Benchmark_Integers_Prime64(void)
{
	uint32 dwTick;
	dwTick = Benchmark_Integers_Prime64_Process();
	PrintResult("Math_IsPrime()[64bits]", dwTick);
}


uint32 Benchmark_Integers_Hash_Process(THash32Proc Hash, uint8 *Data, int lData)
{
	uint32 dwStart, dwDelay;
	int i, k, v, lDataMod = lData / 5;
	
	
	dwStart = Time_GetTick();
	
	for(i=0;i<275*1000;i++)
	{
		k = i % 13;
		v = (k + 1) * i % lDataMod;
		Hash(Data + v, lData - v);
	}
	
	dwDelay = Time_GetElapsedTick(dwStart);
	
	return dwDelay;
}


void Benchmark_Integers_Hash(THash32Proc Hash, char *Name)
{
	uint8 Data[12345], k, v;
	uint32 dwTick;
	int i;
	
	
	// initialize data
	srand(123);
	k = (uint8)(rand() % 123);
	for(i=0;i<12345;i++)
	{
		v = (uint8)(rand() % 123);
		Data[i] = v ^ k;
		k = (v - 1) ^ (k - 1);
	}
	
	// run benchmark
	dwTick = Benchmark_Integers_Hash_Process(Hash, Data, sizeof(Data));
	
	PrintResult(Name, dwTick);
}

void Benchmark_Integers(void)
{
	printf("   integers benchmark\n");
	
	// hash
	Benchmark_Integers_Hash(Hash_CRC32, "Hash_CRC32()");
	Benchmark_Integers_Hash(Hash_SQLITE, "Hash_SQLITE()");
	Benchmark_Integers_Hash(Hash_FNV32, "Hash_FNV32()");
	
	// prime numbers (32 & 64)
	Benchmark_Integers_Prime32();
	Benchmark_Integers_Prime64();
	
	// sqrt
	Benchmark_Integers_SQRT(Math_ISqrt_Dijkstra, "Math_ISqrt_Dijkstra()");
}


typedef struct
{
	THash32Proc	Hash;
	uint8 *		Data;
	int			szData;
	
}TThreadArg;


int Benchmark_Integers_Thread(void *Handle, void *Data)
{
	TThreadArg *Arg = (TThreadArg*)Data;
	uint32 dwTick;
	
	dwTick = Benchmark_Integers_Hash_Process(Arg->Hash, Arg->Data, Arg->szData);
	
	return (int)dwTick;
}


void Benchmark_Integers_Multithread(int nThreads)
{
	printf("   integers benchmark (%d threads)\n", nThreads);
	
	uint8 Data[12345], k, v;
	uint32 dwTick;
	void *Threads[nThreads];
	TThreadArg ThreadArgs;
	int i;
	
	
	// initialize data
	srand(123);
	k = (uint8)(rand() % 123);
	for(i=0;i<12345;i++)
	{
		v = (uint8)(rand() % 123);
		Data[i] = v ^ k;
		k = (v - 1) ^ (k - 1);
	}
	
	ThreadArgs.Hash = Hash_CRC32;
	ThreadArgs.Data = Data;
	ThreadArgs.szData = sizeof(Data);
	
	// run benchmark
	for(i=0;i<nThreads;i++)
		Threads[i] = Thread_Master_Open(Benchmark_Integers_Thread, (void*)&ThreadArgs);
	
	Time_Sleep(1000);
	
	for(i=0;i<nThreads;i++)
		Thread_Master_WaitThreadTermination(Threads[i], 500, 1);
	
	dwTick = 0;
	for(i=0;i<nThreads;i++)
		dwTick += Thread_Master_Close(Threads[i]);
	dwTick /= nThreads;
	
	PrintResult("Hash_CRC32()", dwTick);
}


uint32 Benchmark_Hash_Process(THashInitProc Init, THashUpdateProc Update, THashFinalProc Final, uint8 *Data, int lData)
{
	uint32 dwStart, dwDelay;
	int i, k, v, lDataMod = lData / 5;
	uint8 RawDigest[MAX(MD5_DIGEST_SIZE, SHA1_DIGEST_SIZE)];
	
	
	dwStart = Time_GetTick();
	
	for(i=0;i<80*1000;i++)
	{
		void *Handle;
		
		k = i % 13;
		v = (k + 1) * i % lDataMod;
		
		Handle = Init();
		Update(Handle, Data + v, lData - v);
		Final(Handle, RawDigest);
	}
	
	dwDelay = Time_GetElapsedTick(dwStart);
	
	return dwDelay;
}


void Benchmark_Hash_Array(THashInitProc Init, THashUpdateProc Update, THashFinalProc Final, char *Name)
{
	uint8 Data[12345], k, v;
	uint32 dwTick;
	int i;
	
	
	// initialize data
	srand(123);
	k = (uint8)(rand() % 123);
	for(i=0;i<12345;i++)
	{
		v = (uint8)(rand() % 123);
		Data[i] = v ^ k;
		k = (v - 1) ^ (k - 1);
	}
	
	// run benchmark
	dwTick = Benchmark_Hash_Process(Init, Update, Final, Data, sizeof(Data));
	
	PrintResult(Name, dwTick);
}


void Benchmark_Hash(void)
{
	printf("   hash benchmark\n");
	
	Benchmark_Hash_Array(Hash_MD5_Init, Hash_MD5_Update, Hash_MD5_Final, "Hash_MD5()");
	Benchmark_Hash_Array(Hash_SHA1_Init, Hash_SHA1_Update, Hash_SHA1_Final, "Hash_SHA1()");
}


void Benchmark_Memory_MALLOC()
{
	int i, j, n;
	uint8 *Segments[1024];
	uint32 dwStart, dwDelay;
	
	
	dwStart = Time_GetTick();
	
	for(n=1024;n<128*1024;n*=2)
	{
		for(j=0;j<64;j++)
		{
			for(i=0;i<1024;i++)
			{
				Segments[i] = malloc(n);
			}
			for(i=0;i<1024;i++)
				free(Segments[i]);
		}
	}
	dwDelay = Time_GetElapsedTick(dwStart);
	
	PrintResult("malloc/free", dwDelay);
}

void Benchmark_Memory_MALLOC_MEMCPY()
{
	int i, j, k, l, n;
	uint8 *Segments[1024], *pSegment;
	uint32 dwStart, dwDelay;
	uint8 Pattern[1024];
	
	
	for(i=0;i<1024;i++)
		Pattern[i] = (uint8)(((i % 127) ^ (i + 127)) % 251);
	
	dwStart = Time_GetTick();
	
	for(n=1024;n<128*1024;n*=2)
	{
		for(j=0;j<64;j++)
		{
			for(i=0;i<1024;i++)
			{
				pSegment = malloc(n);
				Segments[i] = pSegment;
				k = n / sizeof(Pattern);
				
				for(l=0;l<k;l++)
					memcpy(pSegment + l * sizeof(Pattern), Pattern, sizeof(Pattern));
			}
			
			for(i=0;i<1024;i++)
				free(Segments[i]);
		}
	}
	dwDelay = Time_GetElapsedTick(dwStart);
	
	PrintResult("malloc/free+memcpy", dwDelay);
}


void Benchmark_Memory(void)
{
	printf("   memory benchmark\n");
	
	// malloc
	Benchmark_Memory_MALLOC();
	Benchmark_Memory_MALLOC_MEMCPY();
}


void Benchmark_FPU_SQRTD(void)
{
	uint32 i, n, n0, n1, r;
	uint32 dwStart, dwDelay;
	
	
	dwStart = Time_GetTick();
	
	// sqrt(2^32) = 2^16
	for(i=0;i<20000;i+=3)
	{
		n0 = i * i;
		n1 = (i + 1) * (i + 1);
		
		//printf("range [%u,%u[\n", n0, n1);
		
		for(n=n0;n<n1;n++)
		{
			r = Math_ISqrt_FPU(n);
			if(r != i)
				printf("error: isqrt result for %u (found %u, should be %u)\n", n, r, i);
		}
		
		//if(i % 64 == 0)
		//	printf("progress: %d%%\r", (int)(i / (20000 / 100)));
	}
	
	dwDelay = Time_GetElapsedTick(dwStart);
	
	PrintResult("Math_ISqrt_FPU()", dwDelay);
}

void Benchmark_FPU(void)
{
	printf("   fpu benchmark\n");
	
	// sqrt
	Benchmark_FPU_SQRTD();
}


int main(void)
{
	TProcessorInfo Info;
	char Buffer[256];
	
	printf("cpu & system benchmark tool, build at %s\n", __DATE__);
	
	printf("compiler info\n");
	printf("   compiler version:         %s\n", System_GetCompilerVersion());
	printf("   compiler version major:   %d\n", System_GetCompilerVersionMajor());
	printf("   compiler version minor:   %d\n", System_GetCompilerVersionMinor());
	printf("   compiler version patch:   %d\n", System_GetCompilerVersionPatch());
	printf("   build date:               %s\n", System_GetBuildDate());
	printf("   build time:               %s\n", System_GetBuildTime());
	
	System_GetProcessorInfo(&Info);
	printf("processor info:\n");
	printf("   name:                     %s\n", Info.Name);
	printf("   vendor id:                %s\n", Info.VendorID);
	printf("   number of processors:     %d\n", Info.NumberOfProcessors);
	printf("   frequency:                %d MHz\n", Info.FrequencyMHz);
	printf("   family:                   %d\n", Info.Family);
	printf("   model:                    %d\n", Info.Model);
	printf("   stepping:                 %d\n", Info.Stepping);
	
	printf("memory info:\n");
	printf("   total physical memory:    %d kB\n", System_GetTotalPhysicalMemory());
	printf("   free physical memory:     %d kB\n", System_GetFreePhysicalMemory());
	
	printf("software info:\n");
	printf("   kernel version:           %s\n", System_GetKernelVersion());
	printf("   runtime version:          %s\n", System_GetRuntimeVersion());
	printf("   system version:           %s\n", System_GetSystemVersion());
	printf("   system arch:              %s\n", System_GetSystemArch());
	printf("   memory address bits:      %d bits\n", System_GetMemoryAddressBits());
	printf("   endianness:               %s\n", System_IsBigEndian() ? "big endian" : "little endian");
	
	printf("host info:\n");
	IP_GetDnsHostname(Buffer, sizeof(Buffer));
	printf("   dns hostname:             %s\n", Buffer);
	IP_GetDnsDomain(Buffer, sizeof(Buffer));
	printf("   dns domain:               %s\n", Buffer);
	
	printf("session info:\n");
	printf("   user:                     %s\n", System_GetCurrentUser());
	
	printf("\n");
	printf("benchmark beginning\n");
	
	Benchmark_Integers();
	Benchmark_Hash();
	
	/*
	Benchmark_FPU();
	Benchmark_Memory();
	
	Benchmark_Integers_Multithread(2);
	Benchmark_Integers_Multithread(4);
	Benchmark_Integers_Multithread(6);
	Benchmark_Integers_Multithread(8);
	*/
	
	printf("benchmark done\n");
	
	return 0;
}
