#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <libshred.h>


typedef struct
{
	char *	Hostname;
	void *	Thread;
	void *	Lock;
	uint32 *dwIPList;
	int		lIPList;
	int		Done;
	
}THostIPAsync;


static int _HostIPAsyncThread(void *Thread, void *Data)
{
	THostIPAsync *Ctx = (THostIPAsync*)Data;
	
	//LOG_TRACE();
	
	//Time_Sleep(2500);
	
	if(IP_GetHostIPEx(Ctx->Hostname, &Ctx->dwIPList, &Ctx->lIPList))
	{
		// nothing to do
		//LOG_DEBUG("ok");
	}
	else
	{
		LOG_DEBUG("error: IP_GetHostIPEx(): %d", Net_GetLastError());
	}
	
	CriticalSection_Enter(Ctx->Lock);
	// thread was previously requested to close, so delete all data
	if(Thread_Slave_GetCloseEvent(Thread))
	{
		//LOG_DEBUG("HostIPAsyncThread: self termination");
		
		CriticalSection_Leave(Ctx->Lock);
		CriticalSection_Close(Ctx->Lock);
		
		if(Ctx->dwIPList)
			free(Ctx->dwIPList);
		
		if(Ctx->Hostname)
			free(Ctx->Hostname);
		
		free(Ctx);
		
		Ctx = NULL;
		
		Thread_Slave_SelfDestruct(Thread);
		// should close thread internally
	}
	else
	{
		//LOG_DEBUG("HostIPAsyncThread: normal termination");
		Ctx->Done = 1;
		CriticalSection_Leave(Ctx->Lock);
	}
	
	//LOG_DEBUG("HostIPAsyncThread: thread exited");
	
	return 0;
}


void * Net_GetHostIPAsyncOpen(char *Hostname)
{
	THostIPAsync *Ctx;
	
	//LOG_TRACE();
	
	if(!Hostname)
		return NULL;
	
	Ctx = malloc(sizeof(THostIPAsync));
	memset(Ctx, 0, sizeof(THostIPAsync));
	
	Ctx->Hostname = strdup(Hostname);
	Ctx->Lock = CriticalSection_Open();
	Ctx->Thread = Thread_Master_Open(_HostIPAsyncThread, (void*)Ctx);
	
	return Ctx;
}


int Net_GetHostIPAsyncCheck(void * Handle, uint32 **pdwIPList, int *plIPList)
{
	THostIPAsync *Ctx = (THostIPAsync*)Handle;
	
	//LOG_TRACE();
	
	if(!Ctx)
		return -1;
	
	CriticalSection_Enter(Ctx->Lock);
	if(!Ctx->Done)
	{
		CriticalSection_Leave(Ctx->Lock);
		return 0;
	}
	
	*pdwIPList = Ctx->dwIPList;
	Ctx->dwIPList = NULL;
	*plIPList = Ctx->lIPList;
	
	CriticalSection_Leave(Ctx->Lock);
	
	if(!Ctx->lIPList)
		return -1;
	
	return 1;
}


void Net_GetHostIPAsyncClose(void * Handle)
{
	THostIPAsync *Ctx = (THostIPAsync*)Handle;
	
	//LOG_TRACE();
	
	if(Ctx)
	{
		CriticalSection_Enter(Ctx->Lock);
		if(Ctx->Done)
		{
			//LOG_DEBUG("child thread done");
			
			CriticalSection_Leave(Ctx->Lock);
			
			// normal close
			Thread_Master_SetCloseEvent(Ctx->Thread);
			Thread_Master_WaitThreadTermination(Ctx->Thread, 20, 1);
			Thread_Master_Close(Ctx->Thread);
			
			CriticalSection_Close(Ctx->Lock);
			
			if(Ctx->dwIPList)
				free(Ctx->dwIPList);
			
			if(Ctx->Hostname)
				free(Ctx->Hostname);
			
			free(Ctx);
		}
		else
		{
			//LOG_DEBUG("child thread not done");
			
			// let run
			Thread_Master_SetCloseEvent(Ctx->Thread);
			CriticalSection_Leave(Ctx->Lock);
		}
	}
}


int main(int nArgs, char **Args)
{
	char *Method = NULL;
	char *Hostname = NULL;
	int Timeout = 2500;
	int ID = -1;
	TCliParserOption Options[] = {
		{"-m", "--method", "method type", "gethostbyname,getaddrinfo,getaddrinfo.async", &Method, ARG_TYPE_STRING, 1},
		{"-h", "--hostname", "hostname", "ip|fqdn", &Hostname, ARG_TYPE_STRING, 1},
		{"-t", "--timeout", "timeout in ms.", "2500", &Timeout, ARG_TYPE_INT, 0},
		{"-i", "--id", "entry id to keep", "0|1...", &ID, ARG_TYPE_INT, 0},
		{NULL,	NULL,	NULL, NULL, 0, 0}
	};
	uint32 dwIP;
	char TmpBuf[20];
	int Success = 0;
	
	
	if(!CliParser_DoOptions(Options, Args + 1, nArgs - 1))
		return 0;
	
	Net_Init();
	
	if(strcmp(Method, "gethostbyname") == 0)
	{
		if(IP_GetHostIP(Hostname, &dwIP))
		{
			LOG_DEBUG("%s => %s", Hostname, IP_Uint32ToIP(dwIP, TmpBuf, sizeof(TmpBuf)));
			Success = 1;
		}
		else
		{
			LOG_DEBUG("error: IP_GetHostIP(): %d", Net_GetLastError());
		}
	}
	else if(strcmp(Method, "getaddrinfo") == 0)
	{
		uint32 *dwIPList = NULL;
		int i, lIPList = 0;
		
		if(IP_GetHostIPEx(Hostname, &dwIPList, &lIPList))
		{
			if(ID < 0)
			{
				LOG_DEBUG("%s => (%d) {", Hostname, lIPList);
				for(i=0;i<lIPList;i++)
				{
					if(i + 1 < lIPList)
						LOG_DEBUG("  %s,", IP_Uint32ToIP(dwIPList[i], TmpBuf, sizeof(TmpBuf)));
					else
						LOG_DEBUG("  %s", IP_Uint32ToIP(dwIPList[i], TmpBuf, sizeof(TmpBuf)));
				}
				LOG_DEBUG("}");
				
				Success = (lIPList > 0);
			}
			else
			{
				if(ID < lIPList)
				{
					LOG_DEBUG("%s => %s", Hostname, IP_Uint32ToIP(dwIPList[ID], TmpBuf, sizeof(TmpBuf)));
					Success = 1;
				}
			}
			
			free(dwIPList);
		}
		else
		{
			LOG_DEBUG("error: IP_GetHostIPEx(): %d", Net_GetLastError());
		}
	}
	else if(strcmp(Method, "getaddrinfo.async") == 0)
	{
		void *Handle = NULL;
		uint32 *dwIPList = NULL;
		int i, lIPList = 0, Result;
		
		Handle = Net_GetHostIPAsyncOpen(Hostname);
		if(!Handle)
		{
			LOG_DEBUG("error: Net_GetHostIPAsyncOpen()");
		}
		else
		{
			uint32 dwStart;
			int Delay;
			
			//Timeout
			dwStart = Time_GetTick();
			while(1)
			{
				Result = Net_GetHostIPAsyncCheck(Handle, &dwIPList, &lIPList);
				if(Result < 0)
				{
					LOG_DEBUG("error: Net_GetHostIPAsyncCheck()");
					break;
				}
				if(Result > 0)
				{
					if(ID < 0)
					{
						LOG_DEBUG("%s => (%d) {", Hostname, lIPList);
						for(i=0;i<lIPList;i++)
						{
							if(i + 1 < lIPList)
								LOG_DEBUG("  %s,", IP_Uint32ToIP(dwIPList[i], TmpBuf, sizeof(TmpBuf)));
							else
								LOG_DEBUG("  %s", IP_Uint32ToIP(dwIPList[i], TmpBuf, sizeof(TmpBuf)));
						}
						LOG_DEBUG("}");
						
						Success = (lIPList > 0);
					}
					else
					{
						if(ID < lIPList)
						{
							LOG_DEBUG("%s => %s", Hostname, IP_Uint32ToIP(dwIPList[ID], TmpBuf, sizeof(TmpBuf)));
							Success = 1;
						}
					}
					
					free(dwIPList);
					
					break;
				}
				
				Time_Sleep(50);
				
				Delay = (int)Time_GetElapsedTick(dwStart);
				if(Delay > Timeout)
				{
					LOG_DEBUG("error: timeout");
					break;
				}
			}
			
			Net_GetHostIPAsyncClose(Handle);
			
			//Time_Sleep(2000);
		}
	}
	else
	{
		LOG_DEBUG("error: invalid method()");
	}
	
	Net_Cleanup();
	
	return !Success;
}
