/*
	Telnet server
	
	Based of busybox 1.00 telnetd
	
	Bjorn Wesen, Axis Communications AB (bjornw@axis.com) 2000
	Vladimir Oleynik <dzo@simtreas.ru> 2001
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <sys/ioctl.h>
#include <netinet/in.h>
#include <signal.h>
#include <termios.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <arpa/telnet.h>
#include <ctype.h>
#include <net/if.h>

#include <libshred.h>

/// features
//#define HAVE_PTMX

#define TTY_NAME_SIZE			32
#define TELNET_PORT_DEFAULT		23
#define LOGIN_PROGRAM_DEFAULT	"/bin/login"


typedef struct TSession
{
	uint32 dwIP;
	uint16 wPort;
	
	int Socket;
	int PTY;
	
	int ShellPID;
	/* two circular buffers */
	char *Buf1, *Buf2;
	int ReadID1, WriteID1, Size1;
	int ReadID2, WriteID2, Size2;
	
	struct TSession *Next;
	
}TSession;

#define BUFSIZE 4000
#define TSESSION_SIZE	(sizeof(TSession) + BUFSIZE * 2)


#if HAVE_PTMX
extern int grantpt(int fd);
extern int unlockpt(int fd);
extern char *ptsname(int __fd);
#endif

/*
   This is how the buffers are used. The arrows indicate the movement
   of data.

   +-------+     WriteID1++     +------+     ReadID1++     +----------+
   |       | <--------------  | Buf1 | <--------------  |          |
   |       |     Size1--      +------+     Size1++      |          |
   |  pty  |                                            |  socket  |
   |       |     ReadID2++     +------+     WriteID2++     |          |
   |       |  --------------> | Buf2 |  --------------> |          |
   +-------+     Size2++      +------+     Size2--      +----------+

   Each session has got two buffers.
*/

static TSession *SessionsRoot = NULL;
static int FDMax;


static char *GetErrorString(void)
{
	static char String[256];
	char ErrorString[256];
	int ErrorID;
	
	ErrorID = System_GetLastErrorID();
	snprintf(String, sizeof(String) - 1, "%u: %s", ErrorID, System_GetLastErrorString(ErrorID, ErrorString, sizeof(ErrorString)));
	
	return String;
}


/*
	Remove all IAC's from the buffer pointed to by bf (recieved IACs are ignored
	and must be removed so as to not be interpreted by the terminal).  Make an
	uninterrupted string of characters fit for the terminal.  Do this by packing
	all characters meant for the terminal sequentially towards the end of bf.
	
	Return a pointer to the beginning of the characters meant for the terminal.
	and make *nToTTY the number of characters that should be sent to
	the terminal.
	
	Note - If an IAC (3 byte quantity) starts before (bf + len) but extends
	past (bf + len) then that IAC will be left unprocessed and *processed will be
	less than len.
	
	FIXME - if we mean to send 0xFF to the terminal then it will be escaped,
	what is the escape character?  We aren't handling that situation here.
	
	CR-LF ->'s CR mapping is also done here, for convenience
*/
static char *RemoveIAC(TSession *pSession, int *pnToTTY)
{
	uint8 *ptr0 = (uint8*)pSession->Buf1 + pSession->WriteID1;
	uint8 *ptr = ptr0;
	uint8 *totty = ptr;
	uint8 *end = ptr + MIN(BUFSIZE - pSession->WriteID1, pSession->Size1);
	int processed;
	int nToTTY;
	
	while(ptr < end)
	{
		if(*ptr != IAC)
		{
			int c = *ptr;
			*totty++ = *ptr++;
			/*
				We now map \r\n ==> \r for pragmatic reasons.
				Many client implementations send \r\n when
				the user hits the CarriageReturn key.
			*/
			if(c == '\r' && (*ptr == '\n' || *ptr == 0) && ptr < end)
				ptr++;
		}
		else
		{
			/*
				TELOPT_NAWS support!
			*/
			if((ptr+2) >= end)
			{
				/*
					only the beginning of the IAC is in the
					buffer we were asked to process, we can't
					process this char.
				*/
				break;
			}
			/*
				IAC -> SB -> TELOPT_NAWS -> 4-byte -> IAC -> SE
			*/
			else if(ptr[1] == SB && ptr[2] == TELOPT_NAWS)
			{
				struct winsize ws;
				if((ptr+8) >= end)
					break; 	// incomplete, can't process
				ws.ws_col = (ptr[3] << 8) | ptr[4];
				ws.ws_row = (ptr[5] << 8) | ptr[6];
				ioctl(pSession->PTY, TIOCSWINSZ, (char *)&ws);
				ptr += 9;
			}
			else
			{
				/* skip 3-byte IAC non-SB cmd */
				ptr += 3;
			}
		}
	}
	
	processed = ptr - ptr0;
	nToTTY = totty - ptr0;
	/*
		the difference between processed and num_to tty
		is all the iacs we removed from the stream.
		Adjust Buf1 accordingly.
	*/
	pSession->WriteID1 += processed - nToTTY;
	pSession->Size1 -= processed - nToTTY;
	*pnToTTY = nToTTY;
	
	// move the chars meant for the terminal towards the end of the buffer.
	return memmove(ptr - nToTTY, ptr0, nToTTY);
}




static int GetPTY(char TTYName[TTY_NAME_SIZE], int *pFD)
{
#if HAVE_PTMX
	int FD = open("/dev/ptmx", 2);
	if(FD > 0)
	{
		grantpt(FD);
		unlockpt(FD);
		strcpy(TTYName, ptsname(FD));
		*pFD = FD;
		return 1;
	}
#else
	char Charset[] = "pqrstuvwxyzabcde";
	struct stat stb;
	int i, j, FD, lCharset;
	
	
	lCharset = strlen(Charset);
	strcpy(TTYName, "/dev/ptyXX");
	for(i=0;i<lCharset;i++)
	{
		TTYName[8] = Charset[i];
		TTYName[9] = '0';
		
		if(stat(TTYName, &stb) < 0)
			continue;
		
		for(j=0;j<lCharset;j++)
		{
			TTYName[9] = j < 10 ? j + '0' : j - 10 + 'a';
			FD = open(TTYName, O_RDWR | O_NOCTTY);
			if(FD >= 0)
			{
				TTYName[5] = 't';
				*pFD = FD;
				return 1;
			}
		}
	}
#endif
	return 0;
}


static void SendIAC(TSession *pSession, uint8 command, int option)
{
	// We rely on that there is space in the buffer for now.
	char *b = pSession->Buf2 + pSession->ReadID2;
	*b++ = IAC;
	*b++ = command;
	*b++ = option;
	pSession->ReadID2 += 3;
	pSession->Size2 += 3;
}


static TSession *Session_Create(int Socket, char *LoginProgram, uint32 dwIP, uint16 wPort)
{
	char TTYName[TTY_NAME_SIZE];
	int PTY, PID;
	TSession *pSession;
	char TmpBuf[64];
	
	
	LOG_DEBUG("new connection from %s", Net_AddrToString(dwIP, wPort, TmpBuf, sizeof(TmpBuf)));
	
	pSession = malloc(TSESSION_SIZE);
	memset(pSession, 0, TSESSION_SIZE);
	
	pSession->dwIP = dwIP;
	pSession->wPort = wPort;
	
	pSession->Socket = Socket;
	
	pSession->Buf1 = (char *)(&pSession[1]);
	pSession->Buf2 = pSession->Buf1 + BUFSIZE;
	
	pSession->ReadID1 = pSession->WriteID1 = pSession->Size1 = 0;
	pSession->ReadID2 = pSession->WriteID2 = pSession->Size2 = 0;
	
	/* Got a new connection, set up a tty and spawn a shell.  */
	
	if(!GetPTY(TTYName, &PTY))
	{
		LOG_DEBUG("error: all network ports in use");
		return NULL;
	}
	LOG_DEBUG("pty: %s (fd: %d)", TTYName, PTY);
	
	if(PTY > FDMax)
		FDMax = PTY;
	
	pSession->PTY = PTY;
	
	/* Make the telnet client understand we will echo characters so it
	 * should not do it locally. We don't tell the client to run linemode,
	 * because we want to handle line editing and tab completion and other
	 * stuff that requires char-by-char support.
	 */
	
	SendIAC(pSession, DO, TELOPT_ECHO);
	SendIAC(pSession, DO, TELOPT_NAWS);
	SendIAC(pSession, DO, TELOPT_LFLOW);
	SendIAC(pSession, WILL, TELOPT_ECHO);
	SendIAC(pSession, WILL, TELOPT_SGA);
	
	PID = fork();
	if(PID < 0)
	{
		LOG_DEBUG("error: fork(): %s", GetErrorString());
	}
	if(PID == 0)
	{
		/* In child, open the child's side of the tty.  */
		char *LoginArgs[] = {LoginProgram, NULL};
		int i;
		
		// close all parent fd, included stdin, stdout, stderr
		for(i=0;i<=FDMax;i++)
			close(i);
		
		// make new process group
		setsid();
		
		if(open(TTYName, O_RDWR /*| O_NOCTTY*/) < 0)
		{
			LOG_DEBUG("error: could not open tty: %s", GetErrorString());
			exit(1);
		}
		dup(0);
		dup(0);
		
		tcsetpgrp(0, getpid());
		
		/*
			The pseudo-terminal allocated to the client is configured to operate in
			cooked mode, and with XTABS CRMOD enabled (see tty(4)).
		*/
		
		{
			struct termios TermBuf;
			
			tcgetattr(0, &TermBuf);
			TermBuf.c_lflag |= ECHO;	// if we use readline we dont want this
			TermBuf.c_oflag |= ONLCR|XTABS;
			TermBuf.c_iflag |= ICRNL;
			TermBuf.c_iflag &= ~IXOFF;
			/*TermBuf.c_lflag &= ~ICANON;*/
			tcsetattr(0, TCSANOW, &TermBuf);
		}
		
		// exec shell, with correct argv and env
		execv(LoginProgram, (char * const *)LoginArgs);
		
		// NOT REACHED
		LOG_DEBUG("error: execv(): %s", GetErrorString());
		exit(1);
	}
	
	pSession->ShellPID = PID;
	
	return pSession;
}


static void Session_Destroy(TSession *pSession)
{
	TSession *pCurrentSession = SessionsRoot;
	char TmpBuf[64];
	
	LOG_DEBUG("connection to %s closed", Net_AddrToString(pSession->dwIP, pSession->wPort, TmpBuf, sizeof(TmpBuf)));
	
	// Unlink this telnet session from the session list
	if(pCurrentSession == pSession)
	{
		SessionsRoot = pSession->Next;
	}
	else
	{
		while(pCurrentSession->Next != pSession)
			pCurrentSession = pCurrentSession->Next;
		pCurrentSession->Next = pSession->Next;
	}
	
	kill(pSession->ShellPID, SIGKILL);
	
	wait4(pSession->ShellPID, NULL, 0, NULL);
	
	close(pSession->PTY);
	Net_CloseSocket(&pSession->Socket);
	
	if(pSession->PTY == FDMax || pSession->Socket == FDMax)
		FDMax--;
	if(pSession->PTY == FDMax || pSession->Socket == FDMax)
		FDMax--;
	
	free(pSession);
}


static int DoExit = 0;

static void Signal_Quit(int SignalNo)
{
	LOG_DEBUG("exiting due to signal %d", SignalNo);
	DoExit = 1;
}


int main(int nArgs, char **Args)
{
	int DoDebug = 0;
	int Port = TELNET_PORT_DEFAULT;
	char *LoginProgram = LOGIN_PROGRAM_DEFAULT;
	char *Interface = NULL;
	TCliParserOption Options[] = {
		{"-p", "--port", "server port", "23", &Port, ARG_TYPE_INT, 0},
		{"-l", "--login-program", "login program", "/bin/login,/bin/sh", &LoginProgram, ARG_TYPE_STRING, 0},
		{"-i", "--interface", "bind to interface", "ethX", &Interface, ARG_TYPE_STRING, 0},
		{"-d", "--debug", "run in foreground", "", &DoDebug, ARG_TYPE_FLAG, 0},
		{NULL,	NULL,	NULL, NULL, 0, 0}
	};
	int ListenSocket = -1;
	
	
	if(!CliParser_DoOptions(Options, Args + 1, nArgs - 1))
		return 0;
	
	{
		TLog Log;
		
		Log_Reset(&Log);
		Log_SetMode(&Log, LOG_AUTO_FLUSH | LOG_THREAD_SAFE | LOG_SHOW_DATE | LOG_SHOW_FILE | LOG_SHOW_LINE | LOG_SHOW_FUNCTION);
		Log_SetFile(&Log, stdout);
		Log_SetPadding(&Log, 70);
		Log_SetDefault(&Log);
	}
	
	Net_Init();
	
	if(access(LoginProgram, X_OK) < 0)
	{
		LOG_DEBUG("error: login path not executable: %s", LoginProgram);
		exit(1);
	}
	
	// prepare main TCP socket
	
	ListenSocket = Net_OpenTCPSocket();
	if(ListenSocket < 0)
	{
		LOG_DEBUG("error: Net_OpenTCPSocket()");
		exit(1);
	}
	
	if(!Net_BindSocket(ListenSocket, 0, Net_ToNet16(Port)))
	{
		LOG_DEBUG("error: Net_BindSocket()");
		exit(1);
	}
	
	if(Interface)
	{
		if(!Net_Option_BindToDevice(ListenSocket, Interface))
		{
			LOG_DEBUG("error: binding to interface %s", Interface);
			exit(1);
		}
		LOG_DEBUG("bound to interface %s", Interface);
	}
	
	if(!Net_Listen(ListenSocket, 4))
	{
		LOG_DEBUG("error: Net_Listen()");
		exit(1);
	}
	LOG_DEBUG("listening on tcp:0.0.0.0:%d", Port);
	
	// daemonize
	
	if(!DoDebug)
	{
		LOG_DEBUG("running in background");
		
		if(!System_Daemonize())
		{
			LOG_DEBUG("error: System_Daemonize()");
			exit(-1);
		}
		
		if(!System_DupOutputToFile("/var/log/telnetd.log", 1, 1))
		{
			LOG_DEBUG("error: System_DupOutputToFile()");
			exit(-1);
		}
	}
	else
	{
		LOG_DEBUG("running in foreground");
		
		signal(SIGINT, Signal_Quit);
		signal(SIGTERM, Signal_Quit);
		signal(SIGQUIT, Signal_Quit);
	}
	
	FDMax = ListenSocket;
	
	// main loop
	while(1)
	{
		TSession *pSession;
		fd_set ReadFDSet, WriteFDSet;
		
		if(DoExit)
			break;
		
		FD_ZERO(&ReadFDSet);
		FD_ZERO(&WriteFDSet);
		
		/*
			select on the master socket, all telnet sockets and their
			ptys if there is room in their respective session buffers.
		*/
		
		FD_SET(ListenSocket, &ReadFDSet);
		
		pSession = SessionsRoot;
		while(pSession)
		{
			/*
				Buf1 is used from socket to pty
				Buf2 is used from pty to socket
			*/
			if(pSession->Size1 > 0)
				FD_SET(pSession->PTY, &WriteFDSet);  /* can write to pty */
			if(pSession->Size1 < BUFSIZE)
				FD_SET(pSession->Socket, &ReadFDSet); /* can read from socket */
			if(pSession->Size2 > 0)
				FD_SET(pSession->Socket, &WriteFDSet); /* can write to socket */
			if(pSession->Size2 < BUFSIZE)
				FD_SET(pSession->PTY, &ReadFDSet);  /* can read from pty */
			pSession = pSession->Next;
		}
		
		{
			int Result = select(FDMax + 1, &ReadFDSet, &WriteFDSet, 0, 0);
			//LOG_DEBUG("select result: %d", Result);
			if(Result <= 0)
				continue;
		}
		
		/* First check for and accept new sessions.  */
		if(FD_ISSET(ListenSocket, &ReadFDSet))
		{
			int Socket = -1;
			uint32 dwIP;
			uint16 wPort;
			
			if(!Net_Accept(ListenSocket, &Socket, &dwIP, &wPort))
				continue;
			
			// Create a new session and link it into our active list.
			TSession *pNewSession = Session_Create(Socket, LoginProgram, dwIP, wPort);
			if(!pNewSession)
			{
				LOG_DEBUG("error: Session_Create()");
				Net_CloseSocket(&Socket);
			}
			else
			{
				pNewSession->Next = SessionsRoot;
				SessionsRoot = pNewSession;
				if(Socket > FDMax)
					FDMax = Socket;
			}
			continue;
		}
		
		// Then check for data tunneling
		pSession = SessionsRoot;
		while(pSession)
		{
			TSession *pSessionNext = pSession->Next;	// in case we free pSession
			
			if(pSession->Size1 && FD_ISSET(pSession->PTY, &WriteFDSet))
			{
				// Write to pty from buffer 1
				int szWrite, nToTTY;
				char *pBuf;
				
				pBuf = RemoveIAC(pSession, &nToTTY);
				szWrite = write(pSession->PTY, pBuf, nToTTY);
				if(szWrite < 0)
				{
					LOG_TRACE();
					Session_Destroy(pSession);
					pSession = pSessionNext;
					continue;
				}
				pSession->WriteID1 += szWrite;
				pSession->Size1 -= szWrite;
				if(pSession->WriteID1 == BUFSIZE)
					pSession->WriteID1 = 0;
			}
			
			if(pSession->Size2 && FD_ISSET(pSession->Socket, &WriteFDSet))
			{
				// Write to socket from buffer 2
				int szWrite, szToWrite;
				
				szToWrite = MIN(BUFSIZE - pSession->WriteID2, pSession->Size2);
				szWrite = write(pSession->Socket, pSession->Buf2 + pSession->WriteID2, szToWrite);
				if(szWrite < 0)
				{
					LOG_TRACE();
					Session_Destroy(pSession);
					pSession = pSessionNext;
					continue;
				}
				pSession->WriteID2 += szWrite;
				pSession->Size2 -= szWrite;
				if(pSession->WriteID2 == BUFSIZE)
					pSession->WriteID2 = 0;
			}
			
			if(pSession->Size1 < BUFSIZE && FD_ISSET(pSession->Socket, &ReadFDSet))
			{
				// Read from socket to buffer 1
				int szRead, szToRead;
				
				szToRead = MIN(BUFSIZE - pSession->ReadID1, BUFSIZE - pSession->Size1);
				szRead = read(pSession->Socket, pSession->Buf1 + pSession->ReadID1, szToRead);
				if(!szRead || (szRead < 0 && errno != EINTR))
				{
					LOG_TRACE();
					Session_Destroy(pSession);
					pSession = pSessionNext;
					continue;
				}
				if(!*(pSession->Buf1 + pSession->ReadID1 + szRead - 1))
				{
					szRead--;
					if(!szRead)
						continue;
				}
				pSession->ReadID1 += szRead;
				pSession->Size1 += szRead;
				if(pSession->ReadID1 == BUFSIZE)
					pSession->ReadID1 = 0;
			}
			
			if(pSession->Size2 < BUFSIZE && FD_ISSET(pSession->PTY, &ReadFDSet))
			{
				// Read from pty to buffer 2
				int szRead, szToRead;
				
				szToRead = MIN(BUFSIZE - pSession->ReadID2, BUFSIZE - pSession->Size2);
				szRead = read(pSession->PTY, pSession->Buf2 + pSession->ReadID2, szToRead);
				if(!szRead || (szRead < 0 && errno != EINTR))
				{
					Session_Destroy(pSession);
					pSession = pSessionNext;
					continue;
				}
				pSession->ReadID2 += szRead;
				pSession->Size2 += szRead;
				if(pSession->ReadID2 == BUFSIZE)
					pSession->ReadID2 = 0;
			}
			
			if(pSession->Size1 == 0)
			{
				pSession->ReadID1 = 0;
				pSession->WriteID1 = 0;
			}
			if(pSession->Size2 == 0)
			{
				pSession->ReadID2 = 0;
				pSession->WriteID2 = 0;
			}
			
			pSession = pSessionNext;
		}
	}
	
	// todo : cleanup structs and sockets
	Net_Cleanup();
	
	return 0;
}
