#ifndef ___INC_NET_HELPERS_H___
#define ___INC_NET_HELPERS_H___

#ifdef WIN32
	#include <winsock2.h>
	#include <ws2tcpip.h>
	#include <windows.h>
	#define socklen_t int
	#define Net_GetLastError()	(WSAGetLastError())	//	winsock error code: http://msdn2.microsoft.com/en-us/library/ms740668.aspx
#else
	#include <unistd.h>
	#include <netinet/in.h>
	#include <arpa/inet.h>
	#include <errno.h>
	#define Net_GetLastError()	(errno)
#endif

#define Net_ToNet16(v)	(htons(v))
#define Net_ToNet32(v)	(htonl(v))
#define Net_ToHost16(v)	(ntohs(v))
#define Net_ToHost32(v)	(ntohl(v))

// the following values are in host order...
#define NET_ADDR_ANY			((uint32)0x00000000)	// Address to accept any incoming messages
#define NET_ADDR_BROADCAST		((uint32)0xffffffff)	// Address to send to all hosts
#define NET_ADDR_LOOPBACK		((uint32)0x7f000001)	// Address to loopback in software to local host: 127.0.0.1


void	Net_Init(void);
void	Net_Cleanup(void);

/*
	Net_Uint32ToIP()
		Converts an uint32 IP address into a string.
	
	Parameters
		dwIP [in]
			Network order uint32 IP address
		TmpBuf [out]
			Buffer to write the IP address, must be at least 16 bytes long: "xxx.xxx.xxx.xxx"
		szTmpBuf [in]
			Size of TmpBuf
	
	Return value
		returns TmpBuf
*/
char *	Net_Uint32ToIP(uint32 dwIP, char *TmpBuf, int szTmpBuf);

/*
	Net_IPToUint32()
		Converts a string IP address into an uint32 value.
	
	Parameters
		IPString [in]
			IP address string
		pdwIP [out]
			point to a network order uint32 IP address
	
	Return value
		returns 1 if successful, 0 otherwise
*/
int		Net_IPToUint32(char *IPString, uint32 *pdwIP);

/*
	Net_AddrToString()
		Converts an uint32 IP address and uint16 port into a string.
	
	Parameters
		dwIP [in]
			Network order uint32 IP address
		wPort [in]
			Network order uint16 port
		TmpBuf [out]
			Buffer to write the IP address, must be at least 22 bytes long: "xxx.xxx.xxx.xxx:xxxxx"
		szTmpBuf [in]
			Size of TmpBuf
	
	Return value
		returns TmpBuf
*/
char *	Net_AddrToString(uint32 dwIP, uint16 wPort, char *TmpBuf, int szTmpBuf);

/*
	Net_StringToAddr()
		Converts a string host into separated uint32 IP address and uint16 port.
	
	Parameters
		Host [in]
			String, format: <hostname|ip>[:port]
		pdwIP [out]
			pointer to a network order uint32 IP address
		pwPort [out]
			pointer to a network order uint16 port
*/
void	Net_StringToAddr(char *Host, uint32 *pdwIP, uint16 *pwPort);

/*
	Net_OpenUDPSocket()
	Net_OpenTCPSocket()
		Returns a new opened socket, -1 on error (should never happen)
*/
int		Net_OpenUDPSocket(void);
int		Net_OpenTCPSocket(void);

/*
	Net_CloseSocket()
		Closes a socket
	
	Parameters
		pSocket [in,out]
			pointer to a socket, the value will be set to -1
*/
void	Net_CloseSocket(int *pSocket);

/*
	Net_GetSocketType()
		Get the type of a socket
	
	Parameters
		Socket [in]
			socket
	
	Return value
		SOCKET_TYPE_TCP if socket is TCP
		SOCKET_TYPE_UDP if socket is UDP
		SOCKET_TYPE_NONE otherwise
*/
#define SOCKET_TYPE_NONE	0
#define SOCKET_TYPE_TCP		1
#define SOCKET_TYPE_UDP		2
int		Net_GetSocketType(int Socket);

/*
	Net_GetSocketError()
		Get the error related to a socket
	
	Parameters
		Socket [in]
			socket
		pError [out]
			error id
	
	Return value
		returns 1 if successful, 0 otherwise
*/
int		Net_GetSocketError(int Socket, int *pError);

/*
	Net_Option_*()
		Set various socket options.
		uint32 IP addresses are in network byte order
		Timeouts are in ms
	
	Return value
		returns 1 if successful, 0 otherwise
*/
int		Net_Option_SetBlock(int Socket, int Enable);
int		Net_Option_SetTimeout(int Socket, int RecvTimeout, int SendTimeout);
#define Net_Option_SetRecvTimeout(s, t)	(Net_Option_SetTimeout(s, t, -1))
#define Net_Option_SetSendTimeout(s, t)	(Net_Option_SetTimeout(s, -1, t))
int		Net_Option_SetBroadcast(int Socket, int Enable);
int		Net_Option_SetKeepAlive(int Socket, int Enable);
int		Net_Option_SetTcpNoDelay(int Socket, int Enable);
int		Net_Option_SetMulticastTTL(int Socket, int TTL);
int		Net_Option_SetMulticastIF(int Socket, uint32 dwIP);
int		Net_Option_AddMulticastMembership(int Socket, uint32 dwIFaceIP, uint32 dwMulticastIP);
int		Net_Option_DropMulticastMembership(int Socket, uint32 dwIFaceIP, uint32 dwMulticastIP);
int		Net_Option_SetRecvBufferSize(int Socket, int Size);
int		Net_Option_SetSendBufferSize(int Socket, int Size);
int		Net_Option_BindToDevice(int Socket, char *Device);	// must be root, use NULL Device to remove binding
int		Net_Option_SetDoNotRoute(int Socket, int Enable);

/*
	Net_Connect()
	Net_ConnectWithTimeout()
		Connect the socket to a remote host.
		The Net_Connect() function blocks until the connection is successful or an error happens.
		The Net_ConnectWithTimeout blocks until the connection is successful, an error or a timeout happens.
		If Timeout is 0, Net_ConnectWithTimeout() is equivalent to Net_Connect()
	
	Parameters
		Socket [in]
			socket
		dwIP [in]
			network order uint32 IP address of the host to connect to
		wPort [in]
			network order uint16 port of the host to connect to
		Timeout [in]
			connection timeout, in ms
	
	Return value
		returns 1 if successful, 0 otherwise
*/
int 	Net_Connect(int Socket, uint32 dwIP, uint16 wPort);
int		Net_ConnectWithTimeout(int Socket, uint32 dwIP, uint16 wPort, int Timeout);

/*
	Net_ConnectNonBlock()
		Connect the socket to a remote host.
		This function is non bloquant. The function Net_CheckNonBlockConnect() must be used to check the connection state.
		The given socket is set to non bloquant state.
	
	Parameters
		Socket [in]
			socket
		dwIP [in]
			network order uint32 IP address of the host to connect to
		wPort [in]
			network order uint16 port of the host to connect to
	
	Return value
		returns 1 if successful, 0 otherwise
*/
int 	Net_ConnectNonBlock(int Socket, uint32 dwIP, uint16 wPort);

/*
	Net_CheckNonBlockConnect()
		Check the connection state of a connecting socket (after a call to Net_ConnectNonBlock()).
		This function returns immediately.
		The given socket is set to bloquant state if the socket is connected.
	
	Parameters
		Socket [in]
			socket
	
	Return value
		1 if the socket is connected
		-1 if an error occured
		0 otherwise
*/
int		Net_CheckNonBlockConnect(int Socket);

/*
	Net_BindSocket()
	Net_BindSocketEx()
		Bind the given socket to the IP address/port.
		Net_BindSocket() will explicitly reuse the address.
		Net_BindSocketEx() will use the given parameter.
		Use 0 for dwIP and wPort to automatically register an address.
	
	Parameters
		Socket [in]
			socket
		dwIP [in]
			network order uint32 IP address of the bound address
		wPort [in]
			network order uint16 port of the bound address
	
	Return value
		returns 1 if successful, 0 otherwise
*/
int 	Net_BindSocketEx(int Socket, uint32 dwIP, uint16 wPort, int ReuseAddr);
#define Net_BindSocket(s, ip, p)	Net_BindSocketEx(s, ip, p, 1)

/*
	Net_GetSockName()
	Net_GetPeerName()
		Get the local/remote socket address.
		The socket should be bound/connected, otherwise returned values will be 0
	
	Parameters
		Socket [in]
			socket
		pdwIP [in]
			pointer to a network order uint32 IP address of the bound address
		pwPort [in]
			pointer to a network order uint16 port of the bound address
	
	Return value
		returns 1 if successful, 0 otherwise
*/
int		Net_GetSockName(int Socket, uint32 *pdwIP, uint16 *pwPort);
int		Net_GetPeerName(int Socket, uint32 *pdwIP, uint16 *pwPort);

/*
	Net_Listen()
		Set listen state for a TCP socket.
	
	Parameters
		Socket [in]
			socket
		Backlog [in]
			maximum length of the queue of pending connections, <= 0 to use the default value
	
	Return value
		returns 1 if successful, 0 otherwise
*/
int		Net_Listen(int Socket, int Backlog);

/*
	Net_Accept()
	Net_AcceptWithTimeout()
		Check accept state for a TCP socket.
		The Net_Accept() function blocks until the connection is successful or an error happens.
		The Net_AcceptWithTimeout blocks until the connection is successful, an error or a timeout happens.
		If Timeout is < 0, Net_AcceptWithTimeout() is equivalent to Net_Accept()
	
	Parameters
		Socket [in]
			socket
	
	Return value
		returns 1 if successful, 0 otherwise
*/
int 	Net_Accept(int ListenSocket, int *pConnSocket, uint32 *pdwIP, uint16 *pwPort);
int		Net_AcceptWithTimeout(int ListenSocket, int *pConnSocket, int Timeout, uint32 *pdwIP, uint16 *pwPort);

/*
	Net_SendTo()
	Net_Write()
	Net_WriteBlock()
		Write functions.
		Net_WriteBlock works only for TCP sockets. It blocks until szBuffer bytes are written, or an error occurred.
	
	Parameters
		Socket [in]
			socket
		Buffer [in]
			socket
		szBuffer [in]
			socket
		dwIP [in]
			network order uint32 IP address of the remote host
		wPort [in]
			network order uint16 port of the remote host
	
	Return value
		returns the number of sent bytes, can be less than szBuffer.
		return <= 0 is an error
*/
int		Net_SendTo(int Socket, uint8 *Buffer, int szBuffer, uint32 dwIP, uint16 wPort);
int		Net_Write(int Socket, uint8 *Buffer, int szBuffer);
int		Net_WriteBlock(int Socket, uint8 *Buffer, int szBuffer);

/*
	Net_RecvFrom()
	Net_Read()
	Net_ReadBlock()
		Read functions.
		Net_ReadBlock works only for TCP sockets. It blocks until szBuffer bytes are read, or an error occurred.
	
	Parameters
		Socket [in]
			socket
		Buffer [out]
			socket
		szBuffer [in]
			socket
		pdwIP [out]
			pointer to a network order uint32 IP address of the remote host
		pwPort [out]
			pointer to a network order uint16 port of the remote host
	
	Return value
		returns the number of received bytes, can be less than szBuffer.
		return <= 0 is an error
*/
int		Net_RecvFrom(int Socket, uint8 *Buffer, int szBuffer, uint32 *pdwIP, uint16 *pwPort);
int		Net_Read(int Socket, uint8 *Buffer, int szBuffer);
int		Net_ReadBlock(int Socket, uint8 *Buffer, int szBuffer);


#define NET_OP_NONE			0
#define NET_OP_ACCEPT		1
#define NET_OP_CONNECT		2
#define NET_OP_READ			3
#define NET_OP_WRITE		4

#undef FD_SETSIZE
#define FD_SETSIZE	1024	// maximum number of sockets that can be used in Net_Select_* functions.

#ifdef WIN32
typedef struct
{
	uint32	Count;
	int		Socket[FD_SETSIZE];
	
}TFDSet;
#endif

typedef struct
{
	int		FD;
	uint8	OpFlag;
	
}TNetSelectFD;

typedef struct
{
	TNetSelectFD	FD[FD_SETSIZE];
	int				lFD;
	
#ifdef WIN32
	TFDSet			ReadFD;
	TFDSet			WriteFD;
	TFDSet			ExceptFD;
#else
	fd_set			ReadFD;
	fd_set			WriteFD;
	fd_set			ExceptFD;
#endif
	
}TNetSelect;

/*
	Net_Select_Reset()
		Clear a TNetSelect structure.
	
	Parameters
		NetSelect [in]
			pointer to a TNetSelect structure.
*/
void	Net_Select_Reset(TNetSelect *NetSelect);

/*
	Net_Select_Add()
		Add a socket to a TNetSelect structure.
		On linux systems, file descriptors are allowed.
	
	Parameters
		NetSelect [in]
			pointer to a TNetSelect structure.
		FD [in]
			a socket, or a file descriptor (under linux)
		Flag [in]
			one of these flags:
			NET_OP_ACCEPT		the socket accept state will be checked
			NET_OP_CONNECT		the socket connect state will be checked (after a call to Net_ConnectNonBlock())
			NET_OP_READ			the socket read state will be checked
			NET_OP_WRITE		the socket write state will be checked
*/
void	Net_Select_Add(TNetSelect *NetSelect, int FD, uint8 Flag);

/*
	Net_Select()
		Wait for an event on any of the given NetSelect structure.
	
	Parameters
		NetSelect [in]
			pointer to a TNetSelect structure.
		Timeout [in]
			time to wait, in ms
			if Timeout is < 0, the function blocks infinitely.
	
	Return value
		returns the total number of socket that have a known state, according to the previously given Flag
		return 0 if a timeout occurs or signal is caught
		return -1 if an error occurs
	
	Remarks
		If the NetSelect structure is empty, the function returns immediatly.
		The function can return 0 before Timeout is reached, when a signal is catched (on linux only)
*/
int		Net_Select(TNetSelect *NetSelect, int Timeout);

/*
	Net_Select_Check()
		Checks the event on a socket.
	
	Parameters
		NetSelect [in]
			pointer to a TNetSelect structure.
		FD [in]
			a socket, or a file descriptor (under linux)
	
	Return value
		return 1 if the check is successful, 0 otherwise
*/
int		Net_Select_Check(TNetSelect *NetSelect, int FD);

/*
	Unix socket functions
	These sockets are compatible with Net_Select_* functions
*/

#if defined(LINUX) || defined(APPLE) || defined(FREEBSD)

int		Net_OpenUnixSocket(void);

int		Net_BindUnix(int Socket, char *Path);
int		Net_AcceptUnix(int ListenSocket, int *pConnSocket);

int		Net_ConnectUnix(int Socket, char *Path);

#endif

#endif
