#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <libshred.h>


#define DEFAULT_SIZE	64


TArray *Array_Create(void)
{
	TArray *Array;
	
	Array = malloc(sizeof(TArray));
	memset(Array, 0, sizeof(TArray));
	
	Array->szElements = DEFAULT_SIZE;
	Array->Elements = malloc(Array->szElements * sizeof(void*));
	
	return Array;
}


TArray *Array_CreateEx(TArrayDestroyElementProc Callback)
{
	TArray *Array;
	
	Array = Array_Create();
	Array->DestroyCallback = Callback;
	
	return Array;
}


static void _Array_DestroyElements(TArray *Array)
{
	if(Array->Elements)
	{
		if(Array->DestroyCallback)
		{
			int i, n;
			void *Element;
			
			n = Array_Count(Array);
			for(i=0;i<n;i++)
			{
				Element = Array_GetAt(Array, i);
				if(Element)
					Array->DestroyCallback(Element);
			}
		}
		free(Array->Elements);
	}
}


void Array_Destroy(TArray *Array)
{
	if(!Array)
		return;
	
	_Array_DestroyElements(Array);
	free(Array);
}


void Array_Clear(TArray *Array)
{
	if(!Array)
		return;
	
	_Array_DestroyElements(Array);
	
	Array->szElements = DEFAULT_SIZE;
	Array->Elements = malloc(Array->szElements * sizeof(void*));
	Array->lElements = 0;
}


void Array_Add(TArray *Array, void *Element)
{
	if(!Array || !Element)
		return;
	
	if(Array->lElements >= Array->szElements)
	{
		// expand array
		Array->szElements *= 2;
		Array->Elements = realloc(Array->Elements, Array->szElements * sizeof(void*));
	}
	Array->Elements[Array->lElements++] = Element;
}


void Array_Insert(TArray *Array, void *Element, int Index)
{
	if(!Array || !Element)
		return;
	
	if(Index < 0 || (Array->lElements && Index >= Array->lElements) || (!Array->lElements && Index > Array->lElements))
	{
		LOG_DEBUG("error: invalid insert offset %d (%d elements)", Index, Array->lElements);
		return;
	}
	
	if(Array->lElements >= Array->szElements)
	{
		// expand array
		Array->szElements *= 2;
		Array->Elements = realloc(Array->Elements, Array->szElements * sizeof(void*));
	}
	memmove(&Array->Elements[Index + 1], &Array->Elements[Index], (Array->lElements - Index) * sizeof(void*));
	Array->Elements[Index] = Element;
	Array->lElements++;
}


void Array_DeleteAt(TArray *Array, int Index)
{
	if(!Array)
		return;
	
	if(Index < 0 || Index >= Array->lElements)
		return;
	
	if(Array->DestroyCallback)
	{
		void *Element = Array_GetAt(Array, Index);
		if(Element)
			Array->DestroyCallback(Element);
	}
	
	memmove(&Array->Elements[Index], &Array->Elements[Index + 1], (Array->lElements - Index - 1) * sizeof(void*));
	Array->lElements--;
	
	if(Array->lElements > DEFAULT_SIZE && Array->lElements < Array->szElements / 3)
	{
		// shrink array
		Array->szElements /= 2;
		Array->Elements = realloc(Array->Elements, Array->szElements * sizeof(void*));
	}
}


void Array_Delete(TArray *Array, void *Element)
{
	int i, n;
	
	if(!Array || !Element)
		return;
	
	n = Array_Count(Array);
	for(i=0;i<n;i++)
	{
		void *ElementRef = Array_GetAt(Array, i);
		if(ElementRef == Element)
		{
			Array_DeleteAt(Array, i);
			return;
		}
	}
}


void Array_Sort(TArray *Array, TArrayCompareProc CompareCallback)
{
	if(!Array || !CompareCallback)
		return;
	
	qsort(Array->Elements, Array->lElements, sizeof(void*), (int (*)(const void *, const void *))CompareCallback);
}
