#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#if defined(LINUX) || defined(APPLE) || defined(FREEBSD)
#include <termios.h>
#ifndef APPLE
#include <termio.h>
#endif
#include <unistd.h>
#include <signal.h>
#endif

#include <libshred.h>


static int CTRLCEvent = 0;


#if defined(LINUX) || defined(APPLE) || defined(FREEBSD)
static void Signal_Terminate(int SignalNo)
{
	CTRLCEvent = 1;
}
#endif


#ifdef WIN32
BOOL CtrlHandler(DWORD dwCtrlType)
{
	switch(dwCtrlType)
	{
		case CTRL_C_EVENT:
			CTRLCEvent = 1;
			return TRUE;
		
		default:
			return FALSE;
	}
}
#endif


void Console_CatchCTRLCEvent(void)
{
#if defined(LINUX) || defined(APPLE) || defined(FREEBSD)
	signal(SIGINT, Signal_Terminate);
#else
	SetConsoleCtrlHandler((PHANDLER_ROUTINE)CtrlHandler, TRUE);
#endif
}


int Console_GetCTRLCEvent(void)
{
	return CTRLCEvent;
}


#ifndef WIN32
typedef void (*TSignalHandler)(int Signal);
#define ESC "\033"

static struct termios TermiosOld;
static TSignalHandler _SigHandler_INT_Old = 0;
static TSignalHandler _SigHandler_QUIT_Old = 0;
static TSignalHandler _SigHandler_TERM_Old = 0;


static void OnInterrupt(int Signal)
{
	tcsetattr(STDERR_FILENO, TCSANOW, &TermiosOld);
	exit(EXIT_FAILURE);
}


static int _Console_GetSize_Linux(int *pWidth, int *pHeight)
{
	struct termios TermiosNew;
	//struct winsize WindowSize = { 0, 0, 0, 0 };
	int CanRead = 0;
	int iRows = 0, iCols = 0;
	
	
	/* We use _stderr_ in order to make resize usable
	 * in shell backticks (those redirect stdout away from tty).
	 * NB: other versions of resize open "/dev/tty"
	 * and operate on it - should we do the same?
	 */
	tcgetattr(STDERR_FILENO, &TermiosOld); /* fiddle echo */
	memcpy(&TermiosNew, &TermiosOld, sizeof(struct termios));
	TermiosNew.c_cflag |= (CLOCAL | CREAD);
	TermiosNew.c_lflag &= ~(ICANON | ECHO | ECHOE | ISIG);
	
	_SigHandler_INT_Old = signal(SIGINT, OnInterrupt);
	_SigHandler_QUIT_Old = signal(SIGQUIT, OnInterrupt);
	_SigHandler_TERM_Old = signal(SIGTERM, OnInterrupt);
	
	// new terminal
	tcsetattr(STDERR_FILENO, TCSANOW, &TermiosNew);
	
	/* save_cursor_pos 7
	 * scroll_whole_screen [r
	 * put_cursor_waaaay_off [$x;$yH
	 * get_cursor_pos [6n
	 * restore_cursor_pos 8
	 */
	fprintf(stderr, ESC"7" ESC"[r" ESC"[999;999H" ESC"[6n");
	
	// Just in case terminal won't answer, handle reply through select()
	{
		TNetSelect NetSelect;
		
		Net_Select_Reset(&NetSelect);
		Net_Select_Add(&NetSelect, STDOUT_FILENO, NET_OP_READ);
		if(Net_Select(&NetSelect, 2000) > 0)
		{
			if(Net_Select_Check(&NetSelect, STDOUT_FILENO))
			{
				CanRead = 1;
				scanf(ESC"[%d;%dR", &iRows, &iCols);
				fprintf(stderr, ESC"8");
			}
		}
	}
	
	// restore terminal
	tcsetattr(STDERR_FILENO, TCSANOW, &TermiosOld);
	
	// restore signals
	signal(SIGINT, _SigHandler_INT_Old);
	signal(SIGQUIT, _SigHandler_QUIT_Old);
	signal(SIGTERM, _SigHandler_TERM_Old);
	
	if(pWidth)
		*pWidth = iCols;
	if(pHeight)
		*pHeight = iRows;
	
	return CanRead;
}
#endif


int Console_GetSize(int *pWidth, int *pHeight)
{
#ifdef WIN32
	CONSOLE_SCREEN_BUFFER_INFO Info;
	HANDLE hStdout;
#endif
	
	if(!pWidth && !pHeight)
		return 0;
	
	if(pWidth)
		*pWidth = 0;
	if(pHeight)
		*pHeight = 0;
	
#ifdef WIN32
	hStdout = GetStdHandle(STD_OUTPUT_HANDLE);
	if(hStdout)
	{
		if(GetConsoleScreenBufferInfo(hStdout, &Info))
		{
			if(pWidth)
				*pWidth = Info.dwSize.X;
			if(pHeight)
				*pHeight = Info.dwSize.Y;
			
			return 1;
		}
	}
#else
	return _Console_GetSize_Linux(pWidth, pHeight);
#endif
	
	return 0;
}
