#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <ctype.h>

#include <libshred.h>


#define MODULUS_MIN			37
#define MODULUS_DEFAULT		1277	// this is enough for keys number below 35


typedef struct
{
	char *	Key;
	int		lKey;
	char *	Value;
	
}TToken;

typedef struct
{
	int		Modulus;
	THashTable *HashTable;
	
}TContext;


static int _Dictionay_CheckKey(char *Key, int *plKey)
{
	int i, lKey;
	
	if(!Key)
	{
		LOG_ERROR("error: empty key");
		return 0;
	}
	
	lKey = strlen(Key);
	for(i=0;i<lKey;i++)
	{
		char c = Key[i];
		if(isalnum(c) || c == '.' || c == '_' || (c == '-' && i + 1 < lKey && !isdigit(Key[i + 1])))
			continue;
		LOG_ERROR("error: invalid key character found at offset %d (%c): %s", i, c, Key);
		return 0;
	}
	
	if(plKey)
		*plKey = lKey;
	
	return 1;
}


TDictionary *Dictionary_CreateEx(int Modulus)
{
	TContext *Context = NULL;
	
	if(Modulus <= 0)
		Modulus = MODULUS_DEFAULT;
	else if(Modulus <= MODULUS_MIN)
		Modulus = MODULUS_MIN;
	
	Context = malloc(sizeof(TContext));
	memset(Context, 0, sizeof(TContext));
	
	Context->Modulus = Modulus;
	// Hash_FNV32 is 1% faster for this process, mainly because of the dispersion
	//Context->HashTable = HashTable_Create(Modulus, OFFSET_OF(TToken, Key), OFFSET_OF(TToken, lKey), Hash_SQLITE);
	Context->HashTable = HashTable_Create(Modulus, OFFSET_OF(TToken, Key), OFFSET_OF(TToken, lKey), Hash_FNV32);
	
	return (TDictionary*)Context;
}


static void _Dictionay_DestroyToken(void *Data)
{
	TToken *Token = (TToken*)Data;
	
	if(Token)
	{
		String_SafeFree(Token->Key);
		String_SafeFree(Token->Value);
		free(Token);
	}
}


void Dictionary_Destroy(TDictionary *Handle)
{
	TContext *Context = (TContext*)Handle;
	
	if(Context)
	{
		HashTable_Destroy(Context->HashTable, _Dictionay_DestroyToken);
		free(Context);
	}
}


TDictionary *Dictionary_Duplicate(TDictionary *Handle)
{
	TContext *Context = (TContext*)Handle;
	TToken *Token = NULL;
	TContext *Context2;
	
	if(!Context)
		return NULL;
	
	// create new context, with same modulus
	Context2 = (TContext*)Dictionary_CreateEx(Context->Modulus);
	
	Token = HashTable_First(Context->HashTable);
	while(Token)
	{
		// duplicate token
		TToken *Token2 = malloc(sizeof(TToken));
		memset(Token2, 0, sizeof(TToken));
		Token2->Key = strdup(Token->Key);
		Token2->lKey = Token->lKey;
		Token2->Value = String_Duplicate(Token->Value);
		
		// add new token to new hash table
		HashTable_Add(Context2->HashTable, Token2);
		
		Token = HashTable_Next(Context->HashTable);
	}
	return (TDictionary*)Context2;
}


void Dictionary_Delete(TDictionary *Handle, char *Key)
{
	TContext *Context = (TContext*)Handle;
	int lKey;
	
	if(!_Dictionay_CheckKey(Key, &lKey))
		return;
	
	HashTable_Delete(Context->HashTable, Key, lKey, _Dictionay_DestroyToken);
}


static void _Dictionary_Set(TContext *Context, char *Key, int lKey, char *Value)
{
	TToken *Token = NULL;
	
	if(lKey < 0)
		lKey = strlen(Key);
	
	Token = HashTable_Search(Context->HashTable, Key, lKey);
	if(!Token)
	{
		Token = malloc(sizeof(TToken));
		memset(Token, 0, sizeof(TToken));
		Token->Key = strdup(Key);
		Token->lKey = lKey;
		
		HashTable_Add(Context->HashTable, Token);
	}
	String_SafeReplace(Token->Value, Value);
}


void Dictionary_Merge(TDictionary *From, TDictionary *To)
{
	TContext *FromDict = (TContext*)From;
	TContext *ToDict = (TContext*)To;
	TToken *Token = NULL;
	
	if(!From || !To)
		return;
	
	Token = HashTable_First(FromDict->HashTable);
	while(Token)
	{
		_Dictionary_Set(ToDict, Token->Key, -1, Token->Value);
		Token = HashTable_Next(FromDict->HashTable);
	}
}


void Dictionary_Set(TDictionary *Handle, char *Key, char *Value)
{
	TContext *Context = (TContext*)Handle;
	int lKey;
	
	if(!_Dictionay_CheckKey(Key, &lKey))
		return;
	
	_Dictionary_Set(Context, Key, lKey, Value);
}


void Dictionary_Set_Int32(TDictionary *Handle, char *Key, int iValue)
{
	char Value[32], *pValue;
	
	pValue = String_IntToString(iValue, Value, sizeof(Value));
	Dictionary_Set(Handle, Key, pValue);
}


void Dictionary_Set_Dict(TDictionary *Handle1, char *Key, TDictionary *Handle2)
{
	char Prefix[256], InsertKey[256];
	char **Keys2 = NULL;
	int i, lKeys2 = 0;
	
	
	if(!Handle1 || !Key || !Handle2)
		return;
	
	if(!_Dictionay_CheckKey(Key, NULL))
		return;
	
	if(!String_EndsWith(Key, "."))
		snprintf(Prefix, sizeof(Prefix) - 1, "%s.", Key);
	else
		String_Copy(Key, Prefix, sizeof(Prefix));
	
	Dictionary_GetKeysEx(Handle2, &Keys2, &lKeys2, 0);
	for(i=0;i<lKeys2;i++)
	{
		char *Key2 = Keys2[i];
		snprintf(InsertKey, sizeof(InsertKey) - 1, "%s%s", Prefix, Key2);
		Dictionary_Set(Handle1, InsertKey, Dictionary_GetValueByRef(Handle2, Key2));
	}
	
	StringArray_Free(&Keys2, &lKeys2);
}


char *Dictionary_GetValueByRef(TDictionary *Handle, char *Key)
{
	TContext *Context = (TContext*)Handle;
	TToken *Token = NULL;
	int lKey;
	
	if(!Context || !_Dictionay_CheckKey(Key, &lKey))
		return NULL;
	
	Token = HashTable_Search(Context->HashTable, Key, lKey);
	if(!Token)
		return NULL;
	return Token->Value;
}


char *Dictionary_GetValue(TDictionary *Handle, char *Key, char *TmpBuf, int szTmpBuf)
{
	char *Value;
	
	Value = Dictionary_GetValueByRef(Handle, Key);
	if(TmpBuf)
	{
		if(szTmpBuf > 0)
		{
			TmpBuf[0] = 0;
			if(Value)
				String_Copy(Value, TmpBuf, szTmpBuf);
			return TmpBuf;
		}
		// should never occur
		return NULL;
	}
	else
	{
		if(!Value)
			Value = "";
		TmpBuf = strdup(Value);
	}
	return TmpBuf;
}


int Dictionary_Get_Int32(TDictionary *Handle, char *Key)
{
	char TmpBuf[32];
	int iValue = 0;
	
	Dictionary_GetValue(Handle, Key, TmpBuf, sizeof(TmpBuf));
	String_GetNumber(TmpBuf, &iValue, NULL);
	
	return iValue;
}


TDictionary *Dictionary_Get_Dict(TDictionary *Handle, char *Key)
{
	TContext *Context = (TContext*)Handle;
	char Prefix[256];
	char **Keys = NULL;
	int i, lKeys = 0;
	TDictionary *SubDict;
	
	
	if(!Handle || !Key)
		return NULL;
	
	if(!_Dictionay_CheckKey(Key, NULL))
		return NULL;
	
	if(!String_EndsWith(Key, "."))
		snprintf(Prefix, sizeof(Prefix) - 1, "%s.", Key);
	else
		String_Copy(Key, Prefix, sizeof(Prefix));
	
	SubDict = Dictionary_CreateEx(Context->Modulus);
	
	Dictionary_GetKeysEx(Handle, &Keys, &lKeys, 0);
	for(i=0;i<lKeys;i++)
	{
		char *_Key = Keys[i];
		char *pTrail;
		
		if(String_StartsWithEx(_Key, Prefix, &pTrail))
			Dictionary_Set(SubDict, pTrail, Dictionary_GetValueByRef(Handle, _Key));
	}
	StringArray_Free(&Keys, &lKeys);
	
	// no keys were found
	if(Dictionary_Count(SubDict, NULL) <= 0)
	{
		Dictionary_Destroy(SubDict);
		return NULL;
	}
	
	return SubDict;
}


static int _Dictionary_KeyCompareClbk(void *pArg0, void *pArg1, void *pData)
{
	char *Str0 = *((char**)pArg0);
	char *Str1 = *((char**)pArg1);
	//return strcmp(Str0, Str1);
	return String_Compare_Natural(Str0, Str1, 0);
}

void Dictionary_GetKeysEx(TDictionary *Handle, char ***pKeys, int *plKeys, int Sort)
{
	TContext *Context = (TContext*)Handle;
	TToken *Token = NULL;
	char **Keys = NULL;
	int lKeys = 0;
	int szKeys = 0;
	
	
	if(!Handle || !pKeys || !plKeys)
		return;
	
	// reserve memory
	szKeys = HashTable_Count(Context->HashTable);
	Keys = malloc(szKeys * sizeof(char*));
	memset(Keys, 0, szKeys * sizeof(char*));
	
	// walk through the whole table
	Token = HashTable_First(Context->HashTable);
	while(Token)
	{
		Keys[lKeys++] = strdup(Token->Key);
		Token = HashTable_Next(Context->HashTable);
	}
	
	// sort results
	if(Sort)
		Sort_HeapSort(Keys, lKeys, sizeof(char*), _Dictionary_KeyCompareClbk, NULL);
	
	*pKeys = Keys;
	*plKeys = lKeys;
}


int Dictionary_Count(TDictionary *Handle, char *KeyPrefix)
{
	TContext *Context = (TContext*)Handle;
	TToken *Token = NULL;
	char KeyPrefixWildcard[256];
	int iMax, lKeyPrefix;
	
	
	if(!Context)
		return 0;
	
	if(!KeyPrefix)
		return HashTable_Count(Context->HashTable);
	
	//LOG_DEBUG("count(%s)", KeyPrefix);
	
	if(!_Dictionay_CheckKey(KeyPrefix, NULL))
		return 0;
	
	lKeyPrefix = strlen(KeyPrefix);
	snprintf(KeyPrefixWildcard, sizeof(KeyPrefixWildcard) - 1, "%s.*.*", KeyPrefix);
	//LOG_DEBUG("  match(%s)", KeyPrefixWildcard);
	
	iMax = 0;
	
	// walk through the whole table
	Token = HashTable_First(Context->HashTable);
	while(Token)
	{
		if(String_Match(Token->Key, KeyPrefixWildcard))
		{
			int k, nDigits;
			
			//LOG_DEBUG("  => [%s]", Token->Key);
			//LOG_DEBUG("  ==> [%s]", Token->Key + lKeyPrefix + 1);
			
			if(String_GetNumber(Token->Key + lKeyPrefix + 1, &k, &nDigits) && k >= 0 && Token->Key[lKeyPrefix + 1 + nDigits] == '.')
			//{
				//LOG_DEBUG("  ===> [%s] [%d] [%c]", Token->Key, k, *(Token->Key + lKeyPrefix + 1 + nDigits));
				iMax = MAX(iMax, k + 1);
			//}
		}
		Token = HashTable_Next(Context->HashTable);
	}
	
	//LOG_DEBUG("  [%d]", iMax);
	
	return iMax;
}


int Dictionary_Equal(TDictionary *Dict0, TDictionary *Dict1)
{
	char **Keys = NULL;
	int i, lKeys = 0, SameKeys = 1;
	
	if(Dict0 && !Dict1)
		return 0;
	if(!Dict0 && Dict1)
		return 0;
	if(Dictionary_Count(Dict0, NULL) != Dictionary_Count(Dict1, NULL))
		return 0;
	
	Dictionary_GetKeysEx(Dict0, &Keys, &lKeys, 0);
	for(i=0;i<lKeys;i++)
	{
		char *Key = Keys[i];
		char *Value0, *Value1;
		Value0 = Dictionary_GetValueByRef(Dict0, Key);
		Value1 = Dictionary_GetValueByRef(Dict1, Key);
		if(!String_Equal(Value0, Value1))
		{
			SameKeys = 0;
			break;
		}
	}
	StringArray_Free(&Keys, &lKeys);
	
	return SameKeys;
}


void Dictionary_Dump(TDictionary *Handle)
{
	//TContext *Context = (TContext*)Handle;
	char **Keys = NULL;
	int i, lKeys = 0;
	
	//HashTable_Dump(Context->HashTable);
	//HashTable_Statistics(Context->HashTable);
	
	LOG_DEBUG("*** dict dump ***");
	
	Dictionary_GetKeys(Handle, &Keys, &lKeys);
	for(i=0;i<lKeys;i++)
	{
		char *Key = Keys[i];
		char *Value = Dictionary_GetValueByRef(Handle, Key);
		LOG_DEBUG("[%02d] [%s] => [%s]", i, Key, Value);
	}
	StringArray_Free(&Keys, &lKeys);
	
	LOG_DEBUG("*** dict dump ***");
}


int Dictionary_WriteToFile(TDictionary *Handle, char *Filename)
{
	char **Keys = NULL;
	int i, lKeys = 0;
	FILE *File;
	
	
	File = fopen(Filename, "w");
	if(!File)
	{
		LOG_DEBUG("error: opening file for writing: %s", Filename);
		return 0;
	}
	
	Dictionary_GetKeys(Handle, &Keys, &lKeys);
	for(i=0;i<lKeys;i++)
	{
		char *Key = Keys[i];
		char *Value = Dictionary_GetValueByRef(Handle, Key);
		fprintf(File, "%s = %s\n", Key, Value);
	}
	StringArray_Free(&Keys, &lKeys);
	
	fclose(File);
	
	return 1;
}


TDictionary *Dictionary_ReadFromFile(char *Filename)
{
	TDictionary *Dict;
	TINIFile *File;
	TINISection *Section;
	
	File = INIFile_Read(Filename);
	if(!File)
	{
		LOG_DEBUG("error: opening file for reading: %s", Filename);
		return NULL;
	}
	
	Dict = Dictionary_Create();
	
	Section = INIFile_FindSection(File, "");
	if(Section)
	{
		int i;
		
		for(i=0;i<Section->nAttributes;i++)
		{
			TINIAttribute *Attr = Section->Attributes[i];
			Dictionary_Set(Dict, Attr->Name, Attr->Value);
		}
	}
	
	INIFile_Destroy(File);
	
	return Dict;
}


char *Dictionary_GetHash(TDictionary *Handle, char *TmpBuf, int szTmpBuf)
{
	TContext *Context = (TContext*)Handle;
	uint8 Digest[MD5_DIGEST_SIZE];
	char DigestStr[MD5_DIGEST_STRING_SIZE];
	TToken *Token = NULL;
	void *MD5;
	char Line[128];
	int lLine;
	
	
	MD5 = Hash_MD5_Init();
	
	lLine = snprintf(Line, sizeof(Line) - 1, "<dict>");
	Hash_MD5_Update(MD5, (uint8*)Line, lLine);
	
	if(Context)
	{
		lLine = snprintf(Line, sizeof(Line) - 1, "<n>%d</n>", HashTable_Count(Context->HashTable));
		Hash_MD5_Update(MD5, (uint8*)Line, lLine);
		
		// walk through the whole table
		Token = HashTable_First(Context->HashTable);
		while(Token)
		{
			Hash_MD5_Update(MD5, (uint8*)"<k>", 3);
			Hash_MD5_Update(MD5, (uint8*)Token->Key, Token->lKey);
			Hash_MD5_Update(MD5, (uint8*)"</k>", 4);
			
			Hash_MD5_Update(MD5, (uint8*)"<v>", 3);
			Hash_MD5_Update(MD5, (uint8*)Token->Value, strlen(Token->Value));
			Hash_MD5_Update(MD5, (uint8*)"</v>", 4);
			
			Token = HashTable_Next(Context->HashTable);
		}
	}
	
	lLine = snprintf(Line, sizeof(Line) - 1, "</dict>");
	Hash_MD5_Update(MD5, (uint8*)Line, lLine);
	
	Hash_MD5_Final(MD5, Digest);
	Hash_MD5_DigestToString(Digest, DigestStr);
	
	String_Copy(DigestStr, TmpBuf, szTmpBuf);
	
	return TmpBuf;
}
