#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <libshred.h>


void Hash_DigestToStringEx(uint8 *RawDigest, int szRawDigest, char *StringDigest)
{
	char *pDigest;
	int lDigest;
	
	lDigest = szRawDigest * 2 + 1;
	pDigest = (char*)StringDigest;
	String_BufferToHex(RawDigest, szRawDigest, &pDigest, &lDigest);
}


/* SHA: NIST's Secure Hash Algorithm */
/*
	This version written November 2000 by David Ireland of 
	DI Management Services Pty Limited <code@di-mgt.com.au>
	
	Adapted from code in the Python Cryptography Toolkit, 
	version 1.0.0 by A.M. Kuchling 1995.
*/
/*
	AM Kuchling's posting:- 
	Based on SHA code originally posted to sci.crypt by Peter Gutmann
	in message <30ajo5$oe8@ccu2.auckland.ac.nz>.
	Modified to test for endianness on creation of SHA objects by AMK.
	Also, the original specification of SHA was found to have a weakness
	by NSA/NIST.  This code implements the fixed version of SHA.
*/
/*
	Here's the first paragraph of Peter Gutmann's posting:
	
	The following is my SHA (FIPS 180) code updated to allow use of the "fixed"
	SHA, thanks to Jim Gillogly and an anonymous contributor for the information on
	what's changed in the new version.  The fix is a simple change which involves
	adding a single rotate in the initial expansion function.  It is unknown
	whether this is an optimal solution to the problem which was discovered in the
	SHA or whether it's simply a bandaid which fixes the problem with a minimum of
	effort (for example the reengineering of a great many Capstone chips).
*/

typedef struct 
{
	uint32 Digest[5];
	uint32 lbCount64;	// Count64: 64-bit bit count
	uint32 hbCount64;
	uint32 Buffer[16];
	
}TSHA1;


// The SHS block size and message Digest sizes, in bytes
#define SHA1_DATA_SIZE		64


/*
	The SHS f()-functions.  The f1 and f3 functions can be optimized to
	save one boolean operation each - thanks to Rich Schroeppel,
	rcs@cs.arizona.edu for discovering this
*/
#define f1(x,y,z)   (z ^ (x & (y ^ z)))			/* Rounds  0-19 */
#define f2(x,y,z)   (x ^ y ^ z)					/* Rounds 20-39 */
#define f3(x,y,z)   ((x & y) | (z & (x | y)))	/* Rounds 40-59 */
#define f4(x,y,z)   (x ^ y ^ z)					/* Rounds 60-79 */

// The SHS Mysterious Constants
#define K1  0x5A827999L							/* Rounds  0-19 */
#define K2  0x6ED9EBA1L							/* Rounds 20-39 */
#define K3  0x8F1BBCDCL							/* Rounds 40-59 */
#define K4  0xCA62C1D6L							/* Rounds 60-79 */

/*
	Note that it may be necessary to add parentheses to these macros if they
	are to be called with expressions as arguments
*/
// 32-bit rotate left - kludged with shifts
#define ROTL(n,X)	(((X) << n) | ((X) >> (32 - n)))

/*
	The initial expanding function.  The hash function is defined over an
	80-UINT2 expanded input array W, where the first 16 are copies of the input
	Buffer, and the remaining 64 are defined by

		W[ i ] = W[ i - 16 ] ^ W[ i - 14 ] ^ W[ i - 8 ] ^ W[ i - 3 ]

	This implementation generates these values on the fly in a circular
	buffer - thanks to Colin Plumb, colin@nyx10.cs.du.edu for this
	optimization.
	
	The updated SHS changes the expanding function by adding a rotate of 1
	bit.  Thanks to Jim Gillogly, jim@rand.org, and an anonymous contributor
	for this information
*/
#define EXPAND(W,i)		(W[i & 15] = ROTL(1, (W[i & 15] ^ W[(i - 14) & 15] ^ W[(i - 8) & 15] ^ W[(i - 3) & 15])))


/*
	The prototype SHS sub-round.  The fundamental sub-round is:
	
	a' = e + ROTL( 5, a ) + f( b, c, d ) + k + buf;
	b' = a;
	c' = ROTL( 30, b);
	d' = c;
	e' = d;
	
	but this is implemented by unrolling the loop 5 times and renaming the
	variables ( e, a, b, c, d ) = ( a', b', c', d', e' ) each iteration.
	This code is then replicated 20 times for each of the 4 functions, using
	the next 20 values from the W[] array each time
*/
#define SUBROUND(a, b, c, d, e, f, k, buf)		(e += ROTL(5, a) + f(b, c, d) + k + buf, b = ROTL(30, b))


/*
	Perform the SHS transformation.  Note that this code, like MD5, seems to
	break some optimizing compilers due to the complexity of the expressions
	and the size of the basic block.  It may be necessary to split it into
	sections, e.g. based on the four subrounds
*/
static void _Hash_SHA1_Transform(uint32 *Digest, uint32 *Buffer)
{
	uint32 A, B, C, D, E;
	uint32 eData[16];
	
	
	A = Digest[0];
	B = Digest[1];
	C = Digest[2];
	D = Digest[3];
	E = Digest[4];
	memcpy((uint8*)eData, (uint8*)Buffer, SHA1_DATA_SIZE);
	
	// heavy mangling, in 4 sub-rounds of 20 interations each.
	SUBROUND(A, B, C, D, E, f1, K1, eData[ 0]);
	SUBROUND(E, A, B, C, D, f1, K1, eData[ 1]);
	SUBROUND(D, E, A, B, C, f1, K1, eData[ 2]);
	SUBROUND(C, D, E, A, B, f1, K1, eData[ 3]);
	SUBROUND(B, C, D, E, A, f1, K1, eData[ 4]);
	SUBROUND(A, B, C, D, E, f1, K1, eData[ 5]);
	SUBROUND(E, A, B, C, D, f1, K1, eData[ 6]);
	SUBROUND(D, E, A, B, C, f1, K1, eData[ 7]);
	SUBROUND(C, D, E, A, B, f1, K1, eData[ 8]);
	SUBROUND(B, C, D, E, A, f1, K1, eData[ 9]);
	SUBROUND(A, B, C, D, E, f1, K1, eData[10]);
	SUBROUND(E, A, B, C, D, f1, K1, eData[11]);
	SUBROUND(D, E, A, B, C, f1, K1, eData[12]);
	SUBROUND(C, D, E, A, B, f1, K1, eData[13]);
	SUBROUND(B, C, D, E, A, f1, K1, eData[14]);
	SUBROUND(A, B, C, D, E, f1, K1, eData[15]);
	SUBROUND(E, A, B, C, D, f1, K1, EXPAND(eData, 16));
	SUBROUND(D, E, A, B, C, f1, K1, EXPAND(eData, 17));
	SUBROUND(C, D, E, A, B, f1, K1, EXPAND(eData, 18));
	SUBROUND(B, C, D, E, A, f1, K1, EXPAND(eData, 19));
	
	SUBROUND(A, B, C, D, E, f2, K2, EXPAND(eData, 20));
	SUBROUND(E, A, B, C, D, f2, K2, EXPAND(eData, 21));
	SUBROUND(D, E, A, B, C, f2, K2, EXPAND(eData, 22));
	SUBROUND(C, D, E, A, B, f2, K2, EXPAND(eData, 23));
	SUBROUND(B, C, D, E, A, f2, K2, EXPAND(eData, 24));
	SUBROUND(A, B, C, D, E, f2, K2, EXPAND(eData, 25));
	SUBROUND(E, A, B, C, D, f2, K2, EXPAND(eData, 26));
	SUBROUND(D, E, A, B, C, f2, K2, EXPAND(eData, 27));
	SUBROUND(C, D, E, A, B, f2, K2, EXPAND(eData, 28));
	SUBROUND(B, C, D, E, A, f2, K2, EXPAND(eData, 29));
	SUBROUND(A, B, C, D, E, f2, K2, EXPAND(eData, 30));
	SUBROUND(E, A, B, C, D, f2, K2, EXPAND(eData, 31));
	SUBROUND(D, E, A, B, C, f2, K2, EXPAND(eData, 32));
	SUBROUND(C, D, E, A, B, f2, K2, EXPAND(eData, 33));
	SUBROUND(B, C, D, E, A, f2, K2, EXPAND(eData, 34));
	SUBROUND(A, B, C, D, E, f2, K2, EXPAND(eData, 35));
	SUBROUND(E, A, B, C, D, f2, K2, EXPAND(eData, 36));
	SUBROUND(D, E, A, B, C, f2, K2, EXPAND(eData, 37));
	SUBROUND(C, D, E, A, B, f2, K2, EXPAND(eData, 38));
	SUBROUND(B, C, D, E, A, f2, K2, EXPAND(eData, 39));
	
	SUBROUND(A, B, C, D, E, f3, K3, EXPAND(eData, 40));
	SUBROUND(E, A, B, C, D, f3, K3, EXPAND(eData, 41));
	SUBROUND(D, E, A, B, C, f3, K3, EXPAND(eData, 42));
	SUBROUND(C, D, E, A, B, f3, K3, EXPAND(eData, 43));
	SUBROUND(B, C, D, E, A, f3, K3, EXPAND(eData, 44));
	SUBROUND(A, B, C, D, E, f3, K3, EXPAND(eData, 45));
	SUBROUND(E, A, B, C, D, f3, K3, EXPAND(eData, 46));
	SUBROUND(D, E, A, B, C, f3, K3, EXPAND(eData, 47));
	SUBROUND(C, D, E, A, B, f3, K3, EXPAND(eData, 48));
	SUBROUND(B, C, D, E, A, f3, K3, EXPAND(eData, 49));
	SUBROUND(A, B, C, D, E, f3, K3, EXPAND(eData, 50));
	SUBROUND(E, A, B, C, D, f3, K3, EXPAND(eData, 51));
	SUBROUND(D, E, A, B, C, f3, K3, EXPAND(eData, 52));
	SUBROUND(C, D, E, A, B, f3, K3, EXPAND(eData, 53));
	SUBROUND(B, C, D, E, A, f3, K3, EXPAND(eData, 54));
	SUBROUND(A, B, C, D, E, f3, K3, EXPAND(eData, 55));
	SUBROUND(E, A, B, C, D, f3, K3, EXPAND(eData, 56));
	SUBROUND(D, E, A, B, C, f3, K3, EXPAND(eData, 57));
	SUBROUND(C, D, E, A, B, f3, K3, EXPAND(eData, 58));
	SUBROUND(B, C, D, E, A, f3, K3, EXPAND(eData, 59));
	
	SUBROUND(A, B, C, D, E, f4, K4, EXPAND(eData, 60));
	SUBROUND(E, A, B, C, D, f4, K4, EXPAND(eData, 61));
	SUBROUND(D, E, A, B, C, f4, K4, EXPAND(eData, 62));
	SUBROUND(C, D, E, A, B, f4, K4, EXPAND(eData, 63));
	SUBROUND(B, C, D, E, A, f4, K4, EXPAND(eData, 64));
	SUBROUND(A, B, C, D, E, f4, K4, EXPAND(eData, 65));
	SUBROUND(E, A, B, C, D, f4, K4, EXPAND(eData, 66));
	SUBROUND(D, E, A, B, C, f4, K4, EXPAND(eData, 67));
	SUBROUND(C, D, E, A, B, f4, K4, EXPAND(eData, 68));
	SUBROUND(B, C, D, E, A, f4, K4, EXPAND(eData, 69));
	SUBROUND(A, B, C, D, E, f4, K4, EXPAND(eData, 70));
	SUBROUND(E, A, B, C, D, f4, K4, EXPAND(eData, 71));
	SUBROUND(D, E, A, B, C, f4, K4, EXPAND(eData, 72));
	SUBROUND(C, D, E, A, B, f4, K4, EXPAND(eData, 73));
	SUBROUND(B, C, D, E, A, f4, K4, EXPAND(eData, 74));
	SUBROUND(A, B, C, D, E, f4, K4, EXPAND(eData, 75));
	SUBROUND(E, A, B, C, D, f4, K4, EXPAND(eData, 76));
	SUBROUND(D, E, A, B, C, f4, K4, EXPAND(eData, 77));
	SUBROUND(C, D, E, A, B, f4, K4, EXPAND(eData, 78));
	SUBROUND(B, C, D, E, A, f4, K4, EXPAND(eData, 79));
	
	// build message digest
	Digest[0] += A;
	Digest[1] += B;
	Digest[2] += C;
	Digest[3] += D;
	Digest[4] += E;
}


#ifdef SYS_BIG_ENDIAN
#define _Hash_SHA1_Reverse(b,l)		// nothing to do for big-endian CPU
#else
// when run on a little-endian CPU we need to perform byte reversal on an array of long words.
static void _Hash_SHA1_Reverse(uint32 *Buffer, int lBuffer)
{
	// little endian code
	uint32 uValue;
	
	lBuffer /= sizeof(uint32);
	while(lBuffer--)
	{
		uValue = *Buffer;
		uValue = ((uValue & 0xFF00FF00L) >> 8) | ((uValue & 0x00FF00FFL) << 8);
		*Buffer++ = (uValue << 16) | (uValue >> 16);
	}
}
#endif


void *Hash_SHA1_Init(void)
{
	TSHA1 *SHA1;
	
	
	SHA1 = malloc(sizeof(TSHA1));
	
	SHA1->lbCount64 = 0;
	SHA1->hbCount64 = 0;
	
	// Set the h-vars to their initial values
	SHA1->Digest[0] = 0x67452301L;
	SHA1->Digest[1] = 0xEFCDAB89L;
	SHA1->Digest[2] = 0x98BADCFEL;
	SHA1->Digest[3] = 0x10325476L;
	SHA1->Digest[4] = 0xC3D2E1F0L;
	
	return SHA1;
}


void Hash_SHA1_Update(void *Handle, uint8 *Buffer, int lBuffer)
{
	TSHA1 *SHA1 = (TSHA1*)Handle;
	uint32 uTmp;
	int lData;
	
	
	// Update bitcount
	uTmp = SHA1->lbCount64;
	if((SHA1->lbCount64 = uTmp + ((uint32)lBuffer << 3)) < uTmp)
		SHA1->hbCount64++;		// Carry from low to high
	SHA1->hbCount64 += lBuffer >> 29;
	
	// Get count of bytes already in Buffer
	lData = (int)(uTmp >> 3) & 0x3F;
	
	// Handle any leading odd-sized chunks
	if(lData)
	{
		uint8 *pBuf = ((uint8*)SHA1->Buffer) + lData;
		
		lData = SHA1_DATA_SIZE - lData;
		if(lBuffer < lData)
		{
			memcpy(pBuf, Buffer, lBuffer);
			return;
		}
		memcpy(pBuf, Buffer, lData);
		_Hash_SHA1_Reverse(SHA1->Buffer, SHA1_DATA_SIZE);
		_Hash_SHA1_Transform(SHA1->Digest, SHA1->Buffer);
		Buffer += lData;
		lBuffer -= lData;
	}
	
	// Process Buffer in SHA1_DATA_SIZE chunks
	while(lBuffer >= SHA1_DATA_SIZE)
	{
		memcpy(SHA1->Buffer, Buffer, SHA1_DATA_SIZE);
		_Hash_SHA1_Reverse(SHA1->Buffer, SHA1_DATA_SIZE);
		_Hash_SHA1_Transform(SHA1->Digest, SHA1->Buffer);
		Buffer += SHA1_DATA_SIZE;
		lBuffer -= SHA1_DATA_SIZE;
	}
	
	// Handle any remaining bytes of Buffer
	memcpy(SHA1->Buffer, Buffer, lBuffer);
}


/*
	Final wrapup - pad to SHA1_DATA_SIZE-byte boundary with the bit pattern
	1 0* (64-bit count of bits processed, MSB-first)
*/
void Hash_SHA1_Final(void *Handle, uint8 Digest[SHA1_DIGEST_SIZE])
{
	TSHA1 *SHA1 = (TSHA1*)Handle;
	int Count;
	uint8 *pBuf;
	
	
	// Compute number of bytes mod 64
	Count = (int)SHA1->lbCount64;
	Count = (Count >> 3) & 0x3F;
	
	// Set the first char of padding to 0x80.
	// This is safe since there is always at least one byte free
	pBuf = ((uint8*)SHA1->Buffer) + Count;
	*pBuf++ = 0x80;
	
	//Bytes of padding needed to make 64 bytes
	Count = SHA1_DATA_SIZE - 1 - Count;
	
	// Pad out to 56 mod 64
	if(Count < 8)
	{
		// Two lots of padding:  Pad the first block to 64 bytes
		memset(pBuf, 0, Count);
		_Hash_SHA1_Reverse(SHA1->Buffer, SHA1_DATA_SIZE);
		_Hash_SHA1_Transform(SHA1->Digest, SHA1->Buffer);

		// Now fill the next block with 56 bytes
		memset(SHA1->Buffer, 0, SHA1_DATA_SIZE - 8);
	}
	else
	{
		// Pad block to 56 bytes
		memset(pBuf, 0, Count - 8);
	}
	
	// Append length in bits and transform
	SHA1->Buffer[14] = SHA1->hbCount64;
	SHA1->Buffer[15] = SHA1->lbCount64;
	
	_Hash_SHA1_Reverse(SHA1->Buffer, SHA1_DATA_SIZE - 8);
	_Hash_SHA1_Transform(SHA1->Digest, SHA1->Buffer);
	
	// Output to an array of bytes
	{
		uint32 i, j;
		
		for(i = 0, j = 0; j < SHA1_DIGEST_SIZE; i++, j += 4) 
		{
			uint32 uTmp = SHA1->Digest[i];
			Digest[j+3] = (uint8)( uTmp        & 0xff);
			Digest[j+2] = (uint8)((uTmp >> 8 ) & 0xff);
			Digest[j+1] = (uint8)((uTmp >> 16) & 0xff);
			Digest[j  ] = (uint8)((uTmp >> 24) & 0xff);
		}
	}
	
	// zeroise sensitive stuff
	memset(SHA1, 0, sizeof(TSHA1));
	free(SHA1);
}


void Hash_SHA1_String(uint8 *Buffer, int lBuffer, char DigestStr[SHA1_DIGEST_STRING_SIZE])
{
	void *Handle;
	uint8 Digest[SHA1_DIGEST_SIZE];
	
	Handle = Hash_SHA1_Init();
	Hash_SHA1_Update(Handle, (uint8*)Buffer, lBuffer);
	Hash_SHA1_Final(Handle, Digest);
	Hash_SHA1_DigestToString(Digest, DigestStr);
}


/*
	Copyright(C) 1991-2, RSA Data Security, Inc. Created 1991. All
	rights reserved.
	
	License to copy and use this software is granted provided that it
	is identified as the "RSA Data Security, Inc. MD5 Message-Digest
	Algorithm" in all material mentioning or referencing this software
	or this function.
	
	License is also granted to make and use derivative works provided
	that such works are identified as "derived from the RSA Data
	Security, Inc. MD5 Message-Digest Algorithm" in all material
	mentioning or referencing the derived work.
	
	RSA Data Security, Inc. makes no representations concerning either
	the merchantability of this software or the suitability of this
	software for any particular purpose. It is provided "as is"
	without express or implied warranty of any kind.
	
	These notices must be retained in any copies of any part of this
	documentation and/or software.
*/

typedef struct
{
	uint32 State[4];	// (ABCD)
	uint32 Count[2];	// number of bits, Modulus 2^64 (lsb first)
	uint8 Buffer[64]; 	// input buffer
	
}TMD5;

#define S11 7
#define S12 12
#define S13 17
#define S14 22
#define S21 5
#define S22 9
#define S23 14
#define S24 20
#define S31 4
#define S32 11
#define S33 16
#define S34 23
#define S41 6
#define S42 10
#define S43 15
#define S44 21

static uint8 _MD5_Padding[64] = {
	0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
};

// F, G, H and I are basic MD5 functions.
#define F(x, y, z)	(((x) & (y)) | ((~x) & (z)))
#define G(x, y, z)	(((x) & (z)) | ((y) & (~z)))
#define H(x, y, z)	((x) ^ (y) ^ (z))
#define I(x, y, z)	((y) ^ ((x) | (~z)))

// ROTATE_LEFT rotates x left n bits.
#define ROTATE_LEFT(x, n)	(((x) << (n)) |((x) >> (32 - (n))))

// FF, GG, HH, and II transformations for rounds 1, 2, 3, and 4.
// Rotation is separate from addition to prevent recomputation.

#define FF(a, b, c, d, x, s, ac) { \
	(a) += F((b), (c), (d)) + (x) + (uint32)(ac); \
	(a) = ROTATE_LEFT((a), (s)); \
	(a) += (b); \
}
#define GG(a, b, c, d, x, s, ac) { \
	(a) += G((b), (c), (d)) +(x) + (uint32)(ac); \
	(a) = ROTATE_LEFT((a), (s)); \
	(a) += (b); \
}
#define HH(a, b, c, d, x, s, ac) { \
	(a) += H((b), (c), (d)) +(x) + (uint32)(ac); \
	(a) = ROTATE_LEFT((a), (s)); \
	(a) += (b); \
}
#define II(a, b, c, d, x, s, ac) { \
	(a) += I((b), (c), (d)) +(x) + (uint32)(ac); \
	(a) = ROTATE_LEFT((a), (s)); \
	(a) += (b); \
}


/*
	Encodes input(uint32) into output(uint8). Assumes len is a multiple of 4.
*/
static void _Hash_MD5_Encode(uint8 *output, uint32 *input, uint32 len)
{
	uint32 i, j;
	
	for(i = 0, j = 0; j < len; i++, j += 4)
	{
		output[j] = (uint8)(input[i] & 0xff);
		output[j+1] = (uint8)((input[i] >> 8) & 0xff);
		output[j+2] = (uint8)((input[i] >> 16) & 0xff);
		output[j+3] = (uint8)((input[i] >> 24) & 0xff);
	}
}


/*
	Decodes input(uint8) into output(uint32). Assumes len is a multiple of 4.
*/
static void _Hash_MD5_Decode(uint32 *output, uint8 *input, uint32 len)
{
	uint32 i, j;
	
	for(i = 0, j = 0; j < len; i++, j += 4)
		output[i] = ((uint32)input[j]) | (((uint32)input[j+1]) << 8) | (((uint32)input[j+2]) << 16) | (((uint32)input[j+3]) << 24);
}


static void _Hash_MD5_Transform(uint32 State[4], uint8 block[64])
{
	uint32 a, b, c, d, x[16];
	
	
	a = State[0];
	b = State[1];
	c = State[2];
	d = State[3];
	
	_Hash_MD5_Decode(x, block, 64);
	
	/* Round 1 */
	FF(a, b, c, d, x[ 0], S11, 0xd76aa478); /* 1 */
	FF(d, a, b, c, x[ 1], S12, 0xe8c7b756); /* 2 */
	FF(c, d, a, b, x[ 2], S13, 0x242070db); /* 3 */
	FF(b, c, d, a, x[ 3], S14, 0xc1bdceee); /* 4 */
	FF(a, b, c, d, x[ 4], S11, 0xf57c0faf); /* 5 */
	FF(d, a, b, c, x[ 5], S12, 0x4787c62a); /* 6 */
	FF(c, d, a, b, x[ 6], S13, 0xa8304613); /* 7 */
	FF(b, c, d, a, x[ 7], S14, 0xfd469501); /* 8 */
	FF(a, b, c, d, x[ 8], S11, 0x698098d8); /* 9 */
	FF(d, a, b, c, x[ 9], S12, 0x8b44f7af); /* 10 */
	FF(c, d, a, b, x[10], S13, 0xffff5bb1); /* 11 */
	FF(b, c, d, a, x[11], S14, 0x895cd7be); /* 12 */
	FF(a, b, c, d, x[12], S11, 0x6b901122); /* 13 */
	FF(d, a, b, c, x[13], S12, 0xfd987193); /* 14 */
	FF(c, d, a, b, x[14], S13, 0xa679438e); /* 15 */
	FF(b, c, d, a, x[15], S14, 0x49b40821); /* 16 */
	
	/* Round 2 */
	GG(a, b, c, d, x[ 1], S21, 0xf61e2562); /* 17 */
	GG(d, a, b, c, x[ 6], S22, 0xc040b340); /* 18 */
	GG(c, d, a, b, x[11], S23, 0x265e5a51); /* 19 */
	GG(b, c, d, a, x[ 0], S24, 0xe9b6c7aa); /* 20 */
	GG(a, b, c, d, x[ 5], S21, 0xd62f105d); /* 21 */
	GG(d, a, b, c, x[10], S22,  0x2441453); /* 22 */
	GG(c, d, a, b, x[15], S23, 0xd8a1e681); /* 23 */
	GG(b, c, d, a, x[ 4], S24, 0xe7d3fbc8); /* 24 */
	GG(a, b, c, d, x[ 9], S21, 0x21e1cde6); /* 25 */
	GG(d, a, b, c, x[14], S22, 0xc33707d6); /* 26 */
	GG(c, d, a, b, x[ 3], S23, 0xf4d50d87); /* 27 */
	
	GG(b, c, d, a, x[ 8], S24, 0x455a14ed); /* 28 */
	GG(a, b, c, d, x[13], S21, 0xa9e3e905); /* 29 */
	GG(d, a, b, c, x[ 2], S22, 0xfcefa3f8); /* 30 */
	GG(c, d, a, b, x[ 7], S23, 0x676f02d9); /* 31 */
	GG(b, c, d, a, x[12], S24, 0x8d2a4c8a); /* 32 */
	
	/* Round 3 */
	HH(a, b, c, d, x[ 5], S31, 0xfffa3942); /* 33 */
	HH(d, a, b, c, x[ 8], S32, 0x8771f681); /* 34 */
	HH(c, d, a, b, x[11], S33, 0x6d9d6122); /* 35 */
	HH(b, c, d, a, x[14], S34, 0xfde5380c); /* 36 */
	HH(a, b, c, d, x[ 1], S31, 0xa4beea44); /* 37 */
	HH(d, a, b, c, x[ 4], S32, 0x4bdecfa9); /* 38 */
	HH(c, d, a, b, x[ 7], S33, 0xf6bb4b60); /* 39 */
	HH(b, c, d, a, x[10], S34, 0xbebfbc70); /* 40 */
	HH(a, b, c, d, x[13], S31, 0x289b7ec6); /* 41 */
	HH(d, a, b, c, x[ 0], S32, 0xeaa127fa); /* 42 */
	HH(c, d, a, b, x[ 3], S33, 0xd4ef3085); /* 43 */
	HH(b, c, d, a, x[ 6], S34,  0x4881d05); /* 44 */
	HH(a, b, c, d, x[ 9], S31, 0xd9d4d039); /* 45 */
	HH(d, a, b, c, x[12], S32, 0xe6db99e5); /* 46 */
	HH(c, d, a, b, x[15], S33, 0x1fa27cf8); /* 47 */
	HH(b, c, d, a, x[ 2], S34, 0xc4ac5665); /* 48 */
	
	/* Round 4 */
	II(a, b, c, d, x[ 0], S41, 0xf4292244); /* 49 */
	II(d, a, b, c, x[ 7], S42, 0x432aff97); /* 50 */
	II(c, d, a, b, x[14], S43, 0xab9423a7); /* 51 */
	II(b, c, d, a, x[ 5], S44, 0xfc93a039); /* 52 */
	II(a, b, c, d, x[12], S41, 0x655b59c3); /* 53 */
	II(d, a, b, c, x[ 3], S42, 0x8f0ccc92); /* 54 */
	II(c, d, a, b, x[10], S43, 0xffeff47d); /* 55 */
	II(b, c, d, a, x[ 1], S44, 0x85845dd1); /* 56 */
	II(a, b, c, d, x[ 8], S41, 0x6fa87e4f); /* 57 */
	II(d, a, b, c, x[15], S42, 0xfe2ce6e0); /* 58 */
	II(c, d, a, b, x[ 6], S43, 0xa3014314); /* 59 */
	II(b, c, d, a, x[13], S44, 0x4e0811a1); /* 60 */
	II(a, b, c, d, x[ 4], S41, 0xf7537e82); /* 61 */
	II(d, a, b, c, x[11], S42, 0xbd3af235); /* 62 */
	II(c, d, a, b, x[ 2], S43, 0x2ad7d2bb); /* 63 */
	II(b, c, d, a, x[ 9], S44, 0xeb86d391); /* 64 */
	
	State[0] += a;
	State[1] += b;
	State[2] += c;
	State[3] += d;
	
	// Zeroize sensitive information.
	memset((uint8*)x, 0, sizeof(x));
}


void *Hash_MD5_Init(void)
{
	TMD5 *MD5;
	
	MD5 = malloc(sizeof(TMD5));
	
	MD5->Count[0] = 0;
	MD5->Count[1] = 0;
	
	// MD5 constants
	MD5->State[0] = 0x67452301;
	MD5->State[1] = 0xefcdab89;
	MD5->State[2] = 0x98badcfe;
	MD5->State[3] = 0x10325476;
	
	return MD5;
}


void Hash_MD5_Update(void *Handle, uint8 *Buffer, int lBuffer)
{
	TMD5 *MD5 = (TMD5*)Handle;
	uint32 i, nBytes, lPart;
	
	
	/* Compute number of bytes mod 64 */
	nBytes = (uint32)((MD5->Count[0] >> 3) & 0x3F);
	
	/* Update number of bits */
	if((MD5->Count[0] += ((uint32)lBuffer << 3)) < ((uint32)lBuffer << 3))
		MD5->Count[1]++;
	MD5->Count[1] += ((uint32)lBuffer >> 29);
	
	lPart = 64 - nBytes;
	
	// Transform as many times as possible.
	if(lBuffer >= lPart)
	{
		memcpy(&MD5->Buffer[nBytes], Buffer, lPart);
		_Hash_MD5_Transform(MD5->State, MD5->Buffer);
		
		for(i = lPart; i + 63 < lBuffer; i += 64)
			_Hash_MD5_Transform(MD5->State, &Buffer[i]);
		
		nBytes = 0;
	}
	else
		i = 0;
	
	// Buffer remaining input
	memcpy((uint8*)&MD5->Buffer[nBytes], (uint8*)&Buffer[i], lBuffer - i);
}


void Hash_MD5_Final(void *Handle, uint8 Digest[MD5_DIGEST_SIZE])
{
	TMD5 *MD5 = (TMD5*)Handle;
	uint8 Bits[8];
	uint32 nBytes, lPadding;
	
	
	// Save number of bits
	_Hash_MD5_Encode(Bits, MD5->Count, 8);
	
	// Pad out to 56 mod 64.
	nBytes = (uint32)((MD5->Count[0] >> 3) & 0x3f);
	lPadding = (nBytes < 56) ? (56 - nBytes) : (120 - nBytes);
	Hash_MD5_Update(MD5, _MD5_Padding, lPadding);
	
	/* Append length(before padding) */
	Hash_MD5_Update(MD5, Bits, 8);
	
	/* Store state in digest */
	_Hash_MD5_Encode(Digest, MD5->State, MD5_DIGEST_SIZE);
	
	// cleanup & zeroize sensitive information
	memset(MD5, 0, sizeof(TMD5));
	free(MD5);
}


void Hash_MD5_String(uint8 *Buffer, int lBuffer, char DigestStr[MD5_DIGEST_STRING_SIZE])
{
	void *Handle;
	uint8 Digest[MD5_DIGEST_SIZE];
	
	Handle = Hash_MD5_Init();
	Hash_MD5_Update(Handle, (uint8*)Buffer, lBuffer);
	Hash_MD5_Final(Handle, Digest);
	Hash_MD5_DigestToString(Digest, DigestStr);
}


/*
	CRC32 (Mark Adler, Cyclic Redundancy Check 32-Bit)
	
	This hash function is based on the good old CRC-32 (Cyclic Redundancy
	Check with 32 Bit) algorithm as invented by Mark Adler. It is one
	of the hash functions with medium performance but with very good
	distribution. So it can be considered as a rock solid general purpose
	hash function. It should be used if good distribution is more
	important than high performance.
*/
uint32 Hash_CRC32(uint8 *Key, int lKey)
{
	static uint32 CRC32Array[256] = {
		0x00000000, 0x77073096, 0xee0e612c, 0x990951ba, 0x076dc419, 0x706af48f, 0xe963a535, 0x9e6495a3,
		0x0edb8832, 0x79dcb8a4, 0xe0d5e91e, 0x97d2d988, 0x09b64c2b, 0x7eb17cbd, 0xe7b82d07, 0x90bf1d91,
		0x1db71064, 0x6ab020f2, 0xf3b97148, 0x84be41de, 0x1adad47d, 0x6ddde4eb, 0xf4d4b551, 0x83d385c7,
		0x136c9856, 0x646ba8c0, 0xfd62f97a, 0x8a65c9ec, 0x14015c4f, 0x63066cd9, 0xfa0f3d63, 0x8d080df5,
		0x3b6e20c8, 0x4c69105e, 0xd56041e4, 0xa2677172, 0x3c03e4d1, 0x4b04d447, 0xd20d85fd, 0xa50ab56b,
		0x35b5a8fa, 0x42b2986c, 0xdbbbc9d6, 0xacbcf940, 0x32d86ce3, 0x45df5c75, 0xdcd60dcf, 0xabd13d59,
		0x26d930ac, 0x51de003a, 0xc8d75180, 0xbfd06116, 0x21b4f4b5, 0x56b3c423, 0xcfba9599, 0xb8bda50f,
		0x2802b89e, 0x5f058808, 0xc60cd9b2, 0xb10be924, 0x2f6f7c87, 0x58684c11, 0xc1611dab, 0xb6662d3d,
		0x76dc4190, 0x01db7106, 0x98d220bc, 0xefd5102a, 0x71b18589, 0x06b6b51f, 0x9fbfe4a5, 0xe8b8d433,
		0x7807c9a2, 0x0f00f934, 0x9609a88e, 0xe10e9818, 0x7f6a0dbb, 0x086d3d2d, 0x91646c97, 0xe6635c01,
		0x6b6b51f4, 0x1c6c6162, 0x856530d8, 0xf262004e, 0x6c0695ed, 0x1b01a57b, 0x8208f4c1, 0xf50fc457,
		0x65b0d9c6, 0x12b7e950, 0x8bbeb8ea, 0xfcb9887c, 0x62dd1ddf, 0x15da2d49, 0x8cd37cf3, 0xfbd44c65,
		0x4db26158, 0x3ab551ce, 0xa3bc0074, 0xd4bb30e2, 0x4adfa541, 0x3dd895d7, 0xa4d1c46d, 0xd3d6f4fb,
		0x4369e96a, 0x346ed9fc, 0xad678846, 0xda60b8d0, 0x44042d73, 0x33031de5, 0xaa0a4c5f, 0xdd0d7cc9,
		0x5005713c, 0x270241aa, 0xbe0b1010, 0xc90c2086, 0x5768b525, 0x206f85b3, 0xb966d409, 0xce61e49f,
		0x5edef90e, 0x29d9c998, 0xb0d09822, 0xc7d7a8b4, 0x59b33d17, 0x2eb40d81, 0xb7bd5c3b, 0xc0ba6cad,
		0xedb88320, 0x9abfb3b6, 0x03b6e20c, 0x74b1d29a, 0xead54739, 0x9dd277af, 0x04db2615, 0x73dc1683,
		0xe3630b12, 0x94643b84, 0x0d6d6a3e, 0x7a6a5aa8, 0xe40ecf0b, 0x9309ff9d, 0x0a00ae27, 0x7d079eb1,
		0xf00f9344, 0x8708a3d2, 0x1e01f268, 0x6906c2fe, 0xf762575d, 0x806567cb, 0x196c3671, 0x6e6b06e7,
		0xfed41b76, 0x89d32be0, 0x10da7a5a, 0x67dd4acc, 0xf9b9df6f, 0x8ebeeff9, 0x17b7be43, 0x60b08ed5,
		0xd6d6a3e8, 0xa1d1937e, 0x38d8c2c4, 0x4fdff252, 0xd1bb67f1, 0xa6bc5767, 0x3fb506dd, 0x48b2364b,
		0xd80d2bda, 0xaf0a1b4c, 0x36034af6, 0x41047a60, 0xdf60efc3, 0xa867df55, 0x316e8eef, 0x4669be79,
		0xcb61b38c, 0xbc66831a, 0x256fd2a0, 0x5268e236, 0xcc0c7795, 0xbb0b4703, 0x220216b9, 0x5505262f,
		0xc5ba3bbe, 0xb2bd0b28, 0x2bb45a92, 0x5cb36a04, 0xc2d7ffa7, 0xb5d0cf31, 0x2cd99e8b, 0x5bdeae1d,
		0x9b64c2b0, 0xec63f226, 0x756aa39c, 0x026d930a, 0x9c0906a9, 0xeb0e363f, 0x72076785, 0x05005713,
		0x95bf4a82, 0xe2b87a14, 0x7bb12bae, 0x0cb61b38, 0x92d28e9b, 0xe5d5be0d, 0x7cdcefb7, 0x0bdbdf21,
		0x86d3d2d4, 0xf1d4e242, 0x68ddb3f8, 0x1fda836e, 0x81be16cd, 0xf6b9265b, 0x6fb077e1, 0x18b74777,
		0x88085ae6, 0xff0f6a70, 0x66063bca, 0x11010b5c, 0x8f659eff, 0xf862ae69, 0x616bffd3, 0x166ccf45,
		0xa00ae278, 0xd70dd2ee, 0x4e048354, 0x3903b3c2, 0xa7672661, 0xd06016f7, 0x4969474d, 0x3e6e77db,
		0xaed16a4a, 0xd9d65adc, 0x40df0b66, 0x37d83bf0, 0xa9bcae53, 0xdebb9ec5, 0x47b2cf7f, 0x30b5ffe9,
		0xbdbdf21c, 0xcabac28a, 0x53b39330, 0x24b4a3a6, 0xbad03605, 0xcdd70693, 0x54de5729, 0x23d967bf,
		0xb3667a2e, 0xc4614ab8, 0x5d681b02, 0x2a6f2b94, 0xb40bbe37, 0xc30c8ea1, 0x5a05df1b, 0x2d02ef8d
	};
	uint32 Hash;
	
	Hash = 0xffffffff;
	while(lKey-->0)
		Hash = CRC32Array[(Hash ^ *Key++) & 0xff] ^ (Hash >> 8);
	Hash ^= 0xffffffff;
	
	return Hash;
}


/*
	Hash from Sqlite
*/
uint32 Hash_SQLITE(uint8 *Key, int lKey)
{
	uint32 Hash;
	
	Hash = 0;
	while(lKey-- > 0)
	{
		Hash = (Hash << 3) ^ Hash ^ (*Key);
		Key++;
	}
	return Hash & 0x7fffffff;
}


/*
	Fowler/Noll/Vo hash
	
	The basis of this hash algorithm was taken from an idea sent
	as reviewer comments to the IEEE POSIX P1003.2 committee by:
	
	     Phong Vo (http://www.research.att.com/info/kpv/)
	     Glenn Fowler (http://www.research.att.com/~gsf/)
	
	In a subsequent ballot round:
	
	     Landon Curt Noll (http://www.isthe.com/chongo/)
	
	improved on their algorithm.  Some people tried this hash
	and found that it worked rather well.  In an EMail message
	to Landon, they named it the ``Fowler/Noll/Vo'' or FNV hash.
	
	FNV hashes are designed to be fast while maintaining a low
	collision rate. The FNV speed allows one to quickly hash lots
	of data while maintaining a reasonable collision rate.  See:
	
	     http://www.isthe.com/chongo/tech/comp/fnv/
	
	for more details as well as other forms of the FNV hash.
*/
// 32 bit magic FNV-0 and FNV-1 prime
#define FNV_32_PRIME	0x01000193

uint32 Hash_FNV32(uint8 *Key, int lKey)
{
	uint8 *KeyEnd = Key + lKey;
	uint32 Hash = FNV_32_PRIME;
	
	
#if 0
	// original implementation, around 3x slower than Hash_SQLITE()
	while(Key < KeyEnd)
	{
		// multiply by the 32 bit FNV magic prime mod 2^32
#if 0
		// this way is faster
		Hash *= FNV_32_PRIME;
#else
		Hash += (Hash << 1) + (Hash << 4) + (Hash << 7) + (Hash << 8) + (Hash << 24);
#endif
		
		// xor the bottom with the current octet
		Hash ^= (uint32)*Key++;
	}
#else
	// my implementation, around 15% only slower than Hash_SQLITE(),
	// which make this hash function a good alternative
	int lKeyMod4 = lKey % 4;
	
	// we just align for 4 block operations...
	switch(lKeyMod4)
	{
		case 3:
			Hash *= FNV_32_PRIME;
			Hash ^= (uint32)*Key++;
		case 2:
			Hash *= FNV_32_PRIME;
			Hash ^= (uint32)*Key++;
		case 1:
			Hash *= FNV_32_PRIME;
			Hash ^= (uint32)*Key++;
	}
	
	while(Key < KeyEnd)
	{
		Hash *= FNV_32_PRIME;
		Hash ^= (uint32)*Key++;
		
		Hash *= FNV_32_PRIME;
		Hash ^= (uint32)*Key++;
		
		Hash *= FNV_32_PRIME;
		Hash ^= (uint32)*Key++;
		
		Hash *= FNV_32_PRIME;
		Hash ^= (uint32)*Key++;
	}
#endif
	
	return Hash;
}


THashTable *HashTable_Create(int Modulus, int StringOffset, int lStringOffset, THash_Callback Callback)
{
	THashTable *HashTable;
	
	if(Modulus <= 0 || StringOffset == lStringOffset || !Callback)
		return NULL;
	
	if(!Math_IsPrime_32(Modulus))
		LOG_DEBUG("warning: hash table modulus should be a prime number");
	
	HashTable = malloc(sizeof(THashTable));
	memset(HashTable, 0, sizeof(THashTable));
	
	HashTable->Hash = malloc(Modulus * sizeof(THashTableRow));
	memset(HashTable->Hash, 0, Modulus * sizeof(THashTableRow));
	
	HashTable->Modulus = Modulus;
	HashTable->StringOffset = StringOffset;
	HashTable->lStringOffset = lStringOffset;
	HashTable->Callback = Callback;
	
	return HashTable;
}


void HashTable_Destroy(THashTable *HashTable, HashTable_DestroyDataCallback DestroyCallback)
{
	int i,j;
	
	if(HashTable)
	{
		if(HashTable->Hash)
		{
			THashTableRow *Row;
			
			for(i=0;i<HashTable->Modulus;i++)
			{
				Row = &HashTable->Hash[i];
				
				for(j=0;j<Row->nData;j++)
				{
					if(Row->Data)
					{
						if(DestroyCallback)
							DestroyCallback(Row->Data[j]);
					}
				}
				if(Row->Data)
					free(Row->Data);
			}
			free(HashTable->Hash);
		}
		free(HashTable);
	}
}


int HashTable_Add(THashTable *HashTable, void *Data)
{
	char *String;
	int lString, nData;
	uint32 iHash;
	THashTableRow *Row;
	
	
	if(!HashTable || !Data)
		return 0;
	
	memcpy(&String, Data + HashTable->StringOffset, sizeof(char*));
	memcpy(&lString, Data + HashTable->lStringOffset, sizeof(int));
	
	iHash = HashTable->Callback((uint8*)String, lString);
	iHash = iHash % HashTable->Modulus;
	Row = &HashTable->Hash[iHash];
	nData = Row->nData;
	Row->Data = realloc(Row->Data, (nData + 1) * sizeof(void*));
	Row->Data[nData] = Data;
	
	Row->nData++;
	HashTable->nItems++;
	
	//LOG_DEBUG("adding [%s](%d)->(%u)", String, lString, iHash);
	
	return 1;
}


int HashTable_Delete(THashTable *HashTable, char *String, int lString, HashTable_DestroyDataCallback DestroyCallback)
{
	char *StringFound;
	int i, j, lStringFound, nData;
	uint32 iHash;
	THashTableRow *Row;
	
	
	if(!HashTable || !String)
		return 0;
	
	iHash = HashTable->Callback((uint8*)String, lString);
	iHash = iHash % HashTable->Modulus;
	
	Row = &HashTable->Hash[iHash];
	
	nData = Row->nData;
	for(i=0;i<nData;i++)
	{
		memcpy(&StringFound, Row->Data[i] + HashTable->StringOffset, sizeof(char*));
		memcpy(&lStringFound, Row->Data[i] + HashTable->lStringOffset, sizeof(int));
		
		if(lStringFound != lString)
			continue;
		
		if(memcmp(StringFound, String, lString) != 0)
			continue;
		
		// delete element
		if(DestroyCallback)
			DestroyCallback(Row->Data[i]);
		// shift down
		for(j=i;j+1<nData;j++)
			Row->Data[j] = Row->Data[j + 1];
		// reduce memory
		Row->Data = realloc(Row->Data, (nData - 1) * sizeof(void*));
		Row->nData--;
		HashTable->nItems--;
		return 1;
	}
	return 0;
}


void * HashTable_Search(THashTable *HashTable, char *String, int lString)
{
	char *StringFound;
	int i, lStringFound, nData;
	uint32 iHash;
	THashTableRow *Row;
	
	
	if(!HashTable || !String)
		return NULL;
	
	iHash = HashTable->Callback((uint8*)String, lString);
	iHash = iHash % HashTable->Modulus;
	
	Row = &HashTable->Hash[iHash];
	
	nData = Row->nData;
	for(i=0;i<nData;i++)
	{
		memcpy(&StringFound, Row->Data[i] + HashTable->StringOffset, sizeof(char*));
		memcpy(&lStringFound, Row->Data[i] + HashTable->lStringOffset, sizeof(int));
		
		if(lStringFound != lString)
			continue;
		
		if(memcmp(StringFound, String, lString) != 0)
			continue;
		
		return Row->Data[i];
	}
	return NULL;
}


void *HashTable_First(THashTable *HashTable)
{
	THashTableRow *Row;
	int i, j;
	
	if(!HashTable)
		return NULL;
	
	HashTable->iCursor = 0;
	HashTable->jCursor = 0;
	
	
	for(i=0;i<HashTable->Modulus;i++)
	{
		Row = &HashTable->Hash[i];
		
		for(j=0;j<Row->nData;j++)
		{
			if(Row->Data[j])
			{
				HashTable->iCursor = i;
				HashTable->jCursor = j;
				return Row->Data[j];
			}
		}
	}
	return NULL;
}


void *HashTable_Next(THashTable *HashTable)
{
	THashTableRow *Row;
	int i,j;
	
	if(!HashTable)
		return NULL;
	
	j = HashTable->jCursor + 1;
	for(i=HashTable->iCursor;i<HashTable->Modulus;i++)
	{
		Row = &HashTable->Hash[i];
		
		for(;j<Row->nData;j++)
		{
			if(Row->Data[j])
			{
				HashTable->iCursor = i;
				HashTable->jCursor = j;
				return Row->Data[j];
			}
		}
		j = 0;
	}
	return NULL;
}


void HashTable_Dump(THashTable *HashTable)
{
	LOG_DEBUG("*** HashTable dump ***");
	if(!HashTable)
	{
		LOG_DEBUG("<empty>");
	}
	else
	{
		THashTableRow *Row;
		int i,j,lString;
		char *String;
		void *Data;
		
		LOG_DEBUG("Modulus:        %d", HashTable->Modulus);
		LOG_DEBUG("StringOffset:   %d", HashTable->StringOffset);
		LOG_DEBUG("lStringOffset:  %d", HashTable->lStringOffset);
		LOG_DEBUG("nItems:         %d", HashTable->nItems);
		
		for(i=0;i<HashTable->Modulus;i++)
		{
			Row = &HashTable->Hash[i];
			LOG_DEBUG("  row %d: %d elements", i, Row->nData);
			
			for(j=0;j<Row->nData;j++)
			{
				Data = Row->Data[j];
				memcpy(&String, Data + HashTable->StringOffset, sizeof(char*));
				memcpy(&lString, Data + HashTable->lStringOffset, sizeof(int));
				LOG_DEBUG("    %d: [%s](%d)", j, String, lString);
			}
		}
	}
	LOG_DEBUG("*** HashTable dump ***");
}


void HashTable_Statistics(THashTable *HashTable)
{
	LOG_DEBUG("*** HashTable statistics ***");
	if(!HashTable)
	{
		LOG_DEBUG("<empty>");
	}
	else
	{
		THashTableRow *Row;
		int i, ItemsByIndexMean, Var;
		
		
		LOG_DEBUG("Modulus:      %d", HashTable->Modulus);
		LOG_DEBUG("nItems:       %d", HashTable->nItems);
		
		ItemsByIndexMean = HashTable->nItems / HashTable->Modulus;
		LOG_DEBUG("Items/Index:  %d", ItemsByIndexMean);
		
		Var = 0;
		for(i=0;i<HashTable->Modulus;i++)
		{
			Row = &HashTable->Hash[i];
			if(Row->nData > ItemsByIndexMean)
			{
				LOG_DEBUG("  row %d: %d elements, +%d", i, Row->nData, Row->nData - ItemsByIndexMean);
				Var += (Row->nData - ItemsByIndexMean) * (Row->nData - ItemsByIndexMean);
			}
			else if(Row->nData < ItemsByIndexMean)
			{
				LOG_DEBUG("  row %d: %d elements, -%d", i, Row->nData, ItemsByIndexMean - Row->nData);
				Var += (ItemsByIndexMean - Row->nData) * (ItemsByIndexMean - Row->nData);
			}
			else
			{
				LOG_DEBUG("  row %d: %d elements, =0", i, Row->nData);
			}
		}
		
		Var = Math_ISqrt(Var / HashTable->Modulus);
		LOG_DEBUG("std:  %d", Var);
		if(HashTable->nItems > 0)
			LOG_DEBUG("std%%: %d%%", (100 * Var * HashTable->Modulus) / HashTable->nItems);
		
	}
	LOG_DEBUG("*** HashTable statistics ***");
}
