#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <libshred.h>

#define LINE_SIZE_MAX			1024
#define DEFAULT_INI_SEPARATOR	'='


TINISection * INIFile_FindSection(TINIFile *INIFile, char *Name)
{
	TINISection *pSection;
	int i;
	
	if(!INIFile || !Name)
		return NULL;
	
	for(i=0;i<INIFile->nSections;i++)
	{
		pSection = INIFile->Sections[i];
		
		if(strcmp(pSection->Name, Name)==0)
		{
			return pSection;
		}
	}
	return NULL;
}


char *INIFile_FindInSection(TINISection *pSection, char *Attribute)
{
	TINIAttribute *pAttribute;
	int i;
	
	if(!pSection || !Attribute)
		return NULL;
	
	for(i=0;i<pSection->nAttributes;i++)
	{
		pAttribute = pSection->Attributes[i];
		
		if(strcmp(pAttribute->Name, Attribute) == 0)
			return pAttribute->Value;
	}
	return NULL;
}


static int _INIFile_ReadLine(char *Buffer, int szBuffer, FILE *File)
{
	if(!Buffer || !File || szBuffer<=0)
		return -1;
	
	Buffer[0] = 0;
	if(!fgets(Buffer, szBuffer, File))
		return -1;
	
	return String_TrimEx(Buffer, szBuffer);
}


static int _INIFile_GetSectionName(char *Buffer, int MaxLength)
{
	int iStart, iEnd;
	
	if(!Buffer)
		return -1;
	
	iStart = 1;
	while(Buffer[iStart] == ' ' || Buffer[iStart] == '\t')
	{
		iStart++;
		if(iStart >= MaxLength)
			break;
	}
	
	iEnd=iStart+1;
	while(Buffer[iEnd] != ']')
	{
		iEnd++;
		if(iEnd >= MaxLength)
			break;
	}
	
	iEnd--;
	while(Buffer[iEnd] == ' ' || Buffer[iEnd] == '\t')
	{
		iEnd--;
		if(iEnd <= 0)
			break;
	}
	iEnd++;
	
	memmove(Buffer, Buffer+iStart, iEnd-iStart);
	Buffer[iEnd-iStart] = 0;
	
	return iEnd-iStart;
}


static int _INIFile_BreakLine(char *Line, char *Attribute, char *Value, int MaxLength, char Separator)
{
	int i, Length;
	
	if(!Line || !Attribute || !Value)
		return 0;
	
	Length = String_TrimEx(Line, MaxLength);
	
	for(i=0;i<Length;i++)
	{
		if(Line[i] == Separator)
		{
			memcpy(Attribute, Line, i);
			Attribute[i] = 0;
			memcpy(Value, Line+i+1, Length);
			Value[Length-i-1] = 0;
			
			String_TrimEx(Attribute, MaxLength);
			String_TrimEx(Value, MaxLength);
			return 1;
		}
	}
	return 0;
}


void INIFile_Destroy(TINIFile *INIFile)
{
	int i,j;
	
	if(!INIFile)
		return;
	
	if(INIFile->Sections)
	{
		for(i=0;i<INIFile->nSections;i++)
		{
			TINISection *pSection = INIFile->Sections[i];
			
			if(pSection)
			{
				if(pSection->Attributes)
				{
					for(j=0;j<pSection->nAttributes;j++)
					{
						TINIAttribute *pAttribute = pSection->Attributes[j];
						
						if(pAttribute)
						{
							if(pAttribute->Name)
								free(pAttribute->Name);
							if(pAttribute->Value)
								free(pAttribute->Value);
							free(pAttribute);
						}
					}
					free(pSection->Attributes);
				}
				
				if(pSection->Name)
					free(pSection->Name);
				
				free(pSection);
			}
		}
		free(INIFile->Sections);
	}
	free(INIFile);
}


char * INIFile_Search(TINIFile *INIFile, char *Section, char *Attribute)
{
	int i,j;
	
	if(!INIFile || !Attribute)
		return NULL;
	
	if(Section)
	{
		// search on one section
		
		for(i=0;i<INIFile->nSections;i++)
		{
			TINISection *pSection = INIFile->Sections[i];
			
			if(strcmp(pSection->Name, Section)==0)
			{
				for(j=0;j<pSection->nAttributes;j++)
				{
					TINIAttribute *pAttribute = pSection->Attributes[j];
					
					if(strcmp(pAttribute->Name, Attribute)==0)
					{
						if(!pAttribute->Value[0])
							return NULL;
						return pAttribute->Value;
					}
				}
			}
		}
	}
	else
	{
		// search on all sections
		
		for(i=0;i<INIFile->nSections;i++)
		{
			TINISection *pSection = INIFile->Sections[i];
			
			for(j=0;j<pSection->nAttributes;j++)
			{
				TINIAttribute *pAttribute = pSection->Attributes[j];
				
				if(strcmp(pAttribute->Name, Attribute)==0)
				{
					if(!pAttribute->Value[0])
						return NULL;
					return pAttribute->Value;
				}
			}
		}
	}
	
	return NULL;
}


char * INIFile_SearchEx(TINIFile *INIFile, char *Section, char *Attribute, int ID)
{
	int i,j,k;
	
	if(!INIFile || !Attribute)
		return NULL;
	
	k = 0;
	if(Section)
	{
		// search on one section
		
		for(i=0;i<INIFile->nSections;i++)
		{
			TINISection *pSection = INIFile->Sections[i];
			
			if(strcmp(pSection->Name, Section)==0)
			{
				for(j=0;j<pSection->nAttributes;j++)
				{
					TINIAttribute *pAttribute = pSection->Attributes[j];
					
					if(strcmp(pAttribute->Name, Attribute)==0)
					{
						if(!pAttribute->Value[0])
						{
							if(ID == k)
								return NULL;
							k++;
						}
						else
						{
							if(ID == k)
								return pAttribute->Value;
							k++;
						}
					}
				}
			}
		}
	}
	else
	{
		// search on all sections
		
		for(i=0;i<INIFile->nSections;i++)
		{
			TINISection *pSection = pSection;
			
			for(j=0;j<pSection->nAttributes;j++)
			{
				TINIAttribute *pAttribute = pSection->Attributes[j];
				
				if(strcmp(pAttribute->Name, Attribute)==0)
				{
					if(!pAttribute->Value[0])
					{
						if(ID == k)
							return NULL;
						k++;
					}
					else
					{
						if(ID == k)
							return pAttribute->Value;
						k++;
					}
				}
			}
		}
	}
	
	return NULL;
}


/* if Section = NULL, search on only GlobalSection, else search on given Section */
/* Variable is the name of the variable without '$' start character */
static char * _INIFile_SearchVariable(TINIFile *INIFile, char *SectionName, char *Variable)
{
	TINISection *pSection;
	int i,j;
	
	
	pSection = INIFile->Sections[0];
	for(j=0;j<pSection->nAttributes;j++)
	{
		TINIAttribute *pAttribute = pSection->Attributes[j];
		
		if(strcmp(&pAttribute->Name[1], Variable) == 0 && pAttribute->Name[0] == '$')
		{
			if(!pAttribute->Value[0])
				return NULL;
			return pAttribute->Value;
		}
	}
	
	if(SectionName)
	{
		// search on one section
		
		for(i=0;i<INIFile->nSections;i++)
		{
			pSection = INIFile->Sections[i];
			
			if(strcmp(pSection->Name, SectionName)==0)
			{
				for(j=0;j<pSection->nAttributes;j++)
				{
					TINIAttribute *pAttribute = pSection->Attributes[j];
					
					if(strcmp(pAttribute->Name + 1, Variable) == 0 && pAttribute->Name[0] == '$')
					{
						if(!pAttribute->Value[0])
							return NULL;
						return pAttribute->Value;
					}
				}
			}
		}
	}
	
	return NULL;
}


static void _INIFile_AddAttributeToSection(TINISection *Section, char *Attribute, char *Value)
{
	TINIAttribute **Attributes, *NewAttribute;
	
	
	NewAttribute = (TINIAttribute*)malloc(sizeof(TINIAttribute));
	NewAttribute->Name = strdup((Attribute ? Attribute : ""));
	NewAttribute->Value = strdup((Value ? Value : ""));
	
	Attributes = (TINIAttribute**)malloc((Section->nAttributes + 1) * sizeof(TINIAttribute*));
	memcpy(Attributes, Section->Attributes, Section->nAttributes * sizeof(TINIAttribute*));
	Attributes[Section->nAttributes] = NewAttribute;
	
	if(Section->Attributes)
		free(Section->Attributes);
	Section->Attributes = Attributes;
	Section->nAttributes++;
}


static void _INIFile_AddSection(TINIFile *INIFile, TINISection *Section)
{
	TINISection **Sections;
	
	Sections = (TINISection**)malloc((INIFile->nSections+1) * sizeof(TINISection*));
	memcpy(Sections, INIFile->Sections, INIFile->nSections * sizeof(TINISection*));
	Sections[INIFile->nSections] = Section;
	
	if(INIFile->Sections)
		free(INIFile->Sections);
	INIFile->Sections = Sections;
	INIFile->nSections++;
}


static TINISection * _INIFile_NewSection(char *SectionName)
{
	TINISection *Section;
	
	if(!SectionName)
		SectionName = "";
	Section = (TINISection*)malloc(sizeof(TINISection));
	memset(Section, 0, sizeof(TINISection));
	
	Section->Name = strdup(SectionName);
	
	return Section;
}


TINIFile *INIFile_Create(void)
{
	TINIFile *INIFile;
	INIFile = malloc(sizeof(TINIFile));
	memset(INIFile, 0, sizeof(TINIFile));
	return INIFile;
}


TINIFile *INIFile_ReadFromMemoryEx(char * Memory, char Separator)
{
	char LineBuffer[LINE_SIZE_MAX], Attribute[LINE_SIZE_MAX], Value[LINE_SIZE_MAX];
	int nRead, Length, LineNo;
	TINIFile *INIFile;
	TINISection *Section, *GlobalSection;
	int i, iFrom, FoundEOS;
	//char AbsPath[LINE_SIZE_MAX];
	
	
	INIFile = INIFile_Create();
	
	if(!Separator)
		Separator = DEFAULT_INI_SEPARATOR;
	
	GlobalSection = _INIFile_NewSection("");
	Section = GlobalSection;
	
	//getcwd(AbsPath, sizeof(AbsPath));
	//_INIFile_AddAttributeToSection(GlobalSection, "$ConfigPath", AbsPath);
	
	LineNo = -1;
	i = 0;
	FoundEOS = 0;
	while(1)
	{
		if(FoundEOS)
			break;
		
		iFrom = i;
		LineBuffer[0] = 0;
		while(1)
		{
			if(Memory[i] == '\n' || !Memory[i])
			{
				if(!Memory[i])
					FoundEOS = 1;
				strncpy(LineBuffer, Memory+iFrom, i-iFrom);
				LineBuffer[i-iFrom] = 0;	/* because strncpy sucks */
				i++;
				break;
			}
			i++;
		}
		nRead = String_TrimEx(LineBuffer, LINE_SIZE_MAX);
		LineNo++;
		
		if(!LineBuffer[0])
			continue;
		
		if(nRead < 0)
			break;
		
		if(!nRead)
			continue;
		
		if(LineBuffer[0] == '#' || LineBuffer[0] == ';')
			continue;
		
		if(LineBuffer[0] == '[')
		{
			Length = _INIFile_GetSectionName(LineBuffer, nRead);
			if(Length <= 0)
			{
				LOG_DEBUG("error: invalid section at line %d", LineNo);
				continue;
			}
			
			_INIFile_AddSection(INIFile, Section);
			Section = _INIFile_NewSection(LineBuffer);
			continue;
		}
		
		Attribute[0] = 0;
		Value[0] = 0;
		if(_INIFile_BreakLine(LineBuffer, Attribute, Value, LINE_SIZE_MAX, Separator))
		{
			_INIFile_AddAttributeToSection(Section, Attribute, Value);
			continue;
		}
		
		LOG_DEBUG("warning: unhandled attribute at line %d", LineNo);
	}
	
	_INIFile_AddSection(INIFile, Section);
	
	if(!INIFile->nSections)
	{
		INIFile_Destroy(INIFile);
		return NULL;
	}
	
	return INIFile;
}


TINIFile *INIFile_ReadEx(char *Filename, char Separator)
{
	FILE *File;
	char LineBuffer[LINE_SIZE_MAX], Attribute[LINE_SIZE_MAX], Value[LINE_SIZE_MAX];
	int nRead, Length, LineNo;
	TINIFile *INIFile;
	TINISection *Section, *GlobalSection;
	
	
	File = fopen(Filename, "r");
	if(!File)
		return NULL;
	
	INIFile = INIFile_Create();
	
	GlobalSection = _INIFile_NewSection("");
	Section = GlobalSection;
	
	if(!Separator)
		Separator = DEFAULT_INI_SEPARATOR;
	
	LineNo = -1;
	while(!feof(File))
	{
		memset(LineBuffer, 0, sizeof(LineBuffer));
		
		nRead = _INIFile_ReadLine(LineBuffer, LINE_SIZE_MAX - 1, File);
		LineNo++;
		
		if(nRead < 0)
			break;
		
		if(!nRead)
			continue;
		
		if(LineBuffer[0] == '#' || LineBuffer[0] == ';')
			continue;
		
		if(LineBuffer[0] == '[')
		{
			Length = _INIFile_GetSectionName(LineBuffer, nRead);
			if(Length <= 0)
			{
				LOG_DEBUG("error: invalid section at line %d", LineNo);
				continue;
			}
			
			_INIFile_AddSection(INIFile, Section);
			Section = _INIFile_NewSection(LineBuffer);
			continue;
		}
		
		Attribute[0] = 0;
		Value[0] = 0;
		if(_INIFile_BreakLine(LineBuffer, Attribute, Value, LINE_SIZE_MAX, Separator))
		{
			_INIFile_AddAttributeToSection(Section, Attribute, Value);
			continue;
		}
		
		LOG_DEBUG("warning: unhandled attribute at line %d", LineNo);
	}
	
	fclose(File);
	
	_INIFile_AddSection(INIFile, Section);
	
	if(!INIFile->nSections)
	{
		INIFile_Destroy(INIFile);
		return NULL;
	}
	
	return INIFile;
}


void INIFile_Dump(TINIFile *INIFile)
{
	int i,j;
	
	if(!INIFile)
		return;
	
	LOG_DEBUG("*** INIFile Dump ***");
	for(i=0;i<INIFile->nSections;i++)
	{
		TINISection *pSection = INIFile->Sections[i];
		
		LOG_DEBUG("Section %d/%d [%s] (%d)", i+1, INIFile->nSections, pSection->Name, pSection->nAttributes);
		for(j=0;j<pSection->nAttributes;j++)
		{
			TINIAttribute *pAttribute = pSection->Attributes[j];
			LOG_DEBUG("   [%s] = [%s]", pAttribute->Name, pAttribute->Value);
		}
	}
	LOG_DEBUG("********************");
}


int INIFile_Set(TINIFile *INIFile, char *Section, char *Attribute, char *Value)
{
	int i;
	TINISection *pSection;
	
	if(!Section)
		Section = "";
	
	if(!INIFile || !Section || !Attribute || !Value)
		return 0;
	
	/* add new section if necessary */
	pSection = INIFile_FindSection(INIFile, Section);
	if(!pSection)
	{
		pSection = _INIFile_NewSection(Section);
		_INIFile_AddSection(INIFile, pSection);
	}
	
	/* add new attribute */
	for(i=0;i<pSection->nAttributes;i++)
	{
		if(strcmp(pSection->Attributes[i]->Name, Attribute) == 0)
		{
			free(pSection->Attributes[i]->Value);
			pSection->Attributes[i]->Value = strdup((Value ? Value : ""));
			return 1;
		}
	}
	_INIFile_AddAttributeToSection(pSection, Attribute, Value);
	return 1;
}


void INIFile_Unset(TINIFile *INIFile, char *Section, char *Attribute)
{
	TINISection *pSection;
	int i;
	
	if(!INIFile || !Attribute)
		return;
	
	pSection = INIFile_FindSection(INIFile, Section);
	if(pSection)
	{
		for(i=0;i<pSection->nAttributes;i++)
		{
			TINIAttribute *pAttribute = pSection->Attributes[i];
			
			if(strcmp(pAttribute->Name, Attribute) == 0)
			{
				int LastParam;
				
				free(pAttribute->Name);
				free(pAttribute->Value);
				free(pAttribute);
				// this solution is faster but makes swap
				LastParam = pSection->nAttributes - 1;
				pSection->Attributes[i] = pSection->Attributes[LastParam];
				pSection->Attributes = realloc(pSection->Attributes, LastParam * sizeof(TINIAttribute*));
				pSection->nAttributes--;
				return;
			}
		}
	}
}


int INIFile_Write(TINIFile *INIFile, char *Filename)
{
	int i, j;
	FILE *File;
	
	if(!INIFile || !Filename)
		return 0;
	
	File = fopen(Filename, "w");
	if(!File)
		return 0;
	
	/* write first empty section */
	for(i=0;i<INIFile->nSections;i++)
	{
		TINISection *pSection = INIFile->Sections[i];
		
		if(pSection->Name[0])
			continue;
		
		//fprintf(File, "# header section\n");
		//fprintf(File, "# %d attributes\n", pSection->nAttributes);
		
		for(j=0;j<pSection->nAttributes;j++)
		{
			TINIAttribute *pAttribute = pSection->Attributes[j];
			fprintf(File, "%s = %s\n", pAttribute->Name, pAttribute->Value);
		}
		
		fprintf(File, "\n");
	}
	
	for(i=0;i<INIFile->nSections;i++)
	{
		TINISection *pSection = INIFile->Sections[i];
		
		if(!pSection->Name[0])
			continue;
		
		fprintf(File, "[%s]\n", pSection->Name);
		//fprintf(File, "# %d attributes\n", pSection->nAttributes);
		for(j=0;j<pSection->nAttributes;j++)
		{
			TINIAttribute *pAttribute = pSection->Attributes[j];
			fprintf(File, "%s = %s\n", pAttribute->Name, pAttribute->Value);
		}
		
		fprintf(File, "\n");
	}
	fclose(File);
	return 1;
}


static void _INIFile_TranslateLine(TINIFile *INIFile, char *SectionName, TINIAttribute *Attribute)
{
	int i, j, k, FoundVars;
	char NewLine[LINE_SIZE_MAX];
	
	
	i = 0;
	j = 0;
	FoundVars = 0;
	memset(NewLine, 0, sizeof(NewLine));
	while(Attribute->Value[i])
	{
		if(Attribute->Value[i] == '$')
		{
			if(Attribute->Value[i+1] != '{')
			{
				// invalid variable call, should be a extern variable
				NewLine[j] = Attribute->Value[i];
				i++;
				j++;
				continue;
			}
			k = i+2;
			while(1)
			{
				if(!Attribute->Value[k])
				{
					// invalid variable call, fatal error
					return;
				}
				if(Attribute->Value[k] == '}')
				{
					char *VarName, *VarValue;
					
					FoundVars = 1;
					Attribute->Value[k] = 0;
					VarName = Attribute->Value+i+2;
					i = k+1;
					
					VarValue = _INIFile_SearchVariable(INIFile, SectionName, VarName);
					if(VarValue)
					{
						k = 0;
						while(VarValue[k])
						{
							NewLine[j] = VarValue[k];
							k++;
							j++;
						}
					}
					else
					{
						//LOG_DEBUG("unknown variable '%s'", VarName);
					}
					
					break;
				}
				k++;
			}
			continue;
		}
		
		NewLine[j] = Attribute->Value[i];
		i++;
		j++;
	}
	
	if(FoundVars)
	{
		free(Attribute->Value);
		Attribute->Value = strdup(NewLine);
	}
}


int INIFile_Translate(TINIFile *INIFile)
{
	int i, j;
	
	
	for(i=0;i<INIFile->nSections;i++)
	{
		TINISection *pSection = INIFile->Sections[i];
		
		for(j=0;j<pSection->nAttributes;j++)
		{
			TINIAttribute *pAttribute = pSection->Attributes[j];
			
			_INIFile_TranslateLine(INIFile, pSection->Name, pAttribute);
		}
	}
	
	return 1;
}
