#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <time.h>
#include <sys/time.h>

#include <libshred.h>

/*
	MWC
	George Marsaglia's The mother of all random number generators
	producing uniformly distributed pseudo random 32 bit values
	witht period about 2^250.
*/
static void MWC_Init(uint32 Seed, uint32 *Seed1, uint32 *Seed2)
{
	uint32 n,*p;
	uint32 sNumber;
	uint32 number;
	
	// avoid stupid bug
	if(!Seed)
		Seed = 1;
	
	// initialize motheri with 9 random values the first time
	sNumber = Seed & 0x0000FFFF;	// the low 16 bits
	number = Seed & 0x7FFFFFFF;		// only want 31 bits
	
	p = Seed1;
	for(n=18;n--;)
	{
		number = 30903 * sNumber + (number >> 16);   // one line multiply-with-cary
		*p++ = sNumber = number & 0xFFFF;
		if(n == 9)
			p = Seed2;
	}
	// make cary 15 bits
	Seed1[0] &= 0x7FFF;
	Seed2[0] &= 0x7FFF;
}


// implementation of George Marsaglia's MWC (multiply-with-carry) random number generators
static uint32 MWC_Random(uint32 *Seed1, uint32 *Seed2)
{
	uint32 n, number1, number2;
	
	// move elements 1 to 8 to 2 to 9
	memmove(Seed1 + 2, Seed1 + 1, 8 * sizeof(int));
	memmove(Seed2 + 2, Seed2 + 1, 8 * sizeof(int));
	
	// Put the carry values in numberi
	number1 = Seed1[0];
	number2 = Seed2[0];
	
	// form the linear combinations
	number1 += 1941*Seed1[2]+1860*Seed1[3]+1812*Seed1[4]+1776*Seed1[5]+1492*Seed1[6]+1215*Seed1[7]+1066*Seed1[8]+12013*Seed1[9];
	number2 += 1111*Seed2[2]+2222*Seed2[3]+3333*Seed2[4]+4444*Seed2[5]+5555*Seed2[6]+6666*Seed2[7]+7777*Seed2[8]+9272*Seed2[9];
	
	// Save the high bits of numberi as the new carry
	Seed1[0] = number1 >> 16;
	Seed2[0] = number2 >> 16;
	
	// Put the low bits of numberi into Seedi[1]
	Seed1[1] = number1 & 0xFFFF;
	Seed2[1] = number2 & 0xFFFF;
	
	// combine the two 16 bit random numbers into one 32 bit
	n = (((uint32)Seed1[1]) << 16) + (uint32)Seed2[1];
	
	return n;
}


void Random_Destroy(TRandom *Rand)
{
	if(Rand)
		free(Rand);
}

TRandom * Random_Create(uint32 Seed)
{
	TRandom *Rand;
	
	Rand = malloc(sizeof(TRandom));
	memset(Rand, 0, sizeof(TRandom));
	
	if(!Seed)
		Seed = Random_PRNG();
	
	MWC_Init(Seed, Rand->Seed1, Rand->Seed2);
	
	return Rand;
}

void Random_Reset(TRandom * Rand, uint32 Seed)
{
	if(!Rand)
		return;
	
	if(!Seed)
		Seed = Random_PRNG();
	
	MWC_Init(Seed, Rand->Seed1, Rand->Seed2);
}

uint32 Random_Process(TRandom *Rand)
{
	int n;
	
	if(!Rand)
		return 0;
	
	n = MWC_Random(Rand->Seed1, Rand->Seed2);
	
	return n;
}


#if defined(LINUX) || defined(APPLE) || defined(FREEBSD)
#define RANDOM_FILE		"/dev/urandom"

uint32 Random_PRNG(void)
{
	FILE *File;
	uint32 i = 0;
	
	
	File = fopen(RANDOM_FILE, "rb");
	if(File)
	{
		fread(&i, sizeof(int), 1, File);
		fclose(File);
	}
	else
	{
		struct timeval tv;
		uint32 i0,i1,i2,i3;
		
		gettimeofday(&tv, NULL);
		
		i0 = ((((uint32)tv.tv_usec) ^ ((uint32)tv.tv_sec)) % 9001 + 1789) & 0x500000ff;
		i1 = ((((uint32)tv.tv_usec) ^ ((uint32)tv.tv_sec)) % 8999 + 2111) & 0x005000ff;
		i2 = ((((uint32)tv.tv_usec) ^ ((uint32)tv.tv_sec)) % 8887 + 3331) & 0x000050ff;
		i3 = ((((uint32)tv.tv_usec) ^ ((uint32)tv.tv_sec)) % 6163 + 3221) & 0x000000ff;
		
		i = ((i0 << 24) | (i1 << 16) | (i2 << 8) | i3);
	}
	
	return i;
}
#endif

#ifdef WIN32
uint32 Random_PRNG(void)
{
	uint32 u32 = 0;
	
	u32 ^= (uint32)GetCurrentThreadId();
#if OS_BITS == 64
	u32 ^= (uint32)((uint64)GetDesktopWindow() & 0xffffffff);
#else
	u32 ^= (uint32)GetDesktopWindow();
#endif
	u32 ^= (uint32)GetTickCount();
	u32 ^= (uint32)time(NULL);
	
	return u32;
}
#endif


#define RANDOM_CHARSET			"abcdefghijklmnopqrstuvwxyz0123456789"
#define RANDOM_CHARSET_LENGTH	(sizeof(RANDOM_CHARSET)-1)

int Random_String(TRandom *Rand, int MinLength, int MaxLength, char *Buffer)
{
	static char Charset[] = RANDOM_CHARSET;
	int i, k, Length, lCharset = RANDOM_CHARSET_LENGTH;
	
	if(MaxLength <= MinLength)
		Length = MinLength;
	else
		Length = MinLength + (Random_Process(Rand) % (MaxLength - MinLength));
	
	for(i=0;i<Length;i++)
	{
		k = Random_Process(Rand) % lCharset;
		Buffer[i] = Charset[k];
	}
	Buffer[i] = 0;
	
	return Length;
}
