#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <libshred.h>


//#define DEBUG_LOCAL


//////
// tools functions
//

static inline void _Uint32ToBuf(uint32 k, uint8 Buffer[4])
{
	int i;
	
	for(i=3;i>=0;i--)
	{
		Buffer[i] = (uint8)(k & 0x000000ff);
		k >>= 8;
	}
}

static inline void _Uint32FromBuf(uint8 Buffer[4], uint32 *pk)
{
	uint32 k;
	int i;
	
	k = 0;
	for(i=0;i<4;i++)
	{
		k <<= 8;
		k += (uint32)Buffer[i];
	}
	*pk = k;
}


static inline void _Uint16ToBuf(uint16 k, uint8 Buffer[2])
{
	int i;
	
	for(i=1;i>=0;i--)
	{
		Buffer[i] = (uint8)(k & 0x00ff);
		k >>= 8;
	}
}


static inline void _Uint16FromBuf(uint8 Buffer[2], uint16 *pk)
{
	uint16 k;
	int i;
	
	k = 0;
	for(i=0;i<2;i++)
	{
		k <<= 8;
		k += (uint16)Buffer[i];
	}
	*pk = k;
}


static char *_IDToString(uint32 ID, char *TmpBuf, int szTmpBuf)
{
	int i;
	
	if(szTmpBuf < 5)
		return NULL;
	
	_Uint32ToBuf(ID, (uint8*)TmpBuf);
	
	// replace \0 par space
	for(i=0;i<4;i++)
	{
		if(!TmpBuf[i])
			TmpBuf[i] = ' ';
	}
	TmpBuf[4] = 0;
	
	return TmpBuf;
}


static int _TLV_IsComplete(uint8 *Buffer, int szBuffer, int *pSize, int DoLog)
{
	int Size;
	
	if(szBuffer < TLV_HEAD_SIZE)
	{
		if(DoLog)
			LOG_DEBUG("error: TLV header is truncated: %d/%d bytes", szBuffer, (int)TLV_HEAD_SIZE);
		return 0;
	}
	
	// after the check of szBuffer, in order to probe for minimal read size
	if(!Buffer)
	{
		if(DoLog)
			LOG_DEBUG("error: NULL buffer");
		return 0;
	}
	
	_Uint32FromBuf(Buffer + 4, (uint32*)&Size);
	if(pSize)
		*pSize = Size;
	
	// check if the tlv is complete
	if(szBuffer - TLV_HEAD_SIZE < Size)
	{
		if(DoLog)
			LOG_DEBUG("error: TLV body is truncated: %d/%d bytes", (int)(szBuffer - TLV_HEAD_SIZE), Size);
		return 0;
	}
	
	return 1;
}


int TLV_IsComplete(uint8 *Buffer, int szBuffer, int *pSize)
{
	return _TLV_IsComplete(Buffer, szBuffer, pSize, 1);
}


// internal version
static inline int _TLV_GetLength(uint8 *Buffer, int szBuffer)
{
	int Size;
	
	if(szBuffer < TLV_HEAD_SIZE)
		return -1;
	
	_Uint32FromBuf(Buffer + 4, (uint32*)&Size);
	
	return Size;
}


int TLV_GetLength(uint8 *Buffer, int szBuffer)
{
	if(!Buffer)
		return -1;
	
	return _TLV_GetLength(Buffer, szBuffer);
}


void TLV_Buffer_Prepare(uint8 **pBuffer, int *pszBuffer)
{
	*pBuffer = NULL;
	*pszBuffer = 0;
}


void TLV_Buffer_Cleanup(uint8 **pBuffer, int *pszBuffer)
{
	if(pBuffer)
	{
		if(*pBuffer)
			free(*pBuffer);
		*pBuffer = NULL;
	}
	if(pszBuffer)
		*pszBuffer = 0;
}


//////
// append functions
//

void TLV_Append_Buffer(uint8 **pBuffer, int *pszBuffer, uint32 ID, uint8 *Buffer, int szBuffer)
{
	uint8 *In, *pIn;
	int lIn;
	
	
	if(!pBuffer || !pszBuffer || (!Buffer && szBuffer > 0) || szBuffer < 0)
	{
		char TmpBuf[5];
		_Uint32ToBuf(ID, (uint8*)TmpBuf);
		TmpBuf[4] = 0;
		LOG_DEBUG("error: invalid arguments for ID '%s' [0x%08x]", TmpBuf, ID);
		return;
	}
	
	In = *pBuffer;
	lIn = *pszBuffer;
	
	// realloc with new header+body
	In = realloc(In, lIn + TLV_HEAD_SIZE + szBuffer);
	
	// pointer to data to append
	pIn = In + lIn;
	
	// update length
	lIn += TLV_HEAD_SIZE + szBuffer;
	
	_Uint32ToBuf(ID, pIn);
	pIn += 4;
	
	_Uint32ToBuf(szBuffer, pIn);
	pIn += 4;
	
	if(szBuffer > 0)
		memcpy(pIn, Buffer, szBuffer);
	
	// update pointers
	*pBuffer = In;
	*pszBuffer = lIn;
}


void TLV_Append_UInt8(uint8 **pBuffer, int *pszBuffer, uint32 ID, uint8 Byte)
{
	TLV_Append_Buffer(pBuffer, pszBuffer, ID, &Byte, 1);
}


void TLV_Append_UInt16(uint8 **pBuffer, int *pszBuffer, uint32 ID, uint16 UInt16)
{
	uint8 Buf2[2];
	_Uint16ToBuf(UInt16, Buf2);	// re-order bytes
	TLV_Append_Buffer(pBuffer, pszBuffer, ID, Buf2, sizeof(Buf2));
}


void TLV_Append_UInt32(uint8 **pBuffer, int *pszBuffer, uint32 ID, uint32 UInt32)
{
	uint8 Buf4[4];
	_Uint32ToBuf(UInt32, Buf4);	// re-order bytes
	TLV_Append_Buffer(pBuffer, pszBuffer, ID, Buf4, sizeof(Buf4));
}


void TLV_Append_UInt64(uint8 **pBuffer, int *pszBuffer, uint32 ID, uint64 UInt64)
{
	uint8 Buf8[8];
	uint32 uHigh, uLow;
	
	uHigh = (UInt64 >> 32) & 0xffffffff;
	uLow = UInt64 & 0xffffffff;
	
	_Uint32ToBuf(uHigh, Buf8);	// re-order bytes
	_Uint32ToBuf(uLow, Buf8 + 4);
	
	TLV_Append_Buffer(pBuffer, pszBuffer, ID, Buf8, sizeof(Buf8));
}


void TLV_Append_String(uint8 **pBuffer, int *pszBuffer, uint32 ID, char *String)
{
	int lString = 0;
	if(String)
		lString = strlen(String);
	TLV_Append_Buffer(pBuffer, pszBuffer, ID, (uint8*)String, lString);
}


//////
// extract functions
//

static int _TLV_Extract(uint8 *Buffer, int szBuffer, uint32 ID, int iRefCount, uint8 **pBuffer, int *pszBuffer)
{
	uint32 TLVID, TLVSize;
	int Result, CurrentRefCount;
	
#ifdef DEBUG_LOCAL
	{
		// debug
		char TmpBuf[5];
		
		LOG_TRACE();
		LOG_DEBUG("  Buffer:      %p", Buffer);
		LOG_DEBUG("  szBuffer:    %d", szBuffer);
		LOG_DEBUG("  ID:          %s (%08x)", _IDToString(ID, TmpBuf, sizeof(TmpBuf)), ID);
		LOG_DEBUG("  iRefCount:   %d", iRefCount);
		LOG_DEBUG("  pBuffer:     %p", pBuffer);
		LOG_DEBUG("  pszBuffer:   %p", pszBuffer);
	}
#endif
	
	if(pBuffer)
		*pBuffer = NULL;
	if(pszBuffer)
		*pszBuffer = 0;
	
	Result = 0;
	CurrentRefCount = 0;
	
	while(1)
	{
		// EOTLV
		if(szBuffer <= 0)
			break;
		
		TLVSize = _TLV_GetLength(Buffer, szBuffer);
		if(TLVSize < 0)
			break;
		
		// get the tlv id
		_Uint32FromBuf(Buffer, &TLVID);
		
		// update buffer to tlv data
		Buffer += TLV_HEAD_SIZE;
		szBuffer -= TLV_HEAD_SIZE;
		
		if(TLVID == ID)
		{
			// we want the buffer data
			if(CurrentRefCount == iRefCount && pBuffer && pszBuffer)
			{
				*pBuffer = Buffer;
				*pszBuffer = TLVSize;
				Result = 1;
				break;
			}
			CurrentRefCount++;
		}
		
		// jump to next tlv
		Buffer += TLVSize;
		szBuffer -= TLVSize;
	}
	
	// we want to count the number of ocurrence
	if(iRefCount < 0)
		return CurrentRefCount;
	
	if(!Result)
	{
		// debug
		char TmpBuf[5];
		LOG_DEBUG("warning: tlv [%s] not found", _IDToString(ID, TmpBuf, sizeof(TmpBuf)));
	}
	
	return Result;
}


int TLV_Extract_RefCount(uint8 *Buffer, int szBuffer, uint32 ID)
{
#ifdef DEBUG_LOCAL
	LOG_TRACE();
#endif
	return _TLV_Extract(Buffer, szBuffer, ID, -1, NULL, NULL);
}


int TLV_Extract_UInt8(uint8 *Buffer, int szBuffer, uint32 ID, int iRefCount, uint8 *pByte)
{
	uint8 *Data;
	int szData;
	
#ifdef DEBUG_LOCAL
	LOG_TRACE();
#endif
	
	*pByte = 0;
	if(_TLV_Extract(Buffer, szBuffer, ID, iRefCount, &Data, &szData))
	{
		if(szData == 1)
		{
			*pByte = Data[0];
			return 1;
		}
	}
	return 0;
}


int TLV_Extract_UInt16(uint8 *Buffer, int szBuffer, uint32 ID, int iRefCount, uint16 *pUInt16)
{
	uint8 *Data;
	int szData;
	
#ifdef DEBUG_LOCAL
	LOG_TRACE();
#endif
	
	*pUInt16 = 0;
	if(_TLV_Extract(Buffer, szBuffer, ID, iRefCount, &Data, &szData))
	{
		if(szData == 2)
		{
			_Uint16FromBuf(Data, pUInt16);
			return 1;
		}
	}
	return 0;
}


int TLV_Extract_UInt32(uint8 *Buffer, int szBuffer, uint32 ID, int iRefCount, uint32 *pUInt32)
{
	uint8 *Data;
	int szData;
	
#ifdef DEBUG_LOCAL
	LOG_TRACE();
#endif
	
	*pUInt32 = 0;
	if(_TLV_Extract(Buffer, szBuffer, ID, iRefCount, &Data, &szData))
	{
		if(szData == 4)
		{
			_Uint32FromBuf(Data, pUInt32);
			return 1;
		}
	}
	return 0;
}


int TLV_Extract_UInt64(uint8 *Buffer, int szBuffer, uint32 ID, int iRefCount, uint64 *pUInt64)
{
	uint8 *Data;
	int szData;
	
#ifdef DEBUG_LOCAL
	LOG_TRACE();
#endif
	
	*pUInt64 = 0;
	if(_TLV_Extract(Buffer, szBuffer, ID, iRefCount, &Data, &szData))
	{
		if(szData == 8)
		{
			uint32 uHigh, uLow;
			
			_Uint32FromBuf(Data, &uHigh);
			_Uint32FromBuf(Data + 4, &uLow);
			
			*pUInt64 = (((uint64)uHigh) << 32) | ((uint64)uLow);
			return 1;
		}
	}
	return 0;
}


int TLV_Extract_String(uint8 *Buffer, int szBuffer, uint32 ID, int iRefCount, char **pString)
{
	uint8 *Data;
	int szData;
	
#ifdef DEBUG_LOCAL
	LOG_TRACE();
#endif
	
	*pString = NULL;
	if(_TLV_Extract(Buffer, szBuffer, ID, iRefCount, &Data, &szData))
	{
		// add a trailing '\0', if not present
		if(szData > 0)
		{
			int szString;
			char *String;
			
			szString = szData;
			if(Data[szData - 1])	// ending '\0' is missing
				szString++;
			
			String = malloc(szString);
			memcpy(String, Data, szData);
			String[szString - 1] = 0;
			*pString = String;
		}
		return 1;
	}
	return 0;
}


int TLV_Extract_Buffer(uint8 *Buffer, int szBuffer, uint32 ID, int iRefCount, uint8 **pChildBuffer, int *pszChildBuffer)
{
	uint8 *Data;
	int szData;
	
#ifdef DEBUG_LOCAL
	LOG_TRACE();
#endif
	
	*pChildBuffer = NULL;
	*pszChildBuffer = 0;
	if(_TLV_Extract(Buffer, szBuffer, ID, iRefCount, &Data, &szData))
	{
		*pChildBuffer = Data;
		*pszChildBuffer = szData;
		return 1;
	}
	return 0;
}


int TLV_Extract_UInt32AsUInt8(uint8 *Buffer, int szBuffer, uint32 ID, int iRefCount, uint32 *pUInt32, uint32 DefaultValue)
{
	uint8 Byte = 0;
	int Result;
	
	Result = TLV_Extract_UInt8(Buffer, szBuffer, ID, iRefCount, &Byte);
	if(Result)
		*pUInt32 = (uint32)Byte;
	else
		*pUInt32 = DefaultValue;
	return Result;
}


int TLV_Extract_UInt32AsUInt16(uint8 *Buffer, int szBuffer, uint32 ID, int iRefCount, uint32 *pUInt32, uint32 DefaultValue)
{
	uint16 UInt16 = 0;
	int Result;
	
	Result = TLV_Extract_UInt16(Buffer, szBuffer, ID, iRefCount, &UInt16);
	if(Result)
		*pUInt32 = (uint32)UInt16;
	else
		*pUInt32 = DefaultValue;
	return Result;
}


char *TLV_IDToString(uint32 ID, char *TmpBuf, int szTmpBuf)
{
	uint8 Buffer[4];
	char IDString[32];	// must be greater than 4*4+1
	int i, lIDString;
	
	_Uint32ToBuf(ID, Buffer);
	
	IDString[0] = 0;
	lIDString = 0;
	for(i=0;i<4;i++)
	{
		uint8 Byte = Buffer[i];
		
		if(Byte >= 32 && Byte <= 126)
			IDString[lIDString++] = Byte;
		else
#if 1
			IDString[lIDString++] = '.';
#else
			lIDString += String_PrintString(IDString + lIDString, sizeof(IDString) - lIDString - 1, "\\x%02x", Byte);
#endif
		IDString[lIDString] = 0;
	}
	
	String_Copy(IDString, TmpBuf, szTmpBuf);
	
	return TmpBuf;
}


static void _TLV_DumpRecursive(uint8 *Buffer, int szBuffer, int Depth, THasChildTLVClbk HasChildTLVClbk)
{
	char Indent[256];
	char IDString[32];
	int i, TLVSize, lIndent;
	uint32 TLVID;
	
	
	Indent[0] = 0;
	lIndent = 0;
	for(i=0;i<Depth;i++)
		lIndent += String_PrintString(Indent + lIndent, sizeof(Indent) - lIndent - 1, "   ");
	
	while(1)
	{
		int HasChildTLV;
		
		// EOTLV
		if(szBuffer <= 0)
			break;
		
		if(!_TLV_IsComplete(Buffer, szBuffer, &TLVSize, 0))
		{
			LOG_DEBUG("%s<payload> (size:%d)", Indent, szBuffer);
			break;
		}
		
		// get the tlv id
		_Uint32FromBuf(Buffer, &TLVID);
		
		HasChildTLV = HasChildTLVClbk && HasChildTLVClbk(TLVID);
		if(HasChildTLV)
			LOG_DEBUG("%s[%s] (size:%d)", Indent, TLV_IDToString(TLVID, IDString, sizeof(IDString)), TLVSize);
		else
		{
#define DUMP_MAX_LEN	16
			char TmpBuf[DUMP_MAX_LEN * 3 + 1];
			String_Escape(Buffer + TLV_HEAD_SIZE, MIN(TLVSize, DUMP_MAX_LEN), TmpBuf, sizeof(TmpBuf), 0);
			LOG_DEBUG("%s[%s] (size:%d) [%s%s]", Indent, TLV_IDToString(TLVID, IDString, sizeof(IDString)), TLVSize, TmpBuf, (TLVSize > DUMP_MAX_LEN) ? "..." : "");
		}
		
		// update buffer to tlv data
		Buffer += TLV_HEAD_SIZE;
		szBuffer -= TLV_HEAD_SIZE;
		
		if(HasChildTLV || !HasChildTLVClbk)
			_TLV_DumpRecursive(Buffer, TLVSize, Depth + 1, HasChildTLVClbk);
		
		// jump to next tlv
		Buffer += TLVSize;
		szBuffer -= TLVSize;
	}
}


void TLV_Dump(uint8 *Buffer, int szBuffer, THasChildTLVClbk HasChildTLVClbk)
{
	LOG_DEBUG("*** TLV_Dump ***");
	if(!Buffer)
	{
		LOG_DEBUG("<empty>");
	}
	else
	{
		_TLV_DumpRecursive(Buffer, szBuffer, 0, HasChildTLVClbk);
	}
	LOG_DEBUG("*** TLV_Dump ***");
}


int TLV_Read_Stream(TTLVStreamReadWriteBlock ReadProc, void *ReadHandle, uint32 ID, uint32 *pID, uint8 **pBuffer, int *pszBuffer)
{
	uint8 Head[TLV_HEAD_SIZE];
	uint8 *Buffer;
	int szBuffer;
	uint32 HeadID;
	
	
	if(!ReadProc || !ReadHandle || !pBuffer || !pszBuffer)
	{
		LOG_DEBUG("error: invalid parameters");
		return 0;
	}
	
	*pBuffer = NULL;
	*pszBuffer = 0;
	
	if(!ReadProc(ReadHandle, Head, sizeof(Head)))
	{
		//LOG_DEBUG("error: reading message header");
		return 0;
	}
	
	_Uint32FromBuf(Head, &HeadID);
	
	szBuffer = _TLV_GetLength(Head, sizeof(Head));
	Buffer = malloc(szBuffer);
	if(!ReadProc(ReadHandle, Buffer, szBuffer))
	{
		LOG_DEBUG("error: reading message body");
		free(Buffer);
		return 0;
	}
	
	if(ID != TLV_ID_ANY && HeadID != ID)
	{
		LOG_DEBUG("error: id mismatch: %08x / %08x", HeadID, ID);
		free(Buffer);
		return 0;
	}
	
	if(pID)
		*pID = HeadID;
	*pBuffer = Buffer;
	*pszBuffer = szBuffer;
	
	return 1;
}


int TLV_Write_Stream(TTLVStreamReadWriteBlock WriteProc, void *WriteHandle, uint32 ID, uint8 *Buffer, int szBuffer)
{
	uint8 *PktBuffer = NULL;
	int szPktBuffer = 0;
	int Result;
	
	// just generate header
	TLV_Buffer_Prepare(&PktBuffer, &szPktBuffer);
	TLV_Append_Buffer(&PktBuffer, &szPktBuffer, ID, Buffer, szBuffer);
	Result = WriteProc(WriteHandle, PktBuffer, szPktBuffer);
	TLV_Buffer_Cleanup(&PktBuffer, &szPktBuffer);
	if(!Result)
	{
		LOG_DEBUG("error: writing message");
		return 0;
	}
	return 1;
}
