#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef WIN32
#include <winsock2.h>
#include <windows.h>
#else
#include <pthread.h>
#include <errno.h>
#include <sys/time.h>
#endif

#include <libshred.h>


typedef struct
{
	void *	CS;
#ifdef WIN32
	HANDLE	hThread;
#endif
	int		IsRunning;
	int		DoStop;
	
	SlaveThreadProc Proc;
	void *	Data;
	
	int		SelfDestruct;
	int		ExitCode;
	
}TThread;

#define LOCK(u)		CriticalSection_Enter(u)
#define UNLOCK(u)	CriticalSection_Leave(u)


// does not work properly on mac os x
TThreadID Thread_GetCurrentThreadId(void)
{
#ifdef WIN32
	return (TThreadID)GetCurrentThreadId();
#else
	return (TThreadID)pthread_self();
#endif
}


void * CriticalSection_Open(void)
{
#ifdef WIN32
	CRITICAL_SECTION *CS;
	
	CS = malloc(sizeof(CRITICAL_SECTION));
	memset(CS, 0, sizeof(CRITICAL_SECTION));
	InitializeCriticalSection(CS);
	return (void*)CS;
#else
	pthread_mutex_t *Mutex;
	
	Mutex = malloc(sizeof(pthread_mutex_t));
	memset(Mutex, 0, sizeof(pthread_mutex_t));
	pthread_mutex_init(Mutex, NULL);
	return (void*)Mutex;
#endif
}


void CriticalSection_Close(void *Handle)
{
	if(!Handle)
	{
		LOG_DEBUG("warning: critical section handle is null");
		return;
	}
	
#ifdef WIN32
	DeleteCriticalSection((CRITICAL_SECTION*)Handle);
	free(Handle);
#else
	pthread_mutex_destroy((pthread_mutex_t*)Handle);
	free(Handle);
#endif
}


void CriticalSection_Enter(void *Handle)
{
	if(!Handle)
	{
		LOG_DEBUG("warning: critical section handle is null");
		return;
	}
	
#ifdef WIN32
	EnterCriticalSection((CRITICAL_SECTION*)Handle);
#else
	pthread_mutex_lock((pthread_mutex_t*)Handle);
#endif
}


void CriticalSection_Leave(void *Handle)
{
	if(!Handle)
	{
		LOG_DEBUG("warning: critical section handle is null");
		return;
	}
	
#ifdef WIN32
	LeaveCriticalSection((CRITICAL_SECTION*)Handle);
#else
	pthread_mutex_unlock((pthread_mutex_t*)Handle);
#endif
}


#if defined(LINUX) || defined(APPLE) || defined(FREEBSD)
// CPU stack size
#define STACK_SIZE	  32768	//	OPTIMIZE_SCALABILITY
//#define STACK_SIZE	  65536

typedef void* (*LPTHREAD_START_ROUTINE)(void *lpParameter);

static int _CreateThreadLinux(LPTHREAD_START_ROUTINE lpStartAddr, void *lpvThreadParam)
{
	pthread_attr_t pth_attr;
	pthread_t thread;
	
	
	pthread_attr_init(&pth_attr);
	pthread_attr_setdetachstate(&pth_attr, PTHREAD_CREATE_DETACHED);
	pthread_attr_setstacksize(&pth_attr, STACK_SIZE);
	
	
#ifdef HAVE_PTHREAD_SIGMASK
	sigset_t newmask, oldmask;
	
	//	The idea is that only the main thread handles all the signals with posix threads.
	//	Signals are blocked for any other thread.
	
	sigemptyset(&newmask);
	sigaddset(&newmask, SIGCHLD);
	sigaddset(&newmask, SIGTERM);
	sigaddset(&newmask, SIGQUIT);
	sigaddset(&newmask, SIGINT);
	sigaddset(&newmask, SIGHUP);
	pthread_sigmask(SIG_BLOCK, &newmask, &oldmask);	//	block signals
#endif
	
	if(pthread_create(&thread, &pth_attr, lpStartAddr, lpvThreadParam))
	{
#ifdef HAVE_PTHREAD_SIGMASK
		pthread_sigmask(SIG_SETMASK, &oldmask, NULL);	// restore the mask
#endif /* HAVE_PTHREAD_SIGMASK */
		return 0;
	}
	
#ifdef HAVE_PTHREAD_SIGMASK
	pthread_sigmask(SIG_SETMASK, &oldmask, NULL);	// restore the mask
#endif /* HAVE_PTHREAD_SIGMASK */
	
	pthread_attr_destroy(&pth_attr);
	
	return 1;
}
#endif


#ifdef WIN32
static uint32 WINAPI _InternalThreadWin32Proc(void *lpParameter)
{
	TThread *pThread = (TThread*)lpParameter;
	int ExitCode;
	
	
	ExitCode = pThread->Proc(pThread, pThread->Data);
	
	pThread->ExitCode = ExitCode;
	pThread->IsRunning = 0;
	
	if(pThread->SelfDestruct)
	{
		CriticalSection_Close(pThread->CS);
		CloseHandle(pThread->hThread);
		free(pThread);
	}
	
	return ExitCode;
}
#else
static void * _InternalThreadLinuxProc(void *lpParameter)
{
	TThread *pThread = (TThread*)lpParameter;
	int ExitCode;
	
	
	pThread->IsRunning = 1;
	
	// suspend
	LOCK(pThread->CS);
	UNLOCK(pThread->CS);
	
	ExitCode = pThread->Proc(pThread, pThread->Data);
	
	pThread->ExitCode = ExitCode;
	pThread->IsRunning = 0;
	
	if(pThread->SelfDestruct)
	{
		CriticalSection_Close(pThread->CS);
		free(pThread);
	}
	
	return NULL;
}
#endif


void * Thread_Master_Open(SlaveThreadProc Proc, void *Data)
{
	TThread *pThread;
#ifdef WIN32
	HANDLE hThread;
#else
	int Result;
#endif
	
	if(!Proc)
	{
		LOG_DEBUG("error: invalid arguments");
		return NULL;
	}
	
	pThread = malloc(sizeof(TThread));
	memset(pThread, 0, sizeof(TThread));
	
	pThread->CS = CriticalSection_Open();
	
	pThread->Proc = Proc;
	pThread->Data = Data;
	
#if defined(LINUX) || defined(APPLE) || defined(FREEBSD)
	// pthread has no suspended state, so use mutex to block child thread
	// lock before calling new thread, so child thread is suspended
	LOCK(pThread->CS);
	Result = _CreateThreadLinux(_InternalThreadLinuxProc, (void*)pThread);
	if(!Result)
	{
		// we garanty that all resources are freed if we return NULL
		LOG_DEBUG("error: CreateThread(): %u", errno);
		UNLOCK(pThread->CS);
		CriticalSection_Close(pThread->CS);
		pThread->CS = NULL;
		free(pThread);
		return NULL;
	}
#endif
	
#ifdef WIN32
	hThread = CreateThread(NULL, 0, (LPTHREAD_START_ROUTINE)_InternalThreadWin32Proc, (void*)pThread, CREATE_SUSPENDED, NULL);
	if(!hThread)
	{
		// we garanty that all resources are freed if we return NULL
		LOG_DEBUG("error: CreateThread(): %u", (uint32)GetLastError());
		CriticalSection_Close(pThread->CS);
		pThread->CS = NULL;
		free(pThread);
		return NULL;
	}
	pThread->hThread = hThread;
	pThread->IsRunning = 1;
	ResumeThread(hThread);
#endif
	
#if defined(LINUX) || defined(APPLE) || defined(FREEBSD)
	// wait that thread is in running state
	while(!pThread->IsRunning)
		Time_Sleep(1);
	
	// release thread
	UNLOCK(pThread->CS);
#endif
	
	// thread is running
	
	return (void*)pThread;
}


void Thread_Master_SetCloseEvent(void *Handle)
{
	TThread *pThread = (TThread*)Handle;
	
	if(!pThread)
		return;
	
	pThread->DoStop = 1;
}


int Thread_Slave_GetCloseEvent(void *Handle)
{
	TThread *pThread = (TThread*)Handle;
	
	if(!pThread)
		return 0;
	
	return (pThread->DoStop != 0);
}


int Thread_Slave_GetCloseEventEx(void *Handle, int Timeout)
{
	TThread *pThread = (TThread*)Handle;
	uint32 uWaitStart;
	
	if(!pThread)
		return 0;
	
	uWaitStart = Time_GetTick();
	while(1)
	{
		if(pThread->DoStop)
			return 1;
		
		if(Time_GetElapsedTick(uWaitStart) >= (uint32)Timeout)
			return 0;
		
		Time_Sleep(250);
	}
}


void Thread_Slave_SelfDestruct(void *Handle)
{
	TThread *pThread = (TThread*)Handle;
	
	if(!pThread)
		return;
	
	pThread->SelfDestruct = 1;
}


int Thread_Master_WaitThreadTermination(void *Handle, int Timeout, int Block)
{
	TThread *pThread = (TThread*)Handle;
	
	
	// fixme: may cause an infinite loop if Block = 0
	if(!pThread)
	{
		LOG_DEBUG("threads: invalid null handle");
		return 0;
	}
	
	if(Block)
	{
		while(pThread->IsRunning)
			Time_Sleep(Timeout);
		return 1;
	}
	else
	{
		if(pThread->IsRunning)
			return 0;
		return 1;
	}
}


int Thread_Master_Close(void *Handle)
{
	TThread *pThread = (TThread*)Handle;
	uint32 ExitCode = 0;
	
	
	if(!pThread)
		return 0;
	
	//GetExitCodeThread(pThread->hThread, &ExitCode);
	CriticalSection_Close(pThread->CS);
	pThread->CS = NULL;
	
#ifdef WIN32
	CloseHandle(pThread->hThread);
#endif
	ExitCode = pThread->ExitCode;
	free(pThread);
	
	return ExitCode;
}


int Thread_Master_IsAlive(void *Handle)
{
	TThread *pThread = (TThread*)Handle;
	int Alive = 0;
	
	
	if(!pThread)
		return 0;
	
	CriticalSection_Enter(pThread->CS);
	Alive = pThread->IsRunning;
	CriticalSection_Leave(pThread->CS);
	
	return Alive;
}


typedef struct
{
#ifdef WIN32
	HANDLE	Handle;
#else
	pthread_cond_t Cond;
	pthread_mutex_t Mutex;
	int		Signaled;
	int		nWaiters;
#endif
	
}TEvent;



void * Thread_Event_Create(void)
{
	TEvent *Event;
	
	Event = malloc(sizeof(TEvent));
	memset(Event, 0, sizeof(TEvent));
	
#ifdef WIN32
	Event->Handle = CreateEvent(NULL, 0, 0, NULL);
	if(!Event->Handle)
	{
		LOG_DEBUG("error: CreateEvent()");
		Thread_Event_Destroy(Event);
		return NULL;
	}
#else
	pthread_cond_init(&Event->Cond, NULL);
	pthread_mutex_init(&Event->Mutex, NULL);
#endif
	
	return Event;
}


void Thread_Event_Destroy(void *Handle)
{
	TEvent *Event = (TEvent*)Handle;
	
	if(Event)
	{
#ifdef WIN32
		if(Event->Handle)
			CloseHandle(Event->Handle);
#else
		pthread_cond_destroy(&Event->Cond);
		pthread_mutex_destroy(&Event->Mutex);
#endif
		free(Event);
	}
}


void Thread_Event_Raise(void *Handle)
{
	TEvent *Event = (TEvent*)Handle;
	
	if(!Event)
		return;
	
#ifdef WIN32
	if(!Event->Handle)
		return;
	
	SetEvent(Event->Handle);
#else
	pthread_mutex_lock(&Event->Mutex);
	if(Event->nWaiters > 0)
		pthread_cond_signal(&Event->Cond);
	else
		Event->Signaled = 1;
	pthread_mutex_unlock(&Event->Mutex);
#endif
}


int Thread_Event_Wait(void *Handle, int Timeout)
{
	TEvent *Event = (TEvent*)Handle;
	int Result = 0;
	
	
	if(!Event)
		return 0;
	
#ifdef WIN32
	{
		uint32 dwTimeout, dwResult;
		
		if(!Event->Handle)
			return 0;
		
		if(Timeout < 0)
			dwTimeout = INFINITE;
		else
			dwTimeout = Timeout;
		
		dwResult = WaitForSingleObject(Event->Handle, dwTimeout);
		Result = 0;
		if(dwResult == WAIT_OBJECT_0)
			Result = 1;
	}
#else
	pthread_mutex_lock(&Event->Mutex);
	Result = 1;
	if(!Event->Signaled)
	{
		// normal behaviour, the event was not raised before we locked the mutex
		struct timespec TS;
		
		Event->nWaiters++;
		
		if(Timeout >= 0)
		{
#ifdef CLOCK_REALTIME
			clock_gettime(CLOCK_REALTIME, &TS);
#else
			struct timeval TV;
			
			gettimeofday(&TV, NULL);
			TS.tv_sec = TV.tv_sec;
			TS.tv_nsec = TV.tv_usec * 1000;
#endif
			
			TS.tv_sec += Timeout / 1000;
			TS.tv_nsec += (Timeout % 1000) * 1000 * 1000;
			if(TS.tv_nsec >= 1000 * 1000 * 1000)
			{
				TS.tv_nsec -= 1000 * 1000 * 1000;
				TS.tv_sec += 1;
			}
			if(pthread_cond_timedwait(&Event->Cond, &Event->Mutex, &TS) == ETIMEDOUT)
				Result = 0;
		}
		else
		{
			pthread_cond_wait(&Event->Cond, &Event->Mutex);
		}
		
		Event->nWaiters--;
	}
	else
	{
		Event->Signaled = 0;
	}
	pthread_mutex_unlock(&Event->Mutex);
#endif
	
	return Result;
}


void *Thread_Mutex_Open(void)
{
#ifdef WIN32
	HANDLE Mutex;
	
	Mutex = CreateMutex(NULL, 0, NULL);
	if(!Mutex)
	{
		LOG_DEBUG("error: CreateMutex()");
		return NULL;
	}
	return Mutex;
#else
	pthread_mutex_t *Mutex;
	
	Mutex = malloc(sizeof(pthread_mutex_t));
	memset(Mutex, 0, sizeof(pthread_mutex_t));
	pthread_mutex_init(Mutex, NULL);
	return (void*)Mutex;
#endif
}


void Thread_Mutex_Close(void *Handle)
{
	if(!Handle)
	{
		LOG_DEBUG("warning: mutex handle is null");
		return;
	}
	
#ifdef WIN32
	CloseHandle((HANDLE)Handle);
#else
	pthread_mutex_destroy((pthread_mutex_t*)Handle);
	free(Handle);
#endif
}


int Thread_Mutex_Wait(void *Handle, int Timeout)
{
	if(!Handle)
	{
		LOG_DEBUG("warning: mutex handle is null");
		return 0;
	}
#ifdef APPLE
	// fixme: MAC OS X does not implement pthread_mutex_timedlock
	// see http://lists.apple.com/archives/Unix-porting/2008/Jan/msg00014.html
	return 0;
#else	// APPLE
#ifdef WIN32
	{
		DWORD dwWaitResult;
		
		// INFINITE = 0xffffffff
		dwWaitResult = WaitForSingleObject((HANDLE)Handle, (DWORD)Timeout);
		switch(dwWaitResult)
		{
			case WAIT_OBJECT_0:
				return 1;
			
			case WAIT_TIMEOUT:
				return 0;
			
			case WAIT_ABANDONED:
				LOG_DEBUG("warning: WaitForSingleObject() returned WAIT_ABANDONED");
				return 0;
		}
		return 0;
	}
#else
	if(!Timeout)
	{
		if(pthread_mutex_trylock((pthread_mutex_t*)Handle) == 0)
			return 1;
		return 0;
	}
	else
	{
		struct timespec TS;
		
#ifdef CLOCK_REALTIME
		clock_gettime(CLOCK_REALTIME, &TS);
#else
		struct timeval TV;
			
		gettimeofday(&TV, NULL);
		TS.tv_sec = TV.tv_sec;
		TS.tv_nsec = TV.tv_usec * 1000;
#endif
		
		if(Timeout < 0)
			Timeout = 2147483647;	// max int32 > 0
		
		TS.tv_sec += Timeout / 1000;
		TS.tv_nsec += (Timeout % 1000) * 1000 * 1000;
		if(TS.tv_nsec >= 1000 * 1000 * 1000)
		{
			TS.tv_nsec -= 1000 * 1000 * 1000;
			TS.tv_sec += 1;
		}
		
		if(pthread_mutex_timedlock((pthread_mutex_t*)Handle, &TS) == 0)
			return 1;
		return 0;
	}
#endif
#endif	// APPLE
}


void Thread_Mutex_Release(void *Handle)
{
	if(!Handle)
	{
		LOG_DEBUG("warning: mutex handle is null");
		return;
	}
	
#ifdef WIN32
	ReleaseMutex((HANDLE)Handle);
#else
	pthread_mutex_unlock((pthread_mutex_t*)Handle);
#endif
}
