#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <libshred.h>


static char *_ToNetmaskString(uint32 dwNetmask, char *TmpBuf, int szTmpBuf)
{
	char Buf[32];
	snprintf(TmpBuf, szTmpBuf - 1, "%s(/%d)", IP_Uint32ToIP(dwNetmask, Buf, sizeof(Buf)), IP_NetmaskToCidr(dwNetmask));
	return TmpBuf;
}


void ShowIP(void)
{
	TInterfaceInfo *InterfaceInfo = NULL;
	int i, szInterfaceInfo;
	
	
	printf("Host ip addresses\n");
	if(IP_GetInterfaceTable(&InterfaceInfo, &szInterfaceInfo))
	{
		if(InterfaceInfo)
		{
			char TmpIP[20], TmpNetmask[20], TmpNetwork[20], TmpBroadcast[20], TmpMAC[20];
			uint32 dwIP, dwNetmask;
			
			
			printf("  %-18s %-16s %-24s %-16s %-16s %-20s %s\n", "MAC", "IP", "Netmask", "Network", "Broadcast", "Name", "Description");
			for(i=0;i<szInterfaceInfo;i++)
			{
				dwIP = InterfaceInfo[i].dwIP;
				dwNetmask = InterfaceInfo[i].dwNetmask;
				
				printf("  %-18s %-16s %-24s %-16s %-16s %-20s %s\n", 
					IP_MACToString(InterfaceInfo[i].bMAC, TmpMAC, sizeof(TmpMAC)), 
					IP_Uint32ToIP(dwIP, TmpIP, sizeof(TmpIP)), 
					_ToNetmaskString(dwNetmask, TmpNetmask, sizeof(TmpNetmask)),
					IP_Uint32ToIP(IP_GetNetworkAddr(dwIP, dwNetmask), TmpNetwork, sizeof(TmpNetwork)), 
					IP_Uint32ToIP(IP_GetBroadcastAddr(dwIP, dwNetmask), TmpBroadcast, sizeof(TmpBroadcast)), 
					InterfaceInfo[i].Name, 
					InterfaceInfo[i].Description
				);
			}
			printf("\n");
			
			IP_FreeInterfaceTable(InterfaceInfo, szInterfaceInfo);
		}
		else
		{
			printf("error: no returned value\n");
		}
	}
	else
	{
		printf("error: IP_GetInterfaceTable()\n");
	}
	printf("\n");
}


void ShowRoutes(void)
{
	TRouteInfo *RouteInfo = NULL;
	int i, szRouteInfo;
	
	
	printf("Host routes table\n");
	if(IP_GetRouteTable(&RouteInfo, &szRouteInfo))
	{
		if(RouteInfo)
		{
			char TmpDestination[20], TmpGateway[20], TmpNetmask[64];
			uint32 dwDestination, dwGateway, dwNetmask;
			
			
			printf("  %-16s %-16s %-16s\n", "Destination", "Gateway", "Netmask");
			for(i=0;i<szRouteInfo;i++)
			{
				dwDestination = RouteInfo[i].dwDestination;
				dwGateway = RouteInfo[i].dwGateway;
				dwNetmask = RouteInfo[i].dwNetmask;
				
				printf("  %-16s %-16s %-24s\n", 
					IP_Uint32ToIP(dwDestination, TmpDestination, sizeof(TmpDestination)), 
					IP_Uint32ToIP(dwGateway, TmpGateway, sizeof(TmpGateway)), 
					_ToNetmaskString(dwNetmask, TmpNetmask, sizeof(TmpNetmask))
				);
			}
			printf("\n");
			
			free(RouteInfo);
		}
		else
		{
			printf("error: no returned value\n");
		}
	}
	else
	{
		printf("error: IP_GetRouteTable()\n");
	}
	printf("\n");
}


void ShowARP(void)
{
	TARPInfo *ARPInfo = NULL;
	int i, szARPInfo;
	
	
	printf("Host arp table\n");
	if(IP_GetARPTable(&ARPInfo, &szARPInfo))
	{
		if(ARPInfo)
		{
			char TmpIP[20], TmpMAC[20];
			uint32 dwIP;
			
			
			printf("  %-16s %-18s\n", "IP", "MAC");
			for(i=0;i<szARPInfo;i++)
			{
				dwIP = ARPInfo[i].dwIP;
				printf("  %-16s %-18s\n", IP_Uint32ToIP(dwIP, TmpIP, sizeof(TmpIP)), IP_MACToString(ARPInfo[i].bMAC, TmpMAC, sizeof(TmpMAC)));
			}
			printf("\n");
			
			free(ARPInfo);
		}
		else
		{
			printf("error: no returned value\n");
		}
	}
	else
	{
		printf("error: IP_GetARPTable()\n");
	}
	printf("\n");
}


void ShowDNS(void)
{
	char Buffer[256];
	uint32 *dwDNSServers;
	int i, lDNSServers;
	
	
	printf("DNS info\n");
	
	IP_GetDnsHostname(Buffer, sizeof(Buffer));
	printf("  hostname: %s\n", Buffer);
	
	IP_GetDnsDomain(Buffer, sizeof(Buffer));
	printf("  domain: %s\n", Buffer);
	
	lDNSServers = 0;
	IP_GetDnsServers(&dwDNSServers, &lDNSServers);
	printf("  dns: %d\n", lDNSServers);
	for(i=0;i<lDNSServers;i++)
	{
		uint32 dwIP = dwDNSServers[i];
		printf("    %-16s\n", IP_Uint32ToIP(dwIP, Buffer, sizeof(Buffer)));
	}
	if(dwDNSServers)
		free(dwDNSServers);
	printf("\n");
}


int main(int nArgs, char **Args)
{
	int DoShowIP = 0;
	int DoShowRoutes = 0;
	int DoShowARP = 0;
	int DoShowDNS = 0;
	TCliParserOption Options[] = {
		{"-i", "--ip", "show ip config", "", &DoShowIP, ARG_TYPE_FLAG, 0},
		{"-r", "--routes", "show routes table", "", &DoShowRoutes, ARG_TYPE_FLAG, 0},
		{"-a", "--arp", "show arp table", "", &DoShowARP, ARG_TYPE_FLAG, 0},
		{"-d", "--dns", "show dns info", "", &DoShowDNS, ARG_TYPE_FLAG, 0},
		{NULL,	NULL,	NULL, NULL, 0, 0}
	};
	
	if(!CliParser_DoOptions(Options, Args + 1, nArgs - 1))
		return 0;
	
	if(!DoShowIP && !DoShowRoutes && !DoShowARP && !DoShowDNS)
	{
		DoShowIP = 1;
		DoShowRoutes = 1;
		DoShowARP = 1;
		DoShowDNS = 1;
	}
	
	if(DoShowIP)
		ShowIP();
	if(DoShowRoutes)
		ShowRoutes();
	if(DoShowARP)
		ShowARP();
	if(DoShowDNS)
		ShowDNS();
	
	return 0;
}
