#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <libshred.h>

#define SMTP_PORT			25
#define SMTP_TIMEOUT		8000	// in ms.


static int SMTP_ReadLine(int Socket, char *Line, int szLine, int ExpectedCode)
{
	int szRead, Offset, lLine;
	
	lLine = 0;
_again_:
	szRead = Net_Read(Socket, (uint8*)(Line + lLine), szLine - lLine);
	if(szRead <= 2)
		return 0;
	
	Offset = szRead - 2;
	if(Line[Offset] == '\r' && Line[Offset + 1] == '\n')
	{
		Line[Offset] = 0;
		
		if(ExpectedCode)
		{
			int Code = 0;
			
			sscanf(Line, "%d", &Code);
			if(Code != ExpectedCode)
				LOG_DEBUG("warning: unexpected server reply: [%s]", Line);
		}
		
		return 1;
	}
	
	lLine += szRead;
	
	// overflow: buffer too small
	if(lLine >= szLine - 2)
	{
		LOG_DEBUG("error: buffer too small: [%.*s]", lLine, Line);
		return 0;
	}
	
	goto _again_;
}


static char *SMTP_GetMessageID(char MessageID[128])
{
	uint8 ID[16];
	char MsgID[128], *pMsgID;
	int i, lMsgID;
	uint32 k;
	
	k = Random_PRNG();
	srand(Time_GetTime());
	for(i=0;i<sizeof(ID);i++)
	{
		k = (k + (rand() % 1237)) ^ (k << 13);
		ID[i] = (uint8)((k << 3) & 0x000000ff);
	}
	lMsgID = sizeof(MsgID);
	pMsgID = &MsgID[0];
	String_BufferToHex(ID, sizeof(ID), &pMsgID, &lMsgID);
	
	String_Copy(MsgID, MessageID, 128);
	
	return &MessageID[0];
}


int main(int nArgs, char **Args)
{
	char *Server = NULL;
	int Port = SMTP_PORT;
	char *From = NULL;
	char *To = NULL;
	char *Cc = NULL;
	char *Subject = NULL;
	char *Body = NULL;
	TCliParserOption Options[] = {
		{"-s", "--server", "server to connect to", "smtp.free.fr", &Server, ARG_TYPE_STRING, 1},
		{"-p", "--port", "port to connect to", "25", &Port, ARG_TYPE_INT, 0},
		{NULL, "--from", "from: address", "", &From, ARG_TYPE_STRING, 1},
		{NULL, "--to", "to: address", "", &To, ARG_TYPE_STRING, 1},
		{NULL, "--cc", "cc: address", "", &Cc, ARG_TYPE_STRING, 0},
		{NULL, "--subject", "subject", "", &Subject, ARG_TYPE_STRING, 1},
		{NULL, "--body", "body as string", "", &Body, ARG_TYPE_STRING, 0},
		{NULL,	NULL,	NULL, NULL, 0, 0}
	};
	uint32 dwServerIP;
	uint16 wServerPort;
	char TmpBuf[32];
	int Socket = -1;
	char HostFQDN[128];
	char Line[128];
	int lLine;
	int Success = 0;
	
	/*
	char *From = "canniclac@free.fr";
	char *To = "canniclac@free.fr";
	char *Cc = NULL;
	char *Subject = "Something Awful !!!";
	char *Body = "Pouet pouet";
	*/
	char *ReplyTo = NULL;
	
	
	if(!CliParser_DoOptions(Options, Args + 1, nArgs - 1))
		return 1;
	
	
	Net_Init();
	
	LOG_DEBUG("resolving host name [%s]", Server);
	if(!IP_GetHostIP(Server, &dwServerIP))
	{
		LOG_DEBUG("error: resolving host name %s", Server);
		goto _end_;
	}
	dwServerIP = Net_ToNet32(dwServerIP);
	wServerPort = Net_ToNet16(Port);
	
	LOG_DEBUG("resolved server: %s => %s", Server, Net_Uint32ToIP(dwServerIP, TmpBuf, sizeof(TmpBuf)));
	
	{
		char Hostname[64];
		char Domain[64];
		
		IP_GetDnsHostname(Hostname, sizeof(Hostname));
		IP_GetDnsDomain(Domain, sizeof(Domain));
		if(Domain[0])
			snprintf(HostFQDN, sizeof(HostFQDN) - 1, "%s.%s", Hostname, Domain);
		else
			snprintf(HostFQDN, sizeof(HostFQDN) - 1, "%s", Hostname);
	}
	LOG_DEBUG("host fqdn: %s", HostFQDN);
	
	
	Socket = Net_OpenTCPSocket();
	if(Socket < 0)
	{
		LOG_DEBUG("error: Net_OpenTCPSocket()");
		goto _end_;
	}
	if(!Net_ConnectWithTimeout(Socket, dwServerIP, wServerPort, SMTP_TIMEOUT))
	{
		LOG_DEBUG("error: connecting to server %s", Server);
		goto _end_;
	}
	
	// server line 0, should be something like '220 smtp2-g21.free.fr ESMTP Postfix'
	if(!SMTP_ReadLine(Socket, Line, sizeof(Line), 220))
	{
		LOG_DEBUG("error: reading server first line");
		goto _end_;
	}
	
	// our first line
	lLine = snprintf(Line, sizeof(Line) - 1, "HELO %s\r\n", HostFQDN);
	if(!Net_WriteBlock(Socket, (uint8*)Line, lLine))
	{
		LOG_DEBUG("error: writing HELO");
		goto _end_;
	}
	
	//---
	
	// server line 1, should be '250 smtp2-g21.free.fr'
	if(!SMTP_ReadLine(Socket, Line, sizeof(Line), 250))
	{
		LOG_DEBUG("error: reading server line");
		goto _end_;
	}
	
	//---
	
	lLine = snprintf(Line, sizeof(Line) - 1, "MAIL FROM: <%s>\r\n", From);
	if(!Net_WriteBlock(Socket, (uint8*)Line, lLine))
	{
		LOG_DEBUG("error: writing header 'MAIL FROM'");
		goto _end_;
	}
	
	// server line, should be '250 2.1.0 Ok'
	if(!SMTP_ReadLine(Socket, Line, sizeof(Line), 250))
	{
		LOG_DEBUG("error: reading server line");
		goto _end_;
	}
	
	//---
	
	lLine = snprintf(Line, sizeof(Line) - 1, "RCPT TO: <%s>\r\n", To);
	if(!Net_WriteBlock(Socket, (uint8*)Line, lLine))
	{
		LOG_DEBUG("error: writing header 'RCPT TO'");
		goto _end_;
	}
	
	// server line, should be '250 2.1.0 Ok'
	if(!SMTP_ReadLine(Socket, Line, sizeof(Line), 250))
	{
		LOG_DEBUG("error: reading server line");
		goto _end_;
	}
	
	//---
	
	lLine = snprintf(Line, sizeof(Line) - 1, "DATA\r\n");
	if(!Net_WriteBlock(Socket, (uint8*)Line, lLine))
	{
		LOG_DEBUG("error: writing header 'DATA'");
		goto _end_;
	}
	
	// server line, should be '354 End data with <CR><LF>.<CR><LF>'
	if(!SMTP_ReadLine(Socket, Line, sizeof(Line), 354))
	{
		LOG_DEBUG("error: reading server line");
		goto _end_;
	}
	
	//---
	{
		TAppendInfo DataAI;
		char *Data;
		int lData;
		char MessageID[128];
		
		
		String_Append_Init(&DataAI, 0);
		String_Append_StringArgs(&DataAI, "Message-ID: <%s@%s>\r\n", SMTP_GetMessageID(MessageID), HostFQDN);
		String_Append_StringArgs(&DataAI, "From: <%s>\r\n", From);
		String_Append_StringArgs(&DataAI, "To: <%s>\r\n", To);
		if(Cc)
			String_Append_StringArgs(&DataAI, "Cc: <%s>\r\n", Cc);
		String_Append_StringArgs(&DataAI, "Subject: %s\r\n", Subject);
		if(ReplyTo)
			String_Append_StringArgs(&DataAI, "Reply-to: <%s>\r\n", ReplyTo);
		
		if(Body)
		{
			// we must add a '\r\n' separator before the real text body
			String_Append_String(&DataAI, "\r\n");
			
			String_Append_String(&DataAI, Body);
		}
		String_Append_String(&DataAI, "\r\n.\r\n");
		
		Data = String_Append_Finalize(&DataAI, &lData);
		//LOG_DEBUG("mail body:\n%s", Data);
		if(!Net_WriteBlock(Socket, (uint8*)Data, lData))
		{
			LOG_DEBUG("error: writing body");
			free(Data);
			goto _end_;
		}
		free(Data);
	}
	
	// server line, should be '250 2.0.0 Ok: queued as 46FF34B03A2'
	if(!SMTP_ReadLine(Socket, Line, sizeof(Line), 250))
	{
		LOG_DEBUG("error: reading server line");
		goto _end_;
	}
	
	//---
	
	lLine = snprintf(Line, sizeof(Line) - 1, "QUIT\r\n");
	if(!Net_WriteBlock(Socket, (uint8*)Line, lLine))
	{
		LOG_DEBUG("error: writing header 'RCPT TO'");
		goto _end_;
	}
	
	// server line, should be '221 2.0.0 Bye'
	if(!SMTP_ReadLine(Socket, Line, sizeof(Line), 221))
	{
		LOG_DEBUG("error: reading server line");
		goto _end_;
	}
	
	LOG_DEBUG("mail sent");
	Success = 1;
	
_end_:
	Net_CloseSocket(&Socket);
	
	Net_Cleanup();
	
	if(Success)
		return 0;
	return 1;
}
