
#include <stdlib.h>
#include <stdio.h>
#include "unicorn.h"

#include "iscsid.h"
#include "ietadm.h"

/* find the matching unicorn ISCSI target */
static int unicorn_target_find(const char *tname,
                               unsigned int *p_utid,
                               int *p_connect)
{
    unicorn_context ctx;

    unicorn_init(&ctx);
    unicorn_set_property_string(&ctx, "iqn", tname);
    if (unicorn_find_resource(&ctx, UNICORN_TARGET) != 0)
        goto error;
    unicorn_get_property_uint32(&ctx, "id", p_utid);
    if (p_connect != NULL)
        unicorn_get_property_boolean(&ctx, "connect", p_connect);
    unicorn_close(&ctx);

    return(0);
error:
    log_error("[unicorn] unknown %s", tname);
    return(-1);
}

/* change the "connect" state of the unicorn ISCSI target */
static int unicorn_target_toggle(unsigned int utid,
                                 int connect)
{
    int res;
    unicorn_context ctx;
    gboolean set = (connect != 0);
    gboolean get;

    unicorn_init(&ctx);
    /* change target "connect" field */
    unicorn_set_property_boolean(&ctx, "connect", set);
    res = unicorn_set_resource(&ctx, UNICORN_TARGET, utid);
    if (res != 0)
    {
        log_error("[unicorn %d] set connect failed", utid);
        goto error;
    }
    /* check target "connect" field */
    unicorn_get_resource(&ctx, UNICORN_TARGET, utid);
    res = unicorn_get_property_boolean(&ctx, "connect", &get);
    if (get != set)
        goto error;

    log_warning("[unicorn %d] toggled to %d", utid, get);
    unicorn_close(&ctx);
    return(0);

error:
    log_error("[unicorn %d] toggle %d failed", utid, set);
    unicorn_close(&ctx);
    return(-1);
}

extern
void ietadm_request_exec(struct ietadm_req *req, struct ietadm_rsp *rsp,
                         void **rsp_data, size_t *rsp_data_sz);

/* attach/detach disk images as target luns */
int unicorn_target_luns(unsigned int utid, int attach, u32 tid)
{
    unicorn_context ctx;
    u32 lun_id;
    const char *list;

    unicorn_init(&ctx);

    /* get unicorn ISCSI target */
    if (unicorn_get_resource(&ctx, UNICORN_TARGET, utid) != 0)
    {
        log_error("[unicorn %d] unknown", utid);
        goto error;
    }

    lun_id = 0;
    /* get the path list of disk images */
    list = unicorn_get_property_string(&ctx, "luns_list");
    /* for each path in the list */
    while((list != NULL) && (*list != '\0'))
    {
        const char *sep, *path;
        struct ietadm_req req;
        struct ietadm_rsp rsp;
        size_t len;

        /* one line for each path */
        sep = strchr(list, '\n');
        if (sep == NULL)
            goto error;
        /* current path */ 
        path = list;
        len = sep - list;
        list = sep + 1;

        /* fill an admin request */
        req.tid = tid;
        req.lun = lun_id;
        if (attach)
        {
            req.rcmnd = C_LUNIT_NEW;
            strcpy(req.u.lunit.args, "Type=fileio,Path=");
            strncat(req.u.lunit.args, path, len);
        }
        else
        {
            req.rcmnd = C_LUNIT_DEL;
            strcpy(req.u.lunit.args, "delete");
        }

        /* send request to kernel module */
        ietadm_request_exec(&req, &rsp, NULL, NULL);
        if (rsp.err != 0)
            log_error("[unicorn %d]: lun %d: %s failed",
                      utid, lun_id, req.u.lunit.args);
        else
            log_warning("[unicorn %d]: lun %d: %s succeeded",
                      utid, lun_id, req.u.lunit.args);

        lun_id++;
    }

    unicorn_close(&ctx);
    return(0);

error:
    unicorn_close(&ctx);
    return(-1);
}

int unicorn_target_activate(u32 tid)
{
    struct target *tgt;
    unsigned int utid;
    int connect;

    tgt = target_find_by_id(tid);
    if (tgt == NULL)
        return(-1);

    /* find unicorn target id */
    if (unicorn_target_find(tgt->name, &utid, &connect) != 0)
        /* ignore target not managed by unicorn */
        return(0);

    if (connect)
    {
        log_warning("[unicorn %d]: already connected", utid);
        /* reject multiple connections */
        return(-1);
    }

    /* start target connection */
    if (unicorn_target_toggle(utid, 1) != 0)
        return(-1);
    /* attach luns */
    return(unicorn_target_luns(utid, 1, tgt->tid));
}

int unicorn_target_sync(struct target *tgt)
{
    /* when target session list is empty */
    if (list_empty(&(tgt->sessions_list)))
    {
        unsigned int utid;

        /* find unicorn target id */
        if (unicorn_target_find(tgt->name, &utid, NULL) != 0)
            /* ignore target not managed by unicorn */
            return(0);

        /* detach luns */
        unicorn_target_luns(utid, 0, tgt->tid);
        /* end target connection */
        return(unicorn_target_toggle(utid, 0));
    }

    return(0);
}
