/*
 * I2C bus driver
 * -------------
 *  This driver emulates an I2C bus over the Marvell's TWSI bus.
 *
 * 2005 - LaCie SA (www.lacie.com)
 *
 * This file is licensed under the terms of the GNU General Public
 * License version 2. This program is licensed "as is" without any
 * warranty of any kind, whether express or implied.
 */

/*
 * !!! FOR DEBUG TRACES
 * Comment or uncomment the following define to disable or enable debug traces
 * !!!FOR DEBUG TRACES
#define DBG_NAME_SPACE  "I2C_TWSI"
#define __DBG_PRN
 */

/* ************************************************************ */
/* ********************* Include Headers ********************** */
/* ************************************************************ */
#include "mvTwsi.h"
#include "i2c-twsi.h"

#include <linux/i2c.h>
#include <linux/platform_device.h>

/* WARNING : guess work - arch/arm/mach-feroceon-kw/kw_family/boardEnv/mvBoardEnvLib.c */
#define TWSI_CHANNEL 0

/* ************************************************************************* */
/* ****************************** Define types ***************************** */
/* ************************************************************************* */
typedef struct _lacieTwsiI2CData {
    struct i2c_adapter adapter;
} lacieTwsiI2CData_t;

/* ************************************************************************* */
/* **************************** Define Functions *************************** */
/* ************************************************************************* */

/* ========================================================================= */
/* ========================= I2C adapter's functions ======================= */
/* ========================================================================= */

/* ////////////////////////////////////////////////////// lacie_twsiI2CStart */
/*
 * This function is used to initiate a transfer on the I2C bus
 *
 * Return: 0 on success, -1 on error
 */
static int
lacie_twsiI2CStart(void)
{
    int l_ret   = 0; /* Assume success */

    /*
     * Tells the TWSI bus to send the transfer Start bit
     */
    if (mvTwsiStartBitSet(TWSI_CHANNEL) != MV_OK)
    {
        /*
         * Lets try again ...
         */
        if (mvTwsiStartBitSet(TWSI_CHANNEL) != MV_OK)
            l_ret   = -1;
    }

    return l_ret;
}

/* ///////////////////////////////////////////////////////// lacie_twsiWrite */
/*
 * This function sends the specified data on the I2C bus.
 *
 * a_target : The I2C ID of the targetted device
 * a_data   : A byte buffer containing the data to send
 * a_length : The number of bytes to send
 *
 * Return : 0 on success, -1 on error
 */
static int 
lacie_twsiWrite ( int a_target, const unsigned char *a_data, int a_length )
{
    MV_TWSI_ADDR    l_slaveAddr;
    int             l_ret   = -1; /* Assume error */

    /*
     * Prepare the I2C device bus address
     */
    l_slaveAddr.type    = ADDR7_BIT;
    l_slaveAddr.address = a_target;

    /*
     * Tell the TWSI bus the targetted device and the access mode (WRITE)
     */
    if (mvTwsiAddrSet(TWSI_CHANNEL, &l_slaveAddr, MV_TWSI_WRITE) == MV_OK)
    {
        /*
         * Tell the TWSI bus to transmit the data
         */
        if (twsiDataTransmit(TWSI_CHANNEL, (uint8_t*)a_data,a_length) == MV_OK)
            l_ret   = 0;
    }

    return l_ret;
}

/* ////////////////////////////////////////////////////////// lacie_twsiRead */
/*
 * This function reads the specified data on the I2C bus.
 *
 * a_target : The I2C ID of the selected device
 * a_data   : A byte buffer to hold the received data
 * a_length : The number of bytes to receive
 *
 * Return : 0 on success, -1 on error
 */
static int 
lacie_twsiRead ( int a_target, unsigned char* a_data, int a_length ) 
{
    MV_TWSI_ADDR    l_slaveAddr;
    int             l_ret   = -1; /* Assume Error */

    /*
     * Prepare the I2C device bus address
     */
    l_slaveAddr.type    = ADDR7_BIT;
    l_slaveAddr.address = a_target;

    /*
     * Tell the TWSI bus the targetted device and the access mode (WRITE)
     */
    if (mvTwsiAddrSet(TWSI_CHANNEL, &l_slaveAddr,MV_TWSI_READ) == MV_OK)
    {
        /*
         * Tell the TWSI bus to receive the data
         */
        if (twsiDataReceive(TWSI_CHANNEL, a_data,a_length) == MV_OK)
        {
            /*
             * Tell the TWSI bus to send the acknowledge bit
             */
            twsiAckBitSet(TWSI_CHANNEL);

            l_ret   = 0;
        }
    }

    return l_ret;
}

/* ////////////////////////////////////////////////////////// lacie_twsiXfer */
/*
 * This function is used to handle an I2C message/transaction.
 *
 * a_adap : I2C adapter instance
 * a_msgs : Array of I2C messages to manage
 * a_num  : The number of I2C messages
 *
 * Return: The number of managed messages if success, negative value if error.
 */
static int
lacie_twsiXfer ( struct i2c_adapter* a_adap, struct i2c_msg* a_msgs, int a_num)
{
    struct i2c_msg* l_pmsg;
    int             l_idx;
    int             l_ret;

    /*
     * Initiate the I2C transfer
     */
    l_ret = lacie_twsiI2CStart();
    if (l_ret == 0 )
    {
        /*
         * Loop through all the messages
         */
        for(l_idx=0; (l_ret == 0) && (l_idx < a_num); l_idx++)
        {
            /*
             * For each messages ...
             */
            l_pmsg = &a_msgs[l_idx];

            /*
             * Initiate a new I2C transfer for the new message
             */
            if (l_idx > 0)
            {
                l_ret   = lacie_twsiI2CStart();
            }

            /*
             * Transfer the new message
             */
            if (l_ret == 0)
            {
                if (l_pmsg->flags & I2C_M_RD)
                {
                    l_ret   = lacie_twsiRead(   l_pmsg->addr,
                                                l_pmsg->buf,
                                                l_pmsg->len);
                }
                else
                {
                    l_ret   = lacie_twsiWrite(  l_pmsg->addr,
                                                l_pmsg->buf,
                                                l_pmsg->len);
                }
            }
        }

        /*
         * Tells the TWSI bus to stop all the transactions
         */
        mvTwsiStopBitSet(TWSI_CHANNEL);

		/*
		 * It seems that we need to wait a little after the Stop command...
		 */
		msleep(6);
    }
    else
    {
        printk(KERN_ERR "Failed to set I2C start bit\n");
    }

    return (l_ret != 0)?l_ret:a_num;
}

/* ///////////////////////////////////////////////// lacie_twsiFunctionality */
/*
 * This function is used to get the I2C adapter's capabilities
 *
 * a_adap : The instance of the I2C adapter
 *
 * Return: A bit mask indicating the capabilities.
 */
static u32
lacie_twsiFunctionality ( struct i2c_adapter* a_adap )
{
    u32 l_func  = I2C_FUNC_I2C | I2C_FUNC_SMBUS_EMUL | I2C_FUNC_10BIT_ADDR;
    return l_func;
}

/* ************************************************************************* */
/* ******************** Define the adapter's interface ********************* */
/* ************************************************************************* */
/* ----------------------------------------------------------- lacieTwsiAlgo */
/*
 * This is the adapter's callbacks
 */
static struct i2c_algorithm lacieTwsiAlgo =
{
    .master_xfer    = lacie_twsiXfer,
    .functionality  = lacie_twsiFunctionality,
};

/* ------------------------------------------------------------ lacieTwsiOps */
/*
 * This is the adapter's definition
 */
static struct i2c_adapter lacieTwsiOps = {
    .owner      = THIS_MODULE,
    .name       = TWSI_I2C_NAME,
    .id         = I2C_ALGO_TWSI,
    .algo       = &lacieTwsiAlgo,
    .class      = I2C_CLASS_HWMON,
    .timeout    = 1,
    .retries    = 1
};

/* ========================================================================= */
/* ========================= I2C driver's functions ======================== */
/* ========================================================================= */

/* ///////////////////////////////////////////////////// lacie_twsiI2CRemove */
/*
 * This function is used to release an adapter instance.
 *
 * a_platDev : This is the platform device referencing the adapter instance.
 *
 * Return : Should return 0
 */
static int
lacie_twsiI2CRemove ( struct platform_device* a_platDev )
{
    lacieTwsiI2CData_t* l_drvData   = dev_get_drvdata(&a_platDev->dev);

    if (l_drvData)
    {
        dev_set_drvdata(&a_platDev->dev, NULL);
        kfree(l_drvData);
    }

    return 0;
}

/* ////////////////////////////////////////////////////// lacie_twsiI2CProbe */
/*
 * This function is used to initialize a new adapter instance.
 *
 * a_platDev : This is the platform device referencing the adapter instance.
 *
 * Return : Should return 0 on succes, negative value on error
 */
static int
lacie_twsiI2CProbe ( struct platform_device* a_platDev )
{
    lacieTwsiI2CData_t* l_drvData   = NULL;
    int                 l_result    = -ENOMEM;   /* Assume error */

    /*
     * Allocate a context buffer for the instance
     */
    l_drvData   = kzalloc(sizeof(lacieTwsiI2CData_t), GFP_KERNEL);
    if(l_drvData != NULL)
    {
        /*
         * Initialize the context buffer
         */
        dev_set_drvdata(&a_platDev->dev,l_drvData);

        /*
         * Reference the instance
         */
        l_drvData->adapter               = lacieTwsiOps;
        l_drvData->adapter.dev.parent    = &a_platDev->dev;
        l_drvData->adapter.nr            = 0; /* use bus 0 */
        i2c_set_adapdata(&l_drvData->adapter,l_drvData);

        l_result    = i2c_add_numbered_adapter(&l_drvData->adapter);
        if (l_result < 0)
        {
            printk(KERN_ERR "i2c-twsi - failed to add adapter\n");

            dev_set_drvdata(&a_platDev->dev, NULL);
            kfree(l_drvData);
        }
    }

    return l_result;
}

/* ------------------------------------------------------ lacieTwsiI2CDriver */
/*
 * Define driver's profile
 */
static struct platform_driver lacieTwsiI2CDriver = {
    .driver     = {
        .name   = TWSI_I2C_NAME,
        .owner  = THIS_MODULE,
        .bus    = &platform_bus_type,
    },
    .probe      = lacie_twsiI2CProbe,
    .remove     = lacie_twsiI2CRemove,
};

/* ------------------------------------------------------ lacieTwsiI2CDevice */
/*
 * Define I2C device's profil
 */
static struct platform_device lacieTwsiI2CDevice = {
    .name           = TWSI_I2C_NAME,
    .id             = 0,
};

/* /////////////////////////////////////////////////////// lacie_twsiI2CInit */
/*
 * This function is used to initialize the module when it is loaded in the
 * kernel environment.
 *
 * Return: 0 on success, <0 on error.
 */
static int __init
lacie_twsiI2CInit(void)
{
    int l_ret;

    /*
     * Register the driver's profile
     */
    l_ret = platform_driver_register(&lacieTwsiI2CDriver);
    if (l_ret == 0)
    {
        /*
         * Register the SoC device
         */

        /*
         * N.B.: In the previous driver (patch for LSP 2.3.1), this SoC device
         *       was added through Marvell's 'mv_init()' function (file
         *       'core.c').
         *       This time, instead of patching (i.e. parasiting) Marvell's
         *       source code, we prefer to do this task here ...
         */

        l_ret   = platform_device_register(&lacieTwsiI2CDevice);
        if (l_ret != 0)
        {
            printk( KERN_ERR "i2c-twsi - device registration failed [%d]\n",
                    l_ret);
        }
    }
    else
        printk(KERN_ERR "i2c-twsi - driver registration failed [%d]\n",l_ret);

    return l_ret;
}

/* /////////////////////////////////////////////////////// lacie_twsiI2CExit */
/*
 * This function is used to release the module's resources when it is unloaded
 * from the kernel environment.
 *
 * Return: 0 on success, <0 on error.
 */
static void __exit
lacie_twsiI2CExit(void)
{
    platform_driver_unregister(&lacieTwsiI2CDriver);
}

module_init(lacie_twsiI2CInit);
module_exit(lacie_twsiI2CExit);


MODULE_AUTHOR("LaCie S.A. <www.lacie.com>");
MODULE_DESCRIPTION
    ("I2C-Bus adapter for Marvell Two Wire Serial Interface (TWSI)");
MODULE_LICENSE("GPL");
