/*
 * arch/arm/mach-feroceon-kw/lacie/netspace_plug_v2-setup.c
 *
 * LaCie Network Space Plug v2 board setup.
 *
 * Copyright (c) 2009 LaCie
 *
 * Author: Simon Guinot <sguinot@lacie.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * The LaCie Network Space Plug v2 board is based on a Marvell Kirkwood
 * SoC 88F6192.
 *
 * Chipset list for revision A:
 * - DDR2-800 1Gbit, 400Mhz
 * - power push button
 * - 1 blue LED
 * - 1 red LED
 * - 1 Mini PCI Express interface (For WiFi n/g module)
 * - 1 Ethernet Gigabit port (PHY MARVELL 88E1116R)
 * - 4 port host USB2
 * - 1 NAND Flash 1Gbit (Hynix H27U1G8F2BTR-BC)
 *
 * revision B changes:
 * - update PHY to MARVELL 88E1318 (SMI address 0x0)
 * - add a GPIO to handle the PHY WoL interrupts
 * - add a GPIO to enable/disable latch mode for the front power button
 */

#include <linux/kernel.h>
#include <linux/platform_device.h>
#include <linux/i2c.h>
#include <linux/input.h>
#include <linux/gpio_keys.h>
#include <linux/leds.h>
#include <linux/gpio.h>
#include <linux/mtd/partitions.h>
#include <linux/mv643xx_eth.h>
#include <asm/mach-types.h>
#include <asm/mach/arch.h>

#include "core.h"
#include "lacie_v2-common.h"
#include "marvell_boardenv/netspace_plug_v2-boardenv.h"
#include "lacie/common.h"
#include "lacie/drivers/gpio-hd-power.h"
#include "lacie/drivers/gpio-usb.h"

extern struct lacie_disk_map *lacie_disk_map;

/* Ethernet */

static struct mv643xx_eth_platform_data netspace_plug_v2_r00_ge00_data = {
	.phy_addr	= MV643XX_ETH_PHY_ADDR(8),
};

static struct mv643xx_eth_platform_data netspace_plug_v2_r01_ge00_data = {
	.phy_addr	= MV643XX_ETH_PHY_ADDR(0),
};

/* NAND flash (boot device) */

static struct mtd_partition netspace_plug_v2_nand_parts[] = {
	{
		.name = "u-boot",
		.offset = 0,
		.size = SZ_1M,
	},
	{
		.name = "updater",
		.offset = MTDPART_OFS_NXTBLK,
		.size = SZ_8M,
	},
	{
		.name = "system kernel",
		.offset = MTDPART_OFS_NXTBLK,
		.size = SZ_4M,
	},
	{
		.name = "system rootfs",
		.offset = MTDPART_OFS_NXTBLK,
		.size = SZ_48M + SZ_8M + SZ_4M,
	},
	{
		.name = "config",
		.offset = MTDPART_OFS_NXTBLK,
		.size = SZ_8M,
	},
	{
		.name = "data",
		.offset = MTDPART_OFS_NXTBLK,
		.size = MTDPART_SIZ_FULL,
	},
};

/* GPIO buttons */

static struct gpio_keys_button netspace_plug_v2_buttons[] = {
	[0] = {
		.code		= KEY_POWER,
		.gpio		= NETSPACE_PLUG_V2_GPIO_KEY_POWER,
		.desc		= "Back push power button",
		.active_low	= 0,
	},
};

static struct gpio_keys_platform_data netspace_plug_v2_button_data = {
	.buttons	= netspace_plug_v2_buttons,
	.nbuttons	= ARRAY_SIZE(netspace_plug_v2_buttons),
};

static struct platform_device netspace_plug_v2_gpio_buttons = {
	.name		= "gpio-keys",
	.id		= -1,
	.dev		= {
		.platform_data 	= &netspace_plug_v2_button_data,
	},
};

/* GPIO LEDs */

static struct gpio_led netspace_plug_v2_gpio_led_pins[] = {
	{
		.name	= "ns_plug_v2:blue:power",
		.gpio	= NETSPACE_PLUG_V2_GPIO_LED_BLUE,
	},
	{
		.name	= "ns_plug_v2:red:power",
		.gpio	= NETSPACE_PLUG_V2_GPIO_LED_RED,
	},
};

static struct gpio_led_platform_data netspace_plug_v2_gpio_leds_data = {
	.num_leds	= ARRAY_SIZE(netspace_plug_v2_gpio_led_pins),
	.leds		= netspace_plug_v2_gpio_led_pins,
};

static struct platform_device netspace_plug_v2_gpio_leds = {
	.name		= "leds-gpio",
	.id		= -1,
	.dev		= {
		.platform_data	= &netspace_plug_v2_gpio_leds_data,
	},
};

/* GPIO USB (with on the go fakery) */

/* Revision A */
static struct gpio_usb netspace_plug_v2_r0_gpio_usb_port0[] = {
	[0] = {
		.name	= "fuse",
		.type	= GPIO_USB_FUSE,
		.num	= NETSPACE_PLUG_V2_GPIO_USB_FUSE,
		.act_low = 1,
	},
	[1] = {
		.name	= "vbus-out0",
		.type	= GPIO_USB_VBUS_OUT,
		.num	= NETSPACE_PLUG_V2_GPIO_USB_VBUS_OUT_0,
	},
	[2] = {
		.name	= "vbus-out1",
		.type	= GPIO_USB_VBUS_OUT,
		.num	= NETSPACE_PLUG_V2_GPIO_USB_VBUS_OUT_1,
	},
	[3] = {
		.name	= "vbus-out2",
		.type	= GPIO_USB_VBUS_OUT,
		.num	= NETSPACE_PLUG_V2_GPIO_USB_VBUS_OUT_2,
	},
	[4] = {
		.name	= "vbus-out3",
		.type	= GPIO_USB_VBUS_OUT,
		.num	= NETSPACE_PLUG_V2_GPIO_USB_VBUS_OUT_3,
	},
};

/* Revision B */
static struct gpio_usb netspace_plug_v2_r1_gpio_usb_port0[] = {
	[0] = {
		.name	= "fuse",
		.type	= GPIO_USB_FUSE,
		.num	= NETSPACE_PLUG_V2_GPIO_USB_FUSE,
		.act_low = 1,
	},
	[1] = {
		.name	= "vbus-out0",
		.type	= GPIO_USB_VBUS_OUT,
		.num	= NETSPACE_PLUG_V2_GPIO_USB_VBUS_OUT_0,
	},
	[2] = {
		.name	= "vbus-out1,2",
		.type	= GPIO_USB_VBUS_OUT,
		.num	= NETSPACE_PLUG_V2_GPIO_USB_VBUS_OUT_1,
	},
	[3] = {
		.name	= "vbus-out3",
		.type	= GPIO_USB_VBUS_OUT,
		.num	= NETSPACE_PLUG_V2_GPIO_USB_VBUS_OUT_3,
	},
};

static struct gpio_usb_port netspace_plug_v2_gpio_usb_ports[] = {
	[0] = {
		.name		= "port0",
		.gpio		= netspace_plug_v2_r1_gpio_usb_port0,
		.num_gpio	= ARRAY_SIZE(netspace_plug_v2_r1_gpio_usb_port0),
		.reload_delay	= 0, /* no fuse but current limit switch. */
		.spurious_delay	= 1000,
	},
};

struct gpio_usb_platform_data netspace_plug_v2_lacie_gpio_usb_data = {
	.port		= netspace_plug_v2_gpio_usb_ports,
	.num_port	= ARRAY_SIZE(netspace_plug_v2_gpio_usb_ports),
};

static struct platform_device netspace_plug_v2_gpio_usb = {
	.name		= "gpio-usb",
	.id		= -1,
	.dev		= {
		.platform_data	= &netspace_plug_v2_lacie_gpio_usb_data,
	},
};

/* I2C eeprom */

static struct i2c_board_info netspace_plug_v2_i2c_info[] = {
	{
		I2C_BOARD_INFO("at24", 0x50),
		.type = "24c04",
	},
};

/* internal disk map */

static struct lacie_disk netspace_plug_v2_disks[] = {
	{
		.host_no = 0,
		.channel = 0,
		.id 	 = 0,
		.letter  = 'a',
	},
};

static struct lacie_disk_map netspace_plug_v2_disk_map = {
	.disks = netspace_plug_v2_disks,
	.num_disks = ARRAY_SIZE(netspace_plug_v2_disks),
};

/* Registration */

static struct platform_device *netspace_plug_v2_gpio_devices[] __initdata = {
	&netspace_plug_v2_gpio_buttons,
	&netspace_plug_v2_gpio_leds,
	&netspace_plug_v2_gpio_usb,
};

static inline void __init netspace_plug_v2_gpio_init(void)
{
	/* Fix USB GPIOs for revision A. */
	if (system_rev == NS2_PLUG_REV_A)
		netspace_plug_v2_gpio_usb_ports[0].gpio =
				netspace_plug_v2_r0_gpio_usb_port0;

        platform_add_devices(netspace_plug_v2_gpio_devices,
			     ARRAY_SIZE(netspace_plug_v2_gpio_devices));
}

static inline void __init netspace_plug_v2_i2c_eeprom_init(void)
{
	i2c_register_board_info(0, netspace_plug_v2_i2c_info,
				ARRAY_SIZE(netspace_plug_v2_i2c_info));
}

static inline void __init netspace_plug_v2_disk_map_init(void)
{
	lacie_disk_map = &netspace_plug_v2_disk_map;
}

static void netspace_plug_v2_power_off(void)
{
	gpio_set_value(NETSPACE_PLUG_V2_GPIO_POWER_OFF, 1);
}

int __init netspace_plug_v2_identify_rev(void)
{
	int err;
	u32 phy_id;

	/*
	 * Ethernet PHY address and ID allow to identify the board revision.
	 *
	 * rev A: phy_addr=0x0, phy_id=0x01410e40
	 * rev B: phy_addr=0x0, phy_id=0x01410e90
	 */
	err = boardrev_get_phy_id(0, 0x0, &phy_id);
	if (err == 0 && phy_id == 0x01410e90)
		return NS2_PLUG_REV_B;
	err = boardrev_get_phy_id(0, 0x8, &phy_id);
	if (err == 0 && phy_id == 0x01410e40)
		return NS2_PLUG_REV_A;

	pr_warning("netspace_plug_v2: fail to match PHY ID, assuming revB\n");

	return NS2_PLUG_REV_B;
}

void __init netspace_plug_v2_init(void)
{
	int err;

	/* Identify revision */
	system_rev = netspace_plug_v2_identify_rev();
	pr_info("netspace_plug_v2: identified HW rev%c\n", 'A' + system_rev);

	/*
	 * Basic setup. Needs to be called early.
	 */
	netspace_plug_v2_boardenv_early_init();

	mv_init();
	if (system_rev == NS2_PLUG_REV_A)
		feroceon_kw_ge00_init(&netspace_plug_v2_r00_ge00_data);
	else
		feroceon_kw_ge00_init(&netspace_plug_v2_r01_ge00_data);
	mv_usb_init();
	feroceon_kw_nand_init(netspace_plug_v2_nand_parts,
			      ARRAY_SIZE(netspace_plug_v2_nand_parts), 25);

	netspace_plug_v2_gpio_init();
	netspace_plug_v2_i2c_eeprom_init();
	netspace_plug_v2_disk_map_init();

	err = gpio_request(NETSPACE_PLUG_V2_GPIO_POWER_OFF, "power-off");
	if (err == 0) {
		err = gpio_direction_output(NETSPACE_PLUG_V2_GPIO_POWER_OFF, 0);
		if (err)
			gpio_free(NETSPACE_PLUG_V2_GPIO_POWER_OFF);
	}
	if (err)
		pr_err("netspace_plug_v2: failed to configure power-off GPIO\n");

	pm_power_off = netspace_plug_v2_power_off;

	return;
}

MACHINE_START(NETSPACE_PLUG_V2, "netspace_plug_v2")
	.phys_io = 0xf1000000,
	.io_pg_offst = ((0xf1000000) >> 18) & 0xfffc,
	.boot_params = 0x00000100,
	.init_machine = netspace_plug_v2_init,
	.map_io = mv_map_io,
	.init_irq = feroceon_kw_init_irq,
	.timer = &mv_timer,
MACHINE_END
