/*******************************************************************************
Copyright (C) Marvell International Ltd. and its affiliates

This software file (the "File") is owned and distributed by Marvell
International Ltd. and/or its affiliates ("Marvell") under the following
alternative licensing terms.  Once you have made an election to distribute the
File under one of the following license alternatives, please (i) delete this
introductory statement regarding license alternatives, (ii) delete the two
license alternatives that you have not elected to use and (iii) preserve the
Marvell copyright notice above.


********************************************************************************
Marvell GPL License Option

If you received this File from Marvell, you may opt to use, redistribute and/or
modify this File in accordance with the terms and conditions of the General
Public License Version 2, June 1991 (the "GPL License"), a copy of which is
available along with the File in the license.txt file or by writing to the Free
Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 or
on the worldwide web at http://www.gnu.org/licenses/gpl.txt.

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE IMPLIED
WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE ARE EXPRESSLY
DISCLAIMED.  The GPL License provides additional details about this warranty
disclaimer.
*******************************************************************************/

#include <linux/autoconf.h>
#include <linux/kernel.h>
#include <linux/pci.h>
#include <linux/ptrace.h>
#include <linux/slab.h>
#include <linux/ioport.h>
#include <linux/interrupt.h>
#include <linux/spinlock.h>
#include <linux/init.h>
#include <linux/platform_device.h>
#include <linux/proc_fs.h>
#include <linux/ctype.h>
#include <asm/io.h>
#include <asm/irq.h>


#include "mvDebug.h"
#include "ctrlEnv/mvCtrlEnvLib.h"
#include "ctrlEnv/sys/mvSysUsb.h"
#include "usb/mvUsb.h"

extern u32 mvIsUsbHost;

#define MV_USB_DMA_MASK     0xffffffff

static char usb_dev_name[]  = "mv_udc";
static char usb_host_name[] = "ehci_marvell";
static char usb_bus_name[]  = "platform";

struct usbconf_data_t
{
    int devnum;
    spinlock_t lock;
};

static struct usbconf_data_t usbconf_dev[MV_USB_MAX_PORTS];
static struct proc_dir_entry *usbconf_proc_dir, *usbconf_proc_entry[MV_USB_MAX_PORTS];
static struct platform_device *mv_usb_pdev[MV_USB_MAX_PORTS];

static int mv_usb_pdevice_register(int devnum);
static void mv_usb_pdevice_unregister(int devnum);
static void mv_usb_release(struct device *dev);

static int proc_read_usbconf(char *page, char **start, off_t off, int count, int *eof, void *data)
{
    struct usbconf_data_t *usbconf = (struct usbconf_data_t *) data;

    return sprintf(page, "%s\n",
                   test_bit(usbconf->devnum, (unsigned long *) &mvIsUsbHost) ?
                   "host" : "device");
}

static int proc_write_usbconf(struct file *file, const char *buffer, unsigned long count, void *data)
{
    struct usbconf_data_t *usbconf = (struct usbconf_data_t *) data;
    char userval[8];
    int i;
    int len = min(count, (unsigned long) sizeof(userval));

    if (len <= 0)
        return -EINVAL;

    if (copy_from_user(userval, buffer, len))
        return -EFAULT;

    userval[len-1] = '\0';
    for (i=0; i<len; i++)
        userval[i] = tolower(userval[i]);

    if (strcmp(userval, "host") == 0)
    {
        if (test_bit(usbconf->devnum, (unsigned long *) &mvIsUsbHost))
            goto exit;
        set_bit(usbconf->devnum, (unsigned long *) &mvIsUsbHost);
    }
    else if (strcmp(userval, "device") == 0)
    {
        if (!test_bit(usbconf->devnum, (unsigned long *) &mvIsUsbHost))
            goto exit;
        clear_bit(usbconf->devnum, (unsigned long *) &mvIsUsbHost);
    }
    else
        return -EINVAL;

    /* FIXME: How clean USB stack and reset USB ports properly */

    spin_lock(&usbconf->lock);
    mv_usb_pdevice_unregister(usbconf->devnum);
    mv_usb_pdevice_register(usbconf->devnum);
    spin_unlock(&usbconf->lock);

exit:
    return len;
}

static int mv_usb_pdevice_register(int devnum)
{
    int status, isHost;
    struct platform_device *pdev;

    if (MV_FALSE == mvCtrlPwrClckGet(USB_UNIT_ID, devnum))
    {
        printk("\nWarning Integrated USB %d is Powered Off\n", devnum);
        status = -ENODEV;
        goto exit_error;
    }
    isHost = mvIsUsbHost & (1 << devnum);

    status = mvUsbInit(devnum, isHost);

    mv_usb_pdev[devnum] = kzalloc(sizeof(struct platform_device), GFP_KERNEL);
    if (!mv_usb_pdev[devnum])
    {
        printk("Can't allocate platform_device structure - %d bytes\n",
               sizeof(struct platform_device));
        status = -ENOMEM;
        goto exit_error;
    }
    pdev = mv_usb_pdev[devnum];

    pdev->name = isHost? usb_host_name : usb_dev_name;
    pdev->id = PCI_VENDOR_ID_MARVELL | (MV_USB_VERSION << 16) | (devnum << 24);
    pdev->num_resources = 2;
    pdev->resource = (struct resource*) kzalloc(2*sizeof(struct resource), GFP_KERNEL);
    if (!pdev->resource)
    {
        printk("Can't allocate 2 resource structure - %d bytes\n",
               2*sizeof(struct resource));
        status = -ENOMEM;
        goto exit_free_pdev;
    }
    pdev->resource[0].start = (INTER_REGS_BASE | MV_USB_CORE_CAP_LENGTH_REG(dev));
    pdev->resource[0].end = ((INTER_REGS_BASE | MV_USB_CORE_CAP_LENGTH_REG(dev)) + 4096);
    pdev->resource[0].flags = IORESOURCE_DMA;
    pdev->resource[1].start = IRQ_USB_CTRL(devnum);
    pdev->resource[1].flags = IORESOURCE_IRQ;

    pdev->dev.dma_mask = kmalloc(sizeof(u64), GFP_KERNEL);
    *pdev->dev.dma_mask = MV_USB_DMA_MASK;
    pdev->dev.coherent_dma_mask  = ~0;
    pdev->dev.release = mv_usb_release;
    dev_set_name(&pdev->dev, "%s", usb_bus_name);

    status = platform_device_register(pdev);
    if (status)
        goto exit_free_resource;

    printk("Register Marvell USB EHCI %s controller #%d\n",
           isHost ? "Host" : "Gadget", devnum);

    return 0;

exit_free_resource:
    kfree(pdev->dev.dma_mask);
    kfree(pdev->resource);
exit_free_pdev:
    kfree(pdev);
    mv_usb_pdev[devnum] = NULL;
exit_error:
    printk("Can't register Marvell USB EHCI controller #%d, status=%d\n",
           devnum, status);

    return status;
}

static void mv_usb_pdevice_unregister(int devnum)
{
    struct platform_device *pdev = mv_usb_pdev[devnum];

    if (!pdev)
        return;

    printk("Unregister USB platform device %s (USB port %d)\n",
           pdev->name, devnum);

    platform_device_unregister(pdev);
    mv_usb_pdev[devnum] = NULL;
}

static void mv_usb_release(struct device *dev)
{
    struct platform_device  *pdev = to_platform_device(dev);

    /* normally not freed */
    printk("mv_usb_release\n");

    kfree(pdev->resource);
    kfree(pdev->dev.dma_mask);
    kfree(pdev);
}

int __init mv_usb_init(void)
{
    int ret;
    int devnum;
    char devnum_str[3];

    usbconf_proc_dir = proc_mkdir("usbconf", NULL);
    if (!usbconf_proc_dir)
        return -ENOMEM;

    for(devnum = 0; devnum < MV_USB_MAX_PORTS; devnum++)
    {
        ret = mv_usb_pdevice_register(devnum);
        if (ret)
            continue;

        snprintf(devnum_str, sizeof(devnum_str), "%d", devnum);
        usbconf_proc_entry[devnum] =
            create_proc_entry(devnum_str, 0644, usbconf_proc_dir);
        if (!usbconf_proc_entry[devnum])
            return -ENOMEM;

        usbconf_dev[devnum].devnum = devnum;
        spin_lock_init(&usbconf_dev[devnum].lock);
        usbconf_proc_entry[devnum]->data = &usbconf_dev[devnum];
        usbconf_proc_entry[devnum]->read_proc = proc_read_usbconf;
        usbconf_proc_entry[devnum]->write_proc = proc_write_usbconf;
    }    
    return 0;
}
