/*
 * lacie/drivers/g762.c
 *
 * g762 - Driver for the Global Mixed-mode Technology Inc.
 * 	  fan speed PWM controller chips from g762 family
 *
 * Copyright (c) 2009 LaCie
 *
 * Author: Olivier Mouchet <omouchet@lacie.com>
 *
 * based on g760a code written by Herbert Valerio Riedel <hvr@gnu.org>
 * Copyright (C) 2007  Herbert Valerio Riedel <hvr@gnu.org>
 *
 * g762: minimal datasheet available at:
 * 	www.gmt.com.tw/product/datasheet/EDS-762_763.pdf
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/device.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/jiffies.h>
#include <linux/i2c.h>
#include <linux/hwmon.h>
#include <linux/hwmon-sysfs.h>
#include <linux/err.h>
#include <linux/mutex.h>
#include <linux/sysfs.h>
#include <linux/kernel.h>

#include "g762.h"

#define DRVNAME "g762"

static const struct i2c_device_id g762_id[] = {
	{ DRVNAME, 0 },
	{ }
};
MODULE_DEVICE_TABLE(i2c, g762_id);

enum g762_regs {
	G762_REG_SET_CNT  = 0x00,
	G762_REG_ACT_CNT  = 0x01,
	G762_REG_FAN_STA  = 0x02,
	G762_REG_SET_OUT  = 0x03,
	G762_REG_FAN_CMD1 = 0x04,
	G762_REG_FAN_CMD2 = 0x05,
};

/*
 * Config register bits
 */
#define G762_REG_FAN_STA_FAIL	0x02	/* fan fail */
#define G762_REG_FAN_STA_OOC	0x01	/* fan out of control */

/*
 * Config registers bits
 */
#define G762_REG_FAN_CMD1_DET_FAN_FAIL 	0x80 /* enable fan_fail signal */
#define G762_REG_FAN_CMD1_DET_FAN_OOC 	0x40 /* enable fan_out_of_control */


#define G762_REG_FAN_CMD1_OUT_MODE 	0x20 /* out mode, pwm or dac */

#define G762_REG_FAN_CMD1_FAN_MODE 	0x10 /* fan mode: close or open loop */

#define G762_REG_FAN_CMD1_CLK_DIV_ID0 	0x04 /* clock divisor value */
#define G762_REG_FAN_CMD1_CLK_DIV_ID1   0x08

/*
 * Config register values
 */
#define OUT_MODE_PWM 		1
#define OUT_MODE_DAC 		0

#define FAN_MODE_CLOSE_LOOP 	1
#define FAN_MODE_OPEN_LOOP 	0

/*
 * g762 default values
 * By default, it is assumed that the fan is set for two pulses per revolution
 */
#define G762_DEFAULT_CLK 	32768
#define G762_DEFAULT_FAN_DIV 	2

/* register data is read (and cached) at most once per second */
#define G762_UPDATE_INTERVAL (HZ)

#define DIV_FROM_REG(reg) (1 << (reg & 7))

/*
 * Client data (each client gets its own)
 */
struct g762_data {
	struct i2c_client *client;
	struct device *hwmon_dev;

	/* update mutex */
	struct mutex update_lock;

	/* board specific parameters */
	u32 clk; /* default 32kHz */

	/* g762 register cache */
	unsigned int valid:1;
	unsigned long last_updated; /* In jiffies */

	u8 set_cnt; /* RPM cmd in close loop control */
	u8 act_cnt; /* formula: cnt = (CLK * 30)/(rpm * P) */
	u8 fan_sta; /* bit 0: set when actual fan speed more than 20%
		     * outside requested fan speed
		     * bit 1: set when fan speed below 1920 rpm */
	u8 set_out; /* output voltage/PWM duty in open loop control */
	u8 fan_cmd1; /* 0: FG_PLS_ID0 FG pulses count per revolution
		      * 1: PWM_POLARITY 1: negative_duty
		      * 		0: positive_duty
		      * 2,3: [FG_CLOCK_ID0, FG_CLK_ID1]
		      * 	00: Divide fan clock by 1
		      * 	01: Divide fan clock by 2
		      * 	10: Divide fan clock by 4
		      * 	11: Divide fan clock by 8
		      * 4: FAN_MODE 1:close-loop 0:open-loop
		      * 5: OUT_MODE 1:PWM, 0:DAC
		      * 6: DET_FAN_OOC enable "fan ooc" status
		      * 7: DET_FAN_FAIL enable "fan fail" status
		      */
	u8 fan_cmd2; /* 0,1: FAN_STARTV 0,1,2,3 -> 0,32,64,96 dac_code
		      * 2,3: FG_GEAR_CODE
		      * 	0,1,2 -> dT=Tfg/2,dT=Tfg,dT=2Tfg
		      * 4: g763 only
		      */
};

#define PWM_FROM_CNT(cnt)	(0xff-(cnt))
#define PWM_TO_CNT(pwm)		(0xff-(pwm))

/*
 * Convert count value from fan controller registy into speed value
 *
 * Process value following parameters of fan controller
 */
static inline unsigned int rpm_from_cnt(u8 cnt,
					u32 clk,
					u16 div)
{
	if (cnt == 0 || cnt == 0xff)
		return 0;

	return (clk*30)/(cnt*div);
}

/*
 * Convert rpm value from sysfs into count value
 *
 * Process value following parameters of fan controller
 */
static inline unsigned char cnt_from_rpm(u32 rpm,
					 u32 clk,
					 u16 div)
{
	return (rpm == 0) ? 0xff : clk*30/(rpm*div);
}

static int g762_probe(struct i2c_client *client, const struct i2c_device_id *id);
static int g762_remove(struct i2c_client *client);

static struct i2c_driver g762_driver = {
	.driver = {
		.name	= DRVNAME,
	},
	.probe	  = g762_probe,
	.remove	  = g762_remove,
	.id_table = g762_id,
};

/* read/write wrappers */
static int g762_read_value(struct i2c_client *client, enum g762_regs reg)
{
	return i2c_smbus_read_byte_data(client, reg);
}

static int g762_write_value(struct i2c_client *client, enum g762_regs reg,
			    u16 value)
{
	return i2c_smbus_write_byte_data(client, reg, value);
}

/*
 * sysfs attributes
 */

static struct g762_data *g762_update_client(struct device *dev)
{
	struct i2c_client *client = to_i2c_client(dev);
	struct g762_data *data = i2c_get_clientdata(client);

	mutex_lock(&data->update_lock);

	if (time_after(jiffies, data->last_updated + G762_UPDATE_INTERVAL)
			|| !data->valid) {
		data->set_cnt = g762_read_value(client, G762_REG_SET_CNT);
		data->act_cnt = g762_read_value(client, G762_REG_ACT_CNT);
		data->fan_sta = g762_read_value(client, G762_REG_FAN_STA);
		data->set_out = g762_read_value(client, G762_REG_SET_OUT);
		data->fan_cmd1 = g762_read_value(client, G762_REG_FAN_CMD1);
		data->fan_cmd2 = g762_read_value(client, G762_REG_FAN_CMD2);

		data->last_updated = jiffies;
		data->valid = 1;
	}
	mutex_unlock(&data->update_lock);

	return data;
}

static ssize_t show_fan(struct device *dev,
			struct device_attribute *da,
			char *buf)
{
	struct g762_data *data = g762_update_client(dev);
	unsigned int rpm = 0;
	unsigned char fan_div;

	mutex_lock(&data->update_lock);
	if (data->fan_sta & G762_REG_FAN_STA_OOC) {
		fan_div = (data->fan_cmd1 &
				(G762_REG_FAN_CMD1_CLK_DIV_ID0 |
				 G762_REG_FAN_CMD1_CLK_DIV_ID1)) >> 2;
		rpm = rpm_from_cnt(data->act_cnt,
				   data->clk,
				   DIV_FROM_REG(fan_div));
	}
	mutex_unlock(&data->update_lock);
	return sprintf(buf, "%u\n", rpm);
}

/*
 * Read fan_fail signal
 *
 * Note: it is not available in close loop control
 */
static ssize_t show_fan_alarm(struct device *dev,
			      struct device_attribute *da,
			      char *buf)
{
	int fan_alarm;
	struct g762_data *data = g762_update_client(dev);

	fan_alarm = (data->fan_sta & G762_REG_FAN_STA_FAIL) ? 1 : 0;

	return sprintf(buf, "%d\n", fan_alarm);
}

/*
 * Read/write functions for pwm1_mode sysfs file
 *
 * Set/get fan speed control mode as pwm or linear
 */
static ssize_t get_pwm_mode(struct device *dev,
			    struct device_attribute *da,
			    char *buf)
{
	int pwm_mode;
	struct g762_data *data = g762_update_client(dev);

	mutex_lock(&data->update_lock);
	pwm_mode =
		(data->fan_cmd1 & G762_REG_FAN_CMD1_OUT_MODE) ? 1 : 0;
	mutex_unlock(&data->update_lock);

	return sprintf(buf, "%d\n", pwm_mode);
}

static ssize_t set_pwm_mode(struct device *dev,
			    struct device_attribute *da,
			    const char *buf,
			    size_t count)
{
	struct i2c_client *client = to_i2c_client(dev);
	struct g762_data *data = g762_update_client(dev);
	unsigned long val;

	if (strict_strtoul(buf, 10, &val))
		return -EINVAL;

	mutex_lock(&data->update_lock);
	if (val == OUT_MODE_PWM)
		data->fan_cmd1 = data->fan_cmd1 | G762_REG_FAN_CMD1_OUT_MODE;
	else if (val == OUT_MODE_DAC)
		data->fan_cmd1 = data->fan_cmd1
				 & (u8) ~G762_REG_FAN_CMD1_OUT_MODE;
	else {
		mutex_unlock(&data->update_lock);
		return -EINVAL;
	}
	g762_write_value(client, G762_REG_FAN_CMD1, data->fan_cmd1);
	mutex_unlock(&data->update_lock);

	return count;
}

/*
 * Read/write functions for fan1_div sysfs file.
 *
 * Get/set fan controller prescaler
 */
static ssize_t set_clock_divisor(struct device *dev,
				 struct device_attribute *da,
				 const char *buf,
				 size_t count)
{
	struct i2c_client *client = to_i2c_client(dev);
	struct g762_data *data = g762_update_client(dev);
	unsigned long val;

	if (strict_strtoul(buf, 10, &val))
		return -EINVAL;

	mutex_lock(&data->update_lock);
	switch (val) {
	case 1:
		data->fan_cmd1 &= (u8) ~G762_REG_FAN_CMD1_CLK_DIV_ID0
			& (u8) ~G762_REG_FAN_CMD1_CLK_DIV_ID1;
		break;
	case 2:
		data->fan_cmd1 |= G762_REG_FAN_CMD1_CLK_DIV_ID0;
		data->fan_cmd1 &= (u8) ~G762_REG_FAN_CMD1_CLK_DIV_ID1;
		break;
	case 4:
		data->fan_cmd1 &= (u8) ~G762_REG_FAN_CMD1_CLK_DIV_ID0;
		data->fan_cmd1 |= G762_REG_FAN_CMD1_CLK_DIV_ID1;
		break;
	case 8:
		data->fan_cmd1 |= G762_REG_FAN_CMD1_CLK_DIV_ID0
			| G762_REG_FAN_CMD1_CLK_DIV_ID1;
		break;
	default:
		mutex_unlock(&data->update_lock);
		return -EINVAL;
	}

	g762_write_value(client, G762_REG_FAN_CMD1, (u16)data->fan_cmd1);
	mutex_unlock(&data->update_lock);

	return count;
}

static ssize_t get_clock_divisor(struct device *dev,
				 struct device_attribute *da,
				 char *buf)
{
	int clk_div;
	struct g762_data *data = g762_update_client(dev);

	mutex_lock(&data->update_lock);
	clk_div = (data->fan_cmd1 & (G762_REG_FAN_CMD1_CLK_DIV_ID0 |
				     G762_REG_FAN_CMD1_CLK_DIV_ID1)) >> 2;
	mutex_unlock(&data->update_lock);

	return sprintf(buf, "%d\n", DIV_FROM_REG(clk_div));
}

/*
 * Following documentation about hwmon's sysfs interface, a fan1_enable node
 * should accept followings:
 * 0: no fan speed control (i.e. fan at full speed)
 * 1: manual fan speed control enabled (use pwm[1-*]) (open-loop)
 * 2+: automatic fan speed control enabled (use fan[1-*]_enable)(close-loop)
 *
 * but we do not accept 0 as "no-control" mode is not supported by g762,
 * -EINVAL is returned in this case.
 */
static ssize_t get_fan_speed_control(struct device *dev,
				     struct device_attribute *da,
				     char *buf)
{
	int fan_mode;
	struct g762_data *data = g762_update_client(dev);

	fan_mode =
		(data->fan_cmd1 & G762_REG_FAN_CMD1_FAN_MODE) ? 1 : 0;

	return sprintf(buf, "%d\n", fan_mode+1);
}

static ssize_t set_fan_speed_control(struct device *dev,
				     struct device_attribute *da,
				     const char *buf,
				     size_t count)
{
	struct i2c_client *client = to_i2c_client(dev);
	struct g762_data *data = g762_update_client(dev);
	unsigned long val;

	if (strict_strtoul(buf, 10, &val))
		return -EINVAL;

	mutex_lock(&data->update_lock);
	if (val-1 == FAN_MODE_CLOSE_LOOP)
		data->fan_cmd1 |= G762_REG_FAN_CMD1_FAN_MODE;
	else if (val-1 == FAN_MODE_OPEN_LOOP)
		data->fan_cmd1 &= (u8) ~G762_REG_FAN_CMD1_FAN_MODE;
	else {
		mutex_unlock(&data->update_lock);
		return -EINVAL;
	}
	g762_write_value(client, G762_REG_FAN_CMD1, data->fan_cmd1);
	mutex_unlock(&data->update_lock);

	return count;
}

/*
 * Get/set the fan speed in open-loop mode using pwm1 sysfs file.
 * Speed is given as a relative value from 0 to 255, where 255 is maximum
 * speed. Note that this is done by writing directly to the chip's DAC,
 * it won't change the closed loop speed set by fan1_target.
 * Also note that due to rounding errors it is possible that you don't read
 * back exactly the value you have set.
 */
static ssize_t get_pwm(struct device *dev,
		       struct device_attribute *da,
		       char *buf)
{
	struct g762_data *data = g762_update_client(dev);

	return sprintf(buf, "%d\n", data->set_out);
}

static ssize_t set_pwm(struct device *dev,
		       struct device_attribute *da,
		       const char *buf,
		       size_t count)
{
	int out_mode;
	struct i2c_client *client = to_i2c_client(dev);
	struct g762_data *data = g762_update_client(dev);
	unsigned long val;

	if (strict_strtoul(buf, 10, &val))
		return -EINVAL;

	mutex_lock(&data->update_lock);
	out_mode = (data->fan_cmd1 & G762_REG_FAN_CMD1_OUT_MODE) ? 1 : 0;
	data->set_out = SENSORS_LIMIT(val, 0, 255);
	g762_write_value(client, G762_REG_SET_OUT, data->set_out);
	mutex_unlock(&data->update_lock);

	return count;
}

/*
 * Get/set the fan speed in close-loop mode using fan1_target sysfs file.
 * Speed is given as a rpm value, then G762 will automatically regulate the fan
 * speed using pulses from fan tachometer.
 *
 * Refer to rpm_from_cnt implementation to get info about count number
 * calculation.
 *
 * Also note that due to rounding errors it is possible that you don't read
 * back exactly the value you have set.
 */
static ssize_t get_fan_target(struct device *dev,
			      struct device_attribute *da,
			      char *buf)
{
	struct g762_data *data = g762_update_client(dev);
	unsigned int rpm;
	unsigned char fan_div;

	fan_div = (data->fan_cmd1 &
			(G762_REG_FAN_CMD1_CLK_DIV_ID0 |
			 G762_REG_FAN_CMD1_CLK_DIV_ID1)) >> 2;
	rpm = rpm_from_cnt(data->set_cnt,
			   data->clk,
			   DIV_FROM_REG(fan_div));

	return sprintf(buf, "%u\n", rpm);
}

static ssize_t set_fan_target(struct device *dev,
			      struct device_attribute *da,
			      const char *buf,
			      size_t count)
{
	struct i2c_client *client = to_i2c_client(dev);
	struct g762_data *data = g762_update_client(dev);
	unsigned int val;
	unsigned char fan_div;

	if (sscanf(buf, "%u", &val) != 1)
		return -EINVAL;

	mutex_lock(&data->update_lock);
	fan_div = (data->fan_cmd1 &
			(G762_REG_FAN_CMD1_CLK_DIV_ID0 |
			 G762_REG_FAN_CMD1_CLK_DIV_ID1)) >> 2;
	data->set_cnt = cnt_from_rpm(val,
					     data->clk,
					     DIV_FROM_REG(fan_div));
	g762_write_value(client, G762_REG_SET_CNT, data->set_cnt);
	mutex_unlock(&data->update_lock);

	return count;
}

static DEVICE_ATTR(pwm1, S_IWUSR | S_IRUGO, get_pwm, set_pwm);
static DEVICE_ATTR(pwm1_mode, S_IWUSR | S_IRUGO,
		   get_pwm_mode, set_pwm_mode);
static DEVICE_ATTR(pwm1_enable, S_IWUSR | S_IRUGO,
		   get_fan_speed_control, set_fan_speed_control);

static DEVICE_ATTR(fan1_input, S_IRUGO, show_fan, NULL);
static DEVICE_ATTR(fan1_alarm, S_IRUGO, show_fan_alarm, NULL);
static DEVICE_ATTR(fan1_target, S_IWUSR | S_IRUGO,
		   get_fan_target, set_fan_target);
static DEVICE_ATTR(fan1_div, S_IWUSR | S_IRUGO,
		   get_clock_divisor, set_clock_divisor);


/*
 * Driver data
 */
static struct attribute *g762_attributes[] = {
	&dev_attr_fan1_input.attr,
	&dev_attr_fan1_alarm.attr,
	&dev_attr_fan1_target.attr,
	&dev_attr_fan1_div.attr,
	&dev_attr_pwm1.attr,
	&dev_attr_pwm1_mode.attr,
	&dev_attr_pwm1_enable.attr,
	NULL
};

static const struct attribute_group g762_group = {
	.attrs = g762_attributes,
};

/*
 * new-style driver model code
 */
static int g762_probe(struct i2c_client *client, const struct i2c_device_id *id)
{
	struct g762_data *data;
	int err;
	if (!i2c_check_functionality(client->adapter,
				     I2C_FUNC_SMBUS_BYTE_DATA))
		return -EIO;

	data = kzalloc(sizeof(struct g762_data), GFP_KERNEL);
	if (!data)
		return -ENOMEM;
	i2c_set_clientdata(client, data);

	data->client = client;
	mutex_init(&data->update_lock);
	/* setup default configuration for now */
	data->clk = G762_DEFAULT_CLK;
	/* enable fan alarms detection */
	data->fan_cmd1 |= (G762_REG_FAN_CMD1_DET_FAN_FAIL |
			   G762_REG_FAN_CMD1_DET_FAN_OOC);

	/* load rpm-to-count value tab from platform data */
	if (data->client->dev.platform_data) {
		struct g762_platform_data *sp_map;
		sp_map = data->client->dev.platform_data;
		data->clk = sp_map->ref_clk;
		dev_info(&data->client->dev,
			 "found a platform specific fan speed array\n");
	}

	/* Register sysfs hooks */
	err = sysfs_create_group(&client->dev.kobj, &g762_group);
	if (err)
		goto error_sysfs_create_group;
	data->hwmon_dev = (struct device *)hwmon_device_register(&client->dev);
	if (IS_ERR(data->hwmon_dev)) {
		err = PTR_ERR(data->hwmon_dev);
		goto error_hwmon_device_register;
	}

	dev_info(&data->client->dev,
		 "device successfully initialized\n");
	return 0;

error_hwmon_device_register:
	sysfs_remove_group(&client->dev.kobj, &g762_group);
error_sysfs_create_group:
	mutex_destroy(&data->update_lock);
	kfree(data);
	i2c_set_clientdata(client, NULL);

	return err;
}

static int g762_remove(struct i2c_client *client)
{
	struct g762_data *data = i2c_get_clientdata(client);

	hwmon_device_unregister(data->hwmon_dev);
	sysfs_remove_group(&client->dev.kobj, &g762_group);
	mutex_destroy(&data->update_lock);
	kfree(data);
	i2c_set_clientdata(client, NULL);

	dev_info(&data->client->dev,
		 "device successfully removed\n");
	return 0;
}

/* module management */

static int __init g762_init(void)
{
	return i2c_add_driver(&g762_driver);
}

static void __exit g762_exit(void)
{
	i2c_del_driver(&g762_driver);
}

MODULE_AUTHOR("Olivier Mouchet <omouchet@lacie.com>");
MODULE_DESCRIPTION("GMT G762 driver");
MODULE_LICENSE("GPL");

module_init(g762_init);
module_exit(g762_exit);
