/*
 * lacie/drivers/gpio_usb.c
 *
 * LaCie GPIO USB driver.
 *
 * Copyright (C) 2009 LaCie
 *
 * Author: Simon Guinot <sguinot@lacie.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * TODO: add support for deferred VBUS-out power on.
 * TODO: add support for several fuses per port and for fuse/VBUS-out
 *       associations.
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/irq.h>
#include <linux/platform_device.h>
#include <linux/kobject.h>
#include <linux/completion.h>
#include <linux/sysfs.h>
#include <linux/gpio.h>

#include "gpio-usb.h"

struct gpio_usb_port_data {
	struct kobject		kobj;
	struct completion	kobj_unregister;
	const char		*name;
	struct platform_device	*pdev;
	struct list_head	list; /* GPIO list */
	struct rw_semaphore	rw_sem; /* lock GPIO list */
};
#define kobj_to_port_data(x) container_of(x, struct gpio_usb_port_data, kobj)

struct gpio_usb_data;

struct gpio_usb_attribute {
	struct attribute attr;
	ssize_t (*show)(struct gpio_usb_data *data, char *buff);
	ssize_t (*store)(struct gpio_usb_data *data, const char *buff,
			 size_t count);
};
#define to_gpio_usb_attr(x) container_of(x, struct gpio_usb_attribute, attr)

struct gpio_usb_data {
	const struct gpio_usb	*gpio;
	struct list_head	node;
	void			*ext;
	struct gpio_usb_attribute attr;
	struct gpio_usb_port_data *port_data;
};
#define attr_to_data(x) container_of(x, struct gpio_usb_data, attr)

struct fuse_ext {
	unsigned long		last_blow; /* in jiffies */
	unsigned long		reload_delay; /* re-enable delay in msecs */
	unsigned long		spurious_delay; /* spurious irq delay in msecs */
	struct delayed_work	dwork;
	struct gpio_usb_data	*data;
};

struct vbus_in_ext {
	struct work_struct	work;
	struct gpio_usb_data	*data;
};

/*
 * gpio_usb_{get,set}_value helpers: abstract GPIO activity level.
 */
static inline int gpio_usb_get_value(const struct gpio_usb *gpio)
{
	BUG_ON(gpio == NULL);

	return gpio->act_low ?
		!gpio_get_value(gpio->num) : gpio_get_value(gpio->num);
}

static inline void gpio_usb_set_value(const struct gpio_usb *gpio, int val)
{
	BUG_ON(gpio == NULL);

	gpio_set_value(gpio->num, gpio->act_low ? !val : val);
}

static struct gpio_usb_data *
data_by_type_lookup(struct gpio_usb_port_data *port_data,
		    enum gpio_usb_types type)
{
	struct gpio_usb_data *data = NULL;
	struct gpio_usb_data *tmp;

	down_read(&port_data->rw_sem);

	list_for_each_entry(tmp, &port_data->list, node) {
		if (tmp->gpio->type == type) {
			data = tmp;
			break;
		}
	}

	up_read(&port_data->rw_sem);

	return data;

}

/*
 * sysfs support for gpio-usb sub-kobject.
 */

static void gpio_usb_sysfs_release(struct kobject *kobj)
{
	struct gpio_usb_port_data *port_data = kobj_to_port_data(kobj);

	complete(&port_data->kobj_unregister);
}

static ssize_t gpio_usb_sysfs_show(struct kobject *kobj,
				   struct attribute *attr, char *buff)
{
	struct gpio_usb_attribute *gpio_usb_attr = to_gpio_usb_attr(attr);
	struct gpio_usb_data *data = attr_to_data(gpio_usb_attr);

	if (!gpio_usb_attr->show)
		return -EIO;

	return gpio_usb_attr->show(data, buff);
}

static ssize_t gpio_usb_sysfs_store(struct kobject *kobj,
				    struct attribute *attr,
				    const char *buff, size_t count)
{
	struct gpio_usb_attribute *gpio_usb_attr = to_gpio_usb_attr(attr);
	struct gpio_usb_data *data = attr_to_data(gpio_usb_attr);

	if (!gpio_usb_attr->store)
		return -EIO;

	return gpio_usb_attr->store(data, buff, count);
}

static struct sysfs_ops gpio_usb_sysfs_ops = {
	.show = gpio_usb_sysfs_show,
	.store = gpio_usb_sysfs_store,
};

static struct kobj_type gpio_usb_ktype = {
	.release = gpio_usb_sysfs_release,
	.sysfs_ops = &gpio_usb_sysfs_ops,
	.default_attrs = NULL,
};

/*
 * Fuse support.
 */

static void fuse_notify(struct work_struct *ws)
{
	int off = 0;
	struct fuse_ext *fuse_ext =
		container_of(ws, struct fuse_ext, dwork.work);
	struct gpio_usb_data *fuse_data = fuse_ext->data;
	struct gpio_usb_port_data *port_data = fuse_data->port_data;
	struct gpio_usb_data *tmp;

	/* If the fuse is active, it means that the fuse interrupt is
	 * not due to a blow (fuse need more time before rearming) but
	 * probably to an electric bounce after an USB device plug. */
	if (gpio_usb_get_value(fuse_data->gpio) == 0) {
		dev_dbg(&port_data->pdev->dev,
			"%s %s: discard spurious %s IRQ\n", __FUNCTION__,
			port_data->name, fuse_data->gpio->name);
		return;
	}

	/* Disable _all_ active VBUS-out GPIOs. */
	down_read(&port_data->rw_sem);

	list_for_each_entry(tmp, &port_data->list, node) {
		if (tmp->gpio->type != GPIO_USB_VBUS_OUT)
			continue;
		if (gpio_usb_get_value(tmp->gpio)) {
			gpio_usb_set_value(tmp->gpio, 0);
			off++;
		}
	}

	up_read(&port_data->rw_sem);

	if (!off) {
		dev_dbg(&port_data->pdev->dev,
			"%s %s: discard %s IRQ due to VBUS-out disable\n",
			__FUNCTION__, port_data->name, fuse_data->gpio->name);
		return;
	}

	fuse_ext->last_blow = jiffies;

	/* Notify userspace. */
	sysfs_notify(&port_data->pdev->dev.kobj, NULL, fuse_data->gpio->name);
	kobject_uevent(&port_data->pdev->dev.kobj, KOBJ_CHANGE);

	dev_dbg(&port_data->pdev->dev, "%s %s: %s signal change\n",
		__FUNCTION__, port_data->name, fuse_data->gpio->name);
}

/*
 * IRQ handler called when the USB fuse blow.
 */
static irqreturn_t fuse_irq_handler(int irq, void *data)
{
	struct gpio_usb_data *fuse_data = data;
	struct fuse_ext *fuse_ext = fuse_data->ext;

	/* FIXME: Discard USB fuse interrupts due to VBUS-out disable. */

	/* A platform specific delay is needed to detect spurious interrupts. */
	if (fuse_ext->spurious_delay)
		schedule_delayed_work(&fuse_ext->dwork,
				msecs_to_jiffies(fuse_ext->spurious_delay));

	return IRQ_HANDLED;
}

static ssize_t fuse_show(struct gpio_usb_data *fuse_data, char *buff)
{
	return sprintf(buff, "%i\n", gpio_usb_get_value(fuse_data->gpio));
}

static int gpio_usb_fuse_init(struct gpio_usb_data *fuse_data,
			      unsigned long reload_delay,
			      unsigned long spurious_delay)
{
	int err;
	int fuse_irq;
	struct gpio_usb_port_data *port_data = fuse_data->port_data;
	struct platform_device *pdev = port_data->pdev;
	struct gpio_usb_attribute *fuse_attr = &fuse_data->attr;
	const struct gpio_usb *gpio = fuse_data->gpio;
	struct fuse_ext *fuse_ext;

	/* Initialize fuse extented data. */

	fuse_ext = kzalloc(sizeof (*fuse_ext), GFP_KERNEL);
	if (!fuse_ext)
		return -ENOMEM;

	INIT_DELAYED_WORK(&fuse_ext->dwork, fuse_notify);
	fuse_ext->reload_delay = reload_delay;
	fuse_ext->spurious_delay = spurious_delay;
	/* Set last_blow to avoid false blow event detection. */
	fuse_ext->last_blow = jiffies - msecs_to_jiffies(reload_delay);
	fuse_ext->data = fuse_data;
	fuse_data->ext = fuse_ext;

	/* Enable fuse GPIO. */

	err = gpio_request(gpio->num, gpio->name);
	if (err) {
		dev_err(&pdev->dev, "%s: failed to request GPIO: %s\n",
			port_data->name, gpio->name);
		goto exit_free_fuse_data;
	}
	err = gpio_direction_input(gpio->num);
	if (err) {
		dev_err(&pdev->dev, "%s: failed to configure GPIO: %s\n",
			port_data->name, gpio->name);
		goto exit_free_gpio;
	}

	/* Create sysfs attribute entry. */

	fuse_attr->attr.name = gpio->name;
	fuse_attr->attr.mode = S_IRUGO;
	fuse_attr->show = fuse_show;

	err = sysfs_create_file(&port_data->kobj, &fuse_attr->attr);
	if (err) {
		dev_err(&pdev->dev, "%s: failed to create sysfs entry %s\n",
			port_data->name, gpio->name);
		goto exit_free_gpio;
	}

	/* Enable fuse IRQ */

	fuse_irq = gpio_to_irq(gpio->num);
	set_irq_type(fuse_irq, gpio->act_low ? IRQ_TYPE_EDGE_FALLING :
					       IRQ_TYPE_EDGE_RISING);
	err = request_irq(fuse_irq, fuse_irq_handler, 0, gpio->name, fuse_data);
	if (err) {
		dev_err(&pdev->dev, "%s: failed to request IRQ for %s\n",
			port_data->name, gpio->name);
		goto exit_free_sysfs;
	}

	return 0;

exit_free_sysfs:
	sysfs_remove_file(&port_data->kobj, &fuse_attr->attr);
exit_free_gpio:
	gpio_free(gpio->num);
exit_free_fuse_data:
	kfree(fuse_data);

	return err;
}

static void gpio_usb_fuse_free(struct gpio_usb_data *fuse_data)
{
	struct gpio_usb_port_data *port_data = fuse_data->port_data;
	struct fuse_ext *fuse_ext = fuse_data->ext;

	free_irq(gpio_to_irq(fuse_data->gpio->num), fuse_data);
	cancel_delayed_work(&fuse_ext->dwork);
	sysfs_remove_file(&port_data->kobj, &fuse_data->attr.attr);
	gpio_free(fuse_data->gpio->num);
	kfree(fuse_ext);
}

/*
 * Mode support (device or host mode configuration).
 */

static ssize_t mode_show(struct gpio_usb_data *mode_data, char *buff)
{
	return sprintf(buff, "%i\n", gpio_usb_get_value(mode_data->gpio));
}

static ssize_t mode_store(struct gpio_usb_data *mode_data,
			  const char *buff, size_t count)
{
	int enable;

	enable = !!simple_strtol(buff, NULL, 10);
	gpio_usb_set_value(mode_data->gpio, enable);

	return count;
}

static int gpio_usb_mode_init(struct gpio_usb_data *mode_data)
{
	int err;
	struct gpio_usb_port_data *port_data = mode_data->port_data;
	struct gpio_usb_attribute *mode_attr = &mode_data->attr;
	struct platform_device *pdev = port_data->pdev;
	const struct gpio_usb *gpio = mode_data->gpio;

	/* Enable mode GPIO. */

	err = gpio_request(gpio->num, gpio->name);
	if (err) {
		dev_err(&pdev->dev, "%s: failed to request GPIO %s\n",
			port_data->name, gpio->name);
		return err;
	}
	/* Keep GPIO value. */
	err = gpio_direction_output(gpio->num, gpio_get_value(gpio->num));
	if (err) {
		dev_err(&pdev->dev, "%s: failed to configure GPIO %s\n",
			port_data->name, gpio->name);
		goto exit_free_gpio;
	}

	/* Create sysfs attribute entry. */

	mode_attr->attr.name = gpio->name;
	mode_attr->attr.mode = S_IRUGO | S_IWUSR;
	mode_attr->show = mode_show;
	mode_attr->store = mode_store;

	err = sysfs_create_file(&port_data->kobj, &mode_attr->attr);
	if (err) {
		dev_err(&pdev->dev, "%s: failed to create sysfs entry %s\n",
			port_data->name, gpio->name);
		goto exit_free_gpio;
	}

	return 0;

exit_free_gpio:
	gpio_free(gpio->num);

	return err;
}

static void gpio_usb_mode_free(struct gpio_usb_data *mode_data)
{
	struct gpio_usb_port_data *port_data = mode_data->port_data;

	sysfs_remove_file(&port_data->kobj, &mode_data->attr.attr);
	gpio_free(mode_data->gpio->num);
}

/*
 * VBus in support.
 */

static void vbus_in_notify(struct work_struct *ws)
{
	struct vbus_in_ext *vbus_in_ext =
		container_of(ws, struct vbus_in_ext, work);
	struct gpio_usb_data *vbus_in_data = vbus_in_ext->data;
	struct gpio_usb_port_data *port_data = vbus_in_data->port_data;
	const struct gpio_usb *gpio = vbus_in_data->gpio;

	dev_dbg(&port_data->pdev->dev, "%s %s: %s signal change\n",
		__FUNCTION__, port_data->name, gpio->name);

	sysfs_notify(&port_data->pdev->dev.kobj, NULL, gpio->name);
	kobject_uevent(&port_data->pdev->dev.kobj, KOBJ_CHANGE);
}

/*
 * IRQ handler called when a USB VBUS change is detected (when a USB
 * host is plugged on a port B for example). Notify userspace process
 * from this change.
 */
static irqreturn_t vbus_in_irq_handler(int irq, void *data)
{
	struct gpio_usb_data *vbus_in_data = data;
	struct vbus_in_ext *vbus_in_ext = vbus_in_data->ext;

	schedule_work(&vbus_in_ext->work);

	return IRQ_HANDLED;
}

static ssize_t vbus_in_show(struct gpio_usb_data *vbus_in_data, char *buff)
{
	return sprintf(buff, "%i\n", gpio_usb_get_value(vbus_in_data->gpio));
}

static int gpio_usb_vbus_in_init(struct gpio_usb_data *vbus_in_data)
{
	int err;
	int vbus_in_irq;
	struct gpio_usb_port_data *port_data = vbus_in_data->port_data;
	struct gpio_usb_attribute *vbus_in_attr = &vbus_in_data->attr;
	struct platform_device *pdev = port_data->pdev;
	struct vbus_in_ext *vbus_in_ext;
	const struct gpio_usb *gpio = vbus_in_data->gpio;

	/* Initialize VBUS-in priv data. */

	vbus_in_ext = kzalloc(sizeof (*vbus_in_ext), GFP_KERNEL);
	if (!vbus_in_ext)
		return -ENOMEM;

	INIT_WORK(&vbus_in_ext->work, vbus_in_notify);
	vbus_in_data->ext = vbus_in_ext;
	vbus_in_ext->data = vbus_in_data;

	/* Enable fuse GPIO. */

	err = gpio_request(gpio->num, gpio->name);
	if (err) {
		dev_err(&pdev->dev, "%s: failed to request GPIO %s\n",
			port_data->name, gpio->name);
		goto exit_free_vbus_in_data;
	}
	err = gpio_direction_input(gpio->num);
	if (err) {
		dev_err(&pdev->dev, "%s: failed to configure GPIO %s\n",
			port_data->name, gpio->name);
		goto exit_free_gpio;
	}

	/* Create sysfs attribute entry. */

	vbus_in_attr->attr.name = gpio->name;
	vbus_in_attr->attr.mode = S_IRUGO;
	vbus_in_attr->show = vbus_in_show;

	err = sysfs_create_file(&port_data->kobj, &vbus_in_attr->attr);
	if (err) {
		dev_err(&pdev->dev, "%s: failed to create sysfs entry %s\n",
			port_data->name, gpio->name);
		goto exit_free_gpio;
	}

	/* Enable VBUS-in IRQ. */

	vbus_in_irq = gpio_to_irq(gpio->num);
	set_irq_type(vbus_in_irq, IRQ_TYPE_EDGE_BOTH);
	err = request_irq(vbus_in_irq, vbus_in_irq_handler, 0, gpio->name,
			  vbus_in_data);
	if (err) {
		dev_err(&pdev->dev, "%s: failed to request IRQ for %s\n",
			port_data->name, gpio->name);
		goto exit_free_sysfs;
	}

	return 0;

exit_free_sysfs:
	sysfs_remove_file(&port_data->kobj, &vbus_in_attr->attr);
exit_free_gpio:
	gpio_free(gpio->num);
exit_free_vbus_in_data:
	kfree(vbus_in_data);

	return err;
}

static void gpio_usb_vbus_in_free(struct gpio_usb_data *vbus_in_data)
{
	struct gpio_usb_port_data *port_data = vbus_in_data->port_data;
	struct vbus_in_ext *vbus_in_ext = vbus_in_data->ext;
	const struct gpio_usb *gpio = vbus_in_data->gpio;

	free_irq(gpio_to_irq(gpio->num), vbus_in_data);
	cancel_work_sync(&vbus_in_ext->work);
	sysfs_remove_file(&port_data->kobj, &vbus_in_data->attr.attr);
	gpio_free(gpio->num);
	kfree(vbus_in_ext);
}

/*
 * VBus out support.
 */

static ssize_t vbus_out_show(struct gpio_usb_data *vbus_out_data, char *buff)
{
	return sprintf(buff, "%i\n", gpio_usb_get_value(vbus_out_data->gpio));
}

static ssize_t vbus_out_store(struct gpio_usb_data *vbus_out_data,
			      const char *buff, size_t count)
{
	int err = 0;
	int enable;
	struct gpio_usb_port_data *port_data = vbus_out_data->port_data;
	struct gpio_usb_data *fuse_data = NULL;
	struct fuse_ext *fuse_ext;

	enable = !!simple_strtol(buff, NULL, 10);

	fuse_data = data_by_type_lookup(port_data, GPIO_USB_FUSE);

	/* Handle simple case: disable VBUS-out or no fuse available. */
	if (!enable || !fuse_data) {
		gpio_usb_set_value(vbus_out_data->gpio, enable);
		goto exit;
	}

	fuse_ext = fuse_data->ext;

	/* After a fuse blow, wait until reload delay has elapsed. */
	if (fuse_ext->reload_delay &&
		time_before(jiffies, fuse_ext->last_blow +
			    msecs_to_jiffies(fuse_ext->reload_delay))) {
		err = -EAGAIN;
		goto exit;
	}

	/*
	 * Check the fuse state 20ms after re-enabling VBUS. If the fuse level
	 * is down it means that a faulty USB device is (still) plugged and an
	 * USB fuse event must be triggered.
	 * This check is needed because an immediate blow don't let enough
	 * time to reach a high level on the fuse pin. As a consequence, a
	 * falling edge interrupt can't happen.
	 */
	gpio_usb_set_value(vbus_out_data->gpio, enable);
	__set_current_state(TASK_UNINTERRUPTIBLE);
	schedule_timeout(msecs_to_jiffies(20));

	if (gpio_usb_get_value(fuse_data->gpio) == 1)
		schedule_delayed_work(&fuse_ext->dwork, 0);

exit:
	return err ? err : count;
}

static int gpio_usb_vbus_out_init(struct gpio_usb_data *vbus_out_data)
{
	int err;
	struct gpio_usb_port_data *port_data = vbus_out_data->port_data;
	struct gpio_usb_attribute *vbus_out_attr = &vbus_out_data->attr;
	struct platform_device *pdev = port_data->pdev;
	const struct gpio_usb *gpio = vbus_out_data->gpio;

	/* Enable VBUS-out GPIO */

	err = gpio_request(gpio->num, gpio->name);
	if (err) {
		dev_err(&pdev->dev, "%s: failed to request GPIO %s\n",
			port_data->name, gpio->name);
		return err;
	}
	/* Keep GPIO value. */
	err = gpio_direction_output(gpio->num, gpio_get_value(gpio->num));
	if (err) {
		dev_err(&pdev->dev, "%s: failed to configure GPIO %s\n",
			port_data->name, gpio->name);
		goto exit_free_gpio;
	}

	/* Create sysfs attribute entry. */

	vbus_out_attr->attr.name = gpio->name;
	vbus_out_attr->attr.mode = S_IRUGO | S_IWUSR;
	vbus_out_attr->show = vbus_out_show;
	vbus_out_attr->store = vbus_out_store;

	err = sysfs_create_file(&port_data->kobj, &vbus_out_attr->attr);
	if (err) {
		dev_err(&pdev->dev, "%s: failed to create sysfs entry %s\n",
			port_data->name, gpio->name);
		goto exit_free_gpio;
	}

	return 0;

exit_free_gpio:
	gpio_free(gpio->num);

	return err;
}

static void gpio_usb_vbus_out_free(struct gpio_usb_data *vbus_out_data)
{
	struct gpio_usb_port_data *port_data = vbus_out_data->port_data;

	sysfs_remove_file(&port_data->kobj, &vbus_out_data->attr.attr);
	gpio_free(vbus_out_data->gpio->num);
}

/*
 * GPIO USB port
 */

static void gpio_usb_port_free(struct gpio_usb_port_data *port_data)
{
	struct gpio_usb_data *data;
	struct gpio_usb_data *tmp;

	down_write(&port_data->rw_sem);

	list_for_each_entry_safe(data, tmp, &port_data->list, node) {
		list_del(&data->node);

		switch (data->gpio->type) {
		case GPIO_USB_FUSE:
			gpio_usb_fuse_free(data);
			break;
		case GPIO_USB_MODE:
			gpio_usb_mode_free(data);
			break;
		case GPIO_USB_VBUS_IN:
			gpio_usb_vbus_in_free(data);
			break;
		case GPIO_USB_VBUS_OUT:
			gpio_usb_vbus_out_free(data);
			break;
		default:
			continue;
		}
		kfree(data);

	}

	up_write(&port_data->rw_sem);
}

static int gpio_usb_port_init(struct gpio_usb_port_data *port_data,
			      struct gpio_usb_port *port)
{
	int err;
	int i;

	init_rwsem(&port_data->rw_sem);
	INIT_LIST_HEAD(&port_data->list);

	for (i = 0; i < port->num_gpio; i++) {
		const struct gpio_usb *gpio = &port->gpio[i];
		struct gpio_usb_data *data;

		/* Except for VBUS-out GPIO, a port can't host several GPIOs
		 * with a same type. */
		if (gpio->type != GPIO_USB_VBUS_OUT &&
		    data_by_type_lookup(port_data, gpio->type)) {
			err = -EINVAL;
			goto exit_free_port;
		}

		data = kzalloc(sizeof(*data), GFP_KERNEL);
		if (!data) {
			err = -ENOMEM;
			goto exit_free_port;
		}

		data->gpio = gpio;
		data->port_data = port_data;

		switch (gpio->type) {
		case GPIO_USB_FUSE:
			err = gpio_usb_fuse_init(data, port->reload_delay,
						 port->spurious_delay);
			break;
		case GPIO_USB_MODE:
			err = gpio_usb_mode_init(data);
			break;
		case GPIO_USB_VBUS_IN:
			err = gpio_usb_vbus_in_init(data);
			break;
		case GPIO_USB_VBUS_OUT:
			err = gpio_usb_vbus_out_init(data);
			break;
		default:
			err = -EINVAL;
			break;
		}
		if (err < 0) {
			kfree(data);
			goto exit_free_port;
		}

		down_write(&port_data->rw_sem);
		list_add(&data->node, &port_data->list);
		up_write(&port_data->rw_sem);

	}

exit_free_port:
	if (err)
		gpio_usb_port_free(port_data);
	return err;
}

static int gpio_usb_probe(struct platform_device *pdev)
{
	struct gpio_usb_platform_data *pdata = pdev->dev.platform_data;
	struct gpio_usb_port_data *ports_data;
	int err;
	int i;

	ports_data = kzalloc(sizeof (struct gpio_usb_port_data) *
			     pdata->num_port, GFP_KERNEL);
	if (!ports_data)
		return -ENOMEM;

	for (i = 0; i < pdata->num_port; i++) {
		struct gpio_usb_port *port = &pdata->port[i];
		struct gpio_usb_port_data *port_data = &ports_data[i];

		init_completion(&port_data->kobj_unregister);

		/*
		 * Create a sysfs sub-directory for each GPIO USB port.
		 */
		err = kobject_init_and_add(&port_data->kobj, &gpio_usb_ktype,
					   &pdev->dev.kobj, "%s", port->name);
		if (err) {
			dev_err(&pdev->dev, "failed to register kobject %s\n",
				port->name);
			goto exit_free_ports;
		}
		kobject_uevent(&port_data->kobj, KOBJ_ADD);

		port_data->pdev = pdev;
		port_data->name = port->name;

		err = gpio_usb_port_init(port_data, port);
		if (err < 0) {
			kobject_put(&port_data->kobj);
			wait_for_completion(&port_data->kobj_unregister);
			goto exit_free_ports;
		}
	}

	platform_set_drvdata(pdev, ports_data);

	dev_info(&pdev->dev, "USB GPIO's initialized\n");

	return 0;

exit_free_ports:
	for (i = i - 1; i >= 0; i--) {
		gpio_usb_port_free(&ports_data[i]);
		kobject_put(&ports_data[i].kobj);
		wait_for_completion(&ports_data[i].kobj_unregister);
	}
	kfree(ports_data);

	return err;
}

static int __devexit gpio_usb_remove(struct platform_device *pdev)
{
	struct gpio_usb_platform_data *pdata = pdev->dev.platform_data;
	struct gpio_usb_port_data *ports_data = platform_get_drvdata(pdev);
	int i;

	for (i = 0; i < pdata->num_port; i++) {
		struct gpio_usb_port_data *port_data = &ports_data[i];

		gpio_usb_port_free(port_data);
		kobject_put(&port_data->kobj);
		wait_for_completion(&port_data->kobj_unregister);
	}

	kfree(ports_data);
	platform_set_drvdata(pdev, NULL);

	dev_info(&pdev->dev, "USB GPIO's removed\n");

	return 0;
}

static struct platform_driver gpio_usb_driver = {
	.probe   = gpio_usb_probe,
	.remove  = __devexit_p(gpio_usb_remove),
	.driver  = {
		.name = "gpio-usb",
	},
};

static int __init gpio_usb_init(void)
{
	return platform_driver_register(&gpio_usb_driver);
}

static void __exit gpio_usb_exit(void)
{
	platform_driver_unregister(&gpio_usb_driver);
}

module_init(gpio_usb_init);
module_exit(gpio_usb_exit);

MODULE_AUTHOR("Simon Guinot <sguinot@lacie.com>");
MODULE_DESCRIPTION("GPIO USB driver");
MODULE_LICENSE("GPL");
