/*
 * leds-ns2l.c - Driver for the Network Space Lite v2 LEDs
 *
 * Copyright (C) 2010 LaCie
 *
 * Author: Simon Guinot <sguinot@lacie.com>
 *
 * Based on leds-gpio.c by Raphael Assenat <raph@8d.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/platform_device.h>
#include <linux/slab.h>
#include <linux/gpio.h>
#include <linux/leds.h>
#include <linux/io.h>

#include "leds-ns2l.h"

/*
 * The Network Space Lite v2 LED is wired to a single MPP (multi purpose
 * pin). Tree states are availables: "on", "off" and "SATA blinking". To
 * enable the "on" and "off" states the MPP is configured to act like an
 * output GPIO. To enable the "SATA blinking" state, the MPP is
 * configured to drive a SATA activity signal.
 */

enum ns2l_led_modes {
	NS2L_LED_OFF,
	NS2L_LED_ON,
	NS2L_LED_SATA,
};

struct ns2l_led_data {
	struct led_classdev	cdev;
	unsigned		mpp;
	unsigned int		gpio_sel;
	unsigned int		sata_sel;
	unsigned char		sata; /* True when SATA mode active. */
	enum ns2l_led_modes	mode;
	rwlock_t		rw_lock; /* Lock GPIOs. */
};

#define MPP_CTRL(i) ((INTER_REGS_BASE | 0x10000) + (i) * 4)

static void ns2l_led_set_mpp_sel(struct ns2l_led_data *led_dat,
				 unsigned int sel)
{
	int shift;
	unsigned mpp = led_dat->mpp;
	u32 mpp_ctrl;

	mpp_ctrl = readl(MPP_CTRL(mpp / 8));

	shift = (mpp & 7) << 2;
	mpp_ctrl &= ~(0xf << shift);
	mpp_ctrl |= sel << shift;
	writel(mpp_ctrl, MPP_CTRL(mpp / 8));
}

static unsigned int ns2l_led_get_mpp_sel(struct ns2l_led_data *led_dat)
{
	int shift;
	u32 mpp_ctrl;

	mpp_ctrl = readl(MPP_CTRL(led_dat->mpp / 8));
	shift = (led_dat->mpp & 7) << 2;

	return (mpp_ctrl >> shift) & 0xf;
}

static void ns2l_led_set_mode(struct ns2l_led_data *led_dat,
			      enum ns2l_led_modes mode)
{
	unsigned long flags;

	write_lock_irqsave(&led_dat->rw_lock, flags);

	if (led_dat->mode == mode)
		return;

	if (mode == NS2L_LED_OFF || mode == NS2L_LED_ON) {
		if (led_dat->mode == NS2L_LED_SATA)
			ns2l_led_set_mpp_sel(led_dat, led_dat->gpio_sel);
		gpio_set_value(led_dat->mpp, mode);
	} else {
		ns2l_led_set_mpp_sel(led_dat, led_dat->sata_sel);
	}
	led_dat->mode = mode;

	write_unlock_irqrestore(&led_dat->rw_lock, flags);
}

static enum ns2l_led_modes ns2l_led_get_mode(struct ns2l_led_data *led_dat)
{
	unsigned mpp_sel;
	enum ns2l_led_modes mode;

	read_lock_irq(&led_dat->rw_lock);

	mpp_sel = ns2l_led_get_mpp_sel(led_dat);
	if (mpp_sel == led_dat->sata_sel)
		mode = NS2L_LED_SATA;
	else if (gpio_get_value(led_dat->mpp))
		mode = NS2L_LED_ON;
	else
		mode = NS2L_LED_OFF;

	read_unlock_irq(&led_dat->rw_lock);

	return mode;
}

static void ns2l_led_set(struct led_classdev *led_cdev,
			enum led_brightness value)
{
	struct ns2l_led_data *led_dat =
		container_of(led_cdev, struct ns2l_led_data, cdev);
	enum ns2l_led_modes mode;

	if (value == LED_OFF)
		mode = NS2L_LED_OFF;
	else if (led_dat->sata)
		mode = NS2L_LED_SATA;
	else
		mode = NS2L_LED_ON;

	ns2l_led_set_mode(led_dat, mode);
}

static ssize_t ns2l_led_sata_store(struct device *dev,
				   struct device_attribute *attr,
				   const char *buff, size_t count)
{
	struct led_classdev *led_cdev = dev_get_drvdata(dev);
	struct ns2l_led_data *led_dat =
		container_of(led_cdev, struct ns2l_led_data, cdev);
	int ret;
	unsigned long enable;
	enum ns2l_led_modes mode;

	ret = strict_strtoul(buff, 10, &enable);
	if (ret < 0)
		return ret;

	enable = !!enable;

	if (led_dat->sata == enable)
		return count;

	mode = ns2l_led_get_mode(led_dat);

	if (enable && mode == NS2L_LED_ON)
		ns2l_led_set_mode(led_dat, NS2L_LED_SATA);
	if (!enable && mode == NS2L_LED_SATA)
		ns2l_led_set_mode(led_dat, NS2L_LED_ON);

	led_dat->sata = enable;

	return count;
}

static ssize_t ns2l_led_sata_show(struct device *dev,
				  struct device_attribute *attr, char *buf)
{
	struct led_classdev *led_cdev = dev_get_drvdata(dev);
	struct ns2l_led_data *led_dat =
		container_of(led_cdev, struct ns2l_led_data, cdev);

	return sprintf(buf, "%d\n", led_dat->sata);
}

static DEVICE_ATTR(sata, 0644, ns2l_led_sata_show, ns2l_led_sata_store);

static int __devinit
create_ns2l_led(struct platform_device *pdev, struct ns2l_led_data *led_dat,
		const struct ns2l_led *template)
{
	int ret;
	enum ns2l_led_modes mode;

	rwlock_init(&led_dat->rw_lock);

	led_dat->cdev.name = template->name;
	led_dat->cdev.default_trigger = template->default_trigger;
	led_dat->cdev.blink_set = NULL;
	led_dat->cdev.brightness_set = ns2l_led_set;
	led_dat->cdev.flags |= LED_CORE_SUSPENDRESUME;
	led_dat->mpp = template->mpp;
	led_dat->gpio_sel = template->gpio_sel;
	led_dat->sata_sel = template->sata_sel;

	mode = ns2l_led_get_mode(led_dat);

	/* Set LED initial state. */
	led_dat->sata = (mode == NS2L_LED_SATA) ? 1 : 0;
	led_dat->cdev.brightness =
		(mode == NS2L_LED_OFF) ? LED_OFF : LED_FULL;

	/*
	 * Even if the MPP will be (re-)configured to drive the SATA activity,
	 * it must be registered within the GPIO layer.
	 */
	feroceon_gpio_set_valid(led_dat->mpp, GPIO_OUTPUT_OK | GPIO_INPUT_OK);

	ret = gpio_request(led_dat->mpp, led_dat->cdev.name);
	if (ret == 0) {
		ret = gpio_direction_output(led_dat->mpp,
					    gpio_get_value(led_dat->mpp));
		if (ret)
			gpio_free(led_dat->mpp);
	}
	if (ret) {
		dev_err(&pdev->dev, "failed to configure GPIO %s\n",
			template->name);
		return ret;
	}

	/* Restore initial state */
	ns2l_led_set_mode(led_dat, mode);

	ret = led_classdev_register(&pdev->dev, &led_dat->cdev);
	if (ret < 0)
		goto exit_free_gpio;

	ret = device_create_file(led_dat->cdev.dev, &dev_attr_sata);
	if (ret < 0)
		goto exit_free_cdev;

	return 0;

exit_free_cdev:
	led_classdev_unregister(&led_dat->cdev);
exit_free_gpio:
	gpio_free(led_dat->mpp);

	return ret;
}

static void __devexit delete_ns2l_led(struct ns2l_led_data *led_dat)
{
	device_remove_file(led_dat->cdev.dev, &dev_attr_sata);
	led_classdev_unregister(&led_dat->cdev);
	/* Select MPP SATA activity. */
	gpio_free(led_dat->mpp);
	feroceon_gpio_set_valid(led_dat->mpp, 0);
	ns2l_led_set_mpp_sel(led_dat, led_dat->sata_sel);
}

static int __devinit ns2l_led_probe(struct platform_device *pdev)
{
	struct ns2l_led_platform_data *pdata = pdev->dev.platform_data;
	struct ns2l_led_data *leds_data;
	int i;
	int ret;

	if (!pdata)
		return -EINVAL;

	leds_data = kzalloc(sizeof(struct ns2l_led_data) *
			    pdata->num_leds, GFP_KERNEL);
	if (!leds_data)
		return -ENOMEM;

	for (i = 0; i < pdata->num_leds; i++) {
		ret = create_ns2l_led(pdev, &leds_data[i], &pdata->leds[i]);
		if (ret < 0)
			goto err;
	}

	platform_set_drvdata(pdev, leds_data);

	return 0;

err:
	for (i = i - 1; i >= 0; i--)
		delete_ns2l_led(&leds_data[i]);

	kfree(leds_data);

	return ret;
}

static int __devexit ns2l_led_remove(struct platform_device *pdev)
{
	int i;
	struct ns2l_led_platform_data *pdata = pdev->dev.platform_data;
	struct ns2l_led_data *leds_data;

	leds_data = platform_get_drvdata(pdev);

	for (i = 0; i < pdata->num_leds; i++)
		delete_ns2l_led(&leds_data[i]);

	kfree(leds_data);
	platform_set_drvdata(pdev, NULL);

	return 0;
}

static struct platform_driver ns2l_led_driver = {
	.probe		= ns2l_led_probe,
	.remove		= __devexit_p(ns2l_led_remove),
	.driver		= {
		.name	= "leds-ns2l",
		.owner	= THIS_MODULE,
	},
};
MODULE_ALIAS("platform:leds-ns2l");

static int __init ns2l_led_init(void)
{
	return platform_driver_register(&ns2l_led_driver);
}

static void __exit ns2l_led_exit(void)
{
	platform_driver_unregister(&ns2l_led_driver);
}

module_init(ns2l_led_init);
module_exit(ns2l_led_exit);

MODULE_AUTHOR("Simon Guinot <sguinot@lacie.com>");
MODULE_DESCRIPTION("Network Space Lite v2 LED driver");
MODULE_LICENSE("GPL");
