/*
 * Copyright (C) 2005-2006 Junjiro Okajima
 *
 * This program, aufs is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/* $Id: i_op.c,v 1.14 2006/10/09 00:28:13 sfjro Exp $ */

//#include <linux/fs.h>
//#include <linux/namei.h>
#include <linux/security.h>
#include <asm/uaccess.h>
#include "aufs.h"

static int hidden_permission(struct inode *hidden_inode, int mask,
			     struct nameidata *fake_nd, int brperm)
{
	int err, submask;
	const int write_mask = (mask & (MAY_WRITE | MAY_APPEND));

	LKTRTrace("ino %lu, mask 0x%x, brperm 0x%x\n",
		  hidden_inode->i_ino, mask, brperm);

	err = -EACCES;
	if (unlikely(write_mask && IS_IMMUTABLE(hidden_inode)))
		goto out;

	/* skip hidden fs test in the case of write to ro branch */
	submask = mask & ~MAY_APPEND;
	if (unlikely((write_mask && !(brperm & MAY_WRITE))
		     || !hidden_inode->i_op
		     || !hidden_inode->i_op->permission))
		err = generic_permission(hidden_inode, submask, NULL);
	else {
		err = hidden_inode->i_op->permission(hidden_inode, submask,
						     fake_nd);
		TraceErr(err);
	}

	if (!err)
		err = security_inode_permission(hidden_inode, mask, fake_nd);

 out:
	TraceErr(err);
	return err;
}

static int aufs_permission(struct inode *inode, int mask, struct nameidata *nd)
{
	int err;
	aufs_bindex_t bindex, bend;
	struct inode *hidden_inode;
	struct super_block *sb;
	struct nameidata fake_nd, *p;
	const int write_mask = (mask & (MAY_WRITE | MAY_APPEND));
	const int nondir = !S_ISDIR(inode->i_mode);

	LKTRTrace("ino %lu, mask 0x%x, nondir %d, write_mask %d\n",
		  inode->i_ino, mask, nondir, write_mask);

	sb = inode->i_sb;
	si_read_lock(sb);
	ii_read_lock(inode);

	if (nd)
		fake_nd = *nd;
	if (/* unlikely */(nondir || write_mask)) {
		hidden_inode = itohi(inode);
		DEBUG_ON(!hidden_inode
			 || ((hidden_inode->i_mode & S_IFMT)
			     != (inode->i_mode & S_IFMT)));
		bindex = ibstart(inode);
		p = fake_dm(&fake_nd, nd, sb, bindex);
		err = hidden_permission(hidden_inode, mask, p,
					sbr_perm(sb, bindex));
		fake_dm_release(p);
		if (write_mask && !err) {
			err = find_rw_br(sb, bindex);
			if (err >= 0)
				err = 0;
		}
		goto out;
	}

	/* non-write to dir */
	err = 0;
	bend = ibend(inode);
	for (bindex = ibstart(inode); !err && bindex <= bend; bindex++) {
		hidden_inode = itohi_index(inode, bindex);
		if (!hidden_inode)
			continue;
		DEBUG_ON(!S_ISDIR(hidden_inode->i_mode));

		p = fake_dm(&fake_nd, nd, sb, bindex);
		err = hidden_permission(hidden_inode, mask, p,
					sbr_perm(sb, bindex));
		fake_dm_release(p);
	}

 out:
	ii_read_unlock(inode);
	si_read_unlock(sb);
	TraceErr(err);
	return err;
}

/* ---------------------------------------------------------------------- */

static struct dentry *aufs_lookup(struct inode *dir, struct dentry *dentry,
				  struct nameidata *nd)
{
	struct dentry *ret, *parent;
	int err, npositive;
	struct inode *inode;

	LKTRTrace("dir %lu, %.*s\n", dir->i_ino, DLNPair(dentry));
	DEBUG_ON(IS_ROOT(dentry));
	IMustLock(dir);

	parent = dentry->d_parent; // dget_parent()
	aufs_read_lock(parent, 0);
	err = alloc_dinfo(dentry);
	//if (LktrCond) err = -1;
	ret = ERR_PTR(err);
	if (unlikely(err))
		goto out;

	err = npositive = lookup_dentry(dentry, dbstart(parent), /* type */ 0);
	//err = -1;
	ret = ERR_PTR(err);
	if (unlikely(err < 0))
		goto out_unlock;
	inode = NULL;
	if (npositive) {
		inode = aufs_new_inode(dentry);
		ret = (void*)inode;
	}
	if (!IS_ERR(inode)) {
		/* d_splice_alias() also supports d_add() */
		ret = d_splice_alias(inode, dentry);
		if (unlikely(IS_ERR(ret) && inode))
			ii_write_unlock(inode);
	}

 out_unlock:
	di_write_unlock(dentry);
 out:
	aufs_read_unlock(parent, 0);
	TraceErrPtr(ret);
	return ret;
}

/* ---------------------------------------------------------------------- */

/*
 * decide the branch and the parent dir where we will create a new entry.
 * returns new bindex or an error.
 * copyup the parent dir if needed.
 */
int wr_dir(struct dentry *dentry, int add_entry, struct dentry *src_dentry,
	   aufs_bindex_t force_btgt, int do_lock_srcdir)
{
	int err;
	aufs_bindex_t bcpup, bstart, src_bstart;
	struct dentry *hidden_parent;
	struct super_block *sb;
	struct dentry *parent, *src_parent = NULL;
	struct inode *dir, *src_dir = NULL;

	LKTRTrace("%.*s, add %d, src %p, force %d, lock_srcdir %d\n",
		  DLNPair(dentry), add_entry, src_dentry, force_btgt,
		  do_lock_srcdir);

	sb = dentry->d_sb;
	parent = dentry->d_parent; // dget_parent()
	bcpup = bstart = dbstart(dentry);
	if (force_btgt < 0) {
		if (src_dentry) {
			src_bstart = dbstart(src_dentry);
			if (src_bstart < bstart)
				bcpup = src_bstart;
		}
		if (test_ro(sb, bcpup, dentry->d_inode)) {
			if (!add_entry)
				di_read_lock(parent, 0);
			bcpup = err = find_rw_parent_br(dentry, bcpup);
			if (!add_entry)
				di_read_unlock(parent, 0);
			//err = -1;
			if (unlikely(err < 0))
				goto out;
		}
	} else {
		DEBUG_ON(bstart <= force_btgt
			 || test_ro(sb, force_btgt, dentry->d_inode));
		bcpup = force_btgt;
	}
	LKTRTrace("bstart %d, bcpup %d\n", bstart, bcpup);

	err = bcpup;
	if (bcpup == bstart)
		goto out; /* success */

	/* copyup the new parent into the branch we process */
	hidden_parent = dtohd(dentry)->d_parent; // dget_parent()
	if (src_dentry) {
		src_parent = src_dentry->d_parent; // dget_parent()
		src_dir = src_parent->d_inode;
		if (do_lock_srcdir)
			di_write_lock(src_parent);
	}

	dir = parent->d_inode;
	if (add_entry) {
		update_dbstart(dentry);
		IMustLock(dir);
		DiMustWriteLock(parent);
		IiMustWriteLock(dir);
	} else
		di_write_lock(parent);

	err = 0;
	if (!dtohd_index(parent, bcpup))
		err = cpup_dirs(dentry, bcpup, src_parent);
	//err = -1;
	if (!err && add_entry) {
		hidden_parent = dtohd_index(dentry->d_parent, bcpup);
		DEBUG_ON(!hidden_parent || !hidden_parent->d_inode);
		i_lock(hidden_parent->d_inode);
		err = lookup_negative(dentry, bcpup);
		//err = -1;
		i_unlock(hidden_parent->d_inode);
	}

	if (!add_entry)
		di_write_unlock(parent);
	if (do_lock_srcdir)
		di_write_unlock(src_parent);
	if (!err)
		err = bcpup; /* success */
	//err = -EPERM;
 out:
	TraceErr(err);
	return err;
}

/* ---------------------------------------------------------------------- */

static int aufs_setattr(struct dentry *dentry, struct iattr *ia)
{
	int err;
	aufs_bindex_t bstart, bcpup;
	struct inode *hidden_inode, *inode, *dir;
	struct dentry *hidden_dentry;

	LKTRTrace("%.*s, ia_valid 0x%x\n", DLNPair(dentry), ia->ia_valid);
	inode = dentry->d_inode;
	IMustLock(inode);

	aufs_read_lock(dentry, AUFS_I_WLOCK);
	dir = dentry->d_parent->d_inode;
	bstart = dbstart(dentry);
	bcpup = err = wr_dir(dentry, 0, NULL, -1, 0);
	//err = -1;
	if (unlikely(err < 0))
		goto out;

	hidden_dentry = dtohd(dentry);
	hidden_inode = hidden_dentry->d_inode;
	i_lock(hidden_inode);

	if (bstart != bcpup) {
		struct inode *dir, *hidden_dir;
		struct dentry *parent;
		loff_t size = -1;

		if ((ia->ia_valid & ATTR_SIZE)
		    && ia->ia_size < i_size_read(inode)) {
			size = ia->ia_size;
			ia->ia_valid &= ~ATTR_SIZE;
		}
		parent = dentry->d_parent; // dget_parent()
		dir = parent->d_inode;
		di_read_unlock(dentry, AUFS_I_WLOCK);
		di_write_lock(dentry);
		di_read_lock(parent, AUFS_I_RLOCK);
		hidden_dir = itohi_index(dir, bcpup);
		i_lock(hidden_dir);
		err = 0;
		if (!dtohd_index(dentry, bcpup))
			err = sio_cpup_simple(dentry, bcpup, size, 1);
			//err = -1;
		if (0 && !err && S_ISDIR(inode->i_mode))
			d_drop(dentry->d_parent); // next time, update i_nlink.

		// lock child inode before unlock parent
		i_unlock(hidden_inode);
		di_downgrade_lock(dentry, AUFS_I_WLOCK);
		hidden_dentry = dtohd(dentry);
		hidden_inode = hidden_dentry->d_inode;
		DEBUG_ON(!hidden_inode);
		i_lock(hidden_inode);

		i_unlock(hidden_dir);
		di_read_unlock(parent, AUFS_I_RLOCK);
		if (err || !ia->ia_valid)
			goto out_cpup;
	}
	err = hidden_notify_change(hidden_dentry, ia);
	//err = -1;

 out_cpup:
	if (!err)
		cpup_attr_changable(inode);
	i_unlock(hidden_inode);
 out:
	aufs_read_unlock(dentry, AUFS_I_WLOCK);
	TraceErr(err);
	return err;
}

/* ---------------------------------------------------------------------- */

static int hidden_readlink(struct dentry *dentry, int bindex,
			   char __user * buf, int bufsiz)
{
	struct super_block *sb;
	struct vfsmount *hidden_mnt;
	struct dentry *hidden_dentry;

	hidden_dentry = dtohd_index(dentry, bindex);
	if (unlikely(!hidden_dentry->d_inode->i_op
		     || !hidden_dentry->d_inode->i_op->readlink))
		return -EINVAL;

	sb = dentry->d_sb;
	hidden_mnt = sbr_mnt(sb, bindex);
	if (!test_ro(sb, bindex, dentry->d_inode)) {
		touch_atime(hidden_mnt, hidden_dentry);
		dentry->d_inode->i_atime = hidden_dentry->d_inode->i_atime;
	}
	return hidden_dentry->d_inode->i_op->readlink
		(hidden_dentry, buf, bufsiz);
}

static int aufs_readlink(struct dentry *dentry, char __user * buf, int bufsiz)
{
	int err;

	LKTRTrace("%.*s, %d\n", DLNPair(dentry), bufsiz);

	aufs_read_lock(dentry, AUFS_I_RLOCK);
	err = hidden_readlink(dentry, dbstart(dentry), buf, bufsiz);
	//err = -1;
	aufs_read_unlock(dentry, AUFS_I_RLOCK);
	TraceErr(err);
	return err;
}

static void *aufs_follow_link(struct dentry *dentry, struct nameidata *nd)
{
	int err;
	char *buf;
	mm_segment_t old_fs;

	LKTRTrace("%.*s, nd %.*s\n", DLNPair(dentry), DLNPair(nd->dentry));

	err = -ENOMEM;
	buf = __getname();
	//buf = NULL;
	if (unlikely(!buf))
		goto out;

	aufs_read_lock(dentry, AUFS_I_RLOCK);
	old_fs = get_fs();
	set_fs(KERNEL_DS);
	err = hidden_readlink(dentry, dbstart(dentry), (char __user *)buf,
			      PATH_MAX);
	//err = -1;
	set_fs(old_fs);
	aufs_read_unlock(dentry, AUFS_I_RLOCK);

	if (err >= 0) {
		buf[err] = 0;
		/* will be freed by put_link */
		nd_set_link(nd, buf);
		return NULL; /* success */
	}
	__putname(buf);

 out:
	path_release(nd);
	TraceErr(err);
	return ERR_PTR(err);
}

static void aufs_put_link(struct dentry *dentry, struct nameidata *nd,
			  void *cookie)
{
	LKTRTrace("%.*s\n", DLNPair(dentry));
	__putname(nd_get_link(nd));
}

/* ---------------------------------------------------------------------- */
#if 0 // comment
struct inode_operations {
	int (*create) (struct inode *,struct dentry *,int, struct nameidata *);
	struct dentry * (*lookup) (struct inode *,struct dentry *, struct nameidata *);
	int (*link) (struct dentry *,struct inode *,struct dentry *);
	int (*unlink) (struct inode *,struct dentry *);
	int (*symlink) (struct inode *,struct dentry *,const char *);
	int (*mkdir) (struct inode *,struct dentry *,int);
	int (*rmdir) (struct inode *,struct dentry *);
	int (*mknod) (struct inode *,struct dentry *,int,dev_t);
	int (*rename) (struct inode *, struct dentry *,
			struct inode *, struct dentry *);
	int (*readlink) (struct dentry *, char __user *,int);
	void * (*follow_link) (struct dentry *, struct nameidata *);
	void (*put_link) (struct dentry *, struct nameidata *, void *);
	void (*truncate) (struct inode *);
	int (*permission) (struct inode *, int, struct nameidata *);
	int (*setattr) (struct dentry *, struct iattr *);
	int (*getattr) (struct vfsmount *mnt, struct dentry *, struct kstat *);
	int (*setxattr) (struct dentry *, const char *,const void *,size_t,int);
	ssize_t (*getxattr) (struct dentry *, const char *, void *, size_t);
	ssize_t (*listxattr) (struct dentry *, char *, size_t);
	int (*removexattr) (struct dentry *, const char *);
	void (*truncate_range)(struct inode *, loff_t, loff_t);
};
#endif

struct inode_operations aufs_symlink_iop = {
	.permission	= aufs_permission,
	.setattr	= aufs_setattr,

	.readlink	= aufs_readlink,
	.follow_link	= aufs_follow_link,
	.put_link	= aufs_put_link
};

struct inode_operations aufs_dir_iop = {
	.create		= aufs_create,
	.lookup		= aufs_lookup,
	.link		= aufs_link,
	.unlink		= aufs_unlink,
	.symlink	= aufs_symlink,
	.mkdir		= aufs_mkdir,
	.rmdir		= aufs_rmdir,
	.mknod		= aufs_mknod,
	.rename		= aufs_rename,

	.permission	= aufs_permission,
	.setattr	= aufs_setattr,

#if 0 // xattr
	.setxattr	= aufs_setxattr,
	.getxattr	= aufs_getxattr,
	.listxattr	= aufs_listxattr,
	.removexattr	= aufs_removexattr
#endif
};

struct inode_operations aufs_iop = {
	.permission	= aufs_permission,
	.setattr	= aufs_setattr,

#if 0 // xattr
	.setxattr	= aufs_setxattr,
	.getxattr	= aufs_getxattr,
	.listxattr	= aufs_listxattr,
	.removexattr	= aufs_removexattr
#endif
};
