/*
 * Copyright (C) 2005-2006 Junjiro Okajima
 *
 * This program, aufs is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/* $Id: dir.c,v 1.22 2006/11/27 04:09:13 sfjro Exp $ */

#include "aufs.h"

static int reopen_dir(struct file *file)
{
	int err;
	struct dentry *dentry, *hidden_dentry;
	aufs_bindex_t bindex, btail;
	struct file *hidden_file;

	dentry = file->f_dentry;
	LKTRTrace("%.*s\n", DLNPair(dentry));
	DEBUG_ON(!S_ISDIR(dentry->d_inode->i_mode));

	/* open all hidden dirs */
	bindex = dbstart(dentry);
	set_fbstart(file, bindex);
	btail = dbtaildir(dentry);
	set_fbend(file, btail);
	for (; bindex <= btail; bindex++) {
		hidden_dentry = dtohd_index(dentry, bindex);
		if (!hidden_dentry)
			continue;
		hidden_file = ftohf_index(file, bindex);
		if (hidden_file) {
			DEBUG_ON(hidden_file->f_dentry != hidden_dentry);
			continue;
		}

		hidden_file = hidden_open(dentry, bindex, file->f_flags);
		// unavailable
		//if (LktrCond) {fput(hidden_file);
		//br_put(stobr(dentry->d_sb, bindex));hidden_file=ERR_PTR(-1);}
		err = PTR_ERR(hidden_file);
		if (IS_ERR(hidden_file))
			goto out; // close all?
		//cpup_file_flags(hidden_file, file);
		set_ftohf_index(file, bindex, hidden_file);
	}
	err = 0;

 out:
	TraceErr(err);
	return err;
}

static int do_open_dir(struct file *file, int flags)
{
	int err;
	aufs_bindex_t bindex, btail;
	struct dentry *dentry, *hidden_dentry;
	struct file *hidden_file;

	dentry = file->f_dentry;
	LKTRTrace("%.*s, 0x%x\n", DLNPair(dentry), flags);
	DEBUG_ON(!dentry->d_inode || !S_ISDIR(dentry->d_inode->i_mode));

	err = 0;
	set_fvdir_cache(file, NULL);
	file->f_version = dentry->d_inode->i_version;
	bindex = dbstart(dentry);
	set_fbstart(file, bindex);
	btail = dbtaildir(dentry);
	set_fbend(file, btail);
	for (; !err && bindex <= btail; bindex++) {
		hidden_dentry = dtohd_index(dentry, bindex);
		if (!hidden_dentry)
			continue;

		hidden_file = hidden_open(dentry, bindex, flags);
		//if (LktrCond) {fput(hidden_file);
		//br_put(stobr(dentry->d_sb, bindex));hidden_file=ERR_PTR(-1);}
		if (!IS_ERR(hidden_file)) {
			set_ftohf_index(file, bindex, hidden_file);
			continue;
		}
		err = PTR_ERR(hidden_file);
	}
	if (!err)
		return 0; /* success */

	/* close all */
	for (bindex = fbstart(file); !err && bindex <= btail; bindex++)
		set_ftohf_index(file, bindex, NULL);
	set_fbstart(file, -1);
	set_fbend(file, -1);
	return err;
}

static int aufs_open_dir(struct inode *inode, struct file *file)
{
	return do_open(inode, file, do_open_dir);
}

static int aufs_release_dir(struct inode *inode, struct file *file)
{
	struct aufs_vdir *vdir_cache;
	struct super_block *sb;

	LKTRTrace("i%lu, %.*s\n", inode->i_ino, DLNPair(file->f_dentry));

	sb = file->f_dentry->d_sb;
	si_read_lock(sb);
	fi_write_lock(file);
	vdir_cache = fvdir_cache(file);
	if (vdir_cache)
		free_vdir(vdir_cache);
	fi_write_unlock(file);
	si_read_unlock(sb);
	fin_finfo(file);
	return 0;
}

static int fsync_dir(struct dentry *dentry, int datasync)
{
	int err;
	struct inode *inode;
	struct super_block *sb;
	aufs_bindex_t bend, bindex;

	LKTRTrace("%.*s, %d\n", DLNPair(dentry), datasync);
	DiMustAnyLock(dentry);
	sb = dentry->d_sb;
	SiMustAnyLock(sb);
	inode = dentry->d_inode;
	IMustLock(inode);
	IiMustAnyLock(inode);

	err = 0;
	bend = dbend(dentry);
	for (bindex = dbstart(dentry); !err && bindex <= bend; bindex++) {
		struct dentry *h_dentry;
		struct inode *h_inode;
		struct file_operations *fop;

		if (test_ro(sb, bindex, inode))
			continue;
		h_dentry = dtohd_index(dentry, bindex);
		if (!h_dentry)
			continue;
		h_inode = h_dentry->d_inode;
		if (!h_inode)
			continue;

		/* cf. fs/nsfd/vfs.c and fs/nfsd/nfs4recover.c */
		i_lock(h_inode);
		fop = (void*)h_inode->i_fop;
		err = filemap_fdatawrite(h_inode->i_mapping);
		if (!err && fop && fop->fsync)
			err = fop->fsync(NULL, h_dentry, datasync);
		if (!err)
			err = filemap_fdatawrite(h_inode->i_mapping);
		i_unlock(h_inode);
	}

	TraceErr(err);
	return err;
}

/*
 * @file may be NULL
 */
static int aufs_fsync_dir(struct file *file, struct dentry *dentry,
			  int datasync)
{
	int err;
	struct inode *inode;
	struct file *hidden_file;
	struct super_block *sb;
	aufs_bindex_t bend, bindex;

	LKTRTrace("%.*s, %d\n", DLNPair(dentry), datasync);
	inode = dentry->d_inode;
	IMustLock(inode);

	err = 0;
	sb = dentry->d_sb;
	si_read_lock(sb);
	if (file) {
		err = reval_and_lock_finfo(file, reopen_dir, /*wlock*/1,
					   /*locked*/1);
		//err = -1;
		if (unlikely(err))
			goto out;
	} else
		di_read_lock(dentry, 0);

	ii_write_lock(inode);
	if (file) {
		bend = fbend(file);
		for (bindex = fbstart(file); !err && bindex <= bend; bindex++) {
			hidden_file = ftohf_index(file, bindex);
			if (!hidden_file || test_ro(sb, bindex, inode))
				continue;

			err = -EINVAL;
			if (hidden_file->f_op && hidden_file->f_op->fsync) {
				i_lock(hidden_file->f_dentry->d_inode);
				err = hidden_file->f_op->fsync
					(hidden_file, hidden_file->f_dentry,
					 datasync);
				//err = -1;
				i_unlock(hidden_file->f_dentry->d_inode);
			}
		}
	} else
		err = fsync_dir(dentry, datasync);
	cpup_attr_timesizes(inode);
	ii_write_unlock(inode);
	if (file)
		fi_write_unlock(file);
	else
		di_read_unlock(dentry, 0);

 out:
	si_read_unlock(sb);
	TraceErr(err);
	return err;
}

/* ---------------------------------------------------------------------- */

static int aufs_readdir(struct file *file, void *dirent, filldir_t filldir)
{
	int err;
	struct dentry *dentry;
	struct inode *inode;
	struct super_block *sb;

	dentry = file->f_dentry;
	LKTRTrace("%.*s, pos %Ld\n", DLNPair(dentry), file->f_pos);
	inode = dentry->d_inode;
	IMustLock(inode);

	sb = dentry->d_sb;
	si_read_lock(sb);
	err = reval_and_lock_finfo(file, reopen_dir, /*wlock*/1, /*locked*/1);
	if (unlikely(err))
		goto out;

	ii_write_lock(inode);
	err = init_vdir(file);
	if (unlikely(err))
		goto out_unlock;
	//DbgVdir(fvdir_cache(file));// goto out_unlock;

	err = fill_de(file, dirent, filldir);
	//DbgVdir(fvdir_cache(file));// goto out_unlock;

 out_unlock:
	inode->i_atime = itohi(inode)->i_atime;
	ii_write_unlock(inode);
	fi_write_unlock(file);
 out:
	si_read_unlock(sb);
	TraceErr(err);
	return err;
}

/* ---------------------------------------------------------------------- */

struct test_empty_arg {
	struct aufs_nhash *whlist;
	int whonly;
	aufs_bindex_t bindex;
	int err, called;
};

static int test_empty_cb(void *__arg, const char *__name, int namelen,
			 loff_t offset, filldir_ino_t ino, unsigned int d_type)
{
	struct test_empty_arg *arg = __arg;
	char *name = (void*)__name;

	LKTRTrace("%.*s\n", namelen, name);

	arg->err = 0;
	arg->called++;
	//smp_mb();
	if (name[0] == '.'
	    && (namelen == 1 || (name[1] == '.' && namelen == 2)))
		return 0; /* success */

	if (namelen <= AUFS_WH_LEN || memcmp(name, AUFS_WH_PFX, AUFS_WH_LEN)) {
		if (arg->whonly && !test_known_wh(arg->whlist, name, namelen))
			arg->err = -ENOTEMPTY;
		goto out;
	}

	name += AUFS_WH_LEN;
	namelen -= AUFS_WH_LEN;
	if (!test_known_wh(arg->whlist, name, namelen))
		arg->err = append_wh(arg->whlist, name, namelen, arg->bindex);

 out:
	//smp_mb();
	TraceErr(arg->err);
	return arg->err;
}

static int do_test_empty(struct dentry *dentry, struct test_empty_arg *arg)
{
	int err;
	struct file *hidden_file;

	LKTRTrace("%.*s, {%p, %d, %d}\n",
		  DLNPair(dentry), arg->whlist, arg->whonly, arg->bindex);

	hidden_file = hidden_open(dentry, arg->bindex,
				  O_RDONLY | O_NONBLOCK | O_DIRECTORY);
	err = PTR_ERR(hidden_file);
	if (IS_ERR(hidden_file))
		goto out;

	//hidden_file->f_pos = 0;
	do {
		arg->err = 0;
		arg->called = 0;
		//smp_mb();
		err = vfs_readdir(hidden_file, test_empty_cb, arg);
		if (err >= 0)
			err = arg->err;
	} while (!err && arg->called);
	fput(hidden_file);
	sbr_put(dentry->d_sb, arg->bindex);

 out:
	TraceErr(err);
	return err;
}

static int sio_test_empty(struct dentry *dentry, struct test_empty_arg *arg)
{
	int err;
	struct dentry *hidden_dentry;
	struct inode *hidden_inode;

	LKTRTrace("%.*s\n", DLNPair(dentry));
	hidden_dentry = dtohd_index(dentry, arg->bindex);
	DEBUG_ON(!hidden_dentry);
	hidden_inode = hidden_dentry->d_inode;
	DEBUG_ON(!hidden_inode || !S_ISDIR(hidden_inode->i_mode));

	if (!test_perm(hidden_inode, MAY_EXEC | MAY_READ))
		err = do_test_empty(dentry, arg);
	else {
		void f(void *completion) {
			err = do_test_empty(dentry, arg);
			complete(completion);
		}
		wkq_wait(f);
	}

	TraceErr(err);
	return err;
}

int test_empty_lower(struct dentry *dentry)
{
	int err;
	struct inode *inode;
	struct test_empty_arg arg;
	struct aufs_nhash whlist;
	aufs_bindex_t bindex, bstart, btail;

	LKTRTrace("%.*s\n", DLNPair(dentry));
	inode = dentry->d_inode;
	DEBUG_ON(!inode || !S_ISDIR(inode->i_mode));

	bstart = dbstart(dentry);
	init_nhash(&whlist);
	arg.whlist = &whlist;
	arg.whonly = 0;
	arg.bindex = bstart;
	err = do_test_empty(dentry, &arg);
	if (unlikely(err))
		goto out;

	arg.whonly = 1;
	btail = dbtaildir(dentry);
	for (bindex = bstart + 1; !err && bindex <= btail; bindex++) {
		struct dentry *hidden_dentry;
		hidden_dentry = dtohd_index(dentry, bindex);
		if (hidden_dentry && hidden_dentry->d_inode) {
			DEBUG_ON(!S_ISDIR(hidden_dentry->d_inode->i_mode));
			arg.bindex = bindex;
			err = do_test_empty(dentry, &arg);
		}
	}

 out:
	free_nhash(&whlist);
	TraceErr(err);
	return err;
}

int test_empty(struct dentry *dentry, struct aufs_nhash *whlist)
{
	int err;
	struct inode *inode;
	struct test_empty_arg arg;
	aufs_bindex_t bindex, btail;

	LKTRTrace("%.*s\n", DLNPair(dentry));
	inode = dentry->d_inode;
	DEBUG_ON(!inode || !S_ISDIR(inode->i_mode));

	err = 0;
	arg.whlist = whlist;
	arg.whonly = 1;
	btail = dbtaildir(dentry);
	for (bindex = dbstart(dentry); !err && bindex <= btail; bindex++) {
		struct dentry *hidden_dentry;
		hidden_dentry = dtohd_index(dentry, bindex);
		if (hidden_dentry && hidden_dentry->d_inode) {
			DEBUG_ON(!S_ISDIR(hidden_dentry->d_inode->i_mode));
			arg.bindex = bindex;
			err = sio_test_empty(dentry, &arg);
		}
	}

	TraceErr(err);
	return err;
}

/* ---------------------------------------------------------------------- */

#if 0 // comment
struct file_operations {
	struct module *owner;
	loff_t (*llseek) (struct file *, loff_t, int);
	ssize_t (*read) (struct file *, char __user *, size_t, loff_t *);
	ssize_t (*aio_read) (struct kiocb *, char __user *, size_t, loff_t);
	ssize_t (*write) (struct file *, const char __user *, size_t, loff_t *);
	ssize_t (*aio_write) (struct kiocb *, const char __user *, size_t, loff_t);
	int (*readdir) (struct file *, void *, filldir_t);
	unsigned int (*poll) (struct file *, struct poll_table_struct *);
	int (*ioctl) (struct inode *, struct file *, unsigned int, unsigned long);
	long (*unlocked_ioctl) (struct file *, unsigned int, unsigned long);
	long (*compat_ioctl) (struct file *, unsigned int, unsigned long);
	int (*mmap) (struct file *, struct vm_area_struct *);
	int (*open) (struct inode *, struct file *);
	int (*flush) (struct file *);
	int (*release) (struct inode *, struct file *);
	int (*fsync) (struct file *, struct dentry *, int datasync);
	int (*aio_fsync) (struct kiocb *, int datasync);
	int (*fasync) (int, struct file *, int);
	int (*lock) (struct file *, int, struct file_lock *);
	ssize_t (*readv) (struct file *, const struct iovec *, unsigned long, loff_t *);
	ssize_t (*writev) (struct file *, const struct iovec *, unsigned long, loff_t *);
	ssize_t (*sendfile) (struct file *, loff_t *, size_t, read_actor_t, void *);
	ssize_t (*sendpage) (struct file *, struct page *, int, size_t, loff_t *, int);
	unsigned long (*get_unmapped_area)(struct file *, unsigned long, unsigned long, unsigned long, unsigned long);
	int (*check_flags)(int);
	int (*dir_notify)(struct file *file, unsigned long arg);
	int (*flock) (struct file *, int, struct file_lock *);
};
#endif

struct file_operations aufs_dir_fop = {
	.read		= generic_read_dir,
	.readdir	= aufs_readdir,
	.open		= aufs_open_dir,
	.release	= aufs_release_dir,
	.flush		= aufs_flush,
	.fsync		= aufs_fsync_dir,
};
