/*
 * Copyright (C) 2005-2006 Junjiro Okajima
 *
 * This program, aufs is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/* $Id: f_op.c,v 1.13 2006/11/20 03:28:53 sfjro Exp $ */

//#include <linux/fsnotify.h>
#include <linux/pagemap.h>
#include <linux/poll.h>
//#include <linux/security.h>
#include <linux/version.h>
#include "aufs.h"

/* common function to regular file and dir */
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,18)
#define FlushArgs	hidden_file, id
int aufs_flush(struct file *file, fl_owner_t id)
#else
#define FlushArgs	hidden_file
int aufs_flush(struct file *file)
#endif
{
	int err;
	struct dentry *dentry;
	aufs_bindex_t bindex, bend;

	dentry = file->f_dentry;
	LKTRTrace("%.*s\n", DLNPair(dentry));

	aufs_read_lock(dentry, !AUFS_I_RLOCK);
	fi_read_lock(file);

	err = 0;
	bend = fbend(file);
	for (bindex = fbstart(file); !err && bindex <= bend; bindex++) {
		struct file *hidden_file;
		hidden_file = ftohf_index(file, bindex);
		if (hidden_file && hidden_file->f_op
		    && hidden_file->f_op->flush)
			err = hidden_file->f_op->flush(FlushArgs);
	}

	fi_read_unlock(file);
	aufs_read_unlock(dentry, !AUFS_I_RLOCK);
	TraceErr(err);
	return err;
}
#undef FlushArgs

/* ---------------------------------------------------------------------- */

static int do_open_nondir(struct file *file, int flags)
{
	int err;
	aufs_bindex_t bindex;
	struct super_block *sb;
	struct file *hidden_file;
	struct dentry *dentry;
	struct inode *inode;
	struct aufs_finfo *finfo;

	dentry = file->f_dentry;
	LKTRTrace("%.*s, flags 0%o\n", DLNPair(dentry), flags);
	FiMustWriteLock(file);
	inode = dentry->d_inode;
	DEBUG_ON(!inode || S_ISDIR(inode->i_mode));

	err = 0;
	finfo = ftopd(file);
	finfo->fi_hidden_vm_ops = NULL;
	atomic_set(&finfo->fi_mapcnt, 0);
	sb = dentry->d_sb;
	bindex = dbstart(dentry);
	DEBUG_ON(!dtohd(dentry)->d_inode);
	/* O_TRUNC is processed already */
	BUG_ON(test_ro(sb, bindex, inode) && (flags & O_TRUNC));

	hidden_file = hidden_open(dentry, bindex, flags);
	//if (LktrCond) {fput(hidden_file); br_put(stobr(dentry->d_sb, bindex));
	//hidden_file = ERR_PTR(-1);}
	if (!IS_ERR(hidden_file)) {
		set_fbstart(file, bindex);
		set_fbend(file, bindex);
		set_ftohf_index(file, bindex, hidden_file);
		return 0; /* success */
	}
	err = PTR_ERR(hidden_file);
	TraceErr(err);
	return err;
}

static int aufs_open_nondir(struct inode *inode, struct file *file)
{
	return do_open(inode, file, do_open_nondir);
}

static int aufs_release_nondir(struct inode *inode, struct file *file)
{
	LKTRTrace("i%lu, %.*s\n", inode->i_ino, DLNPair(file->f_dentry));
	DEBUG_ON(atomic_read(&ftopd(file)->fi_mapcnt));
	fin_finfo(file);
	return 0;
}

/* ---------------------------------------------------------------------- */

static ssize_t aufs_read(struct file *file, char __user *buf, size_t count,
			 loff_t *ppos)
{
	ssize_t err;
	struct dentry *dentry;
	struct file *hidden_file;
	struct super_block *sb;
	readf_t func;

	dentry = file->f_dentry;
	LKTRTrace("%.*s, cnt %d, pos %Ld\n", DLNPair(dentry), count, *ppos);

	sb = dentry->d_sb;
	si_read_lock(sb);
	err = reval_and_lock_finfo(file, reopen_nondir, /*wlock*/0,
				   /*locked*/0);
	//if (LktrCond) {fi_read_unlock(file); err = -1;}
	if (unlikely(err))
		goto out;

	hidden_file = ftohf(file);
	func = find_readf(hidden_file);
	//if (LktrCond) func = ERR_PTR(-1);
	err = PTR_ERR(func);
	if (!IS_ERR(func)) {
		err = func(hidden_file, buf, count, ppos);
		//err = -1;
		memcpy(&file->f_ra, &hidden_file->f_ra, sizeof(file->f_ra)); //??
		dentry->d_inode->i_atime
			= hidden_file->f_dentry->d_inode->i_atime;
	}
	fi_read_unlock(file);

 out:
	si_read_unlock(sb);
	TraceErr(err);
	return err;
}

static ssize_t aufs_write(struct file *file, const char __user *__buf,
			  size_t count, loff_t *ppos)
{
	ssize_t err;
	struct dentry *dentry;
	struct inode *inode;
	struct super_block *sb;
	struct file *hidden_file;
	writef_t func;
	char __user *buf = (char __user*)__buf;

	dentry = file->f_dentry;
	LKTRTrace("%.*s, cnt %d, pos %Ld\n", DLNPair(dentry), count, *ppos);

	inode = dentry->d_inode;
	i_lock(inode);
	sb = dentry->d_sb;
	si_read_lock(sb);
	err = reval_and_lock_finfo(file, reopen_nondir, /*wlock*/1,
				   /*locked*/1);
	//if (LktrCond) {fi_write_unlock(file); err = -1;}
	if (unlikely(err))
		goto out;
	err = ready_to_write(file, -1);
	//if (LktrCond) err = -1;
	if (unlikely(err))
		goto out_unlock;

	hidden_file = ftohf(file);
	func = find_writef(hidden_file);
	err = PTR_ERR(func);
	if (!IS_ERR(func)) {
		if (file->f_flags & O_APPEND)
			*ppos = i_size_read(inode);
		err = func(hidden_file, buf, count, ppos);
		//err = -1;
		ii_write_lock(inode);
		cpup_attr_timesizes(inode);
		ii_write_unlock(inode);
	}

 out_unlock:
	fi_write_unlock(file);
 out:
	si_read_unlock(sb);
	i_unlock(inode);
	TraceErr(err);
	return err;
}

/* ---------------------------------------------------------------------- */

static void aufs_vmaopen(struct vm_area_struct *vma)
{
	LKTRTrace("%.*s, %lx\n",
		  DLNPair(vma->vm_file->f_dentry), vma->vm_start);
	atomic_inc(&ftopd(vma->vm_file)->fi_mapcnt);
}

static void aufs_vmaclose(struct vm_area_struct *vma)
{
	LKTRTrace("%.*s, %lx\n",
		  DLNPair(vma->vm_file->f_dentry), vma->vm_start);
	atomic_dec(&ftopd(vma->vm_file)->fi_mapcnt);
}

static struct file *safe_file(struct vm_area_struct *vma)
{
	struct file *file;

	file = vma->vm_file;
	if (file->private_data && SB_AUFS(file->f_dentry->d_sb))
		return file;
	return NULL;
}

static struct page *aufs_nopage(struct vm_area_struct *vma, unsigned long addr,
				int *type)
{
	struct page *page;
	struct dentry *dentry;
	struct file *file, *hidden_file;
	struct inode *inode;
	static DECLARE_WAIT_QUEUE_HEAD(wq);
	struct aufs_finfo *finfo;

	TraceEnter();
	DEBUG_ON(!vma || !vma->vm_file);
	wait_event(wq, (file = safe_file(vma)));
	DEBUG_ON(!SB_AUFS(file->f_dentry->d_sb));
	dentry = file->f_dentry;
	LKTRTrace("%.*s, addr %lx\n", DLNPair(dentry), addr);
	inode = dentry->d_inode;
	DEBUG_ON(!S_ISREG(inode->i_mode));

	// do not revalidate, nor lock
	finfo = ftopd(file);
	hidden_file = finfo->fi_hfile[0 + finfo->fi_bstart].hf_file;
	DEBUG_ON(!hidden_file || !is_mmapped(file));
	vma->vm_file = hidden_file;
	//smp_mb();
	page = ftopd(file)->fi_hidden_vm_ops->nopage(vma, addr, type);
	vma->vm_file = file;
	smp_mb();
#if 0 //def CONFIG_SMP
	//wake_up_nr(&wq, online_cpu - 1);
	wake_up_all(&wq);
#else
	wake_up(&wq);
#endif
	if (!IS_ERR(page)) {
		//page->mapping = file->f_mapping;
		//get_page(page);
		//file->f_mapping = hidden_file->f_mapping;
		//touch_atime(NULL, dentry);
		//inode->i_atime = hidden_file->f_dentry->d_inode->i_atime;
	}
	TraceErrPtr(page);
	return page;
}

static int aufs_populate(struct vm_area_struct *vma, unsigned long addr,
			 unsigned long len, pgprot_t prot, unsigned long pgoff,
			 int nonblock)
{
	Err("please report me this application\n");
	BUG();
	return ftopd(vma->vm_file)->fi_hidden_vm_ops->populate
		(vma, addr, len, prot, pgoff, nonblock);
}

static struct vm_operations_struct aufs_vm_ops = {
	.open		= aufs_vmaopen,
	.close		= aufs_vmaclose,
	.nopage		= aufs_nopage,
	.populate	= aufs_populate,
	//page_mkwrite(struct vm_area_struct *vma, struct page *page)
};

/* ---------------------------------------------------------------------- */

static int aufs_mmap(struct file *file, struct vm_area_struct *vma)
{
	int err, wlock, mmapped;
	struct dentry *dentry;
	struct super_block *sb;
	struct file *hidden_file;
	struct vm_operations_struct *vm_ops;

	dentry = file->f_dentry;
	LKTRTrace("%.*s, len %lu\n",
		  DLNPair(dentry), vma->vm_end - vma->vm_start);
	DEBUG_ON(!S_ISREG(dentry->d_inode->i_mode));

	mmapped = is_mmapped(file);
	wlock = 0;
	if (file->f_mode & FMODE_WRITE) {
		wlock = VM_SHARED | VM_WRITE;
		wlock = ((wlock & vma->vm_flags) == wlock);
	}

	sb = dentry->d_sb;
	si_read_lock(sb);
	err = reval_and_lock_finfo(file, reopen_nondir,
				   wlock | !mmapped, /*locked*/0);
	//err = -1;
	if (unlikely(err))
		goto out;

	if (wlock) {
		err = ready_to_write(file, -1);
		//err = -1;
		if (unlikely(err))
			goto out_unlock;
	}

	hidden_file = ftohf(file);
	vm_ops = ftopd(file)->fi_hidden_vm_ops;
	if (!mmapped) {
		err = hidden_file->f_op->mmap(hidden_file, vma);
		if (unlikely(err))
			goto out_unlock;
		vm_ops = vma->vm_ops;
		DEBUG_ON(!vm_ops);
		err = do_munmap(current->mm, vma->vm_start,
				vma->vm_end - vma->vm_start);
		if (unlikely(err)) {
			IOErr("failed internal unmapping %.*s, %d\n",
			      DLNPair(hidden_file->f_dentry), err);
			err = -EIO;
			goto out_unlock;
		}
	}
	DEBUG_ON(!vm_ops);

	err = generic_file_mmap(file, vma);
	if (!err) {
		file_accessed(hidden_file);
		dentry->d_inode->i_atime
			= hidden_file->f_dentry->d_inode->i_atime;
		ftopd(file)->fi_hidden_vm_ops = vm_ops;
		vma->vm_ops = &aufs_vm_ops;
		atomic_inc(&ftopd(file)->fi_mapcnt);
	}

 out_unlock:
	if (!wlock && mmapped)
		fi_read_unlock(file);
	else
		fi_write_unlock(file);
 out:
	si_read_unlock(sb);
	TraceErr(err);
	return err;
}

static ssize_t aufs_sendfile(struct file *file, loff_t *ppos,
			     size_t count, read_actor_t actor, void *target)
{
	ssize_t err;
	struct file *hidden_file;
	const char c = current->comm[4];
	/* true if a kernel thread named 'loop[0-9].*' accesses a file */
	const int loopback = (current->mm == NULL
			      && '0' <= c && c <= '9'
			      && strncmp(current->comm, "loop", 4) == 0);
	struct dentry *dentry;
	struct super_block *sb;

	dentry = file->f_dentry;
	LKTRTrace("%.*s, pos %Ld, cnt %d, loopback %d\n",
		  DLNPair(dentry), *ppos, count, loopback);

	sb = dentry->d_sb;
	si_read_lock(sb);
	err = reval_and_lock_finfo(file, reopen_nondir, /*wlock*/0,
				   /*locked*/0);
	if (unlikely(err))
		goto out;

	err = -EINVAL;
	hidden_file = ftohf(file);
	if (hidden_file->f_op && hidden_file->f_op->sendfile) {
		if (/* unlikely */(loopback)) {
			file->f_mapping = hidden_file->f_mapping;
			smp_mb();
		}
		err = hidden_file->f_op->sendfile
			(hidden_file, ppos, count, actor, target);
		dentry->d_inode->i_atime
			= hidden_file->f_dentry->d_inode->i_atime;
	}
	fi_read_unlock(file);

 out:
	si_read_unlock(sb);
	TraceErr(err);
	return err;
}

/* ---------------------------------------------------------------------- */

/* copied from linux/fs/select.h, must match */
#define DEFAULT_POLLMASK (POLLIN | POLLOUT | POLLRDNORM | POLLWRNORM)

static unsigned int aufs_poll(struct file *file, poll_table *wait)
{
	unsigned int mask;
	struct file *hidden_file;
	int err;
	struct dentry *dentry;
	struct super_block *sb;

	dentry = file->f_dentry;
	LKTRTrace("%.*s, wait %p\n", DLNPair(dentry), wait);
	DEBUG_ON(S_ISDIR(dentry->d_inode->i_mode));

	/* We should pretend an error happend. */
	mask = POLLERR /* | POLLIN | POLLOUT */;
	sb = dentry->d_sb;
	si_read_lock(sb);
	err = reval_and_lock_finfo(file, reopen_nondir, /*wlock*/0,
				   /*locked*/0);
	//err = -1;
	if (unlikely(err))
		goto out;

	/* it is not an error of hidden_file has no operation */
	mask = DEFAULT_POLLMASK;
	hidden_file = ftohf(file);
	if (hidden_file->f_op && hidden_file->f_op->poll)
		mask = hidden_file->f_op->poll(hidden_file, wait);
	fi_read_unlock(file);

 out:
	si_read_unlock(sb);
	TraceErr(mask);
	return mask;
}

static int aufs_fsync_nondir(struct file *file, struct dentry *dentry,
			     int datasync)
{
	int err;
	struct inode *inode;
	struct file *hidden_file;
	struct super_block *sb;

	LKTRTrace("%.*s, %d\n", DLNPair(dentry), datasync);
	inode = dentry->d_inode;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,17)
	/* before 2.6.17,
	 * msync(2) calls me without locking i_sem/i_mutex, but fsync(2).
	 */
	IMustLock(inode);
#endif

	sb = dentry->d_sb;
	si_read_lock(sb);
	err = reval_and_lock_finfo(file, reopen_nondir, /*wlock*/1,
				   /*locked*/1);
	//err = -1;
	if (unlikely(err))
		goto out;
	err = ready_to_write(file, -1);
	//err = -1;
	if (unlikely(err))
		goto out_unlock;

	err = -EINVAL;
	hidden_file = ftohf(file);
	if (hidden_file->f_op && hidden_file->f_op->fsync) {
		//file->f_mapping->host->i_mutex
		ii_write_lock(inode);
		i_lock(hidden_file->f_dentry->d_inode);
		err = hidden_file->f_op->fsync
			(hidden_file, hidden_file->f_dentry, datasync);
		//err = -1;
		cpup_attr_timesizes(inode);
		i_unlock(hidden_file->f_dentry->d_inode);
		ii_write_unlock(inode);
	}

 out_unlock:
	fi_write_unlock(file);
 out:
	si_read_unlock(sb);
	TraceErr(err);
	return err;
}

static int aufs_fasync(int fd, struct file *file, int flag)
{
	int err;
	struct file *hidden_file;
	struct dentry *dentry;
	struct super_block *sb;

	dentry = file->f_dentry;
	LKTRTrace("%.*s, %d\n", DLNPair(dentry), flag);

	sb = dentry->d_sb;
	si_read_lock(sb);
	err = reval_and_lock_finfo(file, reopen_nondir, /*wlock*/0,
				   /*locked*/0);
	//err = -1;
	if (unlikely(err))
		goto out;

	hidden_file = ftohf(file);
	if (hidden_file->f_op && hidden_file->f_op->fasync)
		err = hidden_file->f_op->fasync(fd, hidden_file, flag);
	fi_read_unlock(file);

 out:
	si_read_unlock(sb);
	TraceErr(err);
	return err;
}

/* ---------------------------------------------------------------------- */

#if 0 // comment
struct file_operations {
	struct module *owner;
	loff_t (*llseek) (struct file *, loff_t, int);
	ssize_t (*read) (struct file *, char __user *, size_t, loff_t *);
	ssize_t (*aio_read) (struct kiocb *, char __user *, size_t, loff_t);
	ssize_t (*write) (struct file *, const char __user *, size_t, loff_t *);
	ssize_t (*aio_write) (struct kiocb *, const char __user *, size_t, loff_t);
	int (*readdir) (struct file *, void *, filldir_t);
	unsigned int (*poll) (struct file *, struct poll_table_struct *);
	int (*ioctl) (struct inode *, struct file *, unsigned int, unsigned long);
	long (*unlocked_ioctl) (struct file *, unsigned int, unsigned long);
	long (*compat_ioctl) (struct file *, unsigned int, unsigned long);
	int (*mmap) (struct file *, struct vm_area_struct *);
	int (*open) (struct inode *, struct file *);
	int (*flush) (struct file *);
	int (*release) (struct inode *, struct file *);
	int (*fsync) (struct file *, struct dentry *, int datasync);
	int (*aio_fsync) (struct kiocb *, int datasync);
	int (*fasync) (int, struct file *, int);
	int (*lock) (struct file *, int, struct file_lock *);
	ssize_t (*readv) (struct file *, const struct iovec *, unsigned long, loff_t *);
	ssize_t (*writev) (struct file *, const struct iovec *, unsigned long, loff_t *);
	ssize_t (*sendfile) (struct file *, loff_t *, size_t, read_actor_t, void *);
	ssize_t (*sendpage) (struct file *, struct page *, int, size_t, loff_t *, int);
	unsigned long (*get_unmapped_area)(struct file *, unsigned long, unsigned long, unsigned long, unsigned long);
	int (*check_flags)(int);
	int (*dir_notify)(struct file *file, unsigned long arg);
	int (*flock) (struct file *, int, struct file_lock *);
};
#endif

struct file_operations aufs_file_fop = {
	.read		= aufs_read,
	.write		= aufs_write,
	.poll		= aufs_poll,
	.mmap		= aufs_mmap,
	.open		= aufs_open_nondir,
	.flush		= aufs_flush,
	.release	= aufs_release_nondir,
	.fsync		= aufs_fsync_nondir,
	.fasync		= aufs_fasync,
	.sendfile	= aufs_sendfile,
};
