/*
 * Copyright (C) 2005-2006 Junjiro Okajima
 *
 * This program, aufs is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/* $Id: finfo.c,v 1.17 2006/10/16 09:42:57 sfjro Exp $ */

#include "aufs.h"

struct aufs_finfo *ftopd(struct file *file)
{
	struct aufs_finfo *finfo = file->private_data;
	DEBUG_ON(!finfo
		 || !finfo->fi_hfile
		 || (0 < finfo->fi_bend
		     && (/* stopd(file->f_dentry->d_sb)->si_bend < finfo->fi_bend
			  * || */ finfo->fi_bend < finfo->fi_bstart)));
	return finfo;
}

// hard/soft set
aufs_bindex_t fbstart(struct file *file)
{
	FiMustAnyLock(file);
	return ftopd(file)->fi_bstart;
}

aufs_bindex_t fbend(struct file *file)
{
	FiMustAnyLock(file);
	return ftopd(file)->fi_bend;
}

struct aufs_vdir *fvdir_cache(struct file *file)
{
	FiMustAnyLock(file);
	return ftopd(file)->fi_vdir_cache;
}

struct aufs_branch *ftobr_index(struct file *file, aufs_bindex_t bindex)
{
	struct aufs_finfo *finfo = ftopd(file);
	struct aufs_hfile *hf;

	FiMustAnyLock(file);
	DEBUG_ON(!finfo
		 || finfo->fi_bstart < 0
		 || bindex < finfo->fi_bstart
		 || finfo->fi_bend < bindex);
	hf = finfo->fi_hfile + bindex;
	DEBUG_ON(hf->hf_br && br_count(hf->hf_br) <= 0);
	return hf->hf_br;
}

struct file *ftohf_index(struct file *file, aufs_bindex_t bindex)
{
	struct aufs_finfo *finfo = ftopd(file);
	struct aufs_hfile *hf;

	FiMustAnyLock(file);
	DEBUG_ON(!finfo
		 || finfo->fi_bstart < 0
		 || bindex < finfo->fi_bstart
		 || finfo->fi_bend < bindex);
	hf = finfo->fi_hfile + bindex;
	DEBUG_ON(hf->hf_file
		 && file_count(hf->hf_file) <= 0
		 && br_count(hf->hf_br) <= 0);
	return hf->hf_file;
}

struct file *ftohf(struct file *file)
{
	return ftohf_index(file, fbstart(file));
}

void set_fbstart(struct file *file, aufs_bindex_t bindex)
{
	FiMustWriteLock(file);
	DEBUG_ON(sbend(file->f_dentry->d_sb) < bindex);
	ftopd(file)->fi_bstart = bindex;
	//smp_mb();
}

void set_fbend(struct file *file, aufs_bindex_t bindex)
{
	FiMustWriteLock(file);
	DEBUG_ON(sbend(file->f_dentry->d_sb) < bindex
		 || bindex < fbstart(file));
	ftopd(file)->fi_bend = bindex;
	//smp_mb();
}

void set_fvdir_cache(struct file *file, struct aufs_vdir *vdir_cache)
{
	FiMustWriteLock(file);
	DEBUG_ON(!S_ISDIR(file->f_dentry->d_inode->i_mode)
		 || (ftopd(file)->fi_vdir_cache && vdir_cache));
	ftopd(file)->fi_vdir_cache = vdir_cache;
	//smp_mb();
}

void set_ftohf_index(struct file *file, aufs_bindex_t bindex, struct file *val)
{
	struct aufs_finfo *finfo = ftopd(file);
	struct aufs_hfile *hf;

	FiMustWriteLock(file);
	DEBUG_ON(!finfo
		 || finfo->fi_bstart < 0
		 || bindex < finfo->fi_bstart
		 || finfo->fi_bend < bindex);
	DEBUG_ON(val && file_count(val) <= 0);
	hf = finfo->fi_hfile + bindex;
	DEBUG_ON(val && hf->hf_file);
	if (hf->hf_file) {
		fput(hf->hf_file);
		hf->hf_file = NULL;
		DEBUG_ON(!hf->hf_br);
		br_put(hf->hf_br);
		hf->hf_br = NULL;
	}
	if (val) {
		hf->hf_file = val;
		hf->hf_br = stobr(file->f_dentry->d_sb, bindex);
	}
	//smp_mb();
}

void update_figen(struct file *file)
{
	atomic_set(&ftopd(file)->fi_generation, digen(file->f_dentry));
}

void fin_finfo(struct file *file)
{
	struct aufs_finfo *finfo;
	struct dentry *dentry;
	aufs_bindex_t bindex, bend;

	dentry = file->f_dentry;
	LKTRTrace("%.*s\n", DLNPair(dentry));

	si_read_lock(dentry->d_sb); // unnecessary
	fi_write_lock(file);
	bend = fbend(file);
	bindex = fbstart(file);
	if (bindex >= 0)
		for (; bindex <= bend; bindex++)
			set_ftohf_index(file, bindex, NULL);

	finfo = ftopd(file);
#ifdef CONFIG_AUFS_DEBUG
	if (finfo->fi_bstart >= 0) {
		bend = fbend(file);
		for (bindex = finfo->fi_bstart; bindex <= bend; bindex++) {
			struct aufs_hfile *hf;
			hf = finfo->fi_hfile + bindex;
			DEBUG_ON(hf->hf_file || hf->hf_br);
		}
	}
#endif

	kfree(finfo->fi_hfile);
	fi_write_unlock(file);
	rw_destroy(&finfo->fi_rwsem);
	cache_free_finfo(finfo);
	//file->private_data = NULL;
	si_read_unlock(dentry->d_sb);
}

int init_finfo(struct file *file)
{
	struct aufs_finfo *finfo;
	struct dentry *dentry;

	dentry = file->f_dentry;
	LKTRTrace("%.*s\n", DLNPair(dentry));
	DEBUG_ON(!dentry->d_inode);

	finfo = cache_alloc_finfo();
	if (finfo) {
		finfo->fi_hfile = kcalloc(sbend(dentry->d_sb) + 1,
					  sizeof(*finfo->fi_hfile), GFP_KERNEL);
		if (finfo->fi_hfile) {
			rw_init_wlock(&finfo->fi_rwsem);
			finfo->fi_bstart = -1;
			finfo->fi_bend = -1;
			atomic_set(&finfo->fi_generation, digen(dentry));

			file->private_data = finfo;
			return 0; /* success */
		}
		cache_free_finfo(finfo);
	}

	TraceErr(-ENOMEM);
	return -ENOMEM;
}
